/*
 * Copyright 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper DensoVS60 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60

#include <phdlStepper.h>
#include "phdlStepper_DensoVS60_External.h"
#include "phdlStepper_DensoVS60_Cmd.h"
#include "phdlStepper_DensoVS60.h"

phStatus_t phdlStepper_DensoVS60_Cmd_GoDownUntilContact(
                                                phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                int32_t dwMinZPoint,
                                                uint16_t wRobotSpeed
                                                )
{
    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Call the command function */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GoDownUntilContact(pDataParams,dwMinZPoint,wRobotSpeed));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_SplineExecuteMove(
                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                    uint8_t bPathIndex,
                                                    uint8_t bBlocking
                                                    )
{
    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Call the command function */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineExecuteMove(pDataParams,bPathIndex,bBlocking));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_SplineClearPath(
                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                    uint8_t bPathIndex
                                                    )
{

    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Call the command function */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineClearPath(pDataParams,bPathIndex));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_SplineAddPathPoint(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         uint8_t  bPathIndex,
                                                         uint32_t dwPositionX,
                                                         uint32_t dwPositionY,
                                                         uint32_t dwPositionZ
                                                        )

{
    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Call the command function */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineAddPathPoint(pDataParams,bPathIndex,dwPositionX,dwPositionY,dwPositionZ));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_SplineExecuteMoveInverse(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         uint8_t bPathIndex,
                                                         uint8_t bBlocking
                                                           )
{

    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Clear the path 20 which is used to store the inverse path */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineClearPath(pDataParams,20));
    /* Load the inverse points from the selected path to the path 20 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineLoadInversePath(pDataParams,bPathIndex));
    /* Execute move for the path 20 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineExecuteMove(pDataParams,20, bBlocking));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Cmd_WaitUntilMoveFinished(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                          )
{
    phStatus_t statusTmp;
    uint8_t bMotionCompleted;

    do
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckMotionComplete(pDataParams, &bMotionCompleted));
    }while( bMotionCompleted != PH_ON );

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Cmd_SetPort(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         uint8_t* pPortName
                                                          )
{
    if (strlen((char*)pPortName) >= DENSO_VS60_MAX_IP_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_STEPPER);
    }

    /* Assign the IP adresse */
    strcpy((char *)pDataParams->pPortName, (char*)pPortName);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_SetToolDef(
                                                phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                float64_t x,
                                                float64_t y,
                                                float64_t z,
                                                float64_t rx,
                                                float64_t ry,
                                                float64_t rz,
                                                uint16_t wtoolNumber
                                                )
{
    phStatus_t status;
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetToolDef(pDataParams, x, y, z, rx, ry, rz));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetTool(pDataParams, wtoolNumber));

    /* Set current position internally */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_SetConfig(pDataParams, PHDL_STEPPER_DENSO_VS60_CONFIG_POSITION_MODE, PHDL_STEPPER_DENSO_VS60_POS_MODE_SET_CLEAR_ZERO_OFFSET));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_PickupCard(
                                                phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                )
{
    phStatus_t statusTmp;
    uint16_t wOldRobotSpeed;
    phdlStepper_DensoVS60_DataParams_XY_coord_t currentCoordsXY;

    wOldRobotSpeed = pDataParams->wRobotSpeed;

    /* remember position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(pDataParams,&currentCoordsXY));

    /* move robot down until contact */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GoDownUntilContact(pDataParams, 0, 10));

    /* activate pump */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_PUMP, PH_ON));

    /* wait for vacuum */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(3000));

    /* go back to previous position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_SetConfig(pDataParams, PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_AXIS, PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_GoToPosition(pDataParams, wOldRobotSpeed, currentCoordsXY.dwPositionZ, 1));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Cmd_DropCard(
                                              phdlStepper_DensoVS60_DataParams_t * pDataParams
                                              )
{
    phStatus_t statusTmp;
    uint16_t wOldRobotSpeed;
    phdlStepper_DensoVS60_DataParams_XY_coord_t currentCoordsXY;

    wOldRobotSpeed = pDataParams->wRobotSpeed;

    /* remember position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(pDataParams,&currentCoordsXY));

    /* move robot down until contact */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GoDownUntilContact(pDataParams, 0, 10));

    /* deactivate pump */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_PUMP, PH_OFF));

    /* wait for vacuum */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(2000));

    /* go back to previous position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_SetConfig(pDataParams, PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_AXIS, PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_GoToPosition(pDataParams, wOldRobotSpeed, currentCoordsXY.dwPositionZ, 1));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */
