/*
 * Copyright 2017 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#define _USE_MATH_DEFINES

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>
#include <math.h>


#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60

#include "phdlStepper_DensoVS60.h"
#include "phdlStepper_DensoVS60_Int.h"
#include "phdlStepper_DensoVS60_External.h"
#include "../phdlStepper_Int.h"

#ifndef _WIN32
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#pragma warning( disable : 4996 )
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>              /* PRQA S 5124 */
#include <cmath>                /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */
#endif

#pragma warning(push)
#pragma warning( disable : 4996 )

#include "external\CAO.h"

/* Added support of the below function for Visual studio's 2015+.
 * Currently if the below function is not available its throwing build erros
 * with VS2015.
 */
#if _MSC_VER >= 1900
	/* Disabling it for versions lower than 2015. */
	FILE _iob[] = { *stdin, *stdout, *stderr };
	extern "C" FILE * __cdecl __iob_func ( void )
	{
		return _iob;
	}
#endif

#ifdef _DEBUG
#include <time.h>
#define DTTMFMT "%Y-%m-%d %H:%M:%S "
#define DTTMSZ 21


static char *getDtTm (char *buff) {
    time_t t = time (0);
    strftime (buff, DTTMSZ, DTTMFMT, localtime (&t));
    return buff;
}
#endif

phStatus_t phdlStepper_DensoVS60_External_EndProc(
                                                  phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                  HRESULT argHr
                                                 )
{
    /* Initialize status variables */
    phStatus_t status;

 #ifdef _DEBUG
    /* Only use the logger when in debug mode */
    phdlStepper_DensoVS60_External_LogErrorCodes(argHr);
 #endif

    /* Initialize function variables */
    ICaoEngine*     pEng  = (ICaoEngine*)     pDataParams->pEngine;
    ICaoWorkspaces* pWss  = (ICaoWorkspaces*) pDataParams->pWorkspaces;
    ICaoWorkspace*  pWs   = (ICaoWorkspace*)  pDataParams->pWorkspace;
    ICaoController* pCtrl = (ICaoController*) pDataParams->pContoller;

    /* TODO - Before relasing you should turn off engine an release arm contol if it is taken*/
    if(pDataParams->bArmTaken == 1)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_StopMotor(pDataParams));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_GiveArmContol(pDataParams));
    }

    /* Give the control of the arm to the controller */
    pDataParams->bArmTaken = 0;

    if (pCtrl)
    {
        pCtrl->Release();
    }
    if (pWs)
    {
        pWs->Release();
    }
    if (pWss)
    {
        pWss->Release();
    }
    if (pEng)
    {
        pEng->Release();
    }

    if (argHr != S_OK)
    {
        return PH_ADD_COMPCODE(PH_STEPPER_DENSO_VS60_ERR_CUSTOMERROR, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_External_CompleteInternalInitialization(
                                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                                        )
{
    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Create engine, connection, worspace and initialize controller */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CreateController(pDataParams));

    /* Initialize robot arm */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CreateRobotArm(pDataParams));

    /* Take the control over arm */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_TakeArmControl(pDataParams));

    /* Attach default IO units to the robot */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_AttachDefaultIOVariables(pDataParams));

    /* Attach optional IO units to the robot, such as laser, pump and etc. */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_AttachOptionalIOVariables(pDataParams));

    /* Initialize standard coordinates */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SetTool(pDataParams, 0));

    /* Initialize offset of X-Y axis to current position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams, 0));

    /* Initialize offset of Joint axis to current position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams, 1));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_CreateController(
                                                           phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                          )
{
    /*IP Addresse of the robot arm*/
    char cServerConnectionStr[DENSO_VS60_MAX_IP_LENGTH + 7]; /* 7 signs for "Server=" */
    wchar_t wcServerConnectionStr[DENSO_VS60_MAX_IP_LENGTH + 7];
    size_t size;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* Initialize function variables */
    ICaoEngine*     pEng  = NULL;
    ICaoWorkspaces* pWss  = NULL;
    ICaoWorkspace*  pWs   = NULL;
    ICaoController* pCtrl = NULL;

    VARIANT zeroValue;
    long value = 0;

    /* Declare string variables for the connection */
    BSTR cmdRc8;
    BSTR cmdDenso;
    BSTR cmdEmptyString;
    BSTR cmdServer;

    /* Initialize string values for connection */
    cmdRc8         = SysAllocString( L"RC8" );
    cmdDenso       = SysAllocString( L"CaoProv.DENSO.RC8" );
    cmdEmptyString = SysAllocString( L"" );

    strcpy(cServerConnectionStr, "Server=");
    strcat(cServerConnectionStr, (char *)pDataParams->pPortName);
    size = strlen(cServerConnectionStr) + 1;
    mbstowcs(wcServerConnectionStr, cServerConnectionStr, size);

    /* Default ip adresse */
    //cmdServer = SysAllocString( L"Server=192.168.106.208" );
    cmdServer = SysAllocString( wcServerConnectionStr );

    /* Create CaoEngine Instance
       Assign void pointer of the DataParams as a pointer to the ICaoEngine object */
    pEng = (ICaoEngine*) pDataParams->pEngine;

    /* Initialze application context to be accessed from multiple threads */
    CoInitializeEx(NULL, COINIT_MULTITHREADED);

    /* Initialize CaoEnginge Instance */
    hr = CoCreateInstance(CLSID_CaoEngine,NULL,CLSCTX_LOCAL_SERVER,IID_ICaoEngine,(void **) &pEng);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add enginepointer to the dataparams struct */
    pDataParams->pEngine = pEng;

    /* Add workspaces to the engine */
    hr = pEng->get_Workspaces(&pWss);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }
    /* Add workspaces pointer to the dataparams struct */
    pDataParams->pWorkspaces = pWss;

    /* Retrieve workspace in use */
    VariantInit(&zeroValue);
    zeroValue.vt = VT_I4;
    zeroValue.lVal = value;
    hr = pWss->Item(zeroValue, &pWs);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add workspace pointer to the dataparams struct */
    pDataParams->pWorkspace = pWs;

    /* Create Engine Controller */
    hr = pWs->AddController(cmdRc8,cmdDenso,cmdEmptyString,cmdServer,&pCtrl);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add contoller pointer to the dataparams struct */
    pDataParams->pContoller = pCtrl;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_TakeArmControl(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                        )
{
    /* Initialize robot arm */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;


    /* Get armControl authority */
    BSTR cmdTakeArm;
    cmdTakeArm = SysAllocString( L"Takearm" );
    VARIANT vntTakeArm, vntArmReturn;
    ::VariantInit(&vntTakeArm);
    ::VariantInit(&vntArmReturn);
    long lElemArm, lValueArm[] = {0L, 1L};
    vntTakeArm.vt = VT_ARRAY | VT_I4;
    vntTakeArm.parray = SafeArrayCreateVector(VT_I4, 0L, 2L);

    for(lElemArm = 0L; lElemArm < 2L; lElemArm++)
    {
        SafeArrayPutElement(vntTakeArm.parray, &lElemArm, &lValueArm[lElemArm]);
    }

    /* Execute take arm command */
    hr = pArm->Execute(cmdTakeArm, vntTakeArm, &vntArmReturn);
    SysFreeString( cmdTakeArm );

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Control over arm has been taken */
    pDataParams->bArmTaken = 1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_DisconnectController(
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                              )
{
    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t status;

    /* Initialize function variables */
    ICaoEngine*     pEng  = (ICaoEngine*)     pDataParams->pEngine;
    ICaoWorkspaces* pWss  = (ICaoWorkspaces*) pDataParams->pWorkspaces;
    ICaoWorkspace*  pWs   = (ICaoWorkspace*)  pDataParams->pWorkspace;
    ICaoController* pCtrl = (ICaoController*) pDataParams->pContoller;

    /* Release all taken devices of the robot */
    ICaoVariables* ppVar = NULL;
    pCtrl->get_Variables(&ppVar);
    ppVar->Clear();

    /* Remove and delete contoller */
    ICaoControllers* ppCtrl = NULL;
    pWs->get_Controllers(&ppCtrl);
    VARIANT varCtrlIndex;
    ::VariantInit(&varCtrlIndex);
    varCtrlIndex.vt = VT_I4;
    long valueCtrlIndex = 0L;
    varCtrlIndex.lVal = valueCtrlIndex;
    ppCtrl->Remove(varCtrlIndex);

    /* Remove and delete workspace */
    pEng->get_Workspaces(&pWss);
    pWss->Remove(varCtrlIndex);

    /* Dealocate used memory */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_EndProc(pDataParams,hr));

    /* Uninitialize pDataParams */
    pDataParams->bInit = PH_OFF;

    /* Set pointers to all used elements to zero */
    pDataParams->pContoller  = NULL;
    pDataParams->pWorkspace  = NULL;
    pDataParams->pWorkspaces = NULL;
    pDataParams->pEngine     = NULL;

    /* Deinitialize the robot object and environment*/
    CoUninitialize();

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_StartMotor(
                                                     phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                    )
{

    /* Initialize robot arm */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    /* If arm controll is not granted, take the control */
    if(pDataParams->bArmTaken != 1)
    {
        phdlStepper_DensoVS60_External_TakeArmControl(pDataParams);
    }

    /* Start Motor */
    BSTR cmdStartMotor;
    cmdStartMotor = SysAllocString( L"Motor" );
    VARIANT motorStart, returnMotorValue;
    long lElemMotor, lValueMotor[] = {1L, 0L};
    ::VariantInit(&motorStart);
    ::VariantInit(&returnMotorValue);
    motorStart.vt = VT_ARRAY | VT_I4;
    motorStart.parray = SafeArrayCreateVector(VT_I4, 0L, 2L);
    for(lElemMotor = 0L; lElemMotor < 2L; lElemMotor++)
    {
        SafeArrayPutElement(motorStart.parray, &lElemMotor, &lValueMotor[lElemMotor]);
    }

    /* Execute start motor command */
    hr = pArm->Execute(cmdStartMotor,motorStart, &returnMotorValue);
    SysFreeString( cmdStartMotor );

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_StopMotor(
                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                   )
{
    // Initialize robot arm
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    HRESULT hr = S_OK;

    /* Turn off motor */
    BSTR cmdEndMotor;
    cmdEndMotor = SysAllocString( L"Motor" );
    VARIANT motorEnd, returnMotorValue;
    long lElemMotor, lValueMotor[] = {0L, 0L};
    ::VariantInit(&motorEnd);
    ::VariantInit(&returnMotorValue);
    motorEnd.vt = VT_ARRAY | VT_I4;
    motorEnd.parray = SafeArrayCreateVector(VT_I4, 0L, 2L);
    for(lElemMotor = 0L; lElemMotor < 2L; lElemMotor++)
    {
        SafeArrayPutElement(motorEnd.parray, &lElemMotor, &lValueMotor[lElemMotor]);
    }

    /* Execute stop motor command */
    hr = pArm->Execute(cmdEndMotor, motorEnd, &returnMotorValue);
    SysFreeString( cmdEndMotor );

    /* Error check */
    if (FAILED(hr))
    {
        #ifdef _DEBUG
          /* Only use the logger when in debug mode */
          phdlStepper_DensoVS60_External_LogErrorCodes(hr);
        #endif
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_GiveArmContol(
                                                        phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                       )
{
    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    /* Function variables */
    VARIANT returnGiveArm;
    BSTR cmdGivearm;
    VARIANT giveArm;

    /*Release arm control authority */
    cmdGivearm = SysAllocString( L"Givearm" );
    VariantInit( &giveArm );
    giveArm.vt = VT_EMPTY;

    /* Execute give arm control command */
    hr = pArm->Execute(cmdGivearm, giveArm, &returnGiveArm);
    SysFreeString( cmdGivearm );

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Give back the control over arm */
    pDataParams->bArmTaken = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_External_SetIODeviceState(
                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                          int devIndex,
                                                          int valueOnOff
                                                          )
{
    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    if (devIndex > DENSO_VS60_IO_DEVICE_MAX_NUMBER)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Function variables */
    VARIANT varValueOnOff;
    ::VariantInit(&varValueOnOff);
    varValueOnOff.vt = VT_I2;
    varValueOnOff.intVal = valueOnOff;

    /* Set the status of the input device 0-Turn off; 1-Turn on */
    ICaoVariable* varDevice = (ICaoVariable*)(pDataParams->pDevicesArray[devIndex]);
    varDevice->put_Value(varValueOnOff);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_External_GetIODeviceState(
                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                          int devIndex
                                                          )
{
    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    if (devIndex > DENSO_VS60_IO_DEVICE_MAX_NUMBER)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Function variables */
    VARIANT varValueOnOff;
    ::VariantInit(&varValueOnOff);

    /* Get the current status of the input device */
    ICaoVariable* varDevice = (ICaoVariable*)(pDataParams->pDevicesArray[devIndex]);
    varDevice->get_Value(&varValueOnOff);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    int bValueOnOff = varValueOnOff.boolVal;

    if(bValueOnOff == 0)
    {
        return PH_ADD_COMPCODE(PH_OFF, PH_COMP_DL_STEPPER); /* Device is turned off */
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ON, PH_COMP_DL_STEPPER); /* Device is turned on */
    }
}

phStatus_t phdlStepper_DensoVS60_External_SetRobotAcceleration(
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                               float wAcceleration
                                                              )
{

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    /* Check if the value is in the allowed range */
    if((wAcceleration < -100) || (wAcceleration > 100))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Assign the value to the pDataParams object */
    pDataParams->bRobotAcceeration = (uint8_t)wAcceleration;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Robot accelerates */
    if(wAcceleration >= 0)
    {
        hr = pArm->Accelerate(0,wAcceleration, -1L);
    }
    /* Robot decelerates */
    else
    {
        /* Change the sign, since it has to be postive value */
        wAcceleration*=(-1);
        hr = pArm->Accelerate(0, -1L, wAcceleration);
    }

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

}

phStatus_t phdlStepper_DensoVS60_External_SetRobotSpeedInternal(
                                                                phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                                uint16_t wRobotSpeed
                                                               )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;
    float32_t fSpeed;

    /* Internal speed of the robot has to be between 0 and 100 percent */
    if((wRobotSpeed >1000) || (wRobotSpeed <= 0))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Add the speed to the pDataParams value */
    pDataParams->wRobotSpeed = wRobotSpeed;

    /* Divide the value of the speed in percantage in order to get the decimal point of the value */
    fSpeed = (float32_t)(wRobotSpeed/10.0);

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    VARIANT varSpeed;
    ::VariantInit(&varSpeed);
    varSpeed.vt = VT_R4;
    varSpeed.fltVal = fSpeed;

    /* Execute set internal speed of the robot command */
    hr = pArm->Speed( -1L, varSpeed.fltVal );

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SetRobotSpeedExternal(
                                                                phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                                float wRobotSpeed
                                                               )
{

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* External speed of the robot has to be between 0 and 100 percent */
    if((wRobotSpeed >100.0) || (wRobotSpeed <= 0.0))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Function variables */
    BSTR cmdExtSpeed;
    VARIANT varExtSpeed;
    VARIANT returnExtSpeed;

    cmdExtSpeed = SysAllocString( L"ExtSpeed" );
    ::VariantInit(&varExtSpeed);
    varExtSpeed.vt = VT_R4;
    varExtSpeed.fltVal = wRobotSpeed;

    /* Execute set external speed of the robot command */
    hr = pArm->Execute(cmdExtSpeed, varExtSpeed, &returnExtSpeed);

    /* Free allocated string */
    SysFreeString( cmdExtSpeed );

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

}

phStatus_t phdlStepper_DensoVS60_External_CreateRobotArm(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                        )
{

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* Function variables */
    BSTR cmdEmptyString;
    BSTR cmdArm;
    cmdEmptyString = SysAllocString( L"" );
    cmdArm         = SysAllocString( L"Arm" );

    /* Add robot arm to the controller */
    ICaoController* pCtrl = (ICaoController*)pDataParams->pContoller;
    hr = pCtrl->AddRobot(cmdArm,cmdEmptyString,&pArm);
    /* Free allocated string */
    SysFreeString(cmdArm);
    SysFreeString(cmdEmptyString);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add robot arm pointer to the data params robot arm pointer*/
    pDataParams->pArm = pArm;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_AttachDefaultIOVariables(
                                                                   phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                                  )
{
    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* Member variables */
    BSTR varBusyStatus;
    BSTR varOptionalParam;
    BSTR varMotorState;
    ICaoVariable* pVarArmMove = NULL;
    ICaoVariable* pVarMotor   = NULL;

    /* Allocate memory for strings */
    varBusyStatus    = SysAllocString( L"@BUSY_STATUS" );
    varOptionalParam = SysAllocString( L"" );

    /* Add Variable to robot to check busy status */
    hr = (pArm)->AddVariable(varBusyStatus,varOptionalParam,&pVarArmMove);
    SysFreeString(varBusyStatus);
    if (FAILED(hr))
    {
        SysFreeString(varOptionalParam);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add Variable to robot to check Motor State */
    varMotorState = SysAllocString( L"@SERVO_ON" );
    hr = (pArm)->AddVariable(varMotorState,varOptionalParam,&pVarMotor);
    SysFreeString(varMotorState);
    SysFreeString(varOptionalParam);
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Assign pointers of the default variables to the first two places in the array of IO variable pointers  */
    pDataParams->pDevicesArray[0] = (uint64_t)pVarArmMove;
    pDataParams->pDevicesArray[1] = (uint64_t)pVarMotor;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_AttachOptionalIOVariables(
                                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                                   )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* Function members */
    ICaoController* pCtrl          = NULL;
    ICaoVariable*   pVarLaser      = NULL;
    ICaoVariable*   pVarPump       = NULL;
    ICaoVariable*   pVarHeadSensor = NULL;

    /* Initialize member variables */
    BSTR varLaserInput;
    BSTR varPumpInput;
    BSTR varHeadSensorInput;

    /* set empty string for all variables */
    BSTR varOptionalParam;
    varOptionalParam = SysAllocString( L"" );
    pCtrl = (ICaoController*)pDataParams->pContoller;

    /* Add Variable Laser to robot */
    varLaserInput = SysAllocString( L"IO26" );
    hr = pCtrl->AddVariable(varLaserInput,varOptionalParam, &pVarLaser);
    SysFreeString(varLaserInput);

    /* Error check */
    if (FAILED(hr))
    {
        SysFreeString(varOptionalParam);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add Variable Pump to robot */
    varPumpInput = SysAllocString( L"IO25" );
    hr = pCtrl->AddVariable(varPumpInput,varOptionalParam, &pVarPump);
    SysFreeString(varPumpInput);

    /* Error check */
    if (FAILED(hr))
    {
        SysFreeString(varOptionalParam);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Add Variable HeadSensor to robot */
    varHeadSensorInput = SysAllocString( L"IO10" );
    hr = pCtrl->AddVariable(varHeadSensorInput,varOptionalParam, &pVarHeadSensor);

    /* Free allocated memory */
    SysFreeString(varHeadSensorInput);
    SysFreeString(varOptionalParam);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Assign pointers of the optional variables to the next three places in the array of IO variable pointers  */
    pDataParams->pDevicesArray[10] = (uint64_t)pVarHeadSensor;
    pDataParams->pDevicesArray[25] = (uint64_t)pVarPump;
    pDataParams->pDevicesArray[26] = (uint64_t)pVarLaser;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

/*------------------------------------------------------------------------*/
/*----------------------------- MOVE COMMAND -----------------------------*/
/*------------------------------------------------------------------------*/
phStatus_t phdlStepper_DensoVS60_External_MoveToTargetCoordinates(
                                                             phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                             double* arrAbosolutePosition,
                                                             int wAxisType,
                                                             int bBlocking
                                                            )
{

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Create command sting variable */
    wchar_t* wStructCommand = new wchar_t[100];

    /* If X-Y axis go here */
    if(wAxisType == 0)
    {
        /* Check if the target position goes outside the limits of Rz-axis, and normalize it when does */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckIfHeadAngleInRange(pDataParams,arrAbosolutePosition,0));

        /* Check if the command goes through the robot circle */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckBeforeMove(pDataParams));

        /* Move command x-y example : L"@0 P(350,0,450,180,-0.5,0,9)"  */
        swprintf(wStructCommand, L"@0 P(%f,%f,%f,%f,%f,%f,%f)",
                                arrAbosolutePosition[0], arrAbosolutePosition[1], arrAbosolutePosition[2],
                                arrAbosolutePosition[3], arrAbosolutePosition[4], arrAbosolutePosition[5], arrAbosolutePosition[6]);

        /* Additional check if the target position is out of range, only for X-Y axis */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckOutOfRange(pDataParams,wStructCommand));

        /* Move to target position */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteMove(pDataParams, wStructCommand, bBlocking));
    }
    /* If joint axis, go here*/
    else if(wAxisType == 1)
    {
        /* Check if the target position goes outside the limits of J6-axis, and normalize it when does */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckIfHeadAngleInRange(pDataParams,arrAbosolutePosition,1));

        swprintf(wStructCommand, L"@0 J(%f,%f,%f,%f,%f,%f,%f,%f)",
                                arrAbosolutePosition[0], arrAbosolutePosition[1], arrAbosolutePosition[2],
                                arrAbosolutePosition[3], arrAbosolutePosition[4], arrAbosolutePosition[5],
                                arrAbosolutePosition[6], arrAbosolutePosition[7]);

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteMove(pDataParams, wStructCommand, bBlocking));
    }

    /* Release allocated dynamic memory and delete the pointer to it */
    delete wStructCommand;
    wStructCommand = NULL;

    /* Return executed true */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_ExecuteMove(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                      wchar_t* wStructCommand,
                                                      int dBlocking
                                                     )
{

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    BSTR pathStr;
    pathStr = SysAllocString(wStructCommand);//L"@0 P(350,0,450,180,-0.5,0,9)" );

    VARIANT pathVar;
    ::VariantInit(&pathVar);
    pathVar.bstrVal = pathStr;
    pathVar.vt = VT_BSTR;

    /* Execute move command */
    if(dBlocking == 1) /* This move cannont be stoped until is done */
    {
        /* Initialize parameter for blocking move command */
        BSTR cmdMoveArmBlocking;
        cmdMoveArmBlocking = SysAllocString( L"" );

        /* Execute blocking move command */
        hr = pArm->Move( 2L, pathVar, cmdMoveArmBlocking);
        SysFreeString(cmdMoveArmBlocking);

        /* Error check */
        if(hr != S_OK)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_parseErrorCode( pDataParams,hr));
        }
    }
    else /* This move can be stoped before complition */
    {
        /* Initialize parameter for non-blocking move command */
        BSTR cmdMoveArmNonBlocking;
        cmdMoveArmNonBlocking = SysAllocString( L"NEXT" );

        /* Execute non-blocking move command */
        hr = pArm->Move( 2L, pathVar, cmdMoveArmNonBlocking);
        SysFreeString(cmdMoveArmNonBlocking);

        /* Error check */
        if(hr != S_OK)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_parseErrorCode( pDataParams,hr));
        }
    }
    /* Check if function executed correctly and terminate if not */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_ExecuteRotation(
                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                          double dwAngle
                                                         )
{

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /*Param 1: Add rotation plane*/
    wchar_t* rotationPlaneCommand = new wchar_t[100];
    swprintf(rotationPlaneCommand, L"V(1,1,1),V(0,1,1),V(0,0,1)");

    BSTR btsrRotationPath;
    btsrRotationPath = SysAllocString(rotationPlaneCommand);//L"@0 P(350,0,450,180,-0.5,0,9)" );

    VARIANT varRotationPath;
    ::VariantInit(&varRotationPath);
    varRotationPath.bstrVal = btsrRotationPath;
    varRotationPath.vt = VT_BSTR;

    /* Get current value of the J1 coordiante */
    double wPositionJ1;
    PH_CHECK_SUCCESS_FCT(statusTmp,
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,1,&wPositionJ1));

    /*Param 2: Add rotation angle */
    float dwEntireAngle = -(float)(dwAngle-wPositionJ1);

    /*Param 3: Add center point*/
    wchar_t* rotationPlaneCenter = new wchar_t[100];
    swprintf(rotationPlaneCenter, L"V(0,0,0)");

    BSTR btsrRotationPlaneCenter;
    btsrRotationPlaneCenter = SysAllocString(rotationPlaneCenter);

    VARIANT varRotationCenter;
    ::VariantInit(&varRotationCenter);
    varRotationCenter.bstrVal = btsrRotationPlaneCenter;
    varRotationCenter.vt = VT_BSTR;

    /*Param 4: Add optional param */
    wchar_t* rotationPlaneOptParam = new wchar_t[100];
    swprintf(rotationPlaneOptParam, L"@E, pose=2,NEXT");
    BSTR btsrRotationOptionalParam;
    btsrRotationOptionalParam = SysAllocString(rotationPlaneOptParam);

    /* Execute rotation movement */
    hr = pArm->Rotate(varRotationPath,dwEntireAngle,varRotationCenter,btsrRotationOptionalParam);

    /* Clear the dynamic memory */
    delete rotationPlaneCenter;
    rotationPlaneCenter   = NULL;
    delete rotationPlaneOptParam;
    rotationPlaneOptParam = NULL;
    delete rotationPlaneOptParam;
    rotationPlaneOptParam = NULL;

    /* Free allocated string variables */
    SysFreeString(btsrRotationPath);
    SysFreeString(btsrRotationPlaneCenter);
    SysFreeString(btsrRotationOptionalParam);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_parseErrorCode( pDataParams,hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_GoDownUntilContact(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         long dwMinZPoint,
                                                         uint16_t wRobotSpeed
                                                        )
{
    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    /* Allocate strings */
    optionalParam = SysAllocString( L"" );
    cmdCurPos     = SysAllocString( L"CurPosEx" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    /* Execute get current cooridantes for XY axis */
    hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

    /* Free allocated strings from the memory */
    SysFreeString(cmdCurPos);
    SysFreeString(optionalParam);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Create array for the current poisition cooridnates of the XY mode */
    double arrAbosolutePosition[7];
    arrAbosolutePosition[0] = ((double *)currentPos.parray->pvData)[1];
    arrAbosolutePosition[1] = ((double *)currentPos.parray->pvData)[2];
    arrAbosolutePosition[2] = (double)dwMinZPoint/1000;
    arrAbosolutePosition[3] = ((double *)currentPos.parray->pvData)[4];
    arrAbosolutePosition[4] = ((double *)currentPos.parray->pvData)[5];
    arrAbosolutePosition[5] = ((double *)currentPos.parray->pvData)[6];
    arrAbosolutePosition[6] = ((double *)currentPos.parray->pvData)[7];


    /* Initialize parameter for non-blocking move command */
    BSTR cmdMoveArmNonBlocking;
    cmdMoveArmNonBlocking = SysAllocString( L"NEXT" );

    /* The upper limit of the robot speed should not exceed 10 in order for sensor to detect a collision */
    if(wRobotSpeed > 100)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Set the speed of the robot */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SetRobotSpeedInternal( pDataParams, wRobotSpeed));

    wchar_t* wStructCommand = new wchar_t[100];
    swprintf(wStructCommand, L"@0 P(%f,%f,%f,%f,%f,%f,%f)",
                arrAbosolutePosition[0], arrAbosolutePosition[1], arrAbosolutePosition[2],
                arrAbosolutePosition[3], arrAbosolutePosition[4], arrAbosolutePosition[5],
                arrAbosolutePosition[6]);

    BSTR pathStr;
    pathStr = SysAllocString(wStructCommand);//L"@0 P(350,0,450,180,-0.5,0,9)" );

    VARIANT pathVar;
    ::VariantInit(&pathVar);
    pathVar.bstrVal = pathStr;
    pathVar.vt = VT_BSTR;

    /* Execute the non-blocking movement */
    hr = pArm->Move( 2L, pathVar, cmdMoveArmNonBlocking);

    /* Free allocated strings from the memory */
    SysFreeString(pathStr);
    SysFreeString(cmdMoveArmNonBlocking);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Execute non-blocking move command, move until  */
    double dwAbsolutValue = 0;
    statusTmp = 1;
    while(statusTmp != 0)
    {
        /* Get the current status of head sensor, when something is reached it should return 0 */
        statusTmp = phdlStepper_DensoVS60_External_GetIODeviceState(pDataParams,10);
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,3,&dwAbsolutValue);
        /* If there is now obstacle until given border, check if the target postion is reached
            ( deviation can be 0.01mm) and then break the while loop */
        if(abs((double)arrAbosolutePosition[Z_INDEX] - dwAbsolutValue) < 0.01)
        {
            statusTmp = 0;
        }
    }

    /* when the bottom is touched, stop the motion */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteHaltMove(pDataParams));

    /* Update current coordinates values */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(pDataParams, &pDataParams->structTargetXY));

    /* Delete the pointer and release its memory */
    delete wStructCommand;
    wStructCommand = NULL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}
/*------------------------------------------------------------------------*/
/*--------------------------- SPLINE MOVE---------------------------------*/
/*------------------------------------------------------------------------*/
phStatus_t phdlStepper_DensoVS60_External_SplineExecuteMove(
                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                            uint8_t bPathIndex,
                                                            uint8_t bBlocking
                                                            )
{

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    /*Number of path points */
    uint16_t argNumberOfPathPoints;

    /* Index is out of range, can store only 20 values */
    if( bPathIndex < 0 || bPathIndex > 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Get the number of current path points */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPointCount(pDataParams, bPathIndex, &argNumberOfPathPoints));

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Set type of blocking movement */
    BSTR cmdMoveTypeOfBlocking;
    if(bBlocking == PH_ON) /* The movement is in blocking state, it cannot be cancelled before it is finished */
    {
        cmdMoveTypeOfBlocking = SysAllocString( L"" );
    }
    else if (bBlocking == PH_OFF) /* The movement is in non-blocking state, it can be interrupted */
    {
        cmdMoveTypeOfBlocking = SysAllocString( L"NEXT" );
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Assign provided path the the variant variable */
    VARIANT varPathIndex;
    ::VariantInit(&varPathIndex);
    varPathIndex.vt = VT_I2;
    varPathIndex.intVal = bPathIndex;

    /* Execute the move command */
    hr = pArm->Move( 4L, varPathIndex, cmdMoveTypeOfBlocking);

    /* Free allocated strings from the memory */
    SysFreeString(cmdMoveTypeOfBlocking);

     /* When the movement is in blocking state, check if motion is competed */
    if(bBlocking == PH_ON)
    {
        phStatus_t statusTmp1;
        uint8_t bMotionCompleted;

        do
        {
            PH_CHECK_SUCCESS_FCT(statusTmp1, phdlStepper_DensoVS60_External_CheckMotionComplete(pDataParams, &bMotionCompleted));
        }while( bMotionCompleted != PH_ON );
    }
    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SplineAddPathPoint(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         uint8_t  bPathIndex,
                                                         int32_t dwPositionX,
                                                         int32_t dwPositionY,
                                                         int32_t dwPositionZ
                                                        )

{
    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;
    uint16_t argNumberOfPathPoints;

    /* Index is out of range, can store only 20 values */
    if( bPathIndex < 0 || bPathIndex > 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Check if already 5000 points added */
    /* Get the number of points in the selected path */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPointCount(pDataParams, bPathIndex, &argNumberOfPathPoints));
    if (argNumberOfPathPoints + 1 > 5000)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW,PH_COMP_DL_STEPPER);
    }

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;
    //--------------------------------------------------------------------------------------------
    /* Create array for the current poisition cooridnates of the XY mode */
    double arrAbosolutePosition[7];
    arrAbosolutePosition[0] = ((double)dwPositionX)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionX;
    arrAbosolutePosition[1] = ((double)dwPositionY)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionY;
    arrAbosolutePosition[2] = ((double)dwPositionZ)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionZ;
    arrAbosolutePosition[3] =  pDataParams->structTargetXY.offsetCoords.dwPositionRx;
    arrAbosolutePosition[4] =  pDataParams->structTargetXY.offsetCoords.dwPositionRy;
    arrAbosolutePosition[5] =  pDataParams->structTargetXY.offsetCoords.dwPositionRz;
    arrAbosolutePosition[6] =  pDataParams->structTargetXY.offsetCoords.wFigure;

    /* Execute add path point */
    BSTR cmdAddPathPoint;
    cmdAddPathPoint = SysAllocString( L"AddPathPoint" );

    /* Path number variable */
    VARIANT varPathIndex;
    ::VariantInit(&varPathIndex);
    varPathIndex.vt = VT_I2;
    varPathIndex.intVal = bPathIndex;
    //--------------------------------------------------------------------------------------------
    /* Create string for the target position */
    wchar_t* wStructCommand = new wchar_t[100];
    swprintf(wStructCommand, L"P(%f,%f,%f,%f,%f,%f,%f)",
                arrAbosolutePosition[0], arrAbosolutePosition[1], arrAbosolutePosition[2],
                arrAbosolutePosition[3], arrAbosolutePosition[4], arrAbosolutePosition[5],
                arrAbosolutePosition[6]);

    BSTR pathStr;
    pathStr = SysAllocString(wStructCommand);//L"@0 P(350,0,450,180,-0.5,0,9)" );

    VARIANT pathVar;
    ::VariantInit(&pathVar);
    pathVar.bstrVal = pathStr;
    pathVar.vt = VT_BSTR;
    //--------------------------------------------------------------------------------------------
    /*Add two VARIANT objects to the safearray pointer */
    VARIANT varCommandArray;
    ::VariantInit(&varCommandArray);

    VARIANT returnCommandArray;
    ::VariantInit(&returnCommandArray);

    varCommandArray.vt = VT_ARRAY | VT_VARIANT;
    varCommandArray.parray = SafeArrayCreateVector(VT_VARIANT, 0L, 2L);

    long elemFirst = 0L;
    long elemSecond = 1L;
    SafeArrayPutElement(varCommandArray.parray, &elemFirst, &varPathIndex);
    SafeArrayPutElement(varCommandArray.parray, &elemSecond, &pathVar);

    /* Execute add path point to the selected path command */
    hr = pArm->Execute(cmdAddPathPoint,varCommandArray, &returnCommandArray);

    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineLoadOnlyReachablePoints(pDataParams,bPathIndex));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_parseErrorCode( pDataParams,hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SplineClearPath(
                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                    uint8_t bPathIndex
                                                        )
{

    /* Check if index is out of range, can store only 20 values */
    if( bPathIndex < 0 || bPathIndex > 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Initialize status variables */
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Execute add path point */
    BSTR cmdAddPathPoint;
    cmdAddPathPoint = SysAllocString( L"ClrPathPoint" );

    /* Path number variable */
     VARIANT varPathIndex;
    ::VariantInit(&varPathIndex);
    varPathIndex.vt = VT_I4;
    varPathIndex.intVal = bPathIndex;

    /* Return value Variant type*/
    VARIANT varReturnValue;
    ::VariantInit(&varReturnValue);

    /* Clear path from the memory */
    hr = pArm->Execute(cmdAddPathPoint,varPathIndex,&varReturnValue);

    /* Free allocated strings from the memory */
    SysFreeString(cmdAddPathPoint);

    /* Error check */
    if (FAILED(hr))
    {
        #ifdef _DEBUG
          /* Only use the logger when in debug mode */
          phdlStepper_DensoVS60_External_LogErrorCodes(hr);
        #endif
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SplineGetPathPointCount (
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         uint8_t bPathIndex,
                                                         uint16_t *pwNumberOfPoints
                                                                 )
{
    /* Check if index is out of range, can store only 20 values */
    if( bPathIndex < 0 || bPathIndex > 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Initialize status variables */
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Execute add path point */
    BSTR cmdAddPathPoint;
    cmdAddPathPoint = SysAllocString( L"GetPathPointCount" );

    /* Path number variable */
     VARIANT varPathIndex;
    ::VariantInit(&varPathIndex);
    varPathIndex.vt = VT_I2;
    varPathIndex.intVal = bPathIndex;

    /* Return value Variant type*/
    VARIANT varReturnValue;
    ::VariantInit(&varReturnValue);

    /* Execute get number of path points from the selected path command */
    hr = pArm->Execute(cmdAddPathPoint,varPathIndex,&varReturnValue);

    if (FAILED(hr))
    {
        #ifdef _DEBUG
          /* Only use the logger when in debug mode */
          phdlStepper_DensoVS60_External_LogErrorCodes(hr);
        #endif
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Number of path points should not exceed the 5000 */
    if(varReturnValue.intVal > 5000)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW,PH_COMP_DL_STEPPER);
    }

    /* Add the number of path points to the path points reference */
    *pwNumberOfPoints = (uint16_t)varReturnValue.intVal;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS,PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SplineGetPathPoint(
                                                             phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                             phdlStepper_DensoVS60_DataParams_XY_coord_t * sLocalStructTargetXY,
                                                             uint8_t  bPathIndex,
                                                             uint16_t wPointIndex
                                                            )
{
    /* Index is out of range, can store only 20 values, but the last one is for the inverse trajectory */
    if( bPathIndex < 1 || bPathIndex > 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    if( wPointIndex < 0 || wPointIndex > 5000)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW,PH_COMP_DL_STEPPER);
    }

    /* Initialize status variables */
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Execute add path point */
    BSTR cmdGetPathPoint;
    cmdGetPathPoint = SysAllocString( L"GetPathPoint" );

    /* Get path points */
    VARIANT varPathPoints, returnPathPoints;
    long lElemCounter, lArrayValue[] = {bPathIndex, wPointIndex};
    ::VariantInit(&varPathPoints);
    ::VariantInit(&returnPathPoints);
    varPathPoints.vt = VT_ARRAY | VT_I4;
    varPathPoints.parray = SafeArrayCreateVector(VT_I4, 0L, 2L);
    for(lElemCounter = 0L; lElemCounter < 2L; lElemCounter++)
    {
        SafeArrayPutElement(varPathPoints.parray, &lElemCounter, &lArrayValue[lElemCounter]);
    }

    /* Get path pointer from the internal memory */
    hr = pArm->Execute(cmdGetPathPoint,varPathPoints,&returnPathPoints);

    /* Error handling */
    if (FAILED(hr))
    {
        #ifdef _DEBUG
          /* Only use the logger when in debug mode */
          phdlStepper_DensoVS60_External_LogErrorCodes(hr);
        #endif
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Coordinates current position for X-Y axis */
    double arrCurrentPositionXYCoords[3];

    /* Save the intern current position coordinates to the array of current positions */
    for (int counter = 0; counter < 3; counter++)
    {
        arrCurrentPositionXYCoords[counter] = ((double *)returnPathPoints.parray->pvData)[counter];
    }

    /* Save the value of the X,Y and Z cooridantes to the reference of the local structure */
    sLocalStructTargetXY->dwPositionX = (int32_t)((arrCurrentPositionXYCoords[0] - pDataParams->structTargetXY.offsetCoords.dwPositionX)*1000);
    sLocalStructTargetXY->dwPositionY = (int32_t)((arrCurrentPositionXYCoords[1] - pDataParams->structTargetXY.offsetCoords.dwPositionY)*1000);
    sLocalStructTargetXY->dwPositionZ = (int32_t)((arrCurrentPositionXYCoords[2] - pDataParams->structTargetXY.offsetCoords.dwPositionZ)*1000);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SplineLoadInversePath(
                                                                phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                                uint8_t bPathIndex
                                                               )
{

    /* Initialize status variables */
    phStatus_t statusTmp;
    uint16_t argNumberOfPathPoints;

    /* Index is out of range, can store only 20 values */
    if( bPathIndex < 1 || bPathIndex > 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
    }

    /* Get the number of points for the selected path */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPointCount(pDataParams,bPathIndex, &argNumberOfPathPoints));

    /* Create local struct to hold all intermediate points */
    phdlStepper_DensoVS60_DataParams_XY_coord_t structTargetXY;

    /* Take points inverse from the selected path and put them in the path 20 */
    for(int counter = argNumberOfPathPoints; counter > 0; counter--)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPoint(pDataParams, &structTargetXY, bPathIndex, (int16_t)counter));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineAddPathPoint(
                                                          pDataParams,
                                                          20,
                                                          structTargetXY.dwPositionX,
                                                          structTargetXY.dwPositionY,
                                                          structTargetXY.dwPositionZ));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SplineLoadOnlyReachablePoints(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         uint8_t bPathIndex
                                                                 )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    uint16_t argNumberOfPathPoints;

    /* Get the number of points in the selected path */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPointCount(pDataParams, bPathIndex, &argNumberOfPathPoints));

    /* Clear path 20 before entering path points */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineClearPath(pDataParams,20));

    /* Create local struct to hold all intermediate points */
    phdlStepper_DensoVS60_DataParams_XY_coord_t structTargetXY;

    /* Take all points from the selected path except the last one and put them in path 20 */
    for(int counter =  0; counter < argNumberOfPathPoints-1; counter++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPoint(pDataParams, &structTargetXY, bPathIndex, (int16_t)(counter+1)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineAddPathPoint(
                                                          pDataParams,
                                                          20,
                                                          structTargetXY.dwPositionX,
                                                          structTargetXY.dwPositionY,
                                                          structTargetXY.dwPositionZ));
    }

    /* Clear all path points from the selected path */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineClearPath(pDataParams,bPathIndex));

    /* Add all points from the path 20 to the selected path */
    for(int counter =  0; counter < argNumberOfPathPoints-1; counter++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineGetPathPoint(pDataParams, &structTargetXY, 20, (int16_t)(counter+1)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SplineAddPathPoint(
                                                          pDataParams,
                                                          bPathIndex,
                                                          structTargetXY.dwPositionX,
                                                          structTargetXY.dwPositionY,
                                                          structTargetXY.dwPositionZ));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

/*------------------------------------------------------------------------*/
/*--------------------------- CHECK FUNCTIONS ----------------------------*/
/*------------------------------------------------------------------------*/
phStatus_t phdlStepper_DensoVS60_External_CheckIfRotationSafeMove(
                                                           phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                           double dwDistance,
                                                           double* arrTargetPosition,
                                                           int bAxisType
                                                           )
{
    /* Initialize status variables */
    phStatus_t statusTmp;

    /*Check rotation for Rz axis*/
    if(bAxisType == 0)
    {
        /* Find the interim angles for the Rz cooridnate */
        double* arrInterimAngles = phdlStepper_DensoVS60_External_ArrayOfInterimAngles(dwDistance);
        int sizeOfArray = (int)(arrInterimAngles[0]);
        if(dwDistance < 0)
        {
            for(int counter = 0; counter < sizeOfArray; counter++)
            {
                /* Create command sting variable for interim move */
                wchar_t* pInterimMoveCommand = new wchar_t[100];
                swprintf(pInterimMoveCommand, L"@0 P(%f,%f,%f,%f,%f,%f,%f)",
                        arrTargetPosition[0], arrTargetPosition[1], arrTargetPosition[2],
                        arrTargetPosition[3], arrTargetPosition[4], arrInterimAngles[counter+1],
                        arrTargetPosition[6]);

                /* Execute move command for interim values */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteMove(pDataParams, pInterimMoveCommand, 0));

                /* Deallocate the dynamic memory */
                delete pInterimMoveCommand;
                pInterimMoveCommand = NULL;
            }
        }
        else
        {
            for(int counter  = sizeOfArray; counter > 0; counter--)
            {
                /* Create command sting variable for interim move */
                wchar_t* pInterimMoveCommand = new wchar_t[100];
                swprintf(pInterimMoveCommand, L"@0 P(%f,%f,%f,%f,%f,%f,%f)",
                        arrTargetPosition[0], arrTargetPosition[1], arrTargetPosition[2],
                        arrTargetPosition[3], arrTargetPosition[4], arrInterimAngles[counter],
                        arrTargetPosition[6]);

                /* Execute move command for interim values */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteMove(pDataParams, pInterimMoveCommand, 0));

                /* Deallocate the dynamic memory */
                delete pInterimMoveCommand;
                pInterimMoveCommand = NULL;
            }
        }
    }
    /*Check rotation for J6 axis*/
    else if( bAxisType == 1)
    {
        /* Find the interim angles for the Rz cooridnate */
        double* arrInterimAngles = phdlStepper_DensoVS60_External_ArrayOfInterimAngles(dwDistance);
        int sizeOfArray = (int)(arrInterimAngles[0]);
        /* If clockwise rotation, current angle larger from targeted */
        if(dwDistance < 0)
        {
            for(int counter  = 0; counter < sizeOfArray; counter++)
            {
                /* Create command sting variable for interim move */
                wchar_t* pInterimMoveCommand = new wchar_t[100];
                swprintf(pInterimMoveCommand, L"@0 P(%f,%f,%f,%f,%f,%f,%f,%f)",
                        arrTargetPosition[0], arrTargetPosition[1], arrTargetPosition[2],
                        arrTargetPosition[3], arrTargetPosition[4], arrInterimAngles[counter+1],
                        arrTargetPosition[6], arrTargetPosition[7]);

                /* Execute move command for interim values */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteMove(pDataParams, pInterimMoveCommand, 1));

                /* Deallocate the dynamic memory */
                delete pInterimMoveCommand;
                pInterimMoveCommand = NULL;
            }
        }
        /* If counterclockwise rotation, target angle larger than current */
        else
        {
            for(int counter = sizeOfArray; counter > 0; counter--)
            {
                /* Create command sting variable for interim move */
                wchar_t* pInterimMoveCommand = new wchar_t[100];
                swprintf(pInterimMoveCommand, L"@0 P(%f,%f,%f,%f,%f,%f,%f,%f)",
                        arrTargetPosition[0], arrTargetPosition[1], arrTargetPosition[2],
                        arrTargetPosition[3], arrTargetPosition[4], arrInterimAngles[counter],
                        arrTargetPosition[6], arrTargetPosition[7]);

                /* Execute move command for interim values */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ExecuteMove(pDataParams, pInterimMoveCommand, 1));

                /* Deallocate the dynamic memory */
                delete pInterimMoveCommand;
                pInterimMoveCommand = NULL;
            }
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_CheckIfHeadAngleInRange(
                                                                  phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                                  double* arrAbosolutePosition,
                                                                  int bAxisType
                                                                 )
{
    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Current Rz axis value in radian */
    double dwCurrentHeadAngle;

    if(bAxisType == 0)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp,
            phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,6,&dwCurrentHeadAngle));
    }
    else    /* if bAxisType == 1 for Joint axis mode */
    {

        PH_CHECK_SUCCESS_FCT(statusTmp,
            phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,6,&dwCurrentHeadAngle));
    }

    /* Normalize target position of the Rz coordiante*/
    double dwDistance = phdlStepper_DensoVS60_External_NormalizeTargetedValue(dwCurrentHeadAngle,&arrAbosolutePosition[5]);

    /* If angle in this range, rotate between borders for either Rz or J6 coordinates*/
    if(dwDistance > 180 || dwDistance < -180)
    {
        if(bAxisType == 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckIfRotationSafeMove(pDataParams,dwDistance,arrAbosolutePosition,0));
        }
        else /* if bAxisType == 1 for Joint axis mode */
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckIfRotationSafeMove(pDataParams,dwDistance,arrAbosolutePosition,1));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_External_CheckBeforeMove(
                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                         )
{

    /* Initialize status variables */
    phStatus_t statusTmp;

    /* Declare start and end point */
    //int32_t pointStartWithoutOffset[2];
    double pointStart[2];
    double pointEnd[2];

    /* Assign X and Y coordinates values to the start and end points */
    pointEnd[X_INDEX] = ((double)pDataParams->structTargetXY.dwPositionX)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionX;
    pointEnd[Y_INDEX] = ((double)pDataParams->structTargetXY.dwPositionY)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionY;

    /* Calculate absolute values for the X and y coordinate */
    PH_CHECK_SUCCESS_FCT(statusTmp,
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition(pDataParams,0,1,&pointStart[X_INDEX]));
    PH_CHECK_SUCCESS_FCT(statusTmp,
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition(pDataParams,0,2,&pointStart[Y_INDEX]));

    /* Check if move between two points is possible without obstacle */
    int isAllowed = phdlStepper_DensoVS60_External_CheckIfMoveAllowed(pointStart,pointEnd);
    if(isAllowed == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
    else
    {
        /* Auto rotation is not allowed, out of range error */
        if(pDataParams->bActiveRotation == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }

        double dwRadians = atan2(pointEnd[Y_INDEX], pointEnd[X_INDEX]);
        double dwAngle = dwRadians *( 180 / M_PI);

        /* The angle of rotation should not be biger thatn (-)169
            otherwise the head will be in forbidden space */
        if(-dwAngle > J1__ANGLE_LIMIT || dwAngle > J1__ANGLE_LIMIT)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }

        /* Execute rotation move of the J1 cooridante */
        PH_CHECK_SUCCESS_FCT(statusTmp,
            phdlStepper_DensoVS60_External_ExecuteRotation(pDataParams,dwAngle));

        /* Continue to move to the end position */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
}

int phdlStepper_DensoVS60_External_CheckIfMoveAllowed(
                                                      double* startPoint,
                                                      double* endPoint
                                                     )
{
    if (startPoint[X_INDEX] < 0.0 && endPoint[X_INDEX] < 0.0 && (startPoint[Y_INDEX] * endPoint[Y_INDEX]) < 0.0)
        return -1;

    /* Internal check between opposite quadrants, but if both axis change values */
    if (startPoint[X_INDEX] * endPoint[X_INDEX] < 0.0 && (startPoint[Y_INDEX] * endPoint[Y_INDEX]) < 0.0)
    {
        /* Calc crossing of x axis */
        double n = -startPoint[Y_INDEX]/(endPoint[Y_INDEX] - startPoint[Y_INDEX]);
        if (-1.0 <= n && n <= 1.0)
        {
            double crossingXValue = startPoint[X_INDEX] + n*(endPoint[X_INDEX] - startPoint[X_INDEX]);
            if (crossingXValue <= 0)
            {
                return -1;
            }
        }
    }

    /* compute the euclidean distance between A and B */
    double lab = sqrt( pow((endPoint[X_INDEX] - startPoint[X_INDEX]),2) + pow((endPoint[Y_INDEX] - startPoint[Y_INDEX]),2));

    /* Security check for permanent move between two points on the Z-axis */
    if(lab == 0.0)
    {
        return 0;
    }

    /* compute the direction vector D from A to B */
    double dX = (endPoint[X_INDEX] - startPoint[X_INDEX])/lab;
    double dY = (endPoint[Y_INDEX] - startPoint[Y_INDEX])/lab;

    /* Now the line equation is x = Dx*t + Ax, y = Dy*t + Ay with 0 <= t <= 1.
       compute the value t of the closest point to the circle center (Cx, Cy) */
    double t = dX*(-startPoint[X_INDEX]) + dY*(-startPoint[Y_INDEX]);
    if (t < 0.0)
        t = 0.0;
    else if (t > lab)
        t = lab;

    /* compute the coordinates of the point E on line and closest to C  */
    double eX = t*dX+startPoint[X_INDEX];
    double eY = t*dY+startPoint[Y_INDEX];

    /* compute the euclidean distance from E to C */
    double lec = sqrt( pow((eX),2) + pow((eY),2));
    if(lec == 0.0)
    {
        /* Move is not allowed */
        return 0;
    }

    /* Check if the line intersects the circle */
    if(lec < 140)
    {
        /* Move is not allowed */
        return -1;
    }
    else
    {
        /* Move is allowed */
        return 0;
    }
}

phStatus_t phdlStepper_DensoVS60_External_CheckMotionComplete(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                              uint8_t *pMotionCompleted
                                                             )
{
    /* Initialize robot arm */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variable */
    HRESULT hr = S_OK;

    /* Start Motor */
    BSTR cmdMotionComplete;
    cmdMotionComplete = SysAllocString( L"MotionComplete" );

    BSTR cmdEmptyComplete;
    cmdEmptyComplete = SysAllocString( L"" );

    VARIANT varCmdTest;
    ::VariantInit(&varCmdTest);
    varCmdTest.vt = VT_BYREF|VT_I1;
    varCmdTest.bstrVal = cmdMotionComplete;

    VARIANT motionComplete, returnMotionComplete;
    long lElemMotComp, lValueMotComp[] = {-1L, 1L};
    ::VariantInit(&motionComplete);
    ::VariantInit(&returnMotionComplete);
    motionComplete.vt = VT_ARRAY | VT_I4;
    motionComplete.parray = SafeArrayCreateVector(VT_I4, 0L, 2L);
    for(lElemMotComp = 0L; lElemMotComp < 2L; lElemMotComp++)
    {
        SafeArrayPutElement(motionComplete.parray, &lElemMotComp, &lValueMotComp[lElemMotComp]);
    }

    /* Execute check if motion complete command */
    hr = pArm->Execute(cmdMotionComplete,motionComplete, &returnMotionComplete);
    int byte = (int)returnMotionComplete.boolVal;

    /* Release allocated strings from the memory */
    SysFreeString( cmdMotionComplete );
    SysFreeString( cmdEmptyComplete );

    /* Error check */
    if (FAILED(hr))
    {
        #ifdef _DEBUG
          /* Only use the logger when in debug mode */
          phdlStepper_DensoVS60_External_LogErrorCodes(hr);
        #endif
        return PH_ADD_COMPCODE(PH_STEPPER_DENSO_VS60_ERR_CUSTOMERROR, PH_COMP_DL_STEPPER);
    }

    if(byte == 0x00)
    {
        *pMotionCompleted = PH_OFF;
    }
    else
    {
        *pMotionCompleted = PH_ON;
    }

    return PH_ADD_COMPCODE(ERROR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_ExecuteHaltMove(
                                                   phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                  )
{

    /* Initialize robot arm */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    BSTR cmdHalt;
    cmdHalt = SysAllocString( L"" );

    /* Execute halt movement command */
    hr = pArm->Halt(cmdHalt);

    /* Free allocated memory */
    SysFreeString( cmdHalt );

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
        return PH_ADD_COMPCODE(PH_STEPPER_DENSO_VS60_ERR_CUSTOMERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

/*------------------------------------------------------------------------*/
/*--------------------------- SETTER FUNCTIONS ---------------------------*/
/*------------------------------------------------------------------------*/

phStatus_t phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(phdlStepper_DensoVS60_DataParams_t * pDataParams, int bAxisType)
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    optionalParam = SysAllocString( L"" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    /* If it is X-Y axis */
    if(bAxisType == 0)
    {
        cmdCurPos = SysAllocString( L"CurPosEx" );
        /* Execute get current coordinates of the XY axis */
        hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);
        if (FAILED(hr))
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
        }

        /* Coordinates current position for X-Y axis */
        double arrCurrentPositionXYCoords[7];

        /* Save the intern current position coordinates to the array of current positions */
        for (int counter = 0; counter < 7; counter++)
        {
            arrCurrentPositionXYCoords[counter] = ((double *)currentPos.parray->pvData)[counter+1];
        }

        /* Assign current coordinates to the offset array */
        pDataParams->structTargetXY.offsetCoords.dwPositionX  = arrCurrentPositionXYCoords[0];
        pDataParams->structTargetXY.offsetCoords.dwPositionY  = arrCurrentPositionXYCoords[1];
        pDataParams->structTargetXY.offsetCoords.dwPositionZ  = arrCurrentPositionXYCoords[2];
        pDataParams->structTargetXY.offsetCoords.dwPositionRx = arrCurrentPositionXYCoords[3];
        pDataParams->structTargetXY.offsetCoords.dwPositionRy = arrCurrentPositionXYCoords[4];
        pDataParams->structTargetXY.offsetCoords.dwPositionRz = arrCurrentPositionXYCoords[5];
        pDataParams->structTargetXY.offsetCoords.wFigure      = arrCurrentPositionXYCoords[6];

        /* Free allocated string */
        SysFreeString(cmdCurPos);
    }

     else if(bAxisType == 1)
    {
        cmdCurPos = SysAllocString( L"CurJntEx" );
        /* Execute get current coordinates of the joint axis */
        hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);
        if (FAILED(hr))
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
        }

        /* Coordinates current position for joint axis */
        double arrCurrentPositionJointCoords[8];

        /* Save the intern current position coordinates to the array of current positions */
        for (int counter = 0; counter < 8; counter++)
        {
            arrCurrentPositionJointCoords[counter] = ((double *)currentPos.parray->pvData)[counter+1];
        }

        /* Assign current coordinates to the offset array */
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ1 = arrCurrentPositionJointCoords[0];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ2 = arrCurrentPositionJointCoords[1];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ3 = arrCurrentPositionJointCoords[2];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ4 = arrCurrentPositionJointCoords[3];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ5 = arrCurrentPositionJointCoords[4];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ6 = arrCurrentPositionJointCoords[5];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ7 = arrCurrentPositionJointCoords[6];
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ8 = arrCurrentPositionJointCoords[7];

        /* Free allocated string */
        SysFreeString(cmdCurPos);
    }

    /* Free allocated string */
    SysFreeString(optionalParam);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SetCurrentXYPositionWithOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                              phdlStepper_DensoVS60_DataParams_XY_coord_t* structCurrentPositionXYCoords
                                                              )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    optionalParam = SysAllocString( L"" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    cmdCurPos = SysAllocString( L"CurPosEx" );
    /* Execute get current coordinates of the XY axis */
    hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

    /* Free allocated strings from the memory */
    SysFreeString(cmdCurPos);
    SysFreeString(optionalParam);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Save the intern current position coordinates to the array of current positions without offset */
    structCurrentPositionXYCoords->dwPositionX =  phdlStepper_DensoVS60_External_RoundDoubleValue(((double *)currentPos.parray->pvData)[1]);

    structCurrentPositionXYCoords->dwPositionY =  phdlStepper_DensoVS60_External_RoundDoubleValue(((double *)currentPos.parray->pvData)[2]);

    structCurrentPositionXYCoords->dwPositionZ =  phdlStepper_DensoVS60_External_RoundDoubleValue(((double *)currentPos.parray->pvData)[3]);

    structCurrentPositionXYCoords->dwPositionRx = phdlStepper_DensoVS60_External_RoundDoubleValue((0.01)*((double *)currentPos.parray->pvData)[4]);

    structCurrentPositionXYCoords->dwPositionRy = phdlStepper_DensoVS60_External_RoundDoubleValue((0.01)*((double *)currentPos.parray->pvData)[5]);

    structCurrentPositionXYCoords->dwPositionRz = phdlStepper_DensoVS60_External_RoundDoubleValue((0.01)*((double *)currentPos.parray->pvData)[6]);

    structCurrentPositionXYCoords->wFigure = (int16_t)phdlStepper_DensoVS60_External_RoundDoubleValue((0.01)*((double *)currentPos.parray->pvData)[7]);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SetCurrentJointPositionWithOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                              phdlStepper_DensoVS60_DataParams_Joint_coord_t* structCurrentPositionJointCoords
                                                              )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    optionalParam = SysAllocString( L"" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    cmdCurPos = SysAllocString( L"CurJnt" );
    /*Execute get current coordinates of the joint axis */
    hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

    /* Free allocated strings from the memory */
    SysFreeString(cmdCurPos);
    SysFreeString(optionalParam);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Save the intern current position coordinates to the array of current positions */
    structCurrentPositionJointCoords->dwPositionJ1 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[0]);

    structCurrentPositionJointCoords->dwPositionJ2 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[1]);

    structCurrentPositionJointCoords->dwPositionJ3 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[2]);

    structCurrentPositionJointCoords->dwPositionJ4 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[3]);

    structCurrentPositionJointCoords->dwPositionJ5 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[4]);

    structCurrentPositionJointCoords->dwPositionJ6 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[5]);

    structCurrentPositionJointCoords->dwPositionJ7 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[6]);

    structCurrentPositionJointCoords->dwPositionJ8 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[7]);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_ResetOffset(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                      int bAxisType
                                                     )
{
    /* Set new values as the offfset to the dataparam axis type structs */
    if(bAxisType == 0) /* XY axis */
    {
        pDataParams->structTargetXY.offsetCoords.dwPositionX  = 0.0;
        pDataParams->structTargetXY.offsetCoords.dwPositionY  = 0.0;
        pDataParams->structTargetXY.offsetCoords.dwPositionZ  = 0.0;
        pDataParams->structTargetXY.offsetCoords.dwPositionRx = 0.0;
        pDataParams->structTargetXY.offsetCoords.dwPositionRy = 0.0;
        pDataParams->structTargetXY.offsetCoords.dwPositionRz = 0.0;
        pDataParams->structTargetXY.offsetCoords.wFigure      = 0.0;
    }
    else if(bAxisType == 1) /* Joint axis */
    {
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ1 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ2 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ3 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ4 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ5 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ6 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ7 = 0.0;
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ8 = 0.0;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_External_ResetCurrentCooridnates(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                      int bAxisType
                                                     )
{
    /* Set new values as the offfset to the dataparam axis type structs */
    if(bAxisType == 0) /* XY axis */
    {
        pDataParams->structTargetXY.dwPositionX  = 0;
        pDataParams->structTargetXY.dwPositionY  = 0;
        pDataParams->structTargetXY.dwPositionZ  = 0;
        pDataParams->structTargetXY.dwPositionRx = 0;
        pDataParams->structTargetXY.dwPositionRy = 0;
        pDataParams->structTargetXY.dwPositionRz = 0;
        pDataParams->structTargetXY.wFigure      = 0;
    }
    else if(bAxisType == 1) /* Joint axis */
    {
        pDataParams->structTargetJoint.dwPositionJ1 = 0;
        pDataParams->structTargetJoint.dwPositionJ2 = 0;
        pDataParams->structTargetJoint.dwPositionJ3 = 0;
        pDataParams->structTargetJoint.dwPositionJ4 = 0;
        pDataParams->structTargetJoint.dwPositionJ5 = 0;
        pDataParams->structTargetJoint.dwPositionJ6 = 0;
        pDataParams->structTargetJoint.dwPositionJ7 = 0;
        pDataParams->structTargetJoint.dwPositionJ8 = 0;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SetTool(
                                                  phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                  uint16_t wtoolNumber
                                                  )
{
    if(wtoolNumber < 0 || wtoolNumber > 63)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR cmdChange;
    wchar_t * cmd = new wchar_t[10];
    swprintf(cmd, L"Tool%d", wtoolNumber);
    cmdChange = SysAllocString( cmd );

    /* Execute get current coordinates of the XY axis */
    hr = pArm->Change(cmdChange);
    SysFreeString(cmdChange);

    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_SetToolDef(
                                                     phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                     float64_t x,
                                                     float64_t y,
                                                     float64_t z,
                                                     float64_t rx,
                                                     float64_t ry,
                                                     float64_t rz
                                                     )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Execute add path point */
    BSTR cmdSetToolDef;
    cmdSetToolDef = SysAllocString( L"SetToolDef" );

    /* Path number variable */
    VARIANT varToolDefNumber;
    VariantInit(&varToolDefNumber);
    varToolDefNumber.vt = VT_I2;
    varToolDefNumber.intVal = 1;

    /* Create string for the position */
    wchar_t* wStructCommand = new wchar_t[100];
    swprintf(wStructCommand, L"P(%f,%f,%f,%f,%f,%f)", x, y, z, rx, ry, rz);

    BSTR positionStr;
    positionStr = SysAllocString(wStructCommand); /* L"@0 P(350,0,450,180,-0.5,0,9)" ); */

    VARIANT positionVar;
    VariantInit(&positionVar);
    positionVar.bstrVal = positionStr;
    positionVar.vt = VT_BSTR;

    /* Add two VARIANT objects to the safearray pointer */
    VARIANT varCommandArray;
    VariantInit(&varCommandArray);

    VARIANT returnCommandArray;
    VariantInit(&returnCommandArray);

    varCommandArray.vt = VT_ARRAY | VT_VARIANT;
    varCommandArray.parray = SafeArrayCreateVector(VT_VARIANT, 0L, 2L);

    long elemFirst = 0L;
    long elemSecond = 1L;
    SafeArrayPutElement(varCommandArray.parray, &elemFirst, &varToolDefNumber);
    SafeArrayPutElement(varCommandArray.parray, &elemSecond, &positionVar);

    /* Execute */
    hr = pArm->Execute(cmdSetToolDef,varCommandArray, &returnCommandArray);
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

/*------------------------------------------------------------------------*/
/*--------------------------- GETTER FUNCTIONS----------------------------*/
/*------------------------------------------------------------------------*/
phStatus_t phdlStepper_DensoVS60_External_GetCurrentJointPositionWithoutOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                              phdlStepper_DensoVS60_DataParams_Joint_coord_t* structCurrentPositionJointCoords
                                                              )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    optionalParam = SysAllocString( L"" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    cmdCurPos = SysAllocString( L"CurJnt" );
    /* Get current coordinates of the XY axis */
    hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

    /* Free allocated strings from the memory */
    SysFreeString(cmdCurPos);
    SysFreeString(optionalParam);
    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Save the intern current position coordinates to the array of current positions */
    structCurrentPositionJointCoords->dwPositionJ1 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[0]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ1);

    structCurrentPositionJointCoords->dwPositionJ2 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[1]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ2);

    structCurrentPositionJointCoords->dwPositionJ3 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[2]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ3);

    structCurrentPositionJointCoords->dwPositionJ4 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[3]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ4);

    structCurrentPositionJointCoords->dwPositionJ5 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[4]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ5);

    structCurrentPositionJointCoords->dwPositionJ6 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[5]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ6);

    structCurrentPositionJointCoords->dwPositionJ7 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[6]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ7);

    structCurrentPositionJointCoords->dwPositionJ8 = phdlStepper_DensoVS60_External_RoundDoubleValue(((double* )currentPos.parray->pvData)[7]
                                                     - pDataParams->structTargetJoint.offsetCoords.dwPositionJ8);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                              phdlStepper_DensoVS60_DataParams_XY_coord_t* structCurrentPositionXYCoords
                                                              )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    optionalParam = SysAllocString( L"" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    cmdCurPos = SysAllocString( L"CurPosEx" );

    /* Execute an allocation of the current position with XY coordinates
       in the appropriated array */
    hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

    /* Free allocated strings from the memory */
    SysFreeString(cmdCurPos);
    SysFreeString(optionalParam);

    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }

    /* Save the intern current position coordinates to the array of current positions without offset */
    structCurrentPositionXYCoords->dwPositionX  =  phdlStepper_DensoVS60_External_RoundDoubleValue(((double *)currentPos.parray->pvData)[1]
                                                  - pDataParams->structTargetXY.offsetCoords.dwPositionX);

    structCurrentPositionXYCoords->dwPositionY  =  phdlStepper_DensoVS60_External_RoundDoubleValue(((double *)currentPos.parray->pvData)[2]
                                                  - pDataParams->structTargetXY.offsetCoords.dwPositionY);

    structCurrentPositionXYCoords->dwPositionZ  = phdlStepper_DensoVS60_External_RoundDoubleValue(((double *)currentPos.parray->pvData)[3]
                                                  - pDataParams->structTargetXY.offsetCoords.dwPositionZ);

    structCurrentPositionXYCoords->dwPositionRx = phdlStepper_DensoVS60_External_RoundDoubleValue(((0.01)*((double *)currentPos.parray->pvData)[4])
                                                  - (0.01)*(pDataParams->structTargetXY.offsetCoords.dwPositionRx));

    structCurrentPositionXYCoords->dwPositionRy = phdlStepper_DensoVS60_External_RoundDoubleValue(((0.01)*((double *)currentPos.parray->pvData)[5])
                                                  - (0.01)*(pDataParams->structTargetXY.offsetCoords.dwPositionRy));

    structCurrentPositionXYCoords->dwPositionRz = phdlStepper_DensoVS60_External_RoundDoubleValue(((0.01)*((double *)currentPos.parray->pvData)[6])
                                                  - (0.01)*(pDataParams->structTargetXY.offsetCoords.dwPositionRz));

    structCurrentPositionXYCoords->wFigure = (int16_t)phdlStepper_DensoVS60_External_RoundDoubleValue(((0.01)*((double *)currentPos.parray->pvData)[7])
                                                   - (0.01)*(pDataParams->structTargetXY.offsetCoords.wFigure));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                              int bAxisType,
                                                              int bAxisNumber,
                                                              double* dwAbsoluteCurrentCoodinate
                                                              )
{
    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize parameters */
    BSTR optionalParam;
    BSTR cmdCurPos;

    optionalParam = SysAllocString( L"" );

    VARIANT returnCurPos;
    VariantInit( &returnCurPos );
    returnCurPos.vt = VT_EMPTY;

    VARIANT currentPos;
    VariantInit( &currentPos );
    currentPos.vt = VT_R8|VT_ARRAY;

    /* Current value of XY axis coordinates */
    if(bAxisType == 0)
    {
        cmdCurPos = SysAllocString( L"CurPosEx" );

        /* Execute an allocation of the current position with XY coordinates
           in the appropriated array */
        hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

        /* Free allocated strings from the memory */
        SysFreeString(cmdCurPos);
        SysFreeString(optionalParam);
        /* Error check */
        if (FAILED(hr))
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
        }

        switch(bAxisNumber)
        {
        case 1:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[1];
            break;
        case 2:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[2];
            break;
        case 3:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[3];
            break;
        case 4:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[4];
            break;
        case 5:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[5];
            break;
        case 6:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[6];
            break;
        case 7:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[7];
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
        }
    }
    else    /* Current value of Joint axis coordinates */
    {
        cmdCurPos = SysAllocString( L"CurJnt" );

        /* Execute an allocation of the current position with Joint coordinates
           in the appropriated array */
        hr = pArm->Execute(cmdCurPos,returnCurPos,&currentPos);

        /* Free allocated strings from the memory */
        SysFreeString(cmdCurPos);
        SysFreeString(optionalParam);

        /* Error check */
        if (FAILED(hr))
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
        }

        switch(bAxisNumber)
        {
        case 1:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[0];
            break;
        case 2:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[1];
            break;
        case 3:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[2];
            break;
        case 4:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[3];
            break;
        case 5:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[4];
            break;
        case 6:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[5];
            break;
        case 7:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[6];
            break;
        case 8:
            *dwAbsoluteCurrentCoodinate = ((double *)currentPos.parray->pvData)[7];
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

/*------------------------------------------------------------------------*/
/*--------------------------- HELP FUNCTIONS  ----------------------------*/
/*------------------------------------------------------------------------*/
double phdlStepper_DensoVS60_External_NormalizeTargetedValue(
                                                             double dwCurrentValue,
                                                             double * pTargetValue
                                                            )
{

    /* Normalize current value if negative */
    double dwTempTagetValue = *pTargetValue;
    if( dwTempTagetValue > 360 || dwTempTagetValue < -360)
    {
        dwTempTagetValue = (int)dwTempTagetValue % 180;
        if(dwTempTagetValue < 0)
        {
            dwTempTagetValue+=360;
        }
    }

    if(dwTempTagetValue < 0)
    {
        dwTempTagetValue+=360;
    }
    if(dwCurrentValue < 0)
    {
        dwCurrentValue += 360;
    }
    pTargetValue = &dwTempTagetValue;

    /* Determine angle disnace between current and  target value in order to find in which
       direction should the hand rotate (clockwise oder counterclockwise) */
    double dwDistance = dwTempTagetValue - dwCurrentValue;
    return dwDistance;
}

double*  phdlStepper_DensoVS60_External_ArrayOfInterimAngles(
                                                             double dwDistance
                                                            )
{
    /* Find the number of interim values between two angles if the distnace is bigger than
       180 degree or smaller than -180 degree */
    if(dwDistance < 0)
    {
        dwDistance*=-1;
    }

    int bDistance = (int)dwDistance;
    int bRestOfDistnace = bDistance%90;
    int bCount = bDistance - bRestOfDistnace;
    int bSize = bCount/90;
    double* pInterimDistance = new double[bSize+1];
    pInterimDistance[0] = bSize;
    int counter = 1;
    while (bDistance > 0)
    {
        if (bDistance - 90 >= 0)
        {
            double dwAngleValue = bDistance - 90;
            pInterimDistance[counter] = dwAngleValue;
            bDistance -= 90;
            counter++;
        }
        else
        {
            bDistance -= 90;
        }

    }
    return pInterimDistance;
}

phStatus_t phdlStepper_DensoVS60_External_InitializeCoordsStructures(
                                                                    phdlStepper_DensoVS60_DataParams_XY_coord_t* structXYCoords,
                                                                    phdlStepper_DensoVS60_DataParams_Joint_coord_t* structJointCoords
                                                                    )
{
    /* Initialize vales for the XY struct */
    structXYCoords->dwPositionX  = 0;
    structXYCoords->dwPositionY  = 0;
    structXYCoords->dwPositionZ  = 0;
    structXYCoords->dwPositionRx = 0;
    structXYCoords->dwPositionRy = 0;
    structXYCoords->dwPositionRz = 0;
    structXYCoords->wFigure      = 0;

    /* Initialize vales for the Joint struct */
    structJointCoords->dwPositionJ1 = 0;
    structJointCoords->dwPositionJ2 = 0;
    structJointCoords->dwPositionJ3 = 0;
    structJointCoords->dwPositionJ4 = 0;
    structJointCoords->dwPositionJ5 = 0;
    structJointCoords->dwPositionJ6 = 0;
    structJointCoords->dwPositionJ7 = 0;
    structJointCoords->dwPositionJ8 = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_parseErrorCode(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                         HRESULT hr
                                                        )
{

    /* Initialize status variables */
    phStatus_t statusTmp;

    unsigned int bErrorCode = (unsigned int)hr;
    char* pHexString = new char[7];
    itoa(bErrorCode, pHexString, 16);
    int dErrorCode = atoi(pHexString);
    int8_t bSkipRelease = 0;
    if(dErrorCode > 82204010 || dErrorCode < 83205098)
    {
        bSkipRelease = 1;
    }


    if(bSkipRelease == 1)
    {
        #ifdef _DEBUG
          /* Only use the logger when in debug mode */
          phdlStepper_DensoVS60_External_LogErrorCodes(hr);
        #endif
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ClearErrorCode(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_StartMotor(pDataParams));
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

   return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_External_ClearErrorCode(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams
                                                        )
{
    /* Initialize robot controller */
    ICaoController* pCtrl = NULL;
    pCtrl = (ICaoController*)pDataParams->pContoller;

    /* Initialize status variables */
    HRESULT hr = S_OK;
    phStatus_t statusTmp;

    /* Function variables */
    BSTR cmdClearError;
    VARIANT vntClearError, vntErrorReturn;

    /* Initialize variables */
    VariantInit(&vntErrorReturn);
    VariantInit(&vntClearError);
    cmdClearError = SysAllocString( L"ClearError" );
    vntClearError.vt =  VT_EMPTY;

    /* Execute clear error command */
    hr = pCtrl->Execute(cmdClearError, vntClearError, &vntErrorReturn);

    /* Free allocated strings from the memory */
    SysFreeString(cmdClearError);

    /* Error check */
    if (FAILED(hr))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_EndProc(pDataParams, hr));
    }
    /* Wait for a half a second before continue */
    phTools_Sleep(500);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

int32_t phdlStepper_DensoVS60_External_RoundDoubleValue(
                                                        double dwValue
                                                       )
{
    int32_t roundedValue = 0;

    if(dwValue >= 0.0)
    {
        dwValue = dwValue*1000+0.5;
    }
    else
    {
        dwValue = dwValue*1000-0.5;
    }

    roundedValue = (int)dwValue;

    return roundedValue;
}

phStatus_t phdlStepper_DensoVS60_External_CheckIfPointInsideBase(
                                                                double dwXcoord,
                                                                double dwYcoord,
                                                                int wInnerLimit
                                                                )
{

    if(pow(dwXcoord,2) + pow(dwYcoord,2) > pow((double)wInnerLimit,2))
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER,PH_COMP_DL_STEPPER);
}
/*------------------------------------------------------------------------*/
/*------------------------- UNDER CONSTRUCTION ---------------------------*/
/*------------------------------------------------------------------------*/
#ifdef _DEBUG
void phdlStepper_DensoVS60_External_LogErrorCodes(
                                                  HRESULT hr
                                                 )
{
    int dErrorCode = 0;
    if(hr != S_OK)
    {
        /* Parse error code */
        unsigned int bErrorCode = (unsigned int)hr;
        char* pHexString = new char[7];
        itoa(bErrorCode, pHexString, 16);
        dErrorCode = atoi(pHexString);
    }

    /* Time stamp array */
    char buff[DTTMSZ];

    FILE * fh = NULL;
    fh = fopen("DensoVS60.log","a");
    fprintf(fh," Error Code: %d  Time: %s\n", dErrorCode , getDtTm (buff));
    fclose(fh);
}
#endif

phStatus_t phdlStepper_DensoVS60_External_CheckOutOfRange(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                      wchar_t* wStructCommand
                                                     )
{

    /* ICaoRobot Object initialization */
    ICaoRobot* pArm = NULL;
    pArm = (ICaoRobot*)pDataParams->pArm;

    /* Initialize status variables */
    phStatus_t statusTmp;
    HRESULT hr = S_OK;

    /* Target path cooridnates string */
    BSTR pathStr;
    pathStr = SysAllocString(wStructCommand);//L"@0 P(350,0,450,180,-0.5,0,9)" );

    /* Set target path for checking if out of range */
    VARIANT pathVar;
    ::VariantInit(&pathVar);
    pathVar.bstrVal = pathStr;
    pathVar.vt = VT_BSTR;

    /* Return value for the out of range command*/
    VARIANT returnOutRange;
    ::VariantInit(&returnOutRange);

    /* Initialize parameter for blocking move command */
    BSTR cmdOutRange;
    cmdOutRange = SysAllocString( L"OutRange");

    /* Execute out of range check */
    hr = pArm->Execute(cmdOutRange,pathVar, &returnOutRange);
    SysFreeString(cmdOutRange);

    /* Error check if error dont turn off the motor */
    if(hr != S_OK)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_parseErrorCode( pDataParams,hr));
    }

    /* Get the return value if the target position is out of range */
    long isOutOfRange = returnOutRange.lVal;
    if(isOutOfRange != 0)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


/*-------------------------END OF TEST FUNCTIONS--------------------------*/
#pragma warning(pop)
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */
