/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>


#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60

#include "phdlStepper_DensoVS60.h"
#include "phdlStepper_DensoVS60_Int.h"
#include "../phdlStepper_Int.h"

#ifndef _WIN32
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>              /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */
#endif
#pragma warning(push)
#pragma warning( disable : 4189 )

#include "external\CAO.h"




phStatus_t phdlStepper_DensoVS60_Int_ConvertToStep(
                                        phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                        int32_t dwDistance,
                                        int32_t * dwSteps
                                        )
{
    double unit = dwDistance * ((double)pDataParams->wStepsWayRatio / 1000.0);


    if( unit < 0.0 )
    {
        if (unit < -0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwSteps = -(int32_t)((-unit)+0.5);
    }
    else
    {
        if (unit > 0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwSteps = (int32_t)((unit)+0.5);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Int_ConvertFromStep(
                                        phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                        int32_t dwSteps,
                                        int32_t * dwDistance
                                        )
{
    double unit = dwSteps * (1000.0 / (double)pDataParams->wStepsWayRatio);
    if( unit < 0.0 )
    {
        if (unit < -0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwDistance = -(int32_t)((-unit)+0.5);
    }
    else
    {
        if (unit > 0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwDistance = (int32_t)((unit)+0.5);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Int_GoToXYAxis(phdlStepper_DensoVS60_DataParams_t * pDataParams, uint8_t bBlocking)
{
    /* Check if the axis type is joint or X-Y and then return the value*/
    return phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking);
}

phStatus_t phdlStepper_DensoVS60_Int_GoToJointAxis(phdlStepper_DensoVS60_DataParams_t * pDataParams, uint8_t bBlocking)
{
    /* Check if the axis type is joint or X-Y and then return the value*/
    return phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking);
}

phStatus_t phdlStepper_DensoVS60_Int_GoToParameterXY(
                                                     phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                     uint8_t bBlocking
                                                    )
{

    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* Create array for the the target position by converting them
       from uint_32t micrometer to double milimeter */
    double targetPosition[7];
    targetPosition[0] = ((double)pDataParams->structTargetXY.dwPositionX)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionX;
    targetPosition[1] = ((double)pDataParams->structTargetXY.dwPositionY)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionY;
    targetPosition[2] = ((double)pDataParams->structTargetXY.dwPositionZ)/1000 + pDataParams->structTargetXY.offsetCoords.dwPositionZ;
    targetPosition[3] = ((double)pDataParams->structTargetXY.dwPositionRx)/10  + pDataParams->structTargetXY.offsetCoords.dwPositionRx;
    targetPosition[4] = ((double)pDataParams->structTargetXY.dwPositionRy)/10  + pDataParams->structTargetXY.offsetCoords.dwPositionRy;
    targetPosition[5] = ((double)pDataParams->structTargetXY.dwPositionRz)/10  + pDataParams->structTargetXY.offsetCoords.dwPositionRz;
    targetPosition[6] = (double)pDataParams->structTargetXY.wFigure/10  + pDataParams->structTargetXY.offsetCoords.wFigure;

    /* Check the inner radius limits */
    PH_CHECK_SUCCESS_FCT(statusTmp,
        phdlStepper_DensoVS60_External_CheckIfPointInsideBase(targetPosition[0]*1000,targetPosition[1]*1000,PHDL_STEPPER_DENSO_VS60_LIMIT_INNER_RADIUS));

    /* Check the limits For all axis the maximum value is
       if the zero point was set at the oposide side */
    if ( targetPosition[0]*1000 >  PHDL_STEPPER_DENSO_VS60_LIMIT_X   ||
         targetPosition[0]*1000 < -PHDL_STEPPER_DENSO_VS60_LIMIT_X   ||
         targetPosition[1]*1000 >  PHDL_STEPPER_DENSO_VS60_LIMIT_Y   ||
         targetPosition[1]*1000 < -PHDL_STEPPER_DENSO_VS60_LIMIT_Y   ||
         targetPosition[2]*1000 >  PHDL_STEPPER_DENSO_VS60_LIMIT_Z   ||
         targetPosition[2]*1000 < -PHDL_STEPPER_DENSO_VS60_LIMIT_Z   ||
         targetPosition[3]*10   >  PHDL_STEPPER_DENSO_VS60_LIMIT_RX  ||
         targetPosition[3]*10   < -PHDL_STEPPER_DENSO_VS60_LIMIT_RX  ||
         targetPosition[4]*10   >  PHDL_STEPPER_DENSO_VS60_LIMIT_RY  ||
         targetPosition[4]*10   < -PHDL_STEPPER_DENSO_VS60_LIMIT_RY  ||
         targetPosition[5]*10   >  PHDL_STEPPER_DENSO_VS60_LIMIT_RZ  ||
         targetPosition[5]*10   < -PHDL_STEPPER_DENSO_VS60_LIMIT_RZ  ||
         targetPosition[6]*10   >  PHDL_STEPPER_DENSO_VS60_LIMIT_FIG ||
         targetPosition[6]*10   < -PHDL_STEPPER_DENSO_VS60_LIMIT_FIG)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp,phdlStepper_DensoVS60_External_ResetCurrentCooridnates( pDataParams,0));
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    PH_ASSERT_NULL (pDataParams);

    /* Go to targeted position */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlStepper_DensoVS60_External_MoveToTargetCoordinates(pDataParams, targetPosition, 0, bBlocking));

    /* It cannot be blocked, check if the movement is done */
    if( bBlocking == PH_ON )
    {

        /* This check is needed because of the threading problem in case of higher robot speed
           The robot will not achieve the needed precison to go further */
        statusTmp = phdlStepper_DensoVS60_Int_WaitMoveFinished(pDataParams);
        if ( (statusTmp & PH_ERR_MASK) == PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR)
        {
            statusTmp = PH_ERR_SUCCESS;
        }
        PH_CHECK_SUCCESS(statusTmp);

        /* Create structure to hold the current coodirnates of the X-Y axis*/
        phdlStepper_DensoVS60_DataParams_XY_coord_t currentCoordsXY;

        /* Ok we should have reached the position so check if it is ok and also store the reached position as sct position */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(pDataParams,&currentCoordsXY));

        if ((pDataParams->structTargetXY.dwPositionX  - currentCoordsXY.dwPositionX ) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_X  ||
            (pDataParams->structTargetXY.dwPositionX -  currentCoordsXY.dwPositionX ) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_X  ||
            (pDataParams->structTargetXY.dwPositionY  - currentCoordsXY.dwPositionY ) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_Y  ||
            (pDataParams->structTargetXY.dwPositionY -  currentCoordsXY.dwPositionY ) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_Y  ||
            (pDataParams->structTargetXY.dwPositionZ  - currentCoordsXY.dwPositionZ ) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_Z  ||
            (pDataParams->structTargetXY.dwPositionZ -  currentCoordsXY.dwPositionZ ) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_Z  ||
            (pDataParams->structTargetXY.dwPositionRx - currentCoordsXY.dwPositionRx) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_RX ||
            (pDataParams->structTargetXY.dwPositionRx - currentCoordsXY.dwPositionRx) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_RX ||
            (pDataParams->structTargetXY.dwPositionRy - currentCoordsXY.dwPositionRy) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_RY ||
            (pDataParams->structTargetXY.dwPositionRy - currentCoordsXY.dwPositionRy) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_RY ||
            (pDataParams->structTargetXY.dwPositionRz - currentCoordsXY.dwPositionRz) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_RZ ||
            (pDataParams->structTargetXY.dwPositionRz - currentCoordsXY.dwPositionRz) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_RZ)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_Int_GoToParameterJoint(
                                                        phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                        uint8_t bBlocking
                                                       )
{

    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* Check the limits */
    /* For all axis the maximum value is if the zero point was set at the oposide side */
    if ( pDataParams->structTargetJoint.dwPositionJ1 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J1 ||
         pDataParams->structTargetJoint.dwPositionJ1 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J1 ||
         pDataParams->structTargetJoint.dwPositionJ2 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J2 ||
         pDataParams->structTargetJoint.dwPositionJ2 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J2 ||
         pDataParams->structTargetJoint.dwPositionJ3 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J3 ||
         pDataParams->structTargetJoint.dwPositionJ3 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J3 ||
         pDataParams->structTargetJoint.dwPositionJ4 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J4 ||
         pDataParams->structTargetJoint.dwPositionJ4 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J4 ||
         pDataParams->structTargetJoint.dwPositionJ5 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J5 ||
         pDataParams->structTargetJoint.dwPositionJ5 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J5 ||
         pDataParams->structTargetJoint.dwPositionJ6 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J6 ||
         pDataParams->structTargetJoint.dwPositionJ6 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J6 ||
         pDataParams->structTargetJoint.dwPositionJ7 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J7 ||
         pDataParams->structTargetJoint.dwPositionJ7 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J7 ||
         pDataParams->structTargetJoint.dwPositionJ8 >  PHDL_STEPPER_DENSO_VS60_LIMIT_J8 ||
         pDataParams->structTargetJoint.dwPositionJ8 < -PHDL_STEPPER_DENSO_VS60_LIMIT_J8)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    /* Create array for the the target position by converting them
    from uint_32t micrometer to double milimeter */
    double targetPosition[8];
    targetPosition[0] = ((double)pDataParams->structTargetJoint.dwPositionJ1) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ1;
    targetPosition[1] = ((double)pDataParams->structTargetJoint.dwPositionJ2) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ2;
    targetPosition[2] = ((double)pDataParams->structTargetJoint.dwPositionJ3) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ3;
    targetPosition[3] = ((double)pDataParams->structTargetJoint.dwPositionJ4) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ4;
    targetPosition[4] = ((double)pDataParams->structTargetJoint.dwPositionJ5) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ5;
    targetPosition[5] = ((double)pDataParams->structTargetJoint.dwPositionJ6) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ6;
    targetPosition[6] = ((double)pDataParams->structTargetJoint.dwPositionJ7) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ7;
    targetPosition[7] = ((double)pDataParams->structTargetJoint.dwPositionJ8) / 1000 + pDataParams->structTargetJoint.offsetCoords.dwPositionJ8;

    /* Go to targeted position, no need for the figure value, that is why the second param is zero */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlStepper_DensoVS60_External_MoveToTargetCoordinates(pDataParams, targetPosition, 1, bBlocking));

    PH_ASSERT_NULL (pDataParams);

    /* It can be blocked, so check when the move is finished */
    if( bBlocking == PH_ON )
    {

        /* This check is needed because of the threading problem in case of higher robot speed
        The robot will not achieve the needed precison to go further */
        statusTmp = phdlStepper_DensoVS60_Int_WaitMoveFinished(pDataParams);
        if ( (statusTmp & PH_ERR_MASK) == PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR)
        {
            statusTmp = PH_ERR_SUCCESS;
        }
        PH_CHECK_SUCCESS(statusTmp);

        /* Create structure to hold the current coodirnates of the Joint axis*/
        phdlStepper_DensoVS60_DataParams_Joint_coord_t currentCoordsJoint;

        /* Ok we should have reached the position so check if it is ok and also store the reached position as sct position */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentJointPositionWithoutOffset(pDataParams,&currentCoordsJoint));

        if ((pDataParams->structTargetJoint.dwPositionJ1 - currentCoordsJoint.dwPositionJ1) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J1 ||
            (pDataParams->structTargetJoint.dwPositionJ1 - currentCoordsJoint.dwPositionJ1) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J1 ||
            (pDataParams->structTargetJoint.dwPositionJ2 - currentCoordsJoint.dwPositionJ2) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J2 ||
            (pDataParams->structTargetJoint.dwPositionJ2 - currentCoordsJoint.dwPositionJ2) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J2 ||
            (pDataParams->structTargetJoint.dwPositionJ3 - currentCoordsJoint.dwPositionJ3) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J3 ||
            (pDataParams->structTargetJoint.dwPositionJ3 - currentCoordsJoint.dwPositionJ3) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J3 ||
            (pDataParams->structTargetJoint.dwPositionJ4 - currentCoordsJoint.dwPositionJ4) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J4 ||
            (pDataParams->structTargetJoint.dwPositionJ4 - currentCoordsJoint.dwPositionJ4) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J4 ||
            (pDataParams->structTargetJoint.dwPositionJ5 - currentCoordsJoint.dwPositionJ5) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J5 ||
            (pDataParams->structTargetJoint.dwPositionJ5 - currentCoordsJoint.dwPositionJ5) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J5 ||
            (pDataParams->structTargetJoint.dwPositionJ6 - currentCoordsJoint.dwPositionJ6) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J6 ||
            (pDataParams->structTargetJoint.dwPositionJ6 - currentCoordsJoint.dwPositionJ6) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J6 ||
            (pDataParams->structTargetJoint.dwPositionJ7 - currentCoordsJoint.dwPositionJ7) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J7 ||
            (pDataParams->structTargetJoint.dwPositionJ7 - currentCoordsJoint.dwPositionJ7) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J7 ||
            (pDataParams->structTargetJoint.dwPositionJ8 - currentCoordsJoint.dwPositionJ8) >  PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J8 ||
            (pDataParams->structTargetJoint.dwPositionJ8 - currentCoordsJoint.dwPositionJ8) < -PHDL_STEPPER_DENSO_VS60_TOLERANCE_LIMIT_J8
            )
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Int_GetPositionXY(
                                                   phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                   phdlStepper_DensoVS60_DataParams_XY_coord_t* currentCoordsXY
                                                  )
{

    phStatus_t statusTmp;

    /*Update current position of the robot */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(pDataParams, currentCoordsXY));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Int_GetPositionJoint(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                                      phdlStepper_DensoVS60_DataParams_Joint_coord_t* currentCoordsJoint
                                                     )
{
    phStatus_t statusTmp;

    /*Update current position of the robot for joint axis */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_GetCurrentJointPositionWithoutOffset(pDataParams, currentCoordsJoint));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Int_SetAsZero(phdlStepper_DensoVS60_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;

    /* Set the current coordinates as zero position with offset in the extern array */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams,0));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams,1));

    /* Reset the values of the current coordinates */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ResetCurrentCooridnates(pDataParams,0));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_ResetCurrentCooridnates(pDataParams,1));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Int_ManualMode(phdlStepper_DensoVS60_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;

    /* Disconnect the software from the robot, give controll to the remote controllers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_DisconnectController(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_Int_WaitMoveFinished(phdlStepper_DensoVS60_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    uint8_t bMotionCompleted;

    do
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_External_CheckMotionComplete(pDataParams, &bMotionCompleted));
    }while( bMotionCompleted != PH_ON );

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#pragma warning(pop)
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */
