/*
 * Copyright 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper HighZ Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ

#include <phdlStepper.h>
#include "phdlStepper_HighZ.h"
#include "phdlStepper_HighZ_Int.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_HighZ_Init(
                               phdlStepper_HighZ_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams,
                               void * pBalRegDataParams
                               )
{
    uint8_t bAxis;
    if (sizeof(phdlStepper_HighZ_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_STEPPER | PH_COMP_DL_STEPPER_HIGHZ_ID;
    pDataParams->pBalRegDataParams      = pBalRegDataParams;
    pDataParams->bInit                  = PH_OFF;
    pDataParams->bPowerIsOn             = PH_OFF;
    for (bAxis = PHDL_STEPPER_HIGHZ_INT_COORDINATE_X; bAxis <= PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z; bAxis++)
    {
        pDataParams->wMicrometerPerTurn[bAxis] = 6000;
        pDataParams->wStepsPerTurn[bAxis] = 2000;
    }

    pDataParams->dwMaxRangeX = 975000;
    pDataParams->dwMaxRangeY = 460000;
    pDataParams->dwMaxRangeZ = 190000;

    pDataParams->wRampDelayMin = 5;
    pDataParams->wRampDelayMax = 50;
    pDataParams->wRampSpeed = 10;
    pDataParams->bRampEnabled = 1;

    pDataParams->dwPaddingX = 25000;
    pDataParams->dwPaddingY = 25000;
    pDataParams->dwPaddingZ = 25000;

    pDataParams->dwDriveTimeout = PHDL_STEPPER_HIGHZ_INT_DRIVE_TIMEOUT;

    pDataParams->bPositionMode          = PHDL_STEPPER_HIGHZ_POS_MODE_REMOTE;
    pDataParams->bActiveAxis            = PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Z;
    pDataParams->dwPositionX            = 0;
    pDataParams->dwPositionY            = 0;
    pDataParams->dwPositionZ            = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_HighZ_Initialize(
                                      phdlStepper_HighZ_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM status;
    phdlStepper_HighZ_HighZStatus_t highZStatus;
    int32_t x,y,z;
    uint8_t bOutOfBounds;

    PH_ASSERT_NULL (pDataParams);

    /* calibrate zero point and
        move robot to relative zero position according to padding in cncSpecData xml file */

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOn(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadStatus(pDataParams, &highZStatus));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &x, &y, &z));
    if (highZStatus.items.em_stop != PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    if ((x == 0 && y == 0 && z == 0) || highZStatus.items.end_x || highZStatus.items.end_y || highZStatus.items.end_z)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToZeroPositionFast(pDataParams));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CheckOutOfBounds(pDataParams, &bOutOfBounds));
        if (bOutOfBounds == PH_OFF)
        {
            /* If position is correct use faster mode */
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToZeroPositionSafe(pDataParams));
        }
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Calibrate(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToZeroPositionRelative(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOff(pDataParams));

    pDataParams->bInit = PH_ON;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_MoveDistance(
                                       phdlStepper_HighZ_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint8_t bDirection,
                                       uint32_t dwDistance,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwActX;
    int32_t    PH_MEMLOC_REM dwActY;
    int32_t    PH_MEMLOC_REM dwActZ;
    int32_t    PH_MEMLOC_REM dwCalcEndPos;
    int32_t    PH_MEMLOC_REM dwSignedDistance;
    int32_t    PH_MEMLOC_REM dwActPosLimitLow;
    int32_t    PH_MEMLOC_REM dwActPosLimitHigh;

    PH_ASSERT_NULL (pDataParams);
    if( bDirection != PHDL_STEPPER_DIR_DOWN && bDirection != PHDL_STEPPER_DIR_UP )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* limit the distance to the max available number / 2 */
    if (dwDistance > 0x7FFFFFFF)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    /* Only blocking is supported because the buffer needs also to be cleared during movement
     * and one step is limited to 64k engine steps, nonblocking can be done with a thread outside */
    if (bBlocking != PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_SetSpeed(pDataParams, wSpeed));

    if (bDirection == PHDL_STEPPER_DIR_DOWN)
    {
        dwSignedDistance = (int32_t)(0-dwDistance);
        dwActPosLimitLow = -0x7FFFFFFF - dwSignedDistance;
        dwActPosLimitHigh = 0x7FFFFFFF;
    }
    else
    {
        dwSignedDistance = dwDistance;
        dwActPosLimitLow = -0x7FFFFFFF;
        dwActPosLimitHigh = 0x7FFFFFFF - dwSignedDistance;
    }

    /* Get the actual position */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &dwActX, &dwActY, &dwActZ));

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_X:
        /* calculate the end position */
        if (dwActX < dwActPosLimitLow || dwActX > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        dwCalcEndPos = dwActX + dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToAbsolutePosition(pDataParams, dwCalcEndPos, dwActY, dwActZ));
        break;
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Y:
        /* calculate the end position */
        if (dwActY < dwActPosLimitLow || dwActY > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        dwCalcEndPos = dwActY + dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToAbsolutePosition(pDataParams, dwActX, dwCalcEndPos, dwActZ));
        break;
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Z:
        /* calculate the end position */
        if (dwActZ < dwActPosLimitLow || dwActZ > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        dwCalcEndPos = dwActZ + dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToAbsolutePosition(pDataParams, dwActX, dwActY, dwCalcEndPos));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_MoveSteps(
                                    phdlStepper_HighZ_DataParams_t * pDataParams,
                                    uint16_t wSpeed,
                                    uint8_t bDirection,
                                    uint32_t dwSteps,
                                    uint8_t bBlocking
                                    )
{
    int32_t    PH_MEMLOC_REM dwDistance;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* limit the distance to the max available number / 2 */
    if (dwSteps > 0x7FFFFFFF)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_X:
        dwDistance = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, (int32_t)dwSteps, PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
        break;
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Y:
        dwDistance = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, (int32_t)dwSteps, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
        break;
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Z:
        dwDistance = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, (int32_t)dwSteps, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    /* Use MoveDistance to perform task. */
    return phdlStepper_HighZ_MoveDistance(pDataParams, wSpeed, bDirection, dwDistance, bBlocking);
}

phStatus_t phdlStepper_HighZ_GoToPosition(
                                       phdlStepper_HighZ_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint32_t dwPosition,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwActX;
    int32_t    PH_MEMLOC_REM dwActY;
    int32_t    PH_MEMLOC_REM dwActZ;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Only blocking is supported because the buffer needs also to be cleared during movement
     * and one step is limited to 64k engine steps, nonblocking can be done with a thread outside */
    if (bBlocking != PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_SetSpeed(pDataParams, wSpeed));

    /* Get the actual position */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionRel(pDataParams, &dwActX, &dwActY, &dwActZ));

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_X:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, (int32_t)dwPosition, dwActY, dwActZ));
        break;
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Y:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, dwActX, (int32_t)dwPosition, dwActZ));
        break;
    case PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Z:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, dwActX, dwActY, (int32_t)dwPosition));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_SetConfig(
                                    phdlStepper_HighZ_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t wValue
                                    )
{
    phStatus_t  PH_MEMLOC_REM status;
    int32_t x, y, z;

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    /*case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        pDataParams->wStepsWayRatio = wValue;
        break;*/

    case PHDL_STEPPER_HIGHZ_CONFIG_POSITION_MODE:
        switch (wValue)
        {
        case PHDL_STEPPER_HIGHZ_POS_MODE_SET_CURRENT_AS_ZERO:
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WritePosition(pDataParams, 0, 0, 0));
            break;
        case PHDL_STEPPER_HIGHZ_POS_MODE_CLEAR_CURRENT_ZERO:
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &x, &y, &z));
            /* Adjust padding and PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE */
            x -= pDataParams->dwPaddingX;
            y -= pDataParams->dwPaddingY;
            z -= pDataParams->dwPaddingZ;

            x += phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
            y += phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
            z += phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WritePosition(pDataParams, x, y, z));
            break;

        case PHDL_STEPPER_HIGHZ_POS_MODE_REMOTE:
            /* If we start this mode we get the current position because there may be a position change in local mode */
            pDataParams->bPositionMode = PHDL_STEPPER_HIGHZ_POS_MODE_REMOTE;
            break;
        case PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT:
            /* If we start this mode we get the current position because there may be a non blocking move before */
            if (pDataParams->bPositionMode != PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT)
            {
                PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionRel(pDataParams, (int32_t * UNALIGNED)&pDataParams->dwPositionX, (int32_t * UNALIGNED)&pDataParams->dwPositionY, (int32_t * UNALIGNED)&pDataParams->dwPositionZ));
            }
            pDataParams->bPositionMode = PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT;
            break;
        case PHDL_STEPPER_HIGHZ_POS_MODE_START:
            status = phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, pDataParams->dwPositionX, pDataParams->dwPositionY, pDataParams->dwPositionZ);
            if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS) || ((status & PH_ERR_MASK) == PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS))
            {
                pDataParams->bPositionMode = PHDL_STEPPER_HIGHZ_POS_MODE_REMOTE;
            }
            PH_CHECK_SUCCESS(status);
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_ACTIVE_AXIS:
        if( wValue != PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_X &&
            wValue != PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Y &&
            wValue != PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Z )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxis = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_RAMP_DELAY_MIN:
        if (wValue < 2 || wValue > 500)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wRampDelayMin = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_RAMP_DELAY_MAX:
        if (wValue < 2 || wValue > 500)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wRampDelayMax = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_ROBOT_SPEED:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_SetSpeed(pDataParams, wValue));
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_RAMP_SPEED:
        if (wValue < 1 || wValue > 1000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wRampSpeed = wValue;
        break;

    case PHDL_STEPPER_CONFIG_RAMP_ON_OFF:
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bRampEnabled = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_X:
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wMicrometerPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_X] = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_Y:
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wMicrometerPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y] = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_Z:
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wMicrometerPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z] = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_X:
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wStepsPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_X] = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_Y:
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wStepsPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y] = wValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_Z:
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wStepsPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z] = wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_SetConfig32(
                                      phdlStepper_HighZ_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t dwValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_X:
        pDataParams->dwPositionX = (int32_t)dwValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, pDataParams->dwPositionX, pDataParams->dwPositionY, pDataParams->dwPositionZ));
        }
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_Y:
        pDataParams->dwPositionY = (int32_t)dwValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, pDataParams->dwPositionX, pDataParams->dwPositionY, pDataParams->dwPositionZ));
        }
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_Z:
        pDataParams->dwPositionZ = (int32_t)dwValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_HighZ_Int_DriveToRelativePosition(pDataParams, pDataParams->dwPositionX, pDataParams->dwPositionY, pDataParams->dwPositionZ));
        }
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_MAX_X:
        if (dwValue < 100000 || dwValue > 1000000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwMaxRangeX = dwValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_MAX_Y:
        if (dwValue < 100000 || dwValue > 600000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwMaxRangeY = dwValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_MAX_Z:
        if (dwValue < 10000 || dwValue > 200000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwMaxRangeZ = dwValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_X:
        if (dwValue < 1000 || dwValue > 100000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwPaddingX = dwValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_Y:
        if (dwValue < 1000 || dwValue > 100000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwPaddingY = dwValue;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_Z:
        if (dwValue < 1000 || dwValue > 100000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwPaddingZ = dwValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_GetConfig(
                                    phdlStepper_HighZ_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;

    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_HIGHZ_CONFIG_POSITION_MODE:
        *pValue = pDataParams->bPositionMode;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_ACTIVE_AXIS:
        *pValue = pDataParams->bActiveAxis;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_RAMP_DELAY_MIN:
        *pValue = pDataParams->wRampDelayMin;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_RAMP_DELAY_MAX:
        *pValue = pDataParams->wRampDelayMax;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_ROBOT_SPEED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_HighZ_Int_GetSpeed(pDataParams, pValue));
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_RAMP_SPEED:
        *pValue = pDataParams->wRampSpeed;
        break;

    case PHDL_STEPPER_CONFIG_RAMP_ON_OFF:
        *pValue = pDataParams->bRampEnabled;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_X:
        *pValue = pDataParams->wMicrometerPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_X];
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_Y:
        *pValue = pDataParams->wMicrometerPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y];
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_Z:
        *pValue = pDataParams->wMicrometerPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z];
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_X:
        *pValue = pDataParams->wStepsPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_X];
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_Y:
        *pValue = pDataParams->wStepsPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y];
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_Z:
        *pValue = pDataParams->wStepsPerTurn[PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z];
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_GetConfig32(
                                      phdlStepper_HighZ_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t * pValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    int32_t    PH_MEMLOC_REM dwActX;
    int32_t    PH_MEMLOC_REM dwActY;
    int32_t    PH_MEMLOC_REM dwActZ;

    PH_ASSERT_NULL (pDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_HighZ_Int_ReadPositionRel(pDataParams, &dwActX, &dwActY, &dwActZ));

    switch(wConfig)
    {
    case PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_X:
        *pValue = (uint32_t)dwActX;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_Y:
        *pValue = (uint32_t)dwActY;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_Z:
        *pValue = (uint32_t)dwActZ;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_MAX_X:
        *pValue = pDataParams->dwMaxRangeX;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_MAX_Y:
        *pValue = pDataParams->dwMaxRangeY;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_MAX_Z:
        *pValue = pDataParams->dwMaxRangeZ;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_X:
        *pValue = pDataParams->dwPaddingX;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_Y:
        *pValue = pDataParams->dwPaddingY;
        break;

    case PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_Z:
        *pValue = pDataParams->dwPaddingZ;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */
