/*
 * Copyright 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ

#include "phdlStepper_HighZ.h"
#include "phdlStepper_HighZ_Int.h"
#include "../phdlStepper_Int.h"
#include "phbalReg_SerialWin_Cmd.h"

#ifndef _WIN32
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>              /* PRQA S 5124 */
#include <time.h>
#pragma warning(pop)            /* PRQA S 3116 */
#endif

#ifndef min
#define min(a,b)            (((a) < (b)) ? (a) : (b))
#endif

#ifndef max
#define max(a,b)            (((a) > (b)) ? (a) : (b))
#endif

uint8_t PHDL_STEPPER_HIGHZ_INT_COORDINATE_SEQUENCE[] = {
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_X,
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y,
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z,
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_X,
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_X,
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y,
    PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z};

phStatus_t phdlStepper_HighZ_Int_ReadStatus(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        phdlStepper_HighZ_HighZStatus_t *pbStatus)
{
    uint8_t response[1];
    phStatus_t status;

    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_READ_STATUS, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, response, sizeof(response), 1000));
    (*pbStatus).data = response[0];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_Exchange(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint8_t * pCommand,
                                        uint16_t wTxLength,
                                        uint8_t *pResponse,
                                        uint16_t wExpectedLength,
                                        uint16_t wTimeoutMs)
{
    phStatus_t status;
    phStatus_t statusExchange;
    uint16_t wOldTimeout;
    uint16_t wRxLenPart;
    uint16_t wRxLen = 0;

    /* Flush buffer */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SerialWin_Cmd_Flush((phbalReg_SerialWin_DataParams_t *)pDataParams->pBalRegDataParams));

    /* Set timeout */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_GetConfig(pDataParams->pBalRegDataParams,
        PHBAL_REG_CONFIG_READ_TIMEOUT_MS,
        &wOldTimeout));
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalRegDataParams,
        PHBAL_REG_CONFIG_READ_TIMEOUT_MS,
        wTimeoutMs));

    do
    {
        /* Exchange */
        statusExchange = phbalReg_Exchange(pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            pCommand,
            wTxLength,
            wExpectedLength - wRxLen,
            &pResponse[wRxLen],
            &wRxLenPart);

        if ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            wRxLen += wRxLenPart;
        }

        /* Ignore timeout in case of no data expected */
        if (wExpectedLength == 0 && (statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
        {
            statusExchange = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
        }


        /* Clear data */
        pCommand = NULL;
        wTxLength = 0;
    } while ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS && wRxLen < wExpectedLength);

    /* Restore timeout */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalRegDataParams,
        PHBAL_REG_CONFIG_READ_TIMEOUT_MS,
        wOldTimeout));

    return statusExchange;
}


phStatus_t phdlStepper_HighZ_Int_CreateMessage(
                                        uint8_t bCommand,
                                        uint8_t *pData,
                                        uint16_t wDataLength,
                                        uint8_t *pMessage,
                                        uint16_t wMessageSize,
                                        uint16_t *pwMessageLength)
{
    phStatus_t status;
    uint16_t wCrc = 0xFFFF;
    uint8_t bIndex = 0;

    if (wDataLength > wMessageSize - 7 || (wDataLength + 3) > 255)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_STEPPER);
    }

    *pwMessageLength = wDataLength + 7;

    pMessage[*pwMessageLength - 1] = PHDL_STEPPER_HIGHZ_INT_PREFIX_1;
    pMessage[*pwMessageLength - 2] = PHDL_STEPPER_HIGHZ_INT_PREFIX_2;
    pMessage[*pwMessageLength - 3] = PHDL_STEPPER_HIGHZ_INT_PREFIX_3;
    pMessage[*pwMessageLength - 4] = (uint8_t)(wDataLength + 3);
    pMessage[*pwMessageLength - 5] = bCommand;
    for (bIndex = 0; bIndex < wDataLength; bIndex++)
    {
        pMessage[*pwMessageLength - 6 - bIndex] = pData[bIndex];
    }

    PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc16(
        PH_TOOLS_CRC_OPTION_MSB_FIRST,
        wCrc,
        PHDL_STEPPER_HIGHZ_INT_CRC16_POLY,
        &bCommand,
        1,
        &wCrc));
    PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc16(
        PH_TOOLS_CRC_OPTION_MSB_FIRST,
        wCrc,
        PHDL_STEPPER_HIGHZ_INT_CRC16_POLY,
        pData,
        wDataLength,
        &wCrc));

    pMessage[1] = (uint8_t)(wCrc);
    pMessage[0] = (uint8_t)(wCrc >> 8);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

int32_t phdlStepper_HighZ_Int_ConvertMicrometerToSteps(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwMicrometer,
                                        uint8_t bAxis
                                        )
{
    return (int32_t)(((double)dwMicrometer * (double)pDataParams->wStepsPerTurn[bAxis] / (double)pDataParams->wMicrometerPerTurn[bAxis]) + 0.5);
}

int32_t phdlStepper_HighZ_Int_ConvertStepsToMicrometer(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwSteps,
                                        uint8_t bAxis
                                        )
{
    return (int32_t)(((double)dwSteps * (double)pDataParams->wMicrometerPerTurn[bAxis] / (double)pDataParams->wStepsPerTurn[bAxis]) + 0.5);
}

phStatus_t phdlStepper_HighZ_Int_PowerOn(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;

    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    if (pDataParams->bPowerIsOn == PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_POWER_ON, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

    phTools_Sleep(PHDL_STEPPER_HIGHZ_INT_POWER_WAIT);

    pDataParams->bPowerIsOn = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_PowerOff(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;

    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    if (pDataParams->bPowerIsOn == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_POWER_OFF, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

    phTools_Sleep(PHDL_STEPPER_HIGHZ_INT_POWER_WAIT);

    pDataParams->bPowerIsOn = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_ReadPosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t *pdwXPosAbs,
                                        int32_t *pdwYPosAbs,
                                        int32_t *pdwZPosAbs,
                                        int32_t *pdwXPosRel,
                                        int32_t *pdwYPosRel,
                                        int32_t *pdwZPosRel
                                        )
{
    uint8_t response[32];
    uint8_t swapped_response[32];
    uint8_t bIndex;
    phStatus_t status;

    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_READ_POSITION, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, response, sizeof(response), 1000));

    for (bIndex = 0; bIndex < 32; bIndex += 4)
    {
        swapped_response[bIndex + 0] = response[bIndex + 3];
        swapped_response[bIndex + 1] = response[bIndex + 2];
        swapped_response[bIndex + 2] = response[bIndex + 1];
        swapped_response[bIndex + 3] = response[bIndex + 0];
    }

    *pdwXPosAbs = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, *(int32_t *)(&swapped_response[0]), PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    *pdwYPosAbs = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, *(int32_t *)(&swapped_response[4]), PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    *pdwZPosAbs = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, *(int32_t *)(&swapped_response[8]), PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);

    *pdwXPosRel = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, *(int32_t *)(&swapped_response[16]), PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    *pdwYPosRel = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, *(int32_t *)(&swapped_response[20]), PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    *pdwZPosRel = phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, *(int32_t *)(&swapped_response[24]), PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_ReadPositionAbs(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t *pdwXPosAbs,
                                        int32_t *pdwYPosAbs,
                                        int32_t *pdwZPosAbs
                                        )
{
    int32_t tmp;
    return phdlStepper_HighZ_Int_ReadPosition(pDataParams, pdwXPosAbs, pdwYPosAbs, pdwZPosAbs, &tmp, &tmp, &tmp);
}

phStatus_t phdlStepper_HighZ_Int_ReadPositionRel(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t *pdwXPosRel,
                                        int32_t *pdwYPosRel,
                                        int32_t *pdwZPosRel
                                        )
{
    int32_t tmp;
    return phdlStepper_HighZ_Int_ReadPosition(pDataParams, &tmp, &tmp, &tmp, pdwXPosRel, pdwYPosRel, pdwZPosRel);
}

phStatus_t phdlStepper_HighZ_Int_WritePosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwXPos,
                                        int32_t dwYPos,
                                        int32_t dwZPos
                                        )
{
    phStatus_t status;

    uint8_t pbMessage[23];
    uint16_t wMessageLength;

    uint8_t bIndex;

    uint8_t added_data[16];
    uint8_t added_data_swapped[16];
    *(int32_t *)(&added_data_swapped[0]) = phdlStepper_HighZ_Int_ConvertMicrometerToSteps(pDataParams, dwXPos, PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    *(int32_t *)(&added_data_swapped[4]) = phdlStepper_HighZ_Int_ConvertMicrometerToSteps(pDataParams, dwYPos, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    *(int32_t *)(&added_data_swapped[8]) = phdlStepper_HighZ_Int_ConvertMicrometerToSteps(pDataParams, dwZPos, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);
    *(int32_t *)(&added_data_swapped[12]) = 0; /* c_step */

    for (bIndex = 0; bIndex < sizeof(added_data_swapped); bIndex += 4)
    {
        added_data[bIndex + 0] = added_data_swapped[bIndex + 3];
        added_data[bIndex + 1] = added_data_swapped[bIndex + 2];
        added_data[bIndex + 2] = added_data_swapped[bIndex + 1];
        added_data[bIndex + 3] = added_data_swapped[bIndex + 0];
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_WRITE_POSITION, added_data, sizeof(added_data), pbMessage, sizeof(pbMessage), &wMessageLength));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_ReleaseFromEndpoints(
                                        phdlStepper_HighZ_DataParams_t * pDataParams
                                        )
{
    phStatus_t status;
    int release_cnt = 0;
    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOn(pDataParams));

    while (release_cnt < PHDL_STEPPER_HIGHZ_INT_RELEASE_RETRY_MAX)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_RELEASE, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

        status = phdlStepper_HighZ_Int_WaitRelease(pDataParams);
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            break;
        }

        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_GoReference(pDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_WaitRelease(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phdlStepper_HighZ_HighZStatus_t highZStatus;
    uint8_t     pbBuffer[1];
    phStatus_t status;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Read(pDataParams, pbBuffer, sizeof(pbBuffer), PHDL_STEPPER_HIGHZ_INT_RELEASE_WAIT_TIME_MAX));
    highZStatus.data = pbBuffer[0];

    /* power supply disabled */
    if (highZStatus.items.em_stop)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }
    /* endpoints are still triggered */
    if (highZStatus.items.end_x || highZStatus.items.end_y || highZStatus.items.end_z)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }
    /* still moving while endpoint is released */
    if (highZStatus.items.drive_x || highZStatus.items.drive_y || highZStatus.items.drive_z)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_Drive(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;
    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_DRIVE, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_DriveToAbsolutePosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwXPosAbs,
                                        int32_t dwYPosAbs,
                                        int32_t dwZPosAbs
                                        )
{
    phStatus_t status;
    int32_t x_pos_abs, y_pos_abs, z_pos_abs;
    int32_t x_dist, y_dist, z_dist;
    int32_t x_min, y_min, z_min;

    x_min = -phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    y_min = -phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    z_min = -phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);

    if (dwXPosAbs > pDataParams->dwMaxRangeX || dwYPosAbs > pDataParams->dwMaxRangeY || dwZPosAbs > pDataParams->dwMaxRangeZ ||
        dwXPosAbs < x_min || dwYPosAbs < y_min || dwZPosAbs < z_min)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &x_pos_abs, &y_pos_abs, &z_pos_abs));

    x_dist = dwXPosAbs - x_pos_abs;
    y_dist = dwYPosAbs - y_pos_abs;
    z_dist = dwZPosAbs - z_pos_abs;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ExecuteDrive(pDataParams, x_dist, y_dist, z_dist));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_DriveToRelativePosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwX,
                                        int32_t dwY,
                                        int32_t dwZ
                                        )
{
    /* move robot to position specified by x, y, z + padding */

    phStatus_t status;
    int32_t x_pos_rel, y_pos_rel, z_pos_rel;
    int32_t x_pos_abs, y_pos_abs, z_pos_abs;
    int32_t x_dist, y_dist, z_dist;
    int32_t x_min, y_min, z_min;
    int32_t dwXPosAbs, dwYPosAbs, dwZPosAbs;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionRel(pDataParams, &x_pos_rel, &y_pos_rel, &z_pos_rel));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &x_pos_abs, &y_pos_abs, &z_pos_abs));

    x_dist = dwX - x_pos_rel;
    y_dist = dwY - y_pos_rel;
    z_dist = dwZ - z_pos_rel;

    /* Calc abs position for range check */
    dwXPosAbs = x_dist + x_pos_abs;
    dwYPosAbs = y_dist + y_pos_abs;
    dwZPosAbs = z_dist + z_pos_abs;

    x_min = -phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    y_min = -phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    z_min = -phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);

    if (dwXPosAbs > pDataParams->dwMaxRangeX || dwYPosAbs > pDataParams->dwMaxRangeY || dwZPosAbs > pDataParams->dwMaxRangeZ ||
        dwXPosAbs < x_min || dwYPosAbs < y_min || dwZPosAbs < z_min)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }



    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOn(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ExecuteDrive(pDataParams, x_dist, y_dist, z_dist));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOff(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_ExecuteDrive(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwXDist,
                                        int32_t dwYDist,
                                        int32_t dwZDist
                                        )
{
    phStatus_t status;
    uint16_t x_steps, y_steps, z_steps;

    int32_t x_steps_total = phdlStepper_HighZ_Int_ConvertMicrometerToSteps(pDataParams, abs(dwXDist), PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    int32_t y_steps_total = phdlStepper_HighZ_Int_ConvertMicrometerToSteps(pDataParams, abs(dwYDist), PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    int32_t z_steps_total = phdlStepper_HighZ_Int_ConvertMicrometerToSteps(pDataParams, abs(dwZDist), PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);

    /* direction flag may be inverted depending on hardware setup */
    uint8_t x_dir = (uint8_t)((dwXDist >= 0) == PHDL_STEPPER_HIGHZ_INT_FORWARD_X);
    uint8_t y_dir = (uint8_t)((dwYDist >= 0) == PHDL_STEPPER_HIGHZ_INT_FORWARD_Y);
    uint8_t z_dir = (uint8_t)((dwZDist >= 0) == PHDL_STEPPER_HIGHZ_INT_FORWARD_Z);

    /* loop in order to let receive settings command not be affected by buffer overflow */
    while (x_steps_total > 0 || y_steps_total > 0 || z_steps_total > 0)
    {
        x_steps = (uint16_t)(x_steps_total <= PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX ? x_steps_total : PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX);
        y_steps = (uint16_t)(y_steps_total <= PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX ? y_steps_total : PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX);
        z_steps = (uint16_t)(z_steps_total <= PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX ? z_steps_total : PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX);

        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_SendDriveSettings(pDataParams, x_steps, y_steps, z_steps, x_dir, y_dir, z_dir));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Drive(pDataParams));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WaitDriveEnd(pDataParams));

        x_steps_total -= x_steps;
        y_steps_total -= y_steps;
        z_steps_total -= z_steps;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_SendDriveSettings(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint16_t wXStep,
                                        uint16_t wYStep,
                                        uint16_t wZStep,
                                        uint8_t bXDir,
                                        uint8_t bYDir,
                                        uint8_t bZDir
                                        )
{
    /*
    configures drive settings for a drive operation
    has to be executed before DRIVE command
    */

    uint32_t dwMaxStep;
    uint32_t dwStepsDuringRamp;
    uint32_t dwFullSpeedSteps;
    double dTimeout;

    phStatus_t status;
    uint8_t pbMessage[23];
    uint16_t wMessageLength;

    uint8_t added_data[16];

    uint16_t c_steps = 0x00;
    uint8_t direction_byte = 0;

    direction_byte |= bXDir << 0;
    direction_byte |= bYDir << 1;
    direction_byte |= bZDir << 2;

    added_data[0] = (uint8_t)((wXStep >> 8) & 0xFF);
    added_data[1] = (uint8_t)((wXStep >> 0) & 0xFF);
    added_data[2] = (uint8_t)((wYStep >> 8) & 0xFF);
    added_data[3] = (uint8_t)((wYStep >> 0) & 0xFF);
    added_data[4] = (uint8_t)((wZStep >> 8) & 0xFF);
    added_data[5] = (uint8_t)((wZStep >> 0) & 0xFF);
    added_data[6] = (uint8_t)((c_steps >> 8) & 0xFF);
    added_data[7] = (uint8_t)((c_steps >> 0) & 0xFF);

    added_data[8] = (uint8_t)((pDataParams->wRampDelayMin >> 8) & 0xFF);
    added_data[9] = (uint8_t)((pDataParams->wRampDelayMin >> 0) & 0xFF);
    added_data[10] = (uint8_t)((pDataParams->wRampDelayMax >> 8) & 0xFF);
    added_data[11] = (uint8_t)((pDataParams->wRampDelayMax >> 0) & 0xFF);
    added_data[12] = (uint8_t)((pDataParams->wRampSpeed >> 8) & 0xFF);
    added_data[13] = (uint8_t)((pDataParams->wRampSpeed >> 0) & 0xFF);
    added_data[14] = pDataParams->bRampEnabled;

    added_data[15] = direction_byte;

    /* Calculate timeout */
    dwMaxStep = max(wXStep,wYStep);
    dwMaxStep = max(dwMaxStep,wZStep);
    /* Calulate steps during ramp */
    /* Calulate full speed steps */
    if (pDataParams->bRampEnabled == PH_ON)
    {
        dwStepsDuringRamp = min((uint32_t)2*pDataParams->wRampSpeed, dwMaxStep);
        dwFullSpeedSteps = dwMaxStep - dwStepsDuringRamp;
    }
    else
    {
        /* If ramp is disabled it always drive with maximum delay */
        dwStepsDuringRamp = dwMaxStep;
        dwFullSpeedSteps = 0;
    }
    /* measurements:
     *------------------------------
     * drive time | delay | #steps |
     *------------------------------
     *    170sec  |  200  | 33333  |
     *     85sec  |  100  | 33333  |
     *  17.14sec  |   20  | 33333  |
     *    4.9sec  |    5  | 33333  |
     *   2.15sec  |    2  | 33333  |
     *------------------------------
     *
     * -> const offset per step = 13.64us
     * -> time per delay = 25.44us
     */

    /* 10 sec offset */
    dTimeout = 10.0;
    dTimeout += (13.64 * ((double)dwStepsDuringRamp + (double)dwFullSpeedSteps) / 1000000.0);
    dTimeout += (25.44 * ((double)dwStepsDuringRamp*(double)pDataParams->wRampDelayMax + (double)dwFullSpeedSteps*(double)pDataParams->wRampDelayMin) / 1000000.0);
    pDataParams->dwDriveTimeout = (uint32_t)(0.5 + dTimeout);

    /* use the max of the calculated timeout and default timeout */
    pDataParams->dwDriveTimeout = max(pDataParams->dwDriveTimeout, PHDL_STEPPER_HIGHZ_INT_DRIVE_TIMEOUT);

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOn(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_RECEIVE_SETTINGS, added_data, sizeof(added_data), pbMessage, sizeof(pbMessage), &wMessageLength));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_WaitDriveEnd(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    /*
    While driving, the motor sends its status every 200ms.
    The read timeout value should always remain greater than this time frame.
    */

    phStatus_t status;
    phdlStepper_HighZ_HighZStatus_t highZStatus;

    clock_t     start_time;
    clock_t     cur_time;
    double      time_diff;
    uint8_t     pbBuffer[1];

    highZStatus.items.drive_x = 1;
    highZStatus.items.drive_y = 1;
    highZStatus.items.drive_z = 1;

    start_time = clock();
    while (highZStatus.items.drive_x || highZStatus.items.drive_y || highZStatus.items.drive_z)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Read(pDataParams, pbBuffer, sizeof(pbBuffer), 1000));
        highZStatus.data = pbBuffer[0];
        if (highZStatus.items.em_stop)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }

        if (highZStatus.items.end_x || highZStatus.items.end_y || highZStatus.items.end_z)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        cur_time = clock();
        time_diff = ((double)cur_time - start_time) / CLOCKS_PER_SEC;
        if (time_diff > pDataParams->dwDriveTimeout)
        {
            return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_STEPPER);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_WaitEndpoint(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint8_t bWaitX,
                                        uint8_t bWaitY,
                                        uint8_t bWaitZ,
                                        phdlStepper_HighZ_HighZStatus_t *pbStatus)
{
    /*
    While driving, the motor sends its status every 200ms.
    The read timeout value should always remain greater than this time frame.
    */

    phdlStepper_HighZ_HighZStatus_t highZStatus;
    phStatus_t status;

    clock_t     start_time;
    clock_t     cur_time;
    double      time_diff;
    uint8_t     pbBuffer[1];

    highZStatus.items.drive_x = 1;
    highZStatus.items.drive_y = 1;
    highZStatus.items.drive_z = 1;

    highZStatus.items.end_x = 0;
    highZStatus.items.end_y = 0;
    highZStatus.items.end_z = 0;

    start_time = clock();
    while (highZStatus.items.drive_x || highZStatus.items.drive_y || highZStatus.items.drive_z)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Read(pDataParams, pbBuffer, sizeof(pbBuffer), 1000));
        highZStatus.data = pbBuffer[0];
        if (highZStatus.items.em_stop)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        if ((bWaitX && highZStatus.items.end_x) || (bWaitY && highZStatus.items.end_y) || (bWaitZ && highZStatus.items.end_z))
        {
            pbStatus->items.end_x = highZStatus.items.end_x;
            pbStatus->items.end_y = highZStatus.items.end_y;
            pbStatus->items.end_z = highZStatus.items.end_z;
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
        }
        cur_time = clock();
        time_diff = ((double)cur_time - start_time) / CLOCKS_PER_SEC;
        if (time_diff > pDataParams->dwDriveTimeout)
        {
            return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_STEPPER);
        }
    }
    pbStatus->items.end_x = highZStatus.items.end_x;
    pbStatus->items.end_y = highZStatus.items.end_y;
    pbStatus->items.end_z = highZStatus.items.end_z;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_WaitEndpointOnAllCoordinates(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    /*
    While driving, the motor sends its status every 200ms.
    The read timeout value should always remain greater than this time frame.
    */

    phdlStepper_HighZ_HighZStatus_t highZStatus;
    phStatus_t status;

    clock_t     start_time;
    clock_t     cur_time;
    double      time_diff;
    uint8_t     pbBuffer[1];

    highZStatus.items.drive_x = 1;
    highZStatus.items.drive_y = 1;
    highZStatus.items.drive_z = 1;

    highZStatus.items.end_x = 0;
    highZStatus.items.end_y = 0;
    highZStatus.items.end_z = 0;

    start_time = clock();
    while (highZStatus.items.drive_x || highZStatus.items.drive_y || highZStatus.items.drive_z ||
        !(highZStatus.items.end_x && highZStatus.items.end_y && highZStatus.items.end_z))
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Read(pDataParams, pbBuffer, sizeof(pbBuffer), 1000));
        highZStatus.data = pbBuffer[0];
        if (highZStatus.items.em_stop)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        if (highZStatus.items.end_x && highZStatus.items.end_y && highZStatus.items.end_z &&
            (highZStatus.items.drive_x || highZStatus.items.drive_y || highZStatus.items.drive_z))
        {
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOn(pDataParams));
            /* still moving while all endpoint switches reached */
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
        }
        cur_time = clock();
        time_diff = ((double)cur_time - start_time) / CLOCKS_PER_SEC;
        if (time_diff > PHDL_STEPPER_HIGHZ_INT_GO_REFERENCE_TIMEOUT)
        {
            return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_STEPPER);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_Read(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint8_t *pResponse,
                                        uint16_t wExpectedLength,
                                        uint16_t wTimeoutMs)
{
    phStatus_t status;
    phStatus_t statusExchange;
    uint16_t wOldTimeout;
    uint16_t wRxLenPart;
    uint16_t wRxLen = 0;

    /* Set timeout */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_GetConfig(pDataParams->pBalRegDataParams,
        PHBAL_REG_CONFIG_READ_TIMEOUT_MS,
        &wOldTimeout));
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalRegDataParams,
        PHBAL_REG_CONFIG_READ_TIMEOUT_MS,
        wTimeoutMs));

    do
    {
        /* Exchange */
        statusExchange = phbalReg_Exchange(pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            NULL,
            0,
            wExpectedLength - wRxLen,
            &pResponse[wRxLen],
            &wRxLenPart);

        /* Ignore buffer overflow */
        if ((statusExchange & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
        {
            statusExchange = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
        }

        if ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            wRxLen += wRxLenPart;
        }
    } while ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS && wRxLen < wExpectedLength);

    /* Restore timeout */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalRegDataParams,
        PHBAL_REG_CONFIG_READ_TIMEOUT_MS,
        wOldTimeout));

    return statusExchange;
}

phStatus_t phdlStepper_HighZ_Int_GoReference(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;
    uint8_t pbMessage[7];
    uint16_t wMessageLength;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_PowerOn(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_CreateMessage(PHDL_STEPPER_HIGHZ_INT_GO_REFERENCE, NULL, 0, pbMessage, sizeof(pbMessage), &wMessageLength));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Exchange(pDataParams, pbMessage, wMessageLength, NULL, 0, 1000));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WaitEndpointOnAllCoordinates(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_Calibrate(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_GoReference(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReleaseFromEndpoints(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionFast(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;

    uint16_t x = PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX;
    uint16_t y = PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX;
    uint16_t z = PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX;
    uint8_t x_dir = 0;
    uint8_t y_dir = 1;
    uint8_t z_dir = 0;

    uint8_t wait_x = 1;
    uint8_t wait_y = 1;

    phdlStepper_HighZ_HighZStatus_t highZStatus;
    highZStatus.items.end_x = 0;
    highZStatus.items.end_y = 0;
    highZStatus.items.end_z = 0;

    while (highZStatus.items.end_z == 0)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_SendDriveSettings(pDataParams, 0, 0, z, x_dir, y_dir, z_dir));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Drive(pDataParams));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WaitEndpoint(pDataParams, 0, 0, 1, &highZStatus));
    }
    while (!(highZStatus.items.end_x == 1 && highZStatus.items.end_y == 1))
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_SendDriveSettings(pDataParams, x, y, 0, x_dir, y_dir, z_dir));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_Drive(pDataParams));
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WaitEndpoint(pDataParams, wait_x, wait_y, 0, &highZStatus));
        if (highZStatus.items.end_x == 1)
        {
            x = 0;
            wait_x = 0;
        }
        if (highZStatus.items.end_y == 1)
        {
            y = 0;
            wait_y = 0;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionSafe(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    /*
    move to padded z-zero
    then move to padded zero position
    and lastly to absolute zero position
    */

    phStatus_t status;
    int32_t x_pos, y_pos, z_pos;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &x_pos, &y_pos, &z_pos));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToAbsolutePosition(pDataParams, x_pos, y_pos, 0));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToZeroPositionAbsolute(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionRelative(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    /*
    The padding is offset by the distance that the RELEASE command drives away from endpoints (1000 steps)
    to ensure backwards compatibility with the behaviour of dll code.
    */

    phStatus_t status;

    int32_t x = pDataParams->dwPaddingX;
    int32_t y = pDataParams->dwPaddingY;
    int32_t z = pDataParams->dwPaddingZ;

    x -= phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_X);
    y -= phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y);
    z -= phdlStepper_HighZ_Int_ConvertStepsToMicrometer(pDataParams, PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE, PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z);
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_DriveToAbsolutePosition(pDataParams, x, y, z));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_WritePosition(pDataParams, 0, 0, 0));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionAbsolute(phdlStepper_HighZ_DataParams_t * pDataParams)
{
    phStatus_t status;
    status = phdlStepper_HighZ_Int_DriveToAbsolutePosition(pDataParams, 0, 0, 0);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_CheckOutOfBounds(phdlStepper_HighZ_DataParams_t * pDataParams, uint8_t *bOutOfBounds)
{
    /*
    After either an internal position reset or a machine stop between absolute zero point and endpoint switch,
    it is possible for the robot to think that one or more of its coordinates are below zero.
    This method returns true if such a case is detected and is part of the procedure to correct it.
    */

    phStatus_t status;
    int32_t x, y, z;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_HighZ_Int_ReadPositionAbs(pDataParams, &x, &y, &z));
    if (x < 0 || y < 0 || z < 0)
    {
        *bOutOfBounds = PH_ON;
    }
    else
    {
        *bOutOfBounds = PH_OFF;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_HighZ_Int_GetSpeed(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint16_t *pwSpeed)
{
    /* for full speed 2ms used 200, for 1ms use 100 */
    *pwSpeed = (uint16_t)(0.5+200.0/(double)pDataParams->wRampDelayMin);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_HighZ_Int_SetSpeed(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint16_t wSpeed)
{

    switch(wSpeed)
    {
    /* Mapping table for full speed = 2ms */
    case 100:pDataParams->wRampDelayMin = 2; break;
    case 67: pDataParams->wRampDelayMin = 3; break;
    case 50: pDataParams->wRampDelayMin = 4; break;
    case 40: pDataParams->wRampDelayMin = 5; break;
    case 33: pDataParams->wRampDelayMin = 6; break;
    case 29: pDataParams->wRampDelayMin = 7; break;
    case 25: pDataParams->wRampDelayMin = 8; break;
    case 22: pDataParams->wRampDelayMin = 9; break;
    case 20: pDataParams->wRampDelayMin = 10; break;
    case 18: pDataParams->wRampDelayMin = 11; break;
    case 17: pDataParams->wRampDelayMin = 11; break;
    case 15: pDataParams->wRampDelayMin = 13; break;
    case 14: pDataParams->wRampDelayMin = 14; break;
    case 13: pDataParams->wRampDelayMin = 15; break;
    case 12: pDataParams->wRampDelayMin = 16; break;
    case 11: pDataParams->wRampDelayMin = 18; break;
    case 10: pDataParams->wRampDelayMin = 20; break;
    case 9:  pDataParams->wRampDelayMin = 22; break;
    case 8:  pDataParams->wRampDelayMin = 25; break;
    case 7:  pDataParams->wRampDelayMin = 28; break;
    case 6:  pDataParams->wRampDelayMin = 33; break;
    case 5:  pDataParams->wRampDelayMin = 40; break;
    case 4:  pDataParams->wRampDelayMin = 50; break;
    case 3:  pDataParams->wRampDelayMin = 66; break;
    case 2:  pDataParams->wRampDelayMin = 100; break;
    case 1:  pDataParams->wRampDelayMin = 200; break;

    /* Mapping table for full speed = 1ms */
    /*case 100:pDataParams->wRampDelayMin = 1; break;
    case 50: pDataParams->wRampDelayMin = 2; break;
    case 33: pDataParams->wRampDelayMin = 3; break;
    case 25: pDataParams->wRampDelayMin = 4; break;
    case 20: pDataParams->wRampDelayMin = 5; break;
    case 17: pDataParams->wRampDelayMin = 6; break;
    case 14: pDataParams->wRampDelayMin = 7; break;
    case 13: pDataParams->wRampDelayMin = 8; break;
    case 11: pDataParams->wRampDelayMin = 9; break;
    case 10: pDataParams->wRampDelayMin = 10; break;
    case 9:  pDataParams->wRampDelayMin = 11; break;
    case 8:  pDataParams->wRampDelayMin = 12; break;
    case 7:  pDataParams->wRampDelayMin = 14; break;
    case 6:  pDataParams->wRampDelayMin = 16; break;
    case 5:  pDataParams->wRampDelayMin = 20; break;
    case 4:  pDataParams->wRampDelayMin = 25; break;
    case 3:  pDataParams->wRampDelayMin = 33; break;
    case 2:  pDataParams->wRampDelayMin = 50; break;
    case 1:  pDataParams->wRampDelayMin = 100; break;*/
    default: return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* If ramp delay max = 10 time delay min but max 50 */
    if (pDataParams->wRampDelayMin < 5)
    {
        pDataParams->wRampDelayMax = pDataParams->wRampDelayMin * 10;
    }
    else if (pDataParams->wRampDelayMin < 50)
    {
        pDataParams->wRampDelayMax = 50;
    }
    else
    {
        pDataParams->wRampDelayMax = pDataParams->wRampDelayMin;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */
