/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper Microbot Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT

#include <phdlStepper.h>
#include "phdlStepper_Microbot.h"
#include "phdlStepper_Microbot_Int.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_Microbot_Init(
                               phdlStepper_Microbot_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams,
                               void * pBalRegDataParams
                               )
{
    if (sizeof(phdlStepper_Microbot_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_STEPPER | PH_COMP_DL_STEPPER_MICROBOT_ID;
    pDataParams->pBalRegDataParams      = pBalRegDataParams;
    pDataParams->bInit                  = PH_OFF;
    pDataParams->wStepsWayRatio         = 400; /* Use the same as in Stepper V1 */
    pDataParams->bPositionMode          = PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE;
    pDataParams->bActiveAxis            = PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Z;
    pDataParams->dwPositionX            = 0;
    pDataParams->dwPositionY            = 0;
    pDataParams->dwPositionZ            = 0;
    pDataParams->wAngle                 = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_Microbot_Initialize(
                                      phdlStepper_Microbot_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_ASSERT_NULL (pDataParams);

    /* Set Microbot to top position and wait until top position is reached */
    status = phdlStepper_Microbot_Int_WaitMoveFinished(pDataParams);
    /* Ignore error of sensor in head */
    if ( (status & PH_ERR_MASK) == PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR)
    {
        status = PH_ERR_SUCCESS;
    }
    PH_CHECK_SUCCESS(status);
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, 0, 0, 1000000, 0, PH_OFF));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_WaitMoveFinished(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_SetAsZero(pDataParams));

    pDataParams->bInit = PH_ON;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_MoveDistance(
                                       phdlStepper_Microbot_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint8_t bDirection,
                                       uint32_t dwDistance,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwActX;
    int32_t    PH_MEMLOC_REM dwActY;
    int32_t    PH_MEMLOC_REM dwActZ;
    int16_t    PH_MEMLOC_REM wActAngle;
    int32_t    PH_MEMLOC_REM dwCalcEndPos;
    int32_t    PH_MEMLOC_REM dwSignedDistance;
    int32_t    PH_MEMLOC_REM dwActPosLimitLow;
    int32_t    PH_MEMLOC_REM dwActPosLimitHigh;

    PH_ASSERT_NULL (pDataParams);
    if( bDirection != PHDL_STEPPER_DIR_DOWN && bDirection != PHDL_STEPPER_DIR_UP )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* If the position is set local on the robot this is not allowed */
    if (pDataParams->bPositionMode == PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* limit the distance to the max available number / 2 */
    if (dwDistance > 0x7FFFFFFF)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    if (bDirection == PHDL_STEPPER_DIR_DOWN)
    {
        dwSignedDistance = (int32_t)(0-dwDistance);
        dwActPosLimitLow = -0x7FFFFFFF - dwSignedDistance;
        dwActPosLimitHigh = 0x7FFFFFFF;
    }
    else
    {
        dwSignedDistance = dwDistance;
        dwActPosLimitLow = -0x7FFFFFFF;
        dwActPosLimitHigh = 0x7FFFFFFF - dwSignedDistance;
    }

    /* Get the actual position */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GetPosition(pDataParams, &dwActX, &dwActY, &dwActZ, &wActAngle));

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_X:
        /* calculate the end position */
        if (dwActX < dwActPosLimitLow || dwActX > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        dwCalcEndPos = dwActX + dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, dwCalcEndPos, dwActY, dwActZ, wActAngle, bBlocking));
        break;
    case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Y:
        /* calculate the end position */
        if (dwActY < dwActPosLimitLow || dwActY > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        dwCalcEndPos = dwActY + dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, dwActX, dwCalcEndPos, dwActZ, wActAngle, bBlocking));
        break;
    case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Z:
        /* calculate the end position */
        if (dwActZ < dwActPosLimitLow || dwActZ > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        dwCalcEndPos = dwActZ + dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, dwActX, dwActY, dwCalcEndPos, wActAngle, bBlocking));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    /* satisfy compiler */
    if (wSpeed);

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_MoveSteps(
                                    phdlStepper_Microbot_DataParams_t * pDataParams,
                                    uint16_t wSpeed,
                                    uint8_t bDirection,
                                    uint32_t dwSteps,
                                    uint8_t bBlocking
                                    )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwDistance;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_ConvertFromStep(pDataParams, dwSteps, &dwDistance));

    /* Use MoveDistance to perform task. */
    return phdlStepper_Microbot_MoveDistance(pDataParams, wSpeed, bDirection, dwDistance, bBlocking);
}

phStatus_t phdlStepper_Microbot_GoToPosition(
                                       phdlStepper_Microbot_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint32_t dwPosition,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwActX;
    int32_t    PH_MEMLOC_REM dwActY;
    int32_t    PH_MEMLOC_REM dwActZ;
    int16_t    PH_MEMLOC_REM wActAngle;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* If the position is set local on the robot this is not allowed */
    if (pDataParams->bPositionMode == PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Get the actual position */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GetPosition(pDataParams, &dwActX, &dwActY, &dwActZ, &wActAngle));

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_X:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, (int32_t)dwPosition, dwActY, dwActZ, wActAngle, bBlocking));
        break;
    case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Y:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, dwActX, (int32_t)dwPosition, dwActZ, wActAngle, bBlocking));
        break;
    case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Z:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GoToParameter(pDataParams, dwActX, dwActY, (int32_t)dwPosition, wActAngle, bBlocking));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    /* satisfy compiler */
    if (wSpeed);

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_SetConfig(
                                    phdlStepper_Microbot_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t wValue
                                    )
{
    phStatus_t  PH_MEMLOC_REM status;
    int16_t    PH_MEMLOC_REM wActAngle;

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        pDataParams->wStepsWayRatio = wValue;
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG_POSITION_MODE:
        switch (wValue)
        {
        case PHDL_STEPPER_MICROBOT_POS_MODE_SET_CURRENT_AS_ZERO:
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_SetAsZero(pDataParams));
            break;
        case PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE:
            /* If we start this mode we get the current position because there may be a position change in local mode */
            if (pDataParams->bPositionMode != PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE)
            {
                PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GetPosition(pDataParams, (int32_t * UNALIGNED)&pDataParams->dwPositionX, (int32_t * UNALIGNED)&pDataParams->dwPositionY, (int32_t * UNALIGNED)&pDataParams->dwPositionZ, (int16_t * UNALIGNED)&wActAngle));
            }
            pDataParams->bPositionMode = PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE;
            break;
        case PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL:
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_ManualMode(pDataParams));
            pDataParams->bPositionMode = PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL;
            break;
        case PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT:
            /* If we start this mode we get the current position because there may be a non blocking move before */
            if (pDataParams->bPositionMode != PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT)
            {
                PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GetPosition(pDataParams, (int32_t * UNALIGNED)&pDataParams->dwPositionX, (int32_t * UNALIGNED)&pDataParams->dwPositionY, (int32_t * UNALIGNED)&pDataParams->dwPositionZ, (int16_t * UNALIGNED)&wActAngle));
            }
            pDataParams->bPositionMode = PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT;
            break;
        case PHDL_STEPPER_MICROBOT_POS_MODE_START:
            status = phdlStepper_Microbot_Int_GoTo(pDataParams, PH_ON);
            if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS) || ((status & PH_ERR_MASK) == PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS))
            {
                pDataParams->bPositionMode = PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE;
            }
            PH_CHECK_SUCCESS(status);
            break;
        case PHDL_STEPPER_MICROBOT_POS_MODE_START_NONBLOCKING:
            status = phdlStepper_Microbot_Int_GoTo(pDataParams, PH_OFF);
            if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS) || ((status & PH_ERR_MASK) == PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS))
            {
                pDataParams->bPositionMode = PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE;
            }
            PH_CHECK_SUCCESS(status);
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG_ACTIVE_AXIS:
        if( wValue != PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_X &&
            wValue != PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Y &&
            wValue != PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Z )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxis = (uint8_t)wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_SetConfig32(
                                      phdlStepper_Microbot_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t dwValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    int32_t dwSignedValue = (int32_t)dwValue;

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER: /* POSITION_COUNTER uses steps so convert */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_ConvertFromStep(pDataParams, dwSignedValue, &dwSignedValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_GoToPosition(pDataParams, 0, (uint32_t)dwSignedValue, PH_ON));
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_X:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwPositionX = (int32_t)dwValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_GoTo(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_Y:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwPositionY = (int32_t)dwValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_GoTo(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_Z:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->dwPositionZ = (int32_t)dwValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_GoTo(pDataParams, PH_ON));
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_GetConfig(
                                    phdlStepper_Microbot_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        *pValue = pDataParams->wStepsWayRatio;
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG_POSITION_MODE:
        *pValue = pDataParams->bPositionMode;
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG_ACTIVE_AXIS:
        *pValue = pDataParams->bActiveAxis;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_GetConfig32(
                                      phdlStepper_Microbot_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t * pValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    int32_t    PH_MEMLOC_REM dwActX;
    int32_t    PH_MEMLOC_REM dwActY;
    int32_t    PH_MEMLOC_REM dwActZ;
    int32_t    PH_MEMLOC_REM dwAct;
    int16_t    PH_MEMLOC_REM wActAngle;

    PH_ASSERT_NULL (pDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_GetPosition(pDataParams, &dwActX, &dwActY, &dwActZ, &wActAngle));

    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER: /* POSITION_COUNTER uses steps so convert */
        switch (pDataParams->bActiveAxis)
        {
        case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_X:
            dwAct = dwActX;
            break;
        case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Y:
            dwAct = dwActY;
            break;
        case PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Z:
            dwAct = dwActZ;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_ConvertToStep(pDataParams, dwAct, &dwAct));
        *pValue = (uint32_t)dwAct;
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_X:
        *pValue = (uint32_t)dwActX;
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_Y:
        *pValue = (uint32_t)dwActY;
        break;

    case PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_Z:
        *pValue = (uint32_t)dwActZ;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */
