/*
 * Copyright 2017 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT

#include "phdlStepper_Microbot.h"
#include "phdlStepper_Microbot_Int.h"
#include "../phdlStepper_Int.h"

#ifndef _WIN32
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>              /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */
#endif

phStatus_t phdlStepper_Microbot_Int_ConvertToStep(
                                        phdlStepper_Microbot_DataParams_t * pDataParams,
                                        int32_t dwDistance,
                                        int32_t * dwSteps
                                        )
{
    double unit = dwDistance * ((double)pDataParams->wStepsWayRatio / 1000.0);

    if( unit < 0.0 )
    {
        if (unit < -0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwSteps = -(int32_t)((-unit)+0.5);
    }
    else
    {
        if (unit > 0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwSteps = (int32_t)((unit)+0.5);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_ConvertFromStep(
                                        phdlStepper_Microbot_DataParams_t * pDataParams,
                                        int32_t dwSteps,
                                        int32_t * dwDistance
                                        )
{
    double unit = dwSteps * (1000.0 / (double)pDataParams->wStepsWayRatio);
    if( unit < 0.0 )
    {
        if (unit < -0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwDistance = -(int32_t)((-unit)+0.5);
    }
    else
    {
        if (unit > 0x7FFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *dwDistance = (int32_t)((unit)+0.5);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_GetStatus(
                                        phdlStepper_Microbot_DataParams_t * pDataParams,
                                        uint32_t * wStatus
                                        )
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_MICROBOT_INT_SND_STATUS,
        (uint16_t)strlen((char *)PHDL_STEPPER_MICROBOT_INT_SND_STATUS),
        sizeof(buf),
        buf,
        &wRxLen));

    if ( memcmp((const char *) buf, (const char *)PHDL_STEPPER_MICROBOT_INT_SND_STATUS, 4) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf + 4, " 0000 ", 6) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( sscanf( (const char *) buf+10, "%d", wStatus ) == 1 )
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_GoTo(phdlStepper_Microbot_DataParams_t * pDataParams, uint8_t bBlocking)
{
    return phdlStepper_Microbot_Int_GoToParameter(pDataParams, pDataParams->dwPositionX, pDataParams->dwPositionY, pDataParams->dwPositionZ, pDataParams->wAngle, bBlocking);
}

phStatus_t phdlStepper_Microbot_Int_GoToParameter(
                                        phdlStepper_Microbot_DataParams_t * pDataParams,
                                        int32_t x, int32_t y, int32_t z, int16_t angle,
                                        uint8_t bBlocking
                                        )
{
    uint8_t cmd[100];
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;
    int16_t    PH_MEMLOC_REM wActAngle;

    /* Check the limits */
    /* For all axis the maximum value is if the zero point was set at the oposide side */
    if ( x > PHDL_STEPPER_MICROBOT_LIMIT_X || x < -PHDL_STEPPER_MICROBOT_LIMIT_X ||
         y > PHDL_STEPPER_MICROBOT_LIMIT_Y || y < -PHDL_STEPPER_MICROBOT_LIMIT_Y ||
         z > PHDL_STEPPER_MICROBOT_LIMIT_Z || z < -PHDL_STEPPER_MICROBOT_LIMIT_Z )
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    PH_ASSERT_NULL (pDataParams);

    sprintf( (char *) cmd, (const char *)PHDL_STEPPER_MICROBOT_INT_SND_GOTO, x, y, z, DOUBLE angle);
    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        cmd,
        (uint16_t)strlen((char *)cmd),
        sizeof(buf),
        buf,
        &wRxLen));

    if ( wRxLen != 11 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf, (const char *)PHDL_STEPPER_MICROBOT_INT_SND_GOTO, 4) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf + 4, " 0000\r\n", 7) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if( bBlocking == PH_ON )
    {
        statusTmp = phdlStepper_Microbot_Int_WaitMoveFinished(pDataParams);
        if ( (statusTmp & PH_ERR_MASK) == PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR)
        {
            statusTmp = PH_ERR_SUCCESS;
        }
        PH_CHECK_SUCCESS(statusTmp);

        /* Ok we should have reached the position so check if it is ok and also store the reached position as sct position */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Microbot_Int_GetPosition(pDataParams, (int32_t * UNALIGNED)&pDataParams->dwPositionX, (int32_t * UNALIGNED)&pDataParams->dwPositionY, (int32_t * UNALIGNED)&pDataParams->dwPositionZ, (int16_t * UNALIGNED)&wActAngle));

        if ((pDataParams->dwPositionX - x) > PHDL_STEPPER_MICROBOT_TOLERANCE_LIMIT_X || (pDataParams->dwPositionX - x) < -PHDL_STEPPER_MICROBOT_TOLERANCE_LIMIT_X ||
            (pDataParams->dwPositionY - y) > PHDL_STEPPER_MICROBOT_TOLERANCE_LIMIT_Y || (pDataParams->dwPositionY - y) < -PHDL_STEPPER_MICROBOT_TOLERANCE_LIMIT_Y ||
            (pDataParams->dwPositionZ - z) > PHDL_STEPPER_MICROBOT_TOLERANCE_LIMIT_Z || (pDataParams->dwPositionZ - z) < -PHDL_STEPPER_MICROBOT_TOLERANCE_LIMIT_Z)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_GetPosition(
                                        phdlStepper_Microbot_DataParams_t * pDataParams,
                                        int32_t *x,  int32_t *y, int32_t *z, int16_t *angle
                                        )
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_MICROBOT_INT_SND_REQUEST_POSITION,
        (uint16_t)strlen((char *)PHDL_STEPPER_MICROBOT_INT_SND_REQUEST_POSITION),
        sizeof(buf),
        buf,
        &wRxLen));
    /* The minimum length is 19 ("MRRP 0000 0 0 0 0\r\n") */
    if ( wRxLen < 19 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf, (const char *)PHDL_STEPPER_MICROBOT_INT_SND_REQUEST_POSITION, 4) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf + 4, " 0000 ", 6) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    /* At offset 10 the actual position starts */
    if( sscanf( (const char *) buf+10, "%ld %ld %ld %hd", x, y, z, angle ) != 4 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_SetAsZero(phdlStepper_Microbot_DataParams_t * pDataParams)
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_MICROBOT_INT_SND_RESET_POSITION,
        (uint16_t)strlen((char *)PHDL_STEPPER_MICROBOT_INT_SND_RESET_POSITION),
        sizeof(buf),
        buf,
        &wRxLen));

    if ( wRxLen != 11 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf, (const char *)PHDL_STEPPER_MICROBOT_INT_SND_RESET_POSITION, 4) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf + 4, " 0000\r\n", 7) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    pDataParams->dwPositionX = 0;
    pDataParams->dwPositionY = 0;
    pDataParams->dwPositionZ = 0;
    pDataParams->wAngle      = 0;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_ManualMode(phdlStepper_Microbot_DataParams_t * pDataParams)
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_MICROBOT_INT_SND_MANUAL_MODE,
        (uint16_t)strlen((char *)PHDL_STEPPER_MICROBOT_INT_SND_MANUAL_MODE),
        sizeof(buf),
        buf,
        &wRxLen));

    if ( wRxLen != 11 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf, (const char *)PHDL_STEPPER_MICROBOT_INT_SND_MANUAL_MODE, 4) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    if ( memcmp((const char *) buf + 4, " 0000\r\n", 7) != 0 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Microbot_Int_WaitMoveFinished(phdlStepper_Microbot_DataParams_t * pDataParams)
{
    uint32_t wStatus = 128;
    phStatus_t status;

    while( wStatus != 0 )
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Microbot_Int_GetStatus(pDataParams, &wStatus));
        if (wStatus == 0x04) /* Sensor detected */
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR, PH_COMP_DL_STEPPER);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */
