/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_STEPPER_V1

#include "phdlStepper_V1.h"
#include "phdlStepper_V1_Int.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_V1_Int_Initparam(
                                        phdlStepper_V1_DataParams_t * pDataParams
                                        )
{
    phStatus_t status;

    /* write initial parameters to stepper */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_RAMP_DELAY, 2));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_RAMP_ON_OFF, PH_ON));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_DIR, PHDL_STEPPER_DIR_DOWN));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_START_SPEED, 0xF424));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_MAX_SPEED, 0xFC0B));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO, 400));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_DISTANCE, 10));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_MAX_CURRENT, 100));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_STDBY_CURRENT, 35));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_STDBY_CURRENT_FLAG, 0));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_V1_Int_GetLimits(
    phdlStepper_V1_DataParams_t * pDataParams
    )
{
    phStatus_t status;
    uint32_t dwPosCount;

    /* Release the Stepper from limit indicator. */
    PH_CHECK_SUCCESS_FCT(status,  phdlStepper_V1_Int_ReleaseFromLimit(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount));

    /* set Max position to maximum to give the stepper enough range to act. */
    pDataParams->dwMax_Position = 0xFFFFFFFF;

    /* Drive to the upper limit indicator. */
    status = phdlStepper_MoveDistance(pDataParams, 5000, PHDL_STEPPER_DIR_UP, 0xFFFFFFFF, PH_ON);
    if ((status & PH_ERR_MASK) != PHDL_STEPPER_ERR_UPPER_LIMIT_INDICATOR)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_DRIVE_FAILURE, PH_COMP_DL_STEPPER);
    }

    /* Release the Stepper from limit indicator. */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_ReleaseFromLimit(pDataParams));

    /* Set Position_Counter to zero (Point of origin) */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, 0));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Wait(100));

    /* Drive to the upper lower indicator */
    status = phdlStepper_MoveDistance(pDataParams, 4000, PHDL_STEPPER_DIR_DOWN, 0xFFFFFFFF, PH_ON);
    if ((status & PH_ERR_MASK) != PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_DRIVE_FAILURE, PH_COMP_DL_STEPPER);
    }

    /* Release the Stepper from limit indicator. */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_ReleaseFromLimit(pDataParams));

    /*Get current position and set dwMax_Position to this value. */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount));

    /* set maximum value to current position */
    pDataParams->dwMax_Position = dwPosCount;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_V1_Int_ReleaseFromLimit(
    phdlStepper_V1_DataParams_t * pDataParams
    )
{
    phStatus_t status;
    uint8_t bTx[10];
    uint16_t wTxLength;
    uint32_t dwPosCount = 0xFFFFFFFF;
    uint16_t wCounter = 0;

    /* build command */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_STATE_RELEASE_FROM_LIMIT, 0, &bTx[0], &wTxLength));

    /* Send comand. */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Send_Wait(pDataParams, &bTx[0], wTxLength, 20));

    /* request position counter untill the stepper answers again (when stepper is the standby state). */
    do
    {
        /* wait 100ms for stepper to proceed command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Wait(100));

        /* request postion counter */
        status = phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount);

        /* Return error after 50 runs with no reaction. */
        wCounter++;
        if (wCounter > 50)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_DRIVE_FAILURE, PH_COMP_DL_STEPPER);
        }
    } while ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT);

    /* check status for any error else than timeout*/
    PH_CHECK_SUCCESS(status);

    /* wait 2 s to ensure stepper is relased from limits */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Wait(2000));

    pDataParams->bState = PHDL_STEPPER_V1_INT_STATE_STDBY;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_V1_Int_DataLength(
    uint8_t Identifier,
    uint16_t *pLength
    )
{
    /* Send  a state -> no data */
    if (Identifier > PHDL_STEPPER_V1_INT_STATE_BEGIN_STATE_ENUM && Identifier < PHDL_STEPPER_V1_INT_STATE_END_STATE_ENUM)
    {
        *pLength = 0;
    }
    else
    {
        switch (Identifier)
        {
            /* send 8 bit (data: (bool), unsigned char) */
        case PHDL_STEPPER_V1_INT_SND_RAMP_ON_OFF:
        case PHDL_STEPPER_V1_INT_SND_DIRECTION:
        case PHDL_STEPPER_V1_INT_SND_MAX_CURRENT:
        case PHDL_STEPPER_V1_INT_SND_SBY_CURRENT:
        case PHDL_STEPPER_V1_INT_SND_SBY_FLAG_CURRENT:
            *pLength = 1;
            break;

            /* Send 16 bit (data: unsigned int) */
        case PHDL_STEPPER_V1_INT_SND_RAMP_DELAY:
        case PHDL_STEPPER_V1_INT_SND_START_SPEED:
        case PHDL_STEPPER_V1_INT_SND_MAX_SPEED:
        case PHDL_STEPPER_V1_INT_SND_STEPS_WAY_RATIO:
        case PHDL_STEPPER_V1_INT_SND_DISTANCE:
            *pLength = 2;
            break;

            /* send 32 bit (data: unsigned long int, signed long int) */
        case PHDL_STEPPER_V1_INT_SND_STEPS:
        case PHDL_STEPPER_V1_INT_SND_POSITION_COUNTER:
        case PHDL_STEPPER_V1_INT_SND_MAX_POSITION:
        case PHDL_STEPPER_V1_INT_REQ_POSITION_COUNTER:
            *pLength = 4;
            break;

            /* Unknowen Identifier */
        default:
            *pLength = 0;
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_V1_Int_GetFrame(
                                        uint8_t bCommand,
                                        uint32_t dwInput,
                                        uint8_t  * pFrame,
                                        uint16_t * wFrameLenth
                                        )
{
    phStatus_t status;
    uint16_t wCrc;
    uint8_t bI;
    uint8_t bTempbyte;
    uint8_t bData_And_Command [5];
    uint16_t wInput_Length;

    /* Set Preamble */
    *(pFrame+0) = 'P';
    *(pFrame+1) = 'r';
    *(pFrame+2) = 'E';

    /* Get length of datablock */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_DataLength(bCommand, &wInput_Length));

    *(pFrame + 3) = (uint8_t)(wInput_Length + 3);
    *(pFrame + 4) = bCommand;

    bData_And_Command [0]= bCommand;

    for (bI = 0; bI < wInput_Length; bI++)
    {
        *(pFrame + (bI + 5)) = (uint8_t)(dwInput >> (8 * (wInput_Length - 1 - bI)) & 0xFF);
        bData_And_Command [bI + 1] = *(pFrame + (bI + 5));
    }

    PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc16(PH_TOOLS_CRC_OPTION_MSB_FIRST, 0xFFFF, 0x1021, (uint8_t*)&bData_And_Command, (wInput_Length + 1), &wCrc));
    *(pFrame + (bI + 5)) = (uint8_t)(wCrc & 0xFF);
    *(pFrame + (bI + 6)) = (uint8_t)(wCrc >> 8 & 0xFF);

    for (bI = 0; bI < (uint8_t)(wInput_Length + 7) / 2; bI++ )
    {
        bTempbyte = *(pFrame + bI);
        *(pFrame + bI) = *(pFrame + (wInput_Length + 6 - bI));
        *(pFrame + (wInput_Length + 6 - bI)) = bTempbyte;
    }

    *wFrameLenth = wInput_Length + 7;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_V1 */
