/*
 * Copyright 2013, 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Stepper Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phdlAmp.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_STEPPER_V1
#include "V1/phdlStepper_V1.h"
#endif /* NXPBUILD__PHDL_STEPPER_V1 */
#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
#include "Robot/phdlStepper_Robot.h"
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */
#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
#include "Microbot/phdlStepper_Microbot.h"
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */
#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
#include "HighZ/phdlStepper_HighZ.h"
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */
#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
#include "DensoVS60/phdlStepper_DensoVS60.h"
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */
#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
#include "Wachler/phdlStepper_Wachler.h"
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */
#ifdef NXPBUILD__PHDL_STEPPER

phStatus_t phdlStepper_MoveDistance(
                                     void * pDataParams,
                                     uint16_t wSpeed,
                                     uint8_t bDirection,
                                     uint32_t wDistance,
                                     uint8_t bBlocking
                                     )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_MoveDistance");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSpeed);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDirection);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wDistance);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlocking);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSpeed_log, &wSpeed);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDirection_log, &bDirection);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, wDistance_log, &wDistance);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bBlocking_log, &bBlocking);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_MoveDistance((phdlStepper_V1_DataParams_t*)pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_MoveDistance((phdlStepper_Robot_DataParams_t*)pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_MoveDistance((phdlStepper_Microbot_DataParams_t*)pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_MoveDistance((phdlStepper_HighZ_DataParams_t*)pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_MoveDistance((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_MoveDistance((phdlStepper_Wachler_DataParams_t*)pDataParams, wSpeed, bDirection, wDistance, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_MoveSteps(
                                  void * pDataParams,
                                  uint16_t wSpeed,
                                  uint8_t bDirection,
                                  uint32_t dwSteps,
                                  uint8_t bBlocking
                                  )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_MoveSteps");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSpeed);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDirection);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(dwSteps);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlocking);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSpeed_log, &wSpeed);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDirection_log, &bDirection);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, dwSteps_log, &dwSteps);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bBlocking_log, &bBlocking);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_MoveSteps((phdlStepper_V1_DataParams_t*)pDataParams, wSpeed, bDirection, dwSteps, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_MoveSteps((phdlStepper_Robot_DataParams_t*)pDataParams, wSpeed, bDirection, dwSteps, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_MoveSteps((phdlStepper_Microbot_DataParams_t*)pDataParams, wSpeed, bDirection, dwSteps, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_MoveSteps((phdlStepper_HighZ_DataParams_t*)pDataParams, wSpeed, bDirection, dwSteps, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_MoveSteps((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wSpeed, bDirection, dwSteps, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_MoveSteps((phdlStepper_Wachler_DataParams_t*)pDataParams, wSpeed, bDirection, dwSteps, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_GoToPosition(
                                      void * pDataParams,
                                      uint16_t wSpeed,
                                      uint32_t wPosition,
                                      uint8_t bBlocking
                                      )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_GoToPosition");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSpeed);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wPosition);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlocking);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSpeed_log, &wSpeed);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, wPosition_log, &wPosition);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bBlocking_log, &bBlocking);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_GoToPosition((phdlStepper_V1_DataParams_t*)pDataParams, wSpeed, wPosition, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_GoToPosition((phdlStepper_Robot_DataParams_t*)pDataParams, wSpeed, wPosition, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_GoToPosition((phdlStepper_Microbot_DataParams_t*)pDataParams, wSpeed, wPosition, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_GoToPosition((phdlStepper_HighZ_DataParams_t*)pDataParams, wSpeed, wPosition, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_GoToPosition((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wSpeed, wPosition, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_GoToPosition((phdlStepper_Wachler_DataParams_t*)pDataParams, wSpeed, wPosition, bBlocking);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_SetConfig(
                                 void * pDataParams,
                                 uint16_t wConfig,
                                 uint16_t wValue
                                 )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_SetConfig");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wValue_log, &wValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_SetConfig((phdlStepper_V1_DataParams_t*)pDataParams, wConfig, wValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_SetConfig((phdlStepper_Robot_DataParams_t*)pDataParams, wConfig, wValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_SetConfig((phdlStepper_Microbot_DataParams_t*)pDataParams, wConfig, wValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_SetConfig((phdlStepper_HighZ_DataParams_t*)pDataParams, wConfig, wValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_SetConfig((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wConfig, wValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_SetConfig((phdlStepper_Wachler_DataParams_t*)pDataParams, wConfig, wValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_SetConfig32(
                                 void * pDataParams,
                                 uint16_t wConfig,
                                 uint32_t dwValue
                                 )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_SetConfig32");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(dwValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, dwValue_log, &dwValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_SetConfig32((phdlStepper_V1_DataParams_t*)pDataParams, wConfig, dwValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_SetConfig32((phdlStepper_Robot_DataParams_t*)pDataParams, wConfig, dwValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_SetConfig32((phdlStepper_Microbot_DataParams_t*)pDataParams, wConfig, dwValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_SetConfig32((phdlStepper_HighZ_DataParams_t*)pDataParams, wConfig, dwValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_SetConfig32((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wConfig, dwValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_SetConfig32((phdlStepper_Wachler_DataParams_t*)pDataParams, wConfig, dwValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_GetConfig(
                                 void * pDataParams,
                                 uint16_t wConfig,
                                 uint16_t * pValue
                                 )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_GetConfig");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_GetConfig((phdlStepper_V1_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_GetConfig((phdlStepper_Robot_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_GetConfig((phdlStepper_Microbot_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_GetConfig((phdlStepper_HighZ_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_GetConfig((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_GetConfig((phdlStepper_Wachler_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pValue_log, pValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_GetConfig32(
                                 void * pDataParams,
                                 uint16_t wConfig,
                                 uint32_t * pValue
                                 )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlStepper_GetConfig32");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_GetConfig32((phdlStepper_V1_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_GetConfig32((phdlStepper_Robot_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_GetConfig32((phdlStepper_Microbot_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_GetConfig32((phdlStepper_HighZ_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_GetConfig32((phdlStepper_DensoVS60_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_GetConfig32((phdlStepper_Wachler_DataParams_t*)pDataParams, wConfig, pValue);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, pValue_log, pValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlStepper_Initialize(
                                  void * pDataParams
                                  )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "Referencing_Position");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_STEPPER)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_STEPPER_V1
    case PH_COMP_DL_STEPPER_V1_ID:
        status = phdlStepper_V1_Initialize((phdlStepper_V1_DataParams_t*)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT
    case PH_COMP_DL_STEPPER_ROBOT_ID:
        status = phdlStepper_Robot_Initialize((phdlStepper_Robot_DataParams_t*)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT
    case PH_COMP_DL_STEPPER_MICROBOT_ID:
        status = phdlStepper_Microbot_Initialize((phdlStepper_Microbot_DataParams_t*)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ
    case PH_COMP_DL_STEPPER_HIGHZ_ID:
        status = phdlStepper_HighZ_Initialize((phdlStepper_HighZ_DataParams_t*)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60
    case PH_COMP_DL_STEPPER_DENSO_VS60_ID:
        status = phdlStepper_DensoVS60_Initialize((phdlStepper_DensoVS60_DataParams_t*)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER
    case PH_COMP_DL_STEPPER_WACHLER_ID:
        status = phdlStepper_Wachler_Initialize((phdlStepper_Wachler_DataParams_t*)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

#endif /* NXPBUILD__PHDL_STEPPER */
