/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Generic Stepper Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_STEPPER_V1
#include "V1/phdlStepper_V1.h"
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER

#include <phdlStepper.h>
#include "phdlStepper_Int.h"

phStatus_t phdlStepper_Int_Send(
                                phdlStepper_V1_DataParams_t * pDataParams,
                                uint8_t * pTxBuffer,
                                uint16_t wTxLength
                                )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout  */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 1000));

    /* send data */
    status = phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,pTxBuffer ,wTxLength,0, &bRx , &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }

    return status;
}

phStatus_t phdlStepper_Int_Send_Wait(
                                     phdlStepper_V1_DataParams_t * pDataParams,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint16_t wWaittime
                                     )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wWaittime));

    /*send data*/
    status = phbalReg_Exchange(pDataParams->pBalRegDataParams,PH_EXCHANGE_DEFAULT, pTxBuffer ,wTxLength, 1, &bRx , &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }

    return status;
}

phStatus_t phdlStepper_Int_Wait(uint16_t wWaitTime)
{
#ifdef _WIN32
    phTools_Sleep(wWaitTime);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
#else
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
#endif
}
#endif /* NXPBUILD__PHDL_STEPPER */
