/*
 * Copyright 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardare Thermostream ATS545 Component Application of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlThstrm.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_THSTRM_ATS545

#include "phdlThstrm_ATS545.h"
#include "phdlThstrm_ATS545_Int.h"
#include "../phdlThstrm_Int.h"

phStatus_t phdlThstrm_ATS545_Int_SetSetPointTemp(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    int16_t wTemperature
    )
{
    phStatus_t statusTmp;
    uint8_t aCmd[12];

    /* check input */
    if (wTemperature < -999 || wTemperature > 2250)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_SP_WRONG_TEMP, PH_COMP_DL_THSTRM);
    }

    /* prepare command */
    aCmd[0] = 'S';
    aCmd[1] = 'E';
    aCmd[2] = 'T';
    aCmd[3] = 'P';
    aCmd[4] = ' ';

    if (wTemperature < 0)
    {
        wTemperature = -wTemperature;
        aCmd[9] = (uint8_t)((wTemperature %10) + '0');
        wTemperature /= 10;
        aCmd[7] = (uint8_t)((wTemperature %10) + '0');
        wTemperature /= 10;
        aCmd[6] = (uint8_t)((wTemperature %10) + '0');
        aCmd[5] = '-';
    }
    else
    {
        aCmd[9] = (uint8_t)((wTemperature %10) + '0');
        wTemperature /= 10;
        aCmd[7] = (uint8_t)((wTemperature %10) + '0');
        wTemperature /= 10;
        aCmd[6] = (uint8_t)((wTemperature %10) + '0');
        wTemperature /= 10;
        aCmd[5] = (uint8_t)((wTemperature %10) + '0');
    }
    aCmd[8] = '.';
    aCmd[10] = '\n';
    aCmd[11] = 0;

    /* send command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)aCmd, 11));

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetSetPointTemp(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    int16_t * pwTemperature
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"SETP?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

        return phdlThstrm_Int_MultTeni16((int8_t*)bRx, pwTemperature);
}

phStatus_t phdlThstrm_ATS545_Int_CheckCmdExec(
    phdlThstrm_ATS545_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;
    uint8_t bCmd;
    uint16_t wCmd;

    /* read ESR register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"*ESR?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
    bCmd = (uint8_t) wTemp;

    /* evaluate the ESR */
    if ((bCmd & 0x04) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_QUERY_ERROR, PH_COMP_DL_THSTRM);
    }
    if ((bCmd & 0x10) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_EXECUTION_ERROR, PH_COMP_DL_THSTRM);
    }
    if ((bCmd & 0x20) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_COMMAND_ERROR, PH_COMP_DL_THSTRM);
    }

    /* read the EROR register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"EROR?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wCmd));

    /* evaluate the EROR */
    if ((wCmd & 0x0001) != 0)
    {
        return PH_ADD_COMPCODE( PHDL_THSTRM_ERR_DDE_OVERHEAT, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x0002) != 0)
    {
        return PH_ADD_COMPCODE( PHDL_THSTRM_ERR_DDE_AIR_OPEN_LOOP, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x0010) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_DDE_LOW_AIR_PRESSURE, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x0020) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_DDE_NO_PURGE_FLOW, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x0080) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_DDE_INTERNAL_ERROR, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x0200) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_DDE_FLOW_SENSOR_HARDWARE_ERROR, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x0400) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_DDE_PURGE_HEAT_FAILURE, PH_COMP_DL_THSTRM);
    }
    if ((wCmd & 0x4000) != 0)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_DDE_NO_DUT_SENSOR_SELECTED, PH_COMP_DL_THSTRM);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetHead(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bDirection
    )
{
    phStatus_t statusTmp;

    switch(bDirection)
    {
    case PHDL_THSTRM_HEAD_UP:
         /* move head up */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"HEAD 0\n", 7));
        break;

    case PHDL_THSTRM_HEAD_DOWN:
         /* move down */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"HEAD 1\n", 7));
        /* turn off flow immediately */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"FLOW 0\n", 7));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_THSTRM);
    }
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetHead(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbDirection
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"HEAD?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert the response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
    *pbDirection = (uint8_t)wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetFlow(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bFlow
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[4];
    uint16_t wRxlen;
    uint8_t bTmp;

    if (bFlow)
    {
        /* check if head is DOWN */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"HEAD?\n",
            6,
            4,
            (uint8_t*)bRx,
            &wRxlen));

        /* report back if head is still up */
        if (bRx[0] == '0')
        {
            /* check if head is LOCKED */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
                pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                (uint8_t*)"HDLK?\n",
                6,
                4,
                (uint8_t*)bRx,
                &wRxlen));

            if (bRx[0] == '1')
            {
                /* if head is up but locked flow on is allowed */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"FLOW 1\n", 7));
            }

            else
            {
                return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_HEAD_IS_STILL_UP, PH_COMP_DL_THSTRM);
            }
        }

        /* turn on flow */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"FLOW 1\n", 7));
    }
    else
    {
        /* turn off flow */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"FLOW 0\n", 7));
    }

    /* check if cmd is processed correct */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams));

    /* check if value was set correctly */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_GetFlow(pDataParams, &bTmp));

    if (bTmp != bFlow)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_EXECUTION_ERROR, PH_COMP_DL_THSTRM);
    }

    return statusTmp;
}

phStatus_t phdlThstrm_ATS545_Int_GetFlow(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbFlow
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"FLOW?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert the response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
    *pbFlow = (uint8_t) wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetFlowRate(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bFlowRate
    )
{
    phStatus_t statusTmp;
    uint8_t aCmd[9];

    /* check input */
    if (bFlowRate < 5 || bFlowRate > 18)
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_INVALID_FLOWRATE, PH_COMP_DL_THSTRM);

    /* prepare command */
    aCmd[0] = 'F';
    aCmd[1] = 'L';
    aCmd[2] = 'S';
    aCmd[3] = 'E';
    aCmd[4] = ' ';
    aCmd[6] = (uint8_t)((bFlowRate %10) + '0');
    bFlowRate /= 10;
    aCmd[5] = (uint8_t)((bFlowRate %10) + '0');
    aCmd[7] = '\n';
    aCmd[8] = 0;

    /* send command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)aCmd, 8));

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetFlowRate(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbFlowRate
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"FLSE?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert the response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));

    /* devide the return value by 10 to be conform with the input format, as only integer values are possible */
    *pbFlowRate = (uint8_t)(wTemp/ 10);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetSetPointSoakTime(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint16_t wSoakTime
    )
{

    phStatus_t statusTmp;
    uint8_t aCmd[11];

    /* check input */
    if (wSoakTime > 9999)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_SP_INVALID_SOAK_TIME, PH_COMP_DL_THSTRM);
    }

    /* prepare command */
    aCmd[0]  = 'S';
    aCmd[1]  = 'O';
    aCmd[2]  = 'A';
    aCmd[3]  = 'K';
    aCmd[4]  = ' ';
    aCmd[8] = (uint8_t)((wSoakTime %10) + '0');
    wSoakTime /= 10;
    aCmd[7] = (uint8_t)((wSoakTime %10) + '0');
    wSoakTime /= 10;
    aCmd[6] = (uint8_t)((wSoakTime %10) + '0');
    wSoakTime /= 10;
    aCmd[5] = (uint8_t)((wSoakTime %10) + '0');
    aCmd[9] = '\n';
    aCmd[10] = 0;

    /* set soak time for current setpoint */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)aCmd, 10));

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetSetPointSoakTime(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint16_t * pwSoakTime
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"SOAK?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert and return the response */
    return phdlThstrm_Int_Atoui16((int8_t*)bRx, pwSoakTime);
}

phStatus_t phdlThstrm_ATS545_Int_SetSelectedSetPoint(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bSetPoint
    )
{
    phStatus_t statusTmp;

    /* check input */
    if (bSetPoint != PHDL_THSTRM_SETPOINT_HOT && bSetPoint != PHDL_THSTRM_SETPOINT_AMBIENT && bSetPoint != PHDL_THSTRM_SETPOINT_COLD)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_INVALID_SETPOINT, PH_COMP_DL_THSTRM);
    }

    /* set setpoint */
    if (bSetPoint == PHDL_THSTRM_SETPOINT_HOT)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"SETN 0\n", 7));
    }
    else if (bSetPoint == PHDL_THSTRM_SETPOINT_AMBIENT)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"SETN 1\n", 7));
    }
    else /* bSetPoint == PHDL_THSTRM_SETPOINT_COLD) */
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"SETN 2\n", 7));
    }

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetSelectedSetPoint(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbSetPoint
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"SETN?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
    *pbSetPoint = (uint8_t)wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetSetPointWindow(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bTempWindow
    )
{
    phStatus_t statusTmp;
    uint8_t aCmd[10];

    /* check input */
    if (bTempWindow >= 100)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_INVALID_WINDOW, PH_COMP_DL_THSTRM);
    }

    /* prepare command */
    aCmd[0] = 'W';
    aCmd[1] = 'N';
    aCmd[2] = 'D';
    aCmd[3] = 'W';
    aCmd[4] = ' ';
    aCmd[7] = (uint8_t)((bTempWindow %10) + '0');
    bTempWindow /= 10;
    aCmd[6] = '.';
    aCmd[5] = (uint8_t)((bTempWindow %10) + '0');
    aCmd[8] = '\n';
    aCmd[9] = 0;

    /* send command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)aCmd, 9));

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetSetPointWindow(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbTempWindow
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;
    int16_t wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"WNDW?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_MultTeni16((int8_t*)bRx, &wTemp));
    *pbTempWindow = (uint8_t) wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_ConfigureThermostream(
    phdlThstrm_ATS545_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[4];
    uint16_t wRxlen;

    /* Check for cancellation */
    if (pDataParams->bCancellationToken == PH_ON)
    {
        pDataParams->bCancellationToken = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
    }

    /* SET TEMPERATURE LIMITS */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"ULIM 150\n", 9));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"LLIM -90\n", 9));

    /* CHECK THE STATE OF THE COMPRESSOR */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"COOL?\n",
        6,
        4,
        (uint8_t*)bRx,
        &wRxlen));

    if (bRx[0] == '0')
    {
        /* Check for cancellation */
        if (pDataParams->bCancellationToken == PH_ON)
        {
            pDataParams->bCancellationToken = PH_OFF;
            return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
        }

        /* turn on compressor and wait 65 seconds for compressor to be ready */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_SendWait(pDataParams, (uint8_t *)"COOL 1\n", 7, 65535));
    }

    /* check the execution of the command */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetCurrentTemp(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    int16_t * pwTemperature
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"TMPA?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert and copy the response */
    return phdlThstrm_Int_MultTeni16((int8_t*)bRx, pwTemperature);
}

phStatus_t phdlThstrm_ATS545_Int_GetSetPointTempReached(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbTempReached
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;
    uint16_t wTemp;

    /* check for the at temperature (soak time has elapsed)-bit in Temperature event condition register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"TECR?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
    *pbTempReached = (uint8_t)(wTemp & 0x01);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetComprState(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bCompr
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[4];
    uint16_t wRxlen;

    if (bCompr)
    {
        /* check if compressor already on */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"COOL?\n",
            6,
            4,
            (uint8_t*)bRx,
            &wRxlen));

        /* If compressor switched of*/
        if (bRx[0] == '0')
        {
            /* turn on compressor */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"COOL 1\n", 7));
        }


    }
    else
    {
        /* turn off compressor */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"COOL 0\n", 7));
    }

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetComprState(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbComprState
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;
    int16_t wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"COOL?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_MultTeni16((int8_t*)bRx, &wTemp));
    *pbComprState = (uint8_t) wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_Send(
                                       phdlThstrm_ATS545_DataParams_t * pDataParams,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength
                                       )
{
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Send(pDataParams->pBalRegDataParams, pTxBuffer, wTxLength));

    /* check the execution of the command */
    statusTmp = phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);

    /* handle expected querry error (caused by reading and timeout) */
    if ((statusTmp & PH_ERR_MASK) == PHDL_THSTRM_ERR_QUERY_ERROR)
    {
        statusTmp = PH_ERR_SUCCESS;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_DL_THSTRM);
}
phStatus_t phdlThstrm_ATS545_Int_SendWait(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pTxBuffer,
    uint16_t wTxLength,
    uint16_t wWaittime
    )
{
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_SendWait(pDataParams->pBalRegDataParams, pTxBuffer, wTxLength, wWaittime));

    /* check the execution of the command */
    statusTmp = phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);

    /* handle expected querry error (caused by reading and timeout) */
    if ((statusTmp & PH_ERR_MASK) == PHDL_THSTRM_ERR_QUERY_ERROR)
    {
        statusTmp = PH_ERR_SUCCESS;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_ATS545_Int_SetHeadLockedState(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t bState
    )
{
    phStatus_t statusTmp;

    if(bState == 1)
    {
        /* Lock Head */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"HDLK 1\n", 7));
    }
    else if(bState == 0)
    {
        /* Unlock Head */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_ATS545_Int_Send(pDataParams, (uint8_t *)"HDLK 0\n", 7));
    }
    else
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_COMMAND_ERROR, PH_COMP_DL_THSTRM);
    }

    /* check if cmd is processed correct */
    return phdlThstrm_ATS545_Int_CheckCmdExec((phdlThstrm_ATS545_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_ATS545_Int_GetHeadLockedState(
    phdlThstrm_ATS545_DataParams_t * pDataParams,
    uint8_t * pbState
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;

    /* read the value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"HDLK?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert the response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
    *pbState = (uint8_t) wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

#endif /* NXPBUILD__PHDL_THSTRM_ATS545 */
