/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Dut Hal specific definition for Symmetric Crypto commands.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 */


#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTOSYM
#include <ph_RefDefs.h>
#include <phhalHw_DUT_Cmd.h>

#include "../phhalHw_DUT.h"

phStatus_t phhalHw_DUT_Cmd_CryptoSym_Init(phhalHw_DUT_DataParams_t * pDataParams)
{
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_INIT,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_DeInit(phhalHw_DUT_DataParams_t * pDataParams)
{
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DEINIT,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_GetLastStatus(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wStatusMsgLen, int8_t * pStatusMsg,
    int32_t * pStatusCode)
{
    phStatus_t wStatus = 0;
    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_GET_STATUS,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy((uint8_t *) pStatusCode, pResponse, 4);

    /* Copy the status information if available. */
    if(wRspLen > 4)
    {
        wStatusMsgLen = (uint16_t) (pResponse[4] | (pResponse[5] << 8));
        (void) memcpy(pStatusMsg, &pResponse[6], 4);
    }

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_InvalidateKey(phhalHw_DUT_DataParams_t * pDataParams)
{
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_INVALIDATE_KEY,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_SetConfig(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    uint8_t aCmd[4];
    uint8_t bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wValue, 2);
    bCmdLen += 2;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_SET_CONFIG,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_GetConfig(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t wStatus = 0;

    uint8_t aCmd[2];
    uint8_t bCmdLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_GET_CONFIG,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy((uint8_t *) pValue, pResponse, wRspLen);

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_Encrypt(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, const uint8_t * pPlainBuff,
    uint16_t wBuffLen, uint8_t * pEncBuff)
{
    phStatus_t wStatus = 0;

    uint8_t aCmd[4];
    uint8_t bCmdLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wOption, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wBuffLen, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_ENCRYPT,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_ENCRYPT,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) pPlainBuff,
        wBuffLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy(pEncBuff, pResponse, wRspLen);

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_Decrypt(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, const uint8_t * pEncBuff,
    uint16_t wBuffLen, uint8_t * pPlainBuff)
{
    phStatus_t wStatus = 0;

    uint8_t aCmd[4];
    uint8_t bCmdLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wOption, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wBuffLen, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DECRYPT,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DECRYPT,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) pEncBuff,
        wBuffLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy(pPlainBuff, pResponse, wRspLen);

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_CalculateMac(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, const uint8_t* pData,
    uint16_t wDataLen, uint8_t * pMac, uint8_t * pMacLen)
{
    phStatus_t wStatus = 0;

    uint8_t aCmd[4];
    uint8_t bCmdLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wOption, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wDataLen, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_CALCULATE_MAC,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_CALCULATE_MAC,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy(pMac, pResponse, wRspLen);
    *pMacLen = (uint8_t) wRspLen;

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_LoadIv(phhalHw_DUT_DataParams_t * pDataParams, const uint8_t * pIV, uint8_t bIVLen)
{
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_IV,
        PH_EXCHANGE_DEFAULT,
        (uint8_t *) pIV,
        bIVLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_LoadKey(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint16_t wKeyType)
{
    uint8_t aCmd[6];
    uint8_t bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyVer, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen += 2;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_LoadKeyDirect(phhalHw_DUT_DataParams_t * pDataParams, const uint8_t * pKey, uint16_t wKeyType)
{
    phStatus_t wStatus = 0;
    uint8_t bKeyLen = 0;

    uint8_t aCmd[3];
    uint8_t bCmdLen = 0;

    /* Compute Key Size */
    switch(wKeyType)
    {
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_DES:
            bKeyLen = 8;
            break;

        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_2K3DES:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_3K3DES:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES192:
            bKeyLen = 24;
            break;

        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        default:
            bKeyLen = 0;
            break;
    }

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY_DIRECT,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY_DIRECT,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) pKey,
        bKeyLen,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_DiversifyKey(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t * pDivInput, uint8_t bDivInLen, uint8_t * pDivKey, uint8_t * pDivKeyLen)
{
    phStatus_t wStatus = 0;

    uint8_t aCmd[6];
    uint8_t bCmdLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wOption, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyVer, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_KEY,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_KEY,
        PH_EXCHANGE_BUFFER_LAST,
        pDivInput,
        bDivInLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy(pDivKey, pResponse, wRspLen);
    *pDivKeyLen = (uint8_t) wRspLen;

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_CryptoSym_DiversifyDirectKey(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pKey, uint16_t wKeyType,
    uint8_t * pDivInput, uint8_t bDivInLen, uint8_t * pDivKey, uint8_t * pDivKeyLen)
{
    phStatus_t wStatus = 0;
    uint8_t bKeyLen = 0;

    uint8_t aCmd[5];
    uint8_t bCmdLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRspLen = 0;

    /* Compute Key Size */
    switch(wKeyType)
    {
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_DES:
            bKeyLen = 8;
            break;

        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_2K3DES:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_3K3DES:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES192:
            bKeyLen = 24;
            break;

        case PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        default:
            bKeyLen = 0;
            break;
    }

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wOption, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen += 2;

    aCmd[bCmdLen++] = bKeyLen;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_DIRECT_KEY,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_DIRECT_KEY,
        PH_EXCHANGE_BUFFER_CONT,
        pKey,
        bKeyLen,
        &pResponse,
        &wRspLen));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_DIRECT_KEY,
        PH_EXCHANGE_BUFFER_LAST,
        pDivInput,
        bDivInLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy(pDivKey, pResponse, wRspLen);
    *pDivKeyLen = (uint8_t) wRspLen;

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTOSYM */
