/*
 * Copyright 2019 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phTools.h>
#include <phhalHw.h>
#include <math.h>
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <stdlib.h>
#pragma warning(pop)            /* PRQA S 3116 */

#ifdef NXPBUILD__PHHAL_HW_ISO3

#include <time.h>

#include "phhalHw_ISO3_cmd.h"
#include "phhalHw_ISO3_int.h"
#include "phhalHw_ISO3_reg.h"
#include "phhalHw_ISO3.h"


phStatus_t phhalHw_ISO3_Cmd_SetServerTime(
                        phhalHw_ISO3_DataParams_t * pDataParams,
                        struct tm* pServerTime
                        )
{
    phStatus_t statusTmp;
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pBalDispatchDataParams);

    /* Set time of petalinux server (useful for logging) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
        pDataParams,
        PHHAL_HW_ISO3_SERVER_CMD_SET_TIME,
        (uint8_t*) pServerTime,
        sizeof(struct tm)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}


phStatus_t phhalHw_ISO3_Cmd_SendServerCommand(
                        phhalHw_ISO3_DataParams_t * pDataParams,
                        uint32_t dwServerCommand,
                        uint8_t * pParams,
                        uint32_t dwParamsSizeBytes
                        )
{
    phhalHw_ISO3_PlsHeaderPackage_t plsSendHeader;
    uint8_t * buffer;
    phStatus_t status = PH_ERR_SUCCESS;
    uint32_t counter = 0;
    uint32_t num_chunks;
    uint16_t remaining_bytes = 0;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pBalDispatchDataParams);

    if(dwParamsSizeBytes > UINT_MAX - sizeof(dwServerCommand))
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_ISO3);

    plsSendHeader.headerFields.bSenderId = (uint8_t) ISO3_PLS_HOST_SENDER_ID;
    plsSendHeader.headerFields.bSignature = (uint8_t) ISO3_PLS_HEADER_SIGNATURE;
    plsSendHeader.headerFields.wExpectedSize = 0;
    plsSendHeader.headerFields.dwPackageSize = dwParamsSizeBytes + sizeof(dwServerCommand);

    buffer = (uint8_t*) malloc(sizeof(dwServerCommand) + dwParamsSizeBytes);

    /* push command */
    memcpy(buffer, &dwServerCommand, sizeof(dwServerCommand));

    /* push args */
    if (dwParamsSizeBytes > 0)
    {
        memcpy(buffer + sizeof(dwServerCommand), pParams, dwParamsSizeBytes);
    }

    /* send header */
    status = phbalReg_Exchange(
        pDataParams->pBalDispatchDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*) &plsSendHeader,
        sizeof(plsSendHeader),
        0, NULL, NULL);

    if(status != PH_ERR_SUCCESS)
    {
        free(buffer);
        PH_CHECK_SUCCESS(status);
    }

    remaining_bytes = (dwParamsSizeBytes + sizeof(dwServerCommand)) % USHRT_MAX;
    num_chunks = (dwParamsSizeBytes + sizeof(dwServerCommand)) / USHRT_MAX;
    /* send chunks of data from exchange buffer */
    for(counter = 0; counter < num_chunks; counter++)
    {
        status = phbalReg_Exchange(
            pDataParams->pBalDispatchDataParams,
            PH_EXCHANGE_DEFAULT,
            buffer + (counter * USHRT_MAX),
            USHRT_MAX,
            0, NULL, NULL);

        if(status != PH_ERR_SUCCESS)
        {
            free(buffer);
            PH_CHECK_SUCCESS(status);
        }
    }

    if(remaining_bytes > 0)
    {
        status = phbalReg_Exchange(
            pDataParams->pBalDispatchDataParams,
            PH_EXCHANGE_DEFAULT,
            buffer + (counter * USHRT_MAX),
            remaining_bytes,
            0, NULL, NULL);
    }

    free(buffer);
    PH_CHECK_SUCCESS(status);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phhalHw_ISO3_Cmd_RecvServerResponse(
                        phhalHw_ISO3_DataParams_t * pDataParams,
                        uint8_t* pResponse,
                        uint16_t wExpectedSizeBytes
                        )
{
    phStatus_t statusTmp;
    uint16_t wRxBufferSizeBytes = wExpectedSizeBytes;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pBalDispatchDataParams);

     PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDispatchDataParams,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        wRxBufferSizeBytes,
        pResponse,
        &wExpectedSizeBytes
        ));

    if (wExpectedSizeBytes != wRxBufferSizeBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_READ_WRITE_ERROR, PH_COMP_DL_ISO3);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phhalHw_ISO3_Cmd_WriteFpgaRegister(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint32_t dwAddr,
                            uint8_t * pValue,
                            uint32_t dwSizeBytes
                            )
{
    phStatus_t statusTmp;
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pValue);

    /* reset buffer */
    pDataParams->wIntNumValidBytes = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWritePtr(pDataParams, dwAddr, pValue, (uint16_t)dwSizeBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_ReadFpgaRegister(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint32_t dwAddr,
                            uint8_t * pValue,
                            uint32_t dwSizeBytes
                            )
{
    phStatus_t statusTmp;
    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Reset buffer */
    pDataParams->wIntNumValidBytes = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterRead(pDataParams, dwAddr, (uint16_t)dwSizeBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    /* Copy rx data */
    if(pDataParams->wIntNumValidRxBytes == dwSizeBytes)
    {
        memcpy(pValue, pDataParams->pIntBuffer, dwSizeBytes);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_InitFpga(
                            phhalHw_ISO3_DataParams_t * pDataParams
                            )
{
    phStatus_t statusTmp;
    uint16_t wVersion;
    uint16_t wRevision;
    uint32_t fpgsVersion;
    uint16_t wWaveShapePrescalePercent;
    uint32_t dwTmpValue = 0;

    time_t ltime;
    struct tm* timeinfo;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* reset buffer */
    pDataParams->wTxBufLen = 0;
    pDataParams->wIntNumValidBytes = 0;
    pDataParams->wRxBufLen = 0;

    /* Check version of PetaLinux Server */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_GetPLSVersion(
        pDataParams,
        &wVersion,
        &wRevision));

    if(wVersion != PHHAL_HW_ISO3_PLS_CUR_VERSION || wRevision != PHHAL_HW_ISO3_PLS_CUR_REVISION)
    {
        return PH_ADD_COMPCODE(PHHAL_HW_ISO3_ERR_INCOMPATIBLE_SERVER_VERSION, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_FPGA_VERSION, (uint8_t*)(&fpgsVersion), sizeof(uint32_t)));

    if ((fpgsVersion & 0x0000FFFF) != PHHAL_HW_ISO3_FPGA_VERSION)
    {
        return PH_ADD_COMPCODE(PHHAL_HW_ISO3_ERR_INCOMPATIBLE_FPGA_VERSION, PH_COMP_HAL);
    }

    /* Set server time to host system time */
    time(&ltime);
    timeinfo = localtime(&ltime);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SetServerTime(pDataParams, timeinfo));

    /* Init */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_MISC_CTRL, 0x00000020));       /*1  Reset the ADC */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_MISC_CTRL, 0x00000000));       /*0  dont reset ADC */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_CLK_CTRL, 0x002c0040)); /*Reg 0 -> 00 2C 00 4 0 (default value) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_CLK_CTRL, 0x83840002)); /*Reg 2 -> 83 84 00 0 2 (default is 83 40 00 0 2) Output 2 -> divider = 4 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_CLK_CTRL, 0xeb800004)); /*Reg 4 -> eb 80 00 0 4 (default is 81 80 00 0 4) Output 4 -> divider = 2  1 110101 1 vs 1 000000 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_CLK_CTRL, 0x83800317)); /*Reg 7 -> 83 80 03 1 7 (default is EB 04 07 1 7) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_MON_CTRL, 0x12c08080));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_MON_CTRL, 0x13408000));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_DAC_CTRL, 0x00001704));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_ADC_CTRL, 0x00004180));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_ADC_CTRL, 0x000044d8));

    /* Start dac */
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FMC150_CLK_CTRL, 0x83840802);                                  /*Reg 2 -> PH2ADJC1 is set to 1?? 83 84 08 0 2 (default is 83 40 00 0 2) Output 2 -> divider = 4 */
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_MISC_CTRL, PHHAL_HW_ISO3_MISC_CMD_START_DAC);

    /* Init TX_CTRL */
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl);

    /* Init RX_CTRL */
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_RX_CTRL_2, pDataParams->dwRxCtrl2);

    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL, PH_ON);
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_GAIN_CTRL, pDataParams->dwDacGain);


    /* set the venus receiver to the default timeout values */
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_PICC2PCD_DELAY, pDataParams->dwPicc2PcdDelay);
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_PCD2PICC_TIMEOUT, pDataParams->dwPcd2PiccTimeout);

    /* set venus receiver signal detection threshold */
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_DGRM_SIG_DET, pDataParams->dwDgrmSigDetTh);

    /* Set default debug register configuation */
    dwTmpValue = ((0xFF & pDataParams->wDebugConfig3) << 24) |
                 ((0xFF & pDataParams->wDebugConfig2) << 16) |
                 ((0xFF & pDataParams->wDebugConfig1) << 8) |
                 ((0xFF & pDataParams->wDebugConfig0) << 0);
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_DEBUG_CONFIG, dwTmpValue);

    /* Send and reset buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    /* Reset ADC Fifo */
    phTools_Sleep(1);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_MISC_CTRL, PHHAL_HW_ISO3_MISC_CMD_START_DAC | 0x00000100));   /*1  reset ADC FIFO */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_MISC_CTRL, PHHAL_HW_ISO3_MISC_CMD_START_DAC));                /*0  dont reset ADC FIFO */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    /* Disable bitgrid adjustment for Miller Min/Max */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_ConfigMillerTest(pDataParams, 0));

    /* get the prescaled waveshape value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_PRESCALE_PERCENT, &wWaveShapePrescalePercent));

    /* This will load the field on/off waveshape according to the set waveshape percentage */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_PRESCALE_PERCENT, wWaveShapePrescalePercent));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_ReadFpgaSignalData(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            int16_t *  pwSignalData1,
                            int16_t *  pwSignalData2,
                            uint16_t * pwSize
                            )
{
    uint16_t wRingBufferOffsetT1;
    uint16_t wRingBufferOffsetT2;
    uint16_t wCounter;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pwSignalData1);
    PH_ASSERT_NULL(pwSignalData2);
    PH_ASSERT_NULL(pwSize);

    pDataParams->wIntNumValidRxBytes = 0;
    pDataParams->wIntNumValidBytes = 0;
    pDataParams->wRxExpectedInputBytes = 0;

    if (*pwSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    phhalHw_ISO3_Int_BuildRegisterRead(pDataParams, PHHAL_HW_ISO3_REG_FDT_DATA, *pwSize * 4);
    phhalHw_ISO3_Int_DirectExchange(pDataParams);

    wRingBufferOffsetT1 = pDataParams->wLastFdtValueT1 % PHHAL_HW_ISO3_FDT_DATA_COUNT;
    wRingBufferOffsetT2 = pDataParams->wLastFdtValueT2 % PHHAL_HW_ISO3_FDT_DATA_COUNT;

    for (wCounter = 0; wCounter < *pwSize; wCounter++)
    {
        pwSignalData1[wCounter]  = (((uint32_t*)pDataParams->pIntBuffer)[(wRingBufferOffsetT1 + wCounter) % PHHAL_HW_ISO3_FDT_DATA_COUNT] & 0x0000FFFF);
        pwSignalData2[wCounter]  = (((uint32_t*)pDataParams->pIntBuffer)[(wRingBufferOffsetT2 + wCounter) % PHHAL_HW_ISO3_FDT_DATA_COUNT] & 0xFFFF0000) >> 16;
    }

    pDataParams->wIntNumValidRxBytes = 0;
    pDataParams->wIntNumValidBytes = 0;
    pDataParams->wRxExpectedInputBytes = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_WriteFpgaOutBuffer(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t * pTxData,
                            uint16_t  wTxLenghBytes,
                            uint8_t * pRxData
                            )
{
    phStatus_t statusTmp;
    /* parameter checks */
    PH_ASSERT_NULL(pDataParams);

    /* reset buffers */
    pDataParams->wIntNumValidBytes = 0;
    pDataParams->wIntNumValidRxBytes = 0;
    pDataParams->wRxExpectedInputBytes = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildOutBufferWrite(pDataParams, pTxData, wTxLenghBytes))
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    /* Copy rx data */
    if (pDataParams->wIntNumValidRxBytes > 0)
    {
        memcpy(pRxData, pDataParams->pIntBuffer, wTxLenghBytes);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_PiccExchange(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t * pTxData,
                            uint16_t  wTxLenghBytes,
                            uint8_t ** ppRxData,
                            uint16_t * pRxLenghBytes
                            )
{
    phStatus_t statusTmp;
    phStatus_t statusExchange;
    uint32_t dwExchangeCounter;
    uint16_t wWaveShapeSlot;
    phhalHw_ISO3_TxCtrlRegister_t temp;
    uint32_t rxTimeoutUs = 0;
    uint32_t timeETU = 0;
    uint32_t txWaitTime = 0;
    uint32_t ovsFilterCtrl = 0;
    uint32_t dwTxStat;

    /* parameter checks */
    PH_ASSERT_NULL(pDataParams);

    /* reset buffers */
    pDataParams->wIntNumValidBytes = 0;
    pDataParams->wIntNumValidRxBytes = 0;
    pDataParams->wRxExpectedInputBytes = 0;

    pDataParams->dwRxStatReg = 0;

    /* Init rx returns with empty */
    if (pRxLenghBytes != NULL)
    {
        *pRxLenghBytes = 0;
    }
    if (ppRxData != NULL)
    {
        *ppRxData = NULL;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_EXCHANGE, &wWaveShapeSlot));
    if (ISO3_WAVESHAPE_SLOT_GET_IN_USE(wWaveShapeSlot) == 0)
    {
        /* no wave shape loaded since last SetConfig(PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, ...) */
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    if (ISO3_OVS_FILTER_SLOT_GET_IN_USE(wWaveShapeSlot) == 1)
    {
        /* reset first */
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_coeff_enable = PH_OFF;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_coeff_set_select = 0;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_enable = PH_OFF;
		((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_resetn = PH_OFF; /* reset_n -> negative logic -> filter is resetted here */

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_OVS_FILTER_CTRL,
            ovsFilterCtrl));

        /* set overshoot filter to current slot */
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_coeff_enable = PH_ON;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_coeff_set_select = wWaveShapeSlot;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_enable = PH_ON;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_resetn = PH_ON; /* reset_n -> negative logic -> filter is not resetted here */

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_OVS_FILTER_CTRL,
            ovsFilterCtrl));
    }

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* deactivate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL, PH_ON));
    }

    /* tx data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWritePtr(
        pDataParams,
        pDataParams->dwNextTxBufferAddress,
        pTxData,
        wTxLenghBytes));

    /* rx timeout */
    txWaitTime = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXWAIT_US]; /* Not sure how this is used by ISO3 transmitter */
    rxTimeoutUs = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS]*1000 + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US]; /* One of them is 0. MS or US. */
    timeETU = 0;

    timeETU = (uint32_t)( (float)txWaitTime * 13.56f / (float)(128 >> ISO3_REG_TX_CTRL_0.datarate) );
    if (timeETU > 0xFFFF) {
        ISO3_REG_VENUS_RX_TIMEOUT.rxEof2txSoftime = 0xFFFF;
    }
    else
    {
        ISO3_REG_VENUS_RX_TIMEOUT.rxEof2txSoftime = timeETU;
    }

    pDataParams->dwPcd2PiccTimeout = (uint32_t)((float32_t)rxTimeoutUs * 135.6f);

    /* Add the SOF offset depending on the Card Type and DR */
    if (ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_A)
    {
        /* Worst case: SOF */
        pDataParams->dwPcd2PiccTimeout += (uint32_t)((float32_t)(1) * (float32_t)(128 >> ISO3_REG_TX_CTRL_0.datarate) * 10.0f);
    }
    else if (ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_B)
    {
        /* Worst case: SOF */
        pDataParams->dwPcd2PiccTimeout += (uint32_t)((float32_t)(14 /*14 ETU SOF*/) * (float32_t)(128 >> ISO3_REG_TX_CTRL_0.datarate) * 10.0f);
    }
    else if (ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_F)
    {
        /* Worst case: Preamble + sync + 256bytes * 8bits */
        pDataParams->dwPcd2PiccTimeout += (uint32_t)((float32_t)(48 + 2 * 8) * (float32_t)(128 >> ISO3_REG_TX_CTRL_0.datarate) * 10.0f);
    }
    else if (ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_ISO_15693)
    {
        /* Worst case for ISO15693: SOF + flags + parameters */

        /* for ICODE 1 out of 256 coding only the basic data rate (1.6kb/s) is used in operation. */
        if (ISO3_REG_TX_CTRL_0.datarate == 0 && ISO3_REG_TX_CTRL_0.iso15693mode == PHHAL_HW_ISO3_ISO_15693_MODE_256 && ISO3_REG_TX_CTRL_1.double_data_rate == PH_OFF)
        {
            pDataParams->dwPcd2PiccTimeout += (uint32_t)((float32_t)(15 + 8 + 8) * (float32_t)(8192 >> ISO3_REG_TX_CTRL_0.datarate) * 10.0f);
        }
        /* 26.48, 53, 106, 212 */
        else if (ISO3_REG_TX_CTRL_0.datarate < 4 && ISO3_REG_TX_CTRL_0.iso15693mode == PHHAL_HW_ISO3_ISO_15693_MODE_4 && ISO3_REG_TX_CTRL_1.double_data_rate == PH_OFF)
        {
            pDataParams->dwPcd2PiccTimeout += (uint32_t)((float32_t)(15 + 8 + 8) * (float32_t)(1024 >> ISO3_REG_TX_CTRL_0.datarate) * 10.0f);
        }
        else
        {
            /* invalid data rate for the current card type */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }
    else
    {
        /* unknown card type */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_PCD2PICC_TIMEOUT, pDataParams->dwPcd2PiccTimeout));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_PICC2PCD_DELAY, pDataParams->dwPicc2PcdDelay));

    /* Transmit control reg 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_1, pDataParams->dwTxCtrl1));

    /* change target tx data address for next exchange command */
    if (pDataParams->dwNextTxBufferAddress == PHHAL_HW_ISO3_REG_TX_DATA)
    {
        pDataParams->dwNextTxBufferAddress = PHHAL_HW_ISO3_REG_TX_DATA_1;
    }
    else
    {
        pDataParams->dwNextTxBufferAddress = PHHAL_HW_ISO3_REG_TX_DATA;
    }

    /* If startup test is active, calculate powerup time and switch on the transmitter, if not done already */
    if (ISO3_REG_TX_CTRL_0.state == PHHAL_HW_ISO3_TX_STATE_POWER_DOWN)
    {
        /* save current datarate and reset later. Accuracy is best at highest datarate because time is converted to ETUs */
        temp.datarate = ISO3_REG_TX_CTRL_0.datarate;
        temp.cardType = ISO3_REG_TX_CTRL_0.cardType;
        temp.iso15693mode = ISO3_REG_TX_CTRL_0.iso15693mode;

        ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_A;
        ISO3_REG_TX_CTRL_0.datarate = 3;
        ISO3_REG_TX_CTRL_0.iso15693mode = 0;

        /* set pointer to field on / off waveshape */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(
            pDataParams,
            ISO3_WAVESHAPE_TYPE_FIELD_ON));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(
            pDataParams,
            ISO3_WAVESHAPE_TYPE_FIELD_OFF));

        /* Just to make sure that tranmitter is in power off state*/
        ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_DOWN;
        ISO3_REG_TX_CTRL_0.operand = 0x0;
        phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl);

        /* build field on command */
        ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_UP;
        ISO3_REG_TX_CTRL_0.operand = pDataParams->dwFieldRecoveryTimeCycles;
        phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl);

        /* reset actual state */
        ISO3_REG_TX_CTRL_0.datarate = temp.datarate;
        ISO3_REG_TX_CTRL_0.cardType = temp.cardType;
        ISO3_REG_TX_CTRL_0.iso15693mode = temp.iso15693mode;
    }

    /* check current waveshape slot */
    if (pDataParams->dwCurrentWaveShapeType  != ISO3_WAVESHAPE_TYPE_EXCHANGE ||
        pDataParams->bCurrentWaveshapeSlot   != wWaveShapeSlot)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(pDataParams, ISO3_WAVESHAPE_TYPE_EXCHANGE));
    }

    /* 4. build command, that actually transmits the bits from the buffer */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_TRANSMIT;
    ISO3_REG_TX_CTRL_0.operand = wTxLenghBytes * 8;

    if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] > 0)
    {
        ISO3_REG_TX_CTRL_0.operand -= (8 - pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS]);
    }
    ISO3_REG_TX_CTRL_0.operand--;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));

    /* build register write command to wait until the expected receive time has passed */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_UP;
    ISO3_REG_TX_CTRL_0.operand = 0x1;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));

    /* after the last bit was received wait for the inverse frame delay time. */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_UP;
    ISO3_REG_TX_CTRL_0.operand = 0x1;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));

    /* Append exchange header */
    pDataParams->dwExchangeCounter++;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildOutBufferWrite(pDataParams, (uint8_t*) &pDataParams->dwExchangeCounter, sizeof(pDataParams->dwExchangeCounter)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterRead(pDataParams, PHHAL_HW_ISO3_REG_FDT_STAT, PHHAL_HW_ISO3_FDT_STAT_SIZE_BYTE));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterRead(pDataParams, PHHAL_HW_ISO3_REG_TX_STAT, sizeof(dwTxStat)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterRead(pDataParams, PHHAL_HW_ISO3_REG_PEAK_SENS_SIGNAL, sizeof(pDataParams->dwPeakSensSignal)));

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* re-activate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL, PH_OFF));
    }


    /* After transmission, turn off overshoot filter */
    if (ISO3_OVS_FILTER_SLOT_GET_IN_USE(wWaveShapeSlot) == 1)
    {
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_coeff_enable = PH_OFF;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_coeff_set_select = wWaveShapeSlot;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_enable = PH_OFF;
        ((phhalHw_ISO3_OvsFilterCtrl_t*)&ovsFilterCtrl)->ovs_filter_resetn = PH_OFF;

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_OVS_FILTER_CTRL,
            ovsFilterCtrl));
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    /* check rx counter */
    dwExchangeCounter = *((uint32_t*)pDataParams->pIntBuffer);
    if (dwExchangeCounter != pDataParams->dwExchangeCounter)
    {
        return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
    }

    pDataParams->wLastFdtValueT2 = *((uint16_t *) (pDataParams->pIntBuffer + 4));
    pDataParams->wLastFdtValueT1 = *((uint16_t *) (pDataParams->pIntBuffer + 6));

    pDataParams->dwRxStatReg = *((uint32_t *)(pDataParams->pIntBuffer + 8));

    dwTxStat = *((uint32_t *)(pDataParams->pIntBuffer + 12));

    /* Read SENS peak value - It can be used to check if input signal was clipped at ADC input on SENS path */
    pDataParams->dwPeakSensSignal = *((uint32_t *) (pDataParams->pIntBuffer + 16));

    /* Receive rx_data */
    if (ISO3_REG_RX_STAT.wNumBytesReceived > 0x0)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterRead(pDataParams, PHHAL_HW_ISO3_REG_RX_DATA, (uint16_t)ISO3_REG_RX_STAT.wNumBytesReceived));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

        /* double check the response length for Felica card responses. sometimes the Venus receiver receives too many bits */
        if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
        {
            /* Set the number of last bits to zero because its seems that venus indicates wrong values for felica */
            ISO3_REG_RX_STAT.bRxNumLastBits = 0;
            /* first byte contains a length byte (excluding the 2 bytes for crc) */
            if ((uint32_t)pDataParams->pIntBuffer[0] + 2 < ISO3_REG_RX_STAT.wNumBytesReceived)
            {
                ISO3_REG_RX_STAT.wNumBytesReceived = ((uint32_t)pDataParams->pIntBuffer[0] + 2);
            }
        }

        if (pRxLenghBytes != NULL)
        {
            *pRxLenghBytes = (uint16_t)ISO3_REG_RX_STAT.wNumBytesReceived;
        }
        if (ppRxData != NULL)
        {
            *ppRxData = pDataParams->pIntBuffer;
        }
        statusExchange = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    else
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    /* Check if there are any error flags set */
    if ((*((phhalHw_ISO3_TxStatRegister_t*)&(dwTxStat))).dwRxWdtCounter == 0)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_NOISE_ERROR, PH_COMP_HAL);
    }
    else if (ISO3_REG_RX_STAT.bRxLenError == PH_ON)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }
    else if(ISO3_REG_RX_STAT.bRxLatencyError == PH_ON ||
            ISO3_REG_RX_STAT.bRxBmaWriteError == PH_ON)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    else if (ISO3_REG_RX_STAT.bRxBufferOvflError == PH_ON)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    else if (ISO3_REG_RX_STAT.bRxEmdError == PH_ON)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_NOISE_ERROR, PH_COMP_HAL);
    }
    else if (ISO3_REG_RX_STAT.bRxProtError == PH_ON)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
    }
    else if (ISO3_REG_RX_STAT.bRxDataError == PH_ON)
    {
        statusExchange = PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
    }

    return statusExchange;
}

phStatus_t phhalHw_ISO3_Cmd_LoadOVSCoefficients(
    phhalHw_ISO3_DataParams_t * pDataParams,
    uint32_t * pCoefficients,
    uint16_t wSize,
    uint8_t bSlotID
    )
{
    phStatus_t statusTmp;
    uint16_t wCurrentMaxNumSlots;
    phhalHw_ISO3_OvsFilterMemCtrl_t ovsFilterMemCtrl;
    uint16_t i;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    /* Check coefficients length */
    if (wSize != PHHAL_HW_ISO3_OVS_FILTER_COEFFICIENTS_PER_SLOT * sizeof(uint32_t))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check slot range */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, &wCurrentMaxNumSlots));

    if (bSlotID >= wCurrentMaxNumSlots || bSlotID >= PHHAL_HW_ISO3_OVS_FILTER_MAX_SLOT_NUM)
    {
        /* There are only #ISO3_OVS_FILTER_MAX_SLOT_NUM of slots available for overshoot filter coefficients */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    for (i = 0; i < PHHAL_HW_ISO3_OVS_FILTER_COEFFICIENTS_PER_SLOT; i++)
    {
        ovsFilterMemCtrl.wr_enable = PH_ON;
        ovsFilterMemCtrl.value = pCoefficients[i];
        ovsFilterMemCtrl.addr = (bSlotID * PHHAL_HW_ISO3_OVS_FILTER_COEFFICIENTS_PER_SLOT) + i;
        ovsFilterMemCtrl.unused = 0;

		/* Load wave shape into slot */
        phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_OVS_FILTER_MEMORY_CTRL,
            (uint8_t*)&ovsFilterMemCtrl,
			(wSize / 17)
            );
    }

    ovsFilterMemCtrl.wr_enable = PH_OFF;

    phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pDataParams,
        PHHAL_HW_ISO3_REG_OVS_FILTER_MEMORY_CTRL,
        (uint8_t*)&ovsFilterMemCtrl,
        sizeof(ovsFilterMemCtrl)
        );

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_LoadWaveShape(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t * pWaveShape,
                            uint16_t wSizeBytes,
                            uint8_t bSlotID
                            )
{
    phStatus_t statusTmp;
    uint16_t wCurrentSlotID;
    uint16_t wSlotSizeBytes;
    uint32_t dwStartAddress;
    uint32_t dwCounter;
    uint32_t dwShadowBufferStartAddr;
    uint16_t wSeq8Mode;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    /* Check slot size */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_SIZE, &wSlotSizeBytes));

    /* check transmitter mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_SEQ8_MODE, &wSeq8Mode));

    if (wSlotSizeBytes == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Check wave shape length */
    if (wSizeBytes == 0 || wSizeBytes > wSlotSizeBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check slot range */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, &wCurrentSlotID));

    if (wCurrentSlotID <= (uint16_t)bSlotID && wSeq8Mode == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* reset buffer */
    pDataParams->wIntNumValidBytes = 0;

    /* calculate start address */
    /* 0x0800 [ RESERVED | SLOT1 | SLOT2 | ... | SLOTn  ] 0x2800 */
    /* Check if Sequence 8 is enabled */
    if (wSeq8Mode == PH_ON)
    {
        dwStartAddress = ISO3_WAVESHAPE_BUFFER_BASE_ADDRESS;
        /* check size. 22 * 1280 samples is still stable.
           Remove this, once this bug is fixed in the FPGA and the whole buffer can be used */
        if (wSizeBytes > (sizeof(uint16_t) * 22 * 1280))
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
    }
    else
    {
        dwStartAddress = ISO3_WAVESHAPE_BUFFER_BASE_ADDRESS +
                       ISO3_WAVESHAPE_RESERVED_BUFFER_OFFSET +
                       bSlotID * (wSlotSizeBytes / sizeof(uint32_t));
    }

    /* Load wave shape into slot */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWritePtr(
                  pDataParams,
                  dwStartAddress,
                  (uint8_t*)pWaveShape,
                  wSizeBytes));

    /* Set inUse flag for this slot */
    ISO3_WAVESHAPE_SLOT_SET_IN_USE(bSlotID);

    /* If sequence8 mode is enabled, store the sample count of the waveshape in the operand and seq8samples bit fields */
    if ((ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_A) && (ISO3_REG_TX_CTRL_0.iso15693mode == PH_ON))
    {
        ISO3_REG_TX_CTRL_0.operand = (wSizeBytes / 2) & 0x1ffffff;
    }

    dwShadowBufferStartAddr = (dwStartAddress - ISO3_WAVESHAPE_BUFFER_BASE_ADDRESS) * sizeof(uint32_t);
    for ( dwCounter = 0; dwCounter < wSizeBytes; dwCounter++ )
    {
        pDataParams->pWaveShapeShadowBuffer[dwCounter + dwShadowBufferStartAddr] = *(((uint8_t*)pWaveShape) + dwCounter);
    }

    /* upload waveshape into slot */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_PlaySeq8(
                            phhalHw_ISO3_DataParams_t * pDataParams
                            )
{
    phStatus_t statusTmp;
    uint32_t dwTimeout;
    uint32_t dwTmp;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Set pointer to sequence8 waveshape */
    phhalHw_ISO3_Int_BuildSetWaveShapePointer(pDataParams, ISO3_WAVESHAPE_TYPE_EXCHANGE);

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* deactivate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL, PH_ON));
    }

    dwTimeout = 0x00300070;
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_RX_CTRL_2, dwTimeout);

    /* command to transmit sequence8 waveshape
       The values for RX_CTRL2 are taken from the reference examples.
       To force playing the wave shape, the TX_CTRL0 register must be written 2 times - 0x00000001 and the actual TX command */
    dwTmp = 0x00000001;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, dwTmp));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));

    /* configure the receiver timeout in case there was an exchange encoded in the wave shape. */
    dwTimeout = 0x40000006;
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_RX_CTRL_2, dwTimeout);
    dwTimeout = 0x20000006;
    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_RX_CTRL_2, dwTimeout);

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* re-activate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL, PH_OFF));
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_GetPLSVersion(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint16_t * pwVersion,                      /**< [Out] Server Version */
                            uint16_t * pwRevision                      /**< [Out] Server Revision */
                            )
{
    phStatus_t statusTmp;
    uint16_t pwPlsVersion[2];

    /* Check param */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL_PARAM(pwVersion, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pwRevision, PH_COMP_HAL);


    /* Get Version of PLS server */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
        pDataParams,
        PHHAL_HW_ISO3_SERVER_CMD_GET_VERSION,
        0, 0));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_RecvServerResponse(
        pDataParams,
        (uint8_t*)&pwPlsVersion,
        sizeof(pwPlsVersion)));

    *pwVersion  = pwPlsVersion[0];
    *pwRevision = pwPlsVersion[1];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_GetAgcStatistic(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint8_t bReset,                             /**  [In] If the is set to PH_ON the registers will be reseted after readout */
                            uint32_t * pdwMinAgc,                       /**< [Out] minimum value of agc control value */
                            uint32_t * pdwMaxAgc                        /**< [Out] maximum value of agc control value */
                            )
{
    phStatus_t statusTmp;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL_PARAM(pdwMinAgc, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pdwMaxAgc, PH_COMP_HAL);

    if (bReset != PH_ON && bReset != PH_OFF)
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
        pDataParams,
        PHHAL_HW_ISO3_REG_MIN_AGC_REG,
        (uint8_t*)(pdwMinAgc),
        sizeof(uint32_t)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
        pDataParams,
        PHHAL_HW_ISO3_REG_MAX_AGC_REG,
        (uint8_t*)(pdwMaxAgc),
        sizeof(uint32_t)));

    if (bReset == PH_ON)
    {
        uint8_t agc_ctrl[] = {0x02, 0x00, 0x00, 0x00};
        /* Set freeze AGC to correct value */
        if (pDataParams->bDisableAGC == PH_ON)
        {
            agc_ctrl[0] |= 0x01;
        }
        /* reset */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
            agc_ctrl,
            sizeof(agc_ctrl)));

        /* Clear reset */
        agc_ctrl[0] &= 0xFD;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
            agc_ctrl,
            sizeof(agc_ctrl)));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_GetDeviceDna(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint64_t * pdna                             /**< [Out] [Out] device dna value*/
                            )
{
    phStatus_t statusTmp;
    uint32_t ddna_1to32;
    uint32_t ddna_33to57;
    uint64_t ddna_shifted;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_DEVICE_DNA_1to32, (uint8_t*)(&ddna_1to32), sizeof(uint32_t)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_DEVICE_DNA_33to57, (uint8_t*)(&ddna_33to57), sizeof(uint32_t)));

    ddna_shifted = ddna_33to57;
    *pdna = ddna_shifted << 32 | ddna_1to32;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Cmd_SetConfig32(
        phhalHw_ISO3_DataParams_t * pDataParams,	   /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                              /**< [In] Configuration Identifier. */
        uint32_t dwValue                               /**< [In] Configuration Value. */
    )
{
    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch (wConfig)
    {
    case PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_CYCLES:
        /* Check if the given Field Off duration in clk cycles can fit in transmitter TX_CTRL0.operand 25bit register */
        if (dwValue > 0x1FFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        if (dwValue < 35) /*Limit is 35 clock cycles */
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->dwFieldOffTimeCycles = (uint32_t)dwValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    default:
        if (dwValue > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        return phhalHw_ISO3_SetConfig(pDataParams, wConfig, (uint16_t)dwValue);
    }
}

phStatus_t phhalHw_ISO3_Cmd_GetConfig32(
        phhalHw_ISO3_DataParams_t * pDataParams,                                                /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint32_t * pdwValue                                                                     /**< [Out] Configuration Value. */
    )
{
    phStatus_t statusTmp;
    uint16_t wTmp;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch (wConfig)
    {
    case PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_CYCLES:
        /* Return parameter */
        *pdwValue = (uint32_t)pDataParams->dwFieldOffTimeCycles;
        break;
    default:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, wConfig, &wTmp));
        *pdwValue = wTmp;
        break;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_ISO3 */
