/*
 * Copyright 2019 - 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 register definitions for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_ISO3_TYPES_H
#define PHHALHW_ISO3_TYPES_H

#ifdef _WIN32
#pragma warning (disable:4214)
#endif

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#include "ph_TypeDefs.h"

/**
* \brief ISO3 HAL Internal Write Addresses
*/
#define PHHAL_HW_ISO3_REG_FMC150_CLK_CTRL           0x000U        /**< SPI data register Clockgenerator (CDCE72010) (b31:b0 used) */
#define PHHAL_HW_ISO3_REG_FMC150_MON_CTRL           0x001U        /**< SPI data register Voltage/Temp. Monitor (AMC7823) (b31:b0 used) */
#define PHHAL_HW_ISO3_REG_FMC150_DAC_CTRL           0x002U        /**< SPI data register DAC (DAC3283) (b15:0 used) */
#define PHHAL_HW_ISO3_REG_FMC150_ADC_CTRL           0x003U        /**< SPI data register ADC (ADS62P49) (b15:0 used) */
#define PHHAL_HW_ISO3_REG_STEP_ATTEN_CTRL           0x004U        /**< Step attenuator (b7=write to #2, b6=write to #1, b5:b0= data) */
#define PHHAL_HW_ISO3_REG_SEVEN_SEG_DISP            0x005U        /**< Seven-Segment-Display (not yet connected !) */
#define PHHAL_HW_ISO3_REG_MISC_CTRL                 0x006U        /**< FMC150 control (b5=ADC-reset, b4=TX-enable, b3=Monitor-reset, b2=Clockgen-reset, b1=Clockgen-powerdown, b0=Clockgen-Ref-enable) */
#define PHHAL_HW_ISO3_REG_FPGA_VERSION              0x009U        /**< Version of Fpga bitfile */


#define PHHAL_HW_ISO3_REG_TX_CTRL_0                 0x007U        /**< Transmit control register 0 */
#define PHHAL_HW_ISO3_REG_TX_CTRL_1                 0x015U        /**< Transmit control register 1 */
#define PHHAL_HW_ISO3_REG_TX_DATA                   0x400U        /**< Transmit Data Ping Buffer (512*32) */
#define PHHAL_HW_ISO3_REG_TX_DATA_1                 0x600U        /**< Transmit Data Pong Buffer (512*32) */

#define PHHAL_HW_ISO3_REG_RX_CTRL_2                 0x00AU        /**< TODO */
#define PHHAL_HW_ISO3_REG_RX_MOD                    0x00FU        /**< TODO */

#define PHHAL_HW_ISO3_REG_TRIG1_CTRL0               0x00CU        /**< TODO */
#define PHHAL_HW_ISO3_REG_TRIG1_CTRL1               0x00DU        /**< TODO */
#define PHHAL_HW_ISO3_REG_GAIN_CTRL                 0x00EU        /**< TODO */
#define PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL          0x00FU        /**< TODO */

#define PHHAL_HW_ISO3_REG_WAVESHAPE_POINTER         0x010U        /**< Waveshape pointer offset - play next waveshape starting from this offset */
#define PHHAL_HW_ISO3_REG_RF_ON_WAVESHAPE_POINTER   0x069U        /**< RF on waveshape pointer */
#define PHHAL_HW_ISO3_REG_RF_OFF_WAVESHAPE_POINTER  0x06AU        /**< RF off waveshape pointer */
#define PHHAL_HW_ISO3_REG_LOG_CTRL                  0x011U        /**< Control register for logging */
#define PHHAL_HW_ISO3_REG_LOG_SIZE                  0x012U        /**< Register for READ/WRITE the LOG_LENGTH. Number of 16-bit sample count. Default = 0x50000 samples (  */
#define PHHAL_HW_ISO3_REG_MLS_WHITE_NOISE_CTRL      0x013U        /**< Control register to enable MLS, swith TX DAC sample sources and set MLS Seed */
#define PHHAL_HW_ISO3_REG_MLS_SCALING_VALUE         0x014U        /**< Scaling value for DAC output '0' and '1'

/**
* \brief ISO3 HAL Internal Read Addresses
*/
/* #define PHHAL_HW_ISO3_REG_FMC150_RB              0x000U        /**< SPI data register readback (last value written to addr 0-3) */
/* #define PHHAL_HW_ISO3_REG_FMC150_RD_SPI          0x001U        /**< SPI read data register (Data read from previously written FMC150-SPI) */
/* #define PHHAL_HW_ISO3_REG_SPI_STAT               0x002U        /**< SPI/Step attenuator status (b2=Step-attenuator busy, b1=FMC150-SPI busy, b0=FMC150-SPI read buffer full) */
/* #define PHHAL_HW_ISO3_REG_CAL_REF	                0x003U        /**< TODO */
/* #define PHHAL_HW_ISO3_REG_STEP_ATTEN_RB          0x004U        /**< Step attenuator readback (last value wriiten to addr 4) */
/* #define PHHAL_HW_ISO3_REG_SEVEN_SEG_RB           0x005U        /**< Seven-segment readback (last value written to addr 5) */
/* #define PHHAL_HW_ISO3_REG_STAT                   0x006U        /**< TODO */
#define PHHAL_HW_ISO3_REG_FDT_STAT                  0x007U        /**< 64bit : [ 16bit FDT ][ 0x0500 ][18 bit noise power level][ 14bit num_rx_bits ] */
/* #define PHHAL_HW_ISO3_REG_RX_STAT                    0x008U        /**< TODO */

#define PHHAL_HW_ISO3_REG_RX_DATA                   0x400U        /**< Data from receiver */
#define PHHAL_HW_ISO3_REG_FDT_DATA                  0x800U        /**< Measured Signals saved in 32bit words. [16bit Trigger2 Signal][16bit Trigger1 Signal] */

#define PHHAL_HW_ISO3_FDT_DATA_COUNT                 0x800        /**< Number of 32-bit registers or maximum number of samples per trigger */

#define PHHAL_HW_ISO3_FDT_STAT_SIZE_BYTE            0x008U

#define PHHAL_HW_ISO3_REG_CAL_COIL_VOLT             0x00aU        /* Mean sqr voltage measured by adc from calibration coil */
#define PHHAL_HW_ISO3_REG_STEERING_VAR              0x00bU        /* Value of actuating variable of field strength gain control */

#define PHHAL_HW_ISO3_REG_RX_DELAY                  0x046U        /**< Register to control the enable of the rx after tx in 10*fc */
#define PHHAL_HW_ISO3_REG_DEBUG_CONFIG              0x047U        /**< Register to control the debug output */
#define PHHAL_HW_ISO3_REG_RX_MATCH                  0x048U        /**< Register to control Rx match */
#define PHHAL_HW_ISO3_REG_RX_MATCH_MASK             0x049U        /**< Register to control Rx match mask */
#define PHHAL_HW_ISO3_REG_TX_MATCH                  0x04AU        /**< Register to control Tx match */
#define PHHAL_HW_ISO3_REG_TX_MATCH_MASK             0x04BU        /**< Register to control Tx match mask */

#define PHHAL_HW_ISO3_REG_TRIGGER_INPUT             0x04CU        /**< Register to control Config of the bits that are used for as trigger input */
#define PHHAL_HW_ISO3_REG_TRIGGER1_RISE_MASK        0x04DU        /**< Register to control Mask used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER1_FALL_MASK        0x04EU        /**< Register to control Mask used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER1_RISE_EXPECTED    0x04FU        /**< Register to control expected value used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER1_FALL_EXPECTED    0x050U        /**< Register to control expected value used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER1_MODE             0x051U        /**< Register to control Mode selection of trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER2_RISE_MASK        0x052U        /**< Register to control Mask used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER2_FALL_MASK        0x053U        /**< Register to control Mask used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER2_RISE_EXPECTED    0x054U        /**< Register to control expected value used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER2_FALL_EXPECTED    0x055U        /**< Register to control expected value used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER2_MODE             0x056U        /**< Register to control Mode selection of trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER3_RISE_MASK        0x057U        /**< Register to control Mask used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER3_FALL_MASK        0x058U        /**< Register to control Mask used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER3_RISE_EXPECTED    0x059U        /**< Register to control expected value used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER3_FALL_EXPECTED    0x05AU        /**< Register to control expected value used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER3_MODE             0x05BU        /**< Register to control Mode selection of trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER4_RISE_MASK        0x05CU        /**< Register to control Mask used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER4_FALL_MASK        0x05DU        /**< Register to control Mask used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER4_RISE_EXPECTED    0x05EU        /**< Register to control expected value used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER4_FALL_EXPECTED    0x05FU        /**< Register to control expected value used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER4_MODE             0x060U        /**< Register to control Mode selection of trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER5_RISE_MASK        0x061U        /**< Register to control Mask used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER5_FALL_MASK        0x062U        /**< Register to control Mask used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER5_RISE_EXPECTED    0x063U        /**< Register to control expected value used to configure the rising edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER5_FALL_EXPECTED    0x064U        /**< Register to control expected value used to configure the falling edge of the trigger */
#define PHHAL_HW_ISO3_REG_TRIGGER5_MODE             0x065U        /**< Register to control Mode selection of trigger */

#define PHHAL_HW_ISO3_REG_RX_EOF_TIMEOUT_CONFIG     0x066U        /**< Register to control Rx EOF Timeout detection */
#define PHHAL_HW_ISO3_REG_OVS_FILTER_MEMORY_CTRL    0x067U        /**< */
#define PHHAL_HW_ISO3_REG_OVS_FILTER_CTRL           0x068U        /**< */
#define PHHAL_HW_ISO3_REG_DEVICE_DNA_1to32          0x070U        /**< Register 1-32 bits of the device dna*/
#define PHHAL_HW_ISO3_REG_DEVICE_DNA_33to57         0x071U        /**< Register 33-57 bits of the device dna*/
#define PHHAL_HW_ISO3_REG_TX_STAT                   0x072U        /**< Tx Status register */

/**
 * \brief IIR Filter control register
*/
#define PHHAL_HW_ISO3_REG_IIRFILTER_CTRL    0x20 /**< Register address to enable or disable the IIR Filter */
typedef struct IIRFilterCtrlRegister_t
{
    uint32_t enable :  1; /**< 1 (Filter is enabled) or 0 (filter is disabled). The default value is 0 */
    uint32_t rfu    : 31; /**< reserved for future use */
} phhalHw_ISO3_IIRFilterCtrlRegister_t;

#define ISO3_REG_IIRFILTER_CTRL (*((phhalHw_ISO3_IIRFilterCtrlRegister_t*)&(pDataParams->dwIIRFilterCtrl))) /**< Macro to control the IIR Filter */

#define PHHAL_HW_ISO3_REG_IIRFILTER_1B0      0x21 /**< address to set IIR filter coefficient sec1, b0 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_1B1      0x22 /**< address to set IIR filter coefficient sec1, b1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_1B2      0x23 /**< address to set IIR filter coefficient sec1, b2 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_1A1      0x24 /**< address to set IIR filter coefficient sec1, a1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_1A2      0x25 /**< address to set IIR filter coefficient sec1, a2 */

#define PHHAL_HW_ISO3_REG_IIRFILTER_2B0      0x26 /**< address to set IIR filter coefficient sec2, b0 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_2B1      0x27 /**< address to set IIR filter coefficient sec2, b1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_2B2      0x28 /**< address to set IIR filter coefficient sec2, b2 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_2A1      0x29 /**< address to set IIR filter coefficient sec2, a1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_2A2      0x2a /**< address to set IIR filter coefficient sec2, a2 */

#define PHHAL_HW_ISO3_REG_IIRFILTER_3B0      0x2b /**< address to set IIR filter coefficient sec3, b0 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_3B1      0x2c /**< address to set IIR filter coefficient sec3, b1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_3B2      0x2d /**< address to set IIR filter coefficient sec3, b2 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_3A1      0x2e /**< address to set IIR filter coefficient sec3, a1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_3A2      0x2f /**< address to set IIR filter coefficient sec3, a2 */

#define PHHAL_HW_ISO3_REG_IIRFILTER_4B0      0x30 /**< address to set IIR filter coefficient sec4, b0 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_4B1      0x31 /**< address to set IIR filter coefficient sec4, b1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_4B2      0x32 /**< address to set IIR filter coefficient sec4, b2 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_4A1      0x33 /**< address to set IIR filter coefficient sec4, a1 */
#define PHHAL_HW_ISO3_REG_IIRFILTER_4A2      0x34 /**< address to set IIR filter coefficient sec4, a2 */

#define PHHAL_HW_ISO3_REG_IIRFILTER_GAIN     0x35 /**< address to set IIR filter gain factor */

#define PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW 0x36 /**< address to read if IIR Filter overflow occured in adder or multiplexer */

#define PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW_RESET 0x37 /**< address to reset the IIR Filter overflow flag */

#define PHHAL_HW_ISO3_REG_DGRM_SIG_DET       0x41 /**< write only address mapped to the venus dgrm_signal_detection_th register */
#define PHHAL_HW_ISO3_REG_PICC2PCD_DELAY     0x1c /**< address mapped to the 14bit transmitter timeout register */
#define PHHAL_HW_ISO3_REG_PCD2PICC_TIMEOUT   0x45 /**< R/W address mapped to rx_timeout_counter */

#define PHHAL_HW_ISO3_REG_SOFT_SCOPE_BUSY    0x1A /**< read only - address to indicate if the soft scope region in the ddr3 RAM is currently used by the FPGA, return trigger_counter = zero if triggering is finished */
#define PHHAL_HW_ISO3_REG_TRIG1_RISE_COUNTER 0x41 /**< read only - trigger 1 rise counter */
#define PHHAL_HW_ISO3_REG_TRIG1_FALL_COUNTER 0x42 /**< read only - trigger 1 fall counter */
#define PHHAL_HW_ISO3_REG_TRIG2_RISE_COUNTER 0x43 /**< read only - trigger 2 rise counter */
#define PHHAL_HW_ISO3_REG_TRIG2_FALL_COUNTER 0x44 /**< read only - trigger 2 fall counter */

#define PHHAL_HW_ISO3_REG_PEAK_SENS_SIGNAL 0x69 /**< read only - Address of register that stores peak value of SENS signal */
#define PHHAL_HW_ISO3_REG_PEAK_CAL_SIGNAL 0x6A /**< read only - Address of register that stores peak value of CAL signal */

#define PHHAL_HW_ISO3_REG_PRESCALER_CTRL_REG          0x6B /**< Address of register that stores ctrl of prescaler */
#define PHHAL_HW_ISO3_REG_PRESCALER_TIMING_0TO7_REG   0x6C /**< Address of register that stores timing values 0 to 7 of prescaler */
#define PHHAL_HW_ISO3_REG_PRESCALER_TIMING_8TO15_REG  0x6D /**< Address of register that stores timing values 8 to 15 of prescaler */

#define PHHAL_HW_ISO3_REG_MIN_AGC_REG                 0x6E /**< Address of register that stores the min AGC value since the last reset of the value */
#define PHHAL_HW_ISO3_REG_MAX_AGC_REG                 0x6F /**< Address of register that stores the max AGC value since the last reset of the value */

#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG          0x16 /**< one of the 6 register CLIF_RX_CONFIG_REG addresses used to configure the data rate and card type for the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG   0x17 /**< one of the 6 register CLIF_SIGPRO_RM_CONFIG_REG addresses used to configure the data rate and card type for the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG  0x18 /**< one of the 6 register CLIF_SIGPRO_RM_PATTERN_REG addresses used to configure the data rate and card type for the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG  0x19 /**< one of the 6 register CLIF_SIGPRO_RM_ENABLES_REG addresses used to configure the data rate and card type for the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG     0x1a /**< one of the 6 register CLIF_SIGPRO_RM_TECH_REG addresses used to configure the data rate and card type for the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG                      0x1b /**< one of the 6 register parts of CLIF_SIGPRO_CONFIG_REG addresses used to configure the data rate and card type for the venus receiver */

/* ADDR 0x16 mapped to CLIF_RX_CONFIG_REG
 *                                                                      +-RX_PARITY_EMD_ON_SOVER
 *                                                                      | +-RX_MISSING_PARITY_IS_EMD
 *                                                                      | | +-RX_ADVANCED_EMD_ENABLE
 *                                                                      | | | +-RX_PARITY_ERROR_IS_EMD
 *                                                                      | | | | +-RX_EMD_SUP
 *                                                                      | | | | |   +-RX_COLL_IS_DATA_ERROR
 *                                                                      | | | | |   | +-VALUES_AFTER_COLLISION
 *                                                                      | | | | |   | | +-RX_CRC_ALLOW_BITS
 *                                                                      | | | | |   | | | +-RX_FORCE_CRC_WRITE
 *                                                                      | | | | |   | | | | +-RESERVED
 *                                                                      | | | | |   | | | | | +-RX_PARITY_ERR_8BITS_ENABLE
 *                                                                      | | | | |   | | | | | | +-RX_BIT_ALIGN
 *                                                                      | | | | |   | | | | | | |   +-RX_EOF_SYMBOL
 *                                                                      | | | | |   | | | | | | |   |        +-RESERVED
 *                                                                      | | | | |   | | | | | | |   |        | +-RX_EOF_TYPE
 *                                                                      | | | | |   | | | | | | |   |        | | +-RX_STOP_CONDITION
 *                                                                      | | | | |   | | | | | | |   |        | | |  +-RX_MSB_FIRST
 *                                                                      | | | | |   | | | | | | |   |        | | |  | +-RX_STOPBIT_ENABLE
 *                                                                      | | | | |   | | | | | | |   |        | | |  | | +-RX_PARITY_TYPE
 *                                                                      | | | | |   | | | | | | |   |        | | |  | | | +-RX_PARITY_ENABLE
 *                                                                      |3|3|2|2|222|2|2|2|2|2|1|111|11111100|0|0|00|0|0|0|0
 *                                                                      |1|0|9|8|765|4|3|2|1|0|9|876|54321098|7|6|54|3|2|1|0
 *                                                                      |-|-|-|-|---|-|-|-|-|-|-|---|--------|-|-|--|-|-|-|--                           */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A106 0x00080003 /**< |0|0|0|0|000|0|0|0|0|0|1|000|00000000|0|0|00|0|0|1|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A212 0x00080023 /**< |0|0|0|0|000|0|0|0|0|0|1|000|00000000|0|0|10|0|0|1|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A424 0x00080023 /**< |0|0|0|0|000|0|0|0|0|0|1|000|00000000|0|0|10|0|0|1|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A848 0x00080023 /**< |0|0|0|0|000|0|0|0|0|0|1|000|00000000|0|0|10|0|0|1|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B106 0x00000054 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00000000|0|1|01|0|1|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B212 0x00000054 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00000000|0|1|01|0|1|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B424 0x00000054 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00000000|0|1|01|0|1|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B848 0x00000054 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00000000|0|1|01|0|1|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_F212 0x00000018 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00000000|0|0|01|1|0|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_F424 0x00000018 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00000000|0|0|01|1|0|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_I006 0x00001d10 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00011101|0|0|01|0|0|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_I026 0x00001d10 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00011101|0|0|01|0|0|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_I053 0x00001d10 /**< |0|0|0|0|000|0|0|0|0|0|0|000|00011101|0|0|01|0|0|0|0 copied from the ISO3 SAD. Constant to configure the venus receiver */

/**
* \brief Struct for PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG.
*/
typedef struct VenusClifRxConfigRegStruct_t
{
    uint32_t RX_PARITY_ENABLE           : 1; /**< If set to 1. a parity-bit for each byte is expected. will be extracted from data stream and checked for correctness. In case the parity-bit is incorrect. the RX_DATA_INTEGRITY_ERROR flag is set. Nevertheless the reception is continued. Note: This bit is set by the mod-detector if automatic mode detection is enabled and ISO14443A communication is detected. */
    uint32_t RX_PARITY_TYPE             : 1; /**< Defines which type of the parity-bit is used Note: This bit is set by the mod-detector if automatic mode detection is enabled and ISO14443A communication is detected. 0 Even parity calculation is used 1 Odd parity calculation is used */
    uint32_t RX_STOPBIT_ENABLE          : 1; /**< Set to 1. a stop-bit is expected and will be checked and extracted from data stream. Note: The stop-bit is not configurable and fixed to a logic 1. Note: This bit is set by the mod-detector if automatic mode detection is enabled and ISO14443B communication is detected. */
    uint32_t RX_MSB_FIRST               : 1; /**< If set to 1. data bytes are interpreted MSB first for reception */
    uint32_t RX_STOP_CONDITION          : 2; /**< This bit field defines the condition to stop a reception. Note: This bit is set by the mod-detector if automatic mode detection is enabled and the corresponding communication is detected. 00b Reception is stopped only if end of data communication is detected by the signal processing 01b Reception is stopped upon reception of the EOF pattern selected by the register field RX_EOF_TYPE and RX_EOF_SYMBOL Note: This value is set by the mod-detector if a ISO14443A or ISO14443B communication is detected 10b Reception is stopped on detection of an incorrect parity bit. 11b Recption is stopped if the number of received bytes reaches the defined frame length (which extracted from the first data-byte received). Note: This value is set by the mod-detector if a Felica communication is detected */
    uint32_t RX_EOF_TYPE                : 1; /**< Defines the EOF handling 0 EOF as defined in the register field RX_EOF_SYMBOL is expected. 1 EOF as defined in ISO14443B is expected. */
    uint32_t RESERVED1                  : 1; /**< Reserved */
    uint32_t RX_EOF_SYMBOL              : 8; /**< This value defines the pattern of the EOF symbol. Note: All bits are clear upon enabling of the receiver is the automatic mode detection is enabled. If in such a case an ISO14443A communication is detected bit0 is set by the mod-detector. */
    uint32_t RX_BIT_ALIGN               : 3; /**< RxAlign defines the bit position within the byte for the first bit received. Further received bits are stored at the following bit positions. */
    uint32_t RX_PARITY_ERR_8BITS_ENABLE : 1; /**< if set parity error is detected when only 8 bits in last byte received */
    uint32_t RESERVED                   : 1; /**< Reserved */
    uint32_t RX_FORCE_CRC_WRITE         : 1; /**< Set to 1. the received CRC byte(s) are written to ram. In normal operation (if this bit is set to 0) CRC bytes are only checked and removed from the data stream. */
    uint32_t RX_CRC_ALLOW_BITS          : 1; /**< Set to 1. a frame with less than one byte length is writtten to ram. when the CRC is enabled otherwise it is discarted. */
    uint32_t VALUES_AFTER_COLLISION     : 1; /**< This bit defined the value of bits received after a collision occured. 0* All received bits after a collision will be cleared. 1 All received bits after a collision keep their value. */
    uint32_t RX_COLL_IS_DATA_ERROR      : 1; /**< If set to 1. a collision is treated as a data integrity error (especially for ISO14443-4) */
    uint32_t RX_EMD_SUP                 : 3; /**< Defines EMD suppression mechanism 000 Off 001 EMD suppression according to ISO14443 010 EMD suppression according to NFC Forum (with respect to the first 3 characters) 011 EMD suppression according to NFC Forum (with respect to the first 4characters) 100 EMD suppression according to NFC Forum (with respect to the first 4characters all valid frames <4 bytes are ignored) 101-111 RFU */
    uint32_t RX_PARITY_ERROR_IS_EMD     : 1; /**< If set to 1 a parity error in the 3rd/4th byte (depending on RX_EMD_SUP setting) is interpreted as an EMD error. Otherwise it is interpreted as a parity error. */
    uint32_t RX_ADVANCED_EMD_ENABLE     : 1; /**< if set new EMD options are enabled */
    uint32_t RX_MISSING_PARITY_IS_EMD   : 1; /**< if set a missing parity bit in the 4th byte is treated as EMD (EMD options 3/4) */
    uint32_t RX_PARITY_EMD_ON_SOVER     : 1; /**< if set decision if EMD due to parity error is taken at sover (depending on frame length) */
} phhalHw_ISO3_VenusClifRxConfigRegStruct_t;
/**
* \brief Union for PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG.
*/
typedef union VenusClifRxConfigReg_t
{
    phhalHw_ISO3_VenusClifRxConfigRegStruct_t Struct;
    uint32_t Value;
} phhalHw_ISO3_VenusClifRxConfigReg_t;

#define ISO3_REG_VENUS_CLIF_RX_CONFIG (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(pDataParams->dwVenusClifRxConfigReg)))   /**< Makro for rx control 2 register fields */


/* ADDR 0x17 mapped to CLIF_SIGPRO_RM_CONFIG_REG
 *                                                                             +-RM_PREAMBLE_SC_TRIGGER
 *                                                                             |     +-RM_OOK_COL_IS_NOISE
 *                                                                             |     | +-RM_EOFMF_COND_ENABLE
 *                                                                             |     | | +-RM_SILENT_COND_ENABLE
 *                                                                             |     | | | +-RM_BIT2BIT_VALUE_COND_ENABLE
 *                                                                             |     | | | | +-RM_BIT2BIT_RATIO_COND_ENABLE
 *                                                                             |     | | | | | +-RM_DOWNMIX_MODE
 *                                                                             |     | | | | | | +-RM_EGT_RESYNC_ENABLE
 *                                                                             |     | | | | | | | +-RM_EOF_2_FALSE_ALARM
 *                                                                             |     | | | | | | | |    +-RM_OOK_ERROR_MODE
 *                                                                             |     | | | | | | | |    | +-RM_CH_COMB_MODE_SEL
 *                                                                             |     | | | | | | | |    | |  +-RM_EOF_2_VALID_DELAY
 *                                                                             |     | | | | | | | |    | |  |  +-RM_SILENCE_FACTOR
 *                                                                             |     | | | | | | | |    | |  |  |   +-RM_CORR_DISTANCE
 *                                                                             |     | | | | | | | |    | |  |  |   |   +-RM_CORR_FACTOR
 *                                                                             |     | | | | | | | |    | |  |  |   |   |   +-RM_EMD_SENSITIVITY
 *                                                                             |33222|2|2|2|2|2|2|2|1111|1|11|11|100|000|000|00
 *                                                                             |10987|6|5|4|3|2|1|0|9876|5|43|21|098|765|432|10
 *                                                                             |-----|-|-|-|-|-|-|-|----|-|--|--|---|---|---|---                           */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A106 0x04249240 /**< |00000|1|0|0|0|0|1|0|0100|1|00|10|010|010|000|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A212 0x43e0224c /**< |01000|0|1|1|1|1|1|0|0000|0|01|00|010|010|011|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A424 0x53e02228 /**< |01010|0|1|1|1|1|1|0|0000|0|01|00|010|001|010|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A848 0xa3e02209 /**< |10100|0|1|1|1|1|1|0|0000|0|01|00|010|000|010|01 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B106 0xf8f00248 /**< |11111|0|0|0|1|1|1|1|0000|0|00|00|010|010|010|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B212 0x40f00248 /**< |01000|0|0|0|1|1|1|1|0000|0|00|00|010|010|010|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B424 0x80f00248 /**< |10000|0|0|0|1|1|1|1|0000|0|00|00|010|010|010|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B848 0x80f00209 /**< |10000|0|0|0|1|1|1|1|0000|0|00|00|010|000|010|01 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_F212 0xa1e00228 /**< |10100|0|0|1|1|1|1|0|0000|0|00|00|010|001|010|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_F424 0xa1e00228 /**< |10100|0|0|1|1|1|1|0|0000|0|00|00|010|001|010|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_I006 0x43F4124C /**< |01000|0|1|1|1|1|1|1|0100|0|00|10|010|010|011|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_I026 0x43F4124C /**< |01000|0|1|1|1|1|1|1|0100|0|00|10|010|010|011|00 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_I053 0x43F4124C /**< |01000|0|1|1|1|1|1|1|0100|0|00|10|010|010|011|00 copied from the ISO3 SAD. Constant to configure the venus receiver */

/* ADDR 0x18 mapped to CLIF_SIGPRO_RM_PATTERN_REG
 *                                                                              +-RM_SYNC_PATTERN
 *                                                                              |                +-RESERVED
 *                                                                              |                |    +-RM_SOF_PATTERN
 *                                                                              |3322222222221111|1111|110000000000
 *                                                                              |1098765432109876|5432|109876543210
 *                                                                              |----------------|----|-------------                           */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A106 0x00000000 /**< |0000000000000000|0000|000000000000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A212 0x00000000 /**< |0000000000000000|0000|000000000000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A424 0x00000000 /**< |0000000000000000|0000|000000000000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A848 0x00000000 /**< |0000000000000000|0000|000000000000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B106 0x00000003 /**< |0000000000000000|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B212 0x00000003 /**< |0000000000000000|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B424 0x00000003 /**< |0000000000000000|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B848 0x00000003 /**< |0000000000000000|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_F212 0xB24D0000 /**< |1011001001001101|0000|000000000000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_F424 0xB24D0000 /**< |1011001001001101|0000|000000000000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_I006 0xb24d0003 /**< |1011001001001101|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_I026 0xb24d0003 /**< |1011001001001101|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_I053 0xb24d0003 /**< |1011001001001101|0000|000000000011 copied from the ISO3 SAD. Constant to configure the venus receiver */

/* ADDR 0x19 mapped to CLIF_SIGPRO_RM_ENABLES_REG
 *                                                                              +-RESERVED
 *                                                                              |             +-RM_OOK_GAIN_RATIO
 *                                                                              |             |  +-RM_OOK_GUARD_DIST
 *                                                                              |             |  |  +-RM_NOISE_INVALID_TH
 *                                                                              |             |  |  |  +-RM_NOISE_COLL_TH
 *                                                                              |             |  |  |  |     +-RM_LMA_READ_ENABLE
 *                                                                              |             |  |  |  |     | +-RM_LMA_TRACKING_ENABLE
 *                                                                              |             |  |  |  |     | | +-RM_EMD_RESET_ENABLE
 *                                                                              |             |  |  |  |     | | | +-RM_SILENT_RESET_ENABLE
 *                                                                              |             |  |  |  |     | | | | +-RM_COLLISION_RESET_ENABLE
 *                                                                              |             |  |  |  |     | | | | | +-RM_FALSE_ALARM_SOF_RESET_ENABLE
 *                                                                              |             |  |  |  |     | | | | | | +-RM_RESYNC_RESET_ENABLE
 *                                                                              |             |  |  |  |     | | | | | | | +-RM_WDT_RESET_ENABLE
 *                                                                              |3322222222221|11|11|11|11100|0|0|0|0|0|0|0|0
 *                                                                              |1098765432109|87|65|43|21098|7|6|5|4|3|2|1|0
 *                                                                              |-------------|--|--|--|-----|-|-|-|-|-|-|-|--                           */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A106 0x0004b47f /**< |0000000000000|10|01|01|10100|0|1|1|1|1|1|1|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A212 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A424 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A848 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B106 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B212 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B424 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B848 0x00000055 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_F212 0x00000051 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|0|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_F424 0x00000051 /**< |0000000000000|00|00|00|00000|0|1|0|1|0|0|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_I006 0x0004B47D /**< |0000000000000|10|01|01|10100|0|1|1|1|1|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_I026 0x0004B47D /**< |0000000000000|10|01|01|10100|0|1|1|1|1|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_I053 0x0004B47D /**< |0000000000000|10|01|01|10100|0|1|1|1|1|1|0|1 copied from the ISO3 SAD. Constant to configure the venus receiver */

/* ADDR 0x1A mapped to CLIF_SIGPRO_RM_TECH_REG
 *                                                                           +-RM_NCO_PERIOD_SEL
 *                                                                           |  +-RM_WAIT_RES_PERIOD_SEL
 *                                                                           |  |   +-RM_EGT_WINDOW_TH_SEL
 *                                                                           |  |   | +-RESERVED
 *                                                                           |  |   | | +-RM_DOWNSAMPLE_RATE_SEL
 *                                                                           |  |   | | |  +-RM_SOF_NUM_CYCLES_SEL
 *                                                                           |  |   | | |  |   +-RM_MF_S0_SEL
 *                                                                           |  |   | | |  |   |   +-RM_MF_S1_SEL
 *                                                                           |  |   | | |  |   |   |   +-RESERVED
 *                                                                           |  |   | | |  |   |   |   |   +RM_AVG_FILT_SEL
 *                                                                           |  |   | | |  |   |   |   |   | +-RM_SYNC_FILT_IN_SEL
 *                                                                           |  |   | | |  |   |   |   |   | |  +-RM_SYNC_FILT_SEL
 *                                                                           |  |   | | |  |   |   |   |   | |  |  +-RM_WATCH_DOG_PERIOD_SEL
 *                                                                           |  |   | | |  |   |   |   |   | |  |  |   +-RESERVED
 *                                                                           |33|222|2|2|22|222|111|111|111|1|00|00|000|000
 *                                                                           |10|987|6|5|43|210|987|654|321|0|98|76|543|210
 *                                                                           |--|---|-|-|--|---|---|---|---|-|--|--|---|----                           */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A106 0x488a00a0 /**< |01|001|0|0|01|000|101|000|000|0|00|10|100|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A212 0x8886c158 /**< |10|001|0|0|01|000|011|011|000|0|01|01|011|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A424 0x88048548 /**< |10|001|0|0|00|000|010|010|000|1|01|01|001|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A848 0x40024500 /**< |01|000|0|0|00|000|001|001|000|1|01|00|000|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B106 0x95090270 /**< |10|010|1|0|10|000|100|100|000|0|10|01|110|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B212 0x9486c258 /**< |10|010|1|0|01|000|011|011|000|0|10|01|011|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B424 0x94048648 /**< |10|010|1|0|00|000|010|010|000|1|10|01|001|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B848 0x5c024600 /**< |01|011|1|0|00|000|001|001|000|1|10|00|000|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_F212 0x01800128 /**< |00|000|0|0|11|000|000|000|000|0|01|00|101|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_F424 0x48824550 /**< |01|001|0|0|01|000|001|001|000|1|01|01|010|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_I006 0xd90600f8 /**< |11|011|0|0|10|000|011|000|000|0|00|11|111|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_I026 0x891600f0 /**< |10|001|0|0|10|001|011|000|000|0|00|11|110|000 copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_I053 0x412a0098 /**< |01|000|0|0|10|010|101|000|000|0|00|10|011|000 copied from the ISO3 SAD. Constant to configure the venus receiver */

/* ADDR 0x1B mapped to CLIF_SIGPRO_CONFIG_REG SIGPRO_BAUDRATE and SIGPRO_FRAMING
 *wire   wr_rm_reg      = wr_sel_ctrl & (s_axi_awaddr[9:2] == 8'b00011011);    // 0x1b RM_REG
 *    .framing_i(~r_rm_reg[25]?((~r_rm_reg[29]&r_rm_reg[28])?'d3:r_rm_reg[29:28]):'d4),
 *    .baudrate_i(~r_rm_reg[25]?(r_rm_reg[27:26]+'d3):(r_rm_reg[27:26])),
 *                                                          33|22|22|2|2
 *                                                          10|98|76|5|4   frame   baudrate
 *                                                          ---------------------------------  */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_A106 0xc0000000 /**< 11|00|00|0|0   0       3           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_A212 0xc4000000 /**< 11|00|01|0|0   0       4           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_A424 0xc8000000 /**< 11|00|10|0|0   0       5           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_A848 0xcc000000 /**< 11|00|11|0|0   0       6           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_B106 0xd0000000 /**< 11|01|00|0|0   3       3           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_B212 0xd4000000 /**< 11|01|01|0|0   3       4           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_B424 0xd8000000 /**< 11|01|10|0|0   3       5           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_B848 0xdc000000 /**< 11|01|11|0|0   3       6           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_F212 0xe4000000 /**< 11|10|01|0|0   2       4           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_F424 0xe8000000 /**< 11|10|10|0|0   2       5           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_I006 0x02000000 /**< 00|00|00|1|0   4       0           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_I026 0x06000000 /**< 00|00|01|1|0   4       1           copied from the ISO3 SAD. Constant to configure the venus receiver */
#define PHHAL_HW_ISO3_REG_VENUS_RM_REG_I053 0x0a000000 /**< 00|00|10|1|0   4       2           copied from the ISO3 SAD. Constant to configure the venus receiver */

/**
* \brief Struct for trigger control register 0. The scope function of the FPGA will sample chosen signals starting from
*  the rising edge of a trigger until the falling edge of the same trigger.
*/
typedef struct TriggerCtrlRegister0_t
{
    uint32_t conditionT1Rise : 4; /**< Condition for the rising edge of trigger 1 \see #TriggerModeIso3 */
    uint32_t conditionT1Fall : 4; /**< Condition for the falling edge of trigger 1 \see #TriggerModeIso3 */
    uint32_t conditionT2Rise : 4; /**< Condition for the rising edge of trigger 2 \see #TriggerModeIso3 */
    uint32_t conditionT2Fall : 4; /**< Condition for the falling edge of trigger 2 \see #TriggerModeIso3 */
    uint32_t signalIdT1      : 3; /**< The identifier of the signal to be captured if trigger 1 contition holds \see #Trigger1SignalId */
    uint32_t signalIdT2      : 3; /**< The identifier of the signal to be captured if trigger 2 contition holds \see #Trigger2SignalId */
    uint32_t externEnableT1  : 1; /**< Enable the trigger 1 signal externally so it can be monitored on a oscilloscope */
    uint32_t externEnableT2  : 1; /**< Enable the trigger 2 signal externally so it can be monitored on a oscilloscope */
    uint32_t fdtEnable       : 1; /**< Enable for Fame Delay Time measurement. SoftScope switches source from CAL to SENS before PICC response*/
    uint32_t rfu             : 7; /**< reserved for future use */
} phhalHw_ISO3_TriggerCtrlRegister0_t;

#define ISO3_REG_TRIGGER_CTRL_0 (*((phhalHw_ISO3_TriggerCtrlRegister0_t*)&(pDataParams->dwTriggerCtrl0Reg)))   /**< Makro for trigger control 0 register fields */

/**
* \brief ISO3 HAL shadow struct for trigger control register 1
*/
typedef struct TriggerCtrlRegister1_t
{
    uint32_t triggerRiseOnBitNumber : 16; /**< If the falling condition of a trigger is set to \see #PHHAL_HW_ISO3_TRIG_MODE_RX_BIT_NUM then use this bit number. */
    uint32_t triggerFallOnBitNumber : 16; /**< If the rising condition of a trigger is set to \see #PHHAL_HW_ISO3_TRIG_MODE_RX_BIT_NUM then use this bit number. */
} phhalHw_ISO3_TriggerCtrlRegister1_t;

#define ISO3_REG_TRIGGER_CTRL_1 (*((phhalHw_ISO3_TriggerCtrlRegister1_t*)&(pDataParams->dwTriggerCtrl1Reg))) /**< Makro for trigger control 1 register fields */

/**
* \brief 32 bit register to control the transmitter.
*/
typedef struct TxCtrlRegister_t
{
    uint32_t operand                    :25; /**< operand The value of bits 0-13 depends on the current state. Either a time period or the number (TX_BITS - 1) / Value of SEQ8 sample count[13:0] */
    uint32_t iso15693mode               :1;  /**< 0 (1 out of 256 mode for ISO15693) or 1 (1 out of 4 mode for ISO15693) */
    uint32_t datarate                   :2;  /**< The current data rate of the transmitter. 0 (106kb/s), 1 (212kb/s), 2 (424kb/s), 3 (848kb/s) If the current card type is ISO15693, then the transmitter can be set to 0 (1 out of 256 coding) or 1 (1 out of 4 coding) */
    uint32_t cardType                   :2;  /**< The current card type. 0 (14443-A), 1 (14443-B), 2 (Felica), 3 (ISO15693) */
    uint32_t state                      :2;  /**< The current state can be 0 (Fied off, no carrier), 1 (field on, plain carrier, unmodulated), 3 (transmit, modulate \see #operand -1 bits in mode \see #cardType */
} phhalHw_ISO3_TxCtrlRegister_t;

#define ISO3_REG_TX_CTRL_0 (*((phhalHw_ISO3_TxCtrlRegister_t*)&(pDataParams->dwTxCtrl)))       /**< Makro for tx control register fields */

typedef struct TxCtrlRegister_1_t
{
    uint32_t bitBoundary                :8;  /**< Set the bit boundary for 14443-B transmissions in a range of -128 to +127. Default value is 0 */
    uint32_t egt                        :13; /**< Set the extra guard time (EGT) for 14443-B transmissions. Default value is 0, maximum is 5875, unit is ETU */
    uint32_t dutyCycle                  :3;  /**< Set the tx duty cycle. \see #TxtDutyCycle for possible values */
    uint32_t rfu                        :6;  /**< reserved for future use */
    uint32_t double_data_rate           :1;  /**< 1 (double the configured data rate in \see #phhalHw_ISO3_TxCtrlRegister_t datarate) or 0 (do not double the data rate) */
    uint32_t disableParity              :1;  /**< Disables the Tx parity if set to 1 */
} phhalHw_ISO3_TxCtrlRegister_1_t;

#define ISO3_REG_TX_CTRL_1 (*((phhalHw_ISO3_TxCtrlRegister_1_t*)&(pDataParams->dwTxCtrl1)))       /**< Makro for tx control register 1 fields */

typedef struct RxCtrlRegister2_t
{
    uint32_t rxActivityTracker   :  8; /** rx activity tracker for coefficient A (bits 17 - 10) */
    uint32_t symbolTracking      :  1; /** 0 (symbol tracking disabled) or 1 (symbol tracking enabled) */
    uint32_t noisePowerThreshold : 18; /** Threshold for manual rx noise power detection */
    uint32_t noisePowerDetector  :  1; /** 0 (manual noise power threshold is used) or 1 (automatic noise power detection) */
    uint32_t parityDecode        :  1; /** Legacy - currently not used. 0 (do not decode parity bits) or 1 (decode parity bits) */
    uint32_t rfu                 :  3; /** reserved for future use */
} phhalHw_ISO3_RxCtrlRegister2_t;

#define ISO3_REG_RX_CTRL_2 (*((phhalHw_ISO3_RxCtrlRegister2_t*)&(pDataParams->dwRxCtrl2)))   /**< Makro for rx control 2 register fields */

/**
* \brief Log Ctrl register addr 0x11 of ISO3 Box. It is used to configure the soft scope trigger and signal sources
*/
typedef struct LogCtrlRegister_t
{
    uint32_t trig0           :  1; /**< enable trigger 1 configuration from \see #TriggerCtrlRegister0_t */
    uint32_t trig1           :  1; /**< enable trigger 2 configuration from \see #TriggerCtrlRegister0_t */
    uint32_t trigForce       :  1; /**< force trigger */
    uint32_t trigMls         :  1; /**< trigger on maximum length sequence (MLS) */
    uint32_t rfu1            :  1; /**< reserved for future use */
    uint32_t logRx           :  1; /**< data source: rx from ADC_IN */
    uint32_t logCal          :  1; /**< data source: CAL from ADC_IN */
    uint32_t logCos          :  1; /**< data source: internal cosine wave */
    uint32_t logSin          :  1; /**< data source: internal sine wave */
    uint32_t logAgcO         :  1; /**< data source: AGC output */
    uint32_t logIqO          :  1; /**< data source: IQ modulator output */
    uint32_t logDacO         :  1; /**< data source: DAC output */
    uint32_t rfu2            : 19; /**< reserved for future use */
    uint32_t reset           :  1; /**< toggle this bit to reset trigger counter and logger states */
} phhalHw_ISO3_LogCtrlRegister_t;

#define ISO3_REG_LOG_CTRL (*((phhalHw_ISO3_LogCtrlRegister_t*)&(pDataParams->dwLogCtrlReg)))   /**< Makro for log ctrl register fields */

/**
* \brief Mls white noise ctrl register addr 0x13 of ISO3 Box
*/
typedef struct MlsWhiteNoiseCtrlReg_t
{
    uint32_t mlsSeed         :  18; /**< initial stage register value */
    uint32_t rfu             :  12; /**< reserved for future use */
    uint32_t switchTx        :  1;  /**< switch TX DAC sample source. 0 (IQ output) or 1 (MLS source). Default value is 0 */
    uint32_t enableMls       :  1;  /**< start generating MLS if set to 1 */
} phhalHw_ISO3_MlsWhiteNoiseCtrlReg_t;

#define ISO3_REG_MLS_WHITE_NOISE_CTRL (*((phhalHw_ISO3_MlsWhiteNoiseCtrlReg_t*)&(pDataParams->dwMlsWhiteNoiseCtrlReg)))   /**< Makro for white noise ctrl register fields */

/**
* \brief Mls scaling value register addr 0x14 of ISO3 Box
*/
typedef struct MlsScalingCtrlReg_t
{
    uint32_t dacMls1 : 16; /**< scaled DAC sample value for MLS output '1'. Default: 0x7FFF */
    uint32_t dacMls0 : 16; /**< scaled DAC sample value for MLS output '0'. Default: 0x8001 */
} phhalHw_ISO3_MlsScalingCtrlReg_t;

/**
* \brief Mls scaling value register addr 0x12 of ISO3 Box
*/
typedef struct LogSizeReg_t
{
    uint32_t length : 32; /**< soft scope 16-bit sample count. Default 0x50000 (327k samples). Maximum 0x2000000 (500m samples) */
} phhalHw_ISO3_LogSizeReg_t;

/**
* \brief Mls scaling value register addr 0x12 of ISO3 Box
*/
typedef struct DgrmSigDet_t
{
    uint32_t dgrm_signal_detection_th : 16; /**< routed to the signal detection threshold of the venus receiver */
    uint32_t rfu                      : 16; /**< reserved for future use */
} phhalHw_ISO3_DgrmSigDet_t;
#define ISO3_REG_VENUS_DGRM_SIG_DET_TH (*((phhalHw_ISO3_DgrmSigDet_t*)&(pDataParams->dwDgrmSigDetTh)))

typedef struct RxTimeoutReg_t
{
    uint32_t rxTimeoutMax : 16;    /**< Deprecated. Not used any more */
    uint32_t rxEof2txSoftime : 16; /**< Maximum time in ETU for timeout of RX state before its transition to IDLE state */
} phhalHw_ISO3_RxTimeoutReg_t;
#define ISO3_REG_VENUS_RX_TIMEOUT (*((phhalHw_ISO3_RxTimeoutReg_t*)&(pDataParams->dwPicc2PcdDelay)))


#define ISO3_REG_OVS_ 3
/**
* \32 bit register to store the overshoot filter coefficients.
*/
typedef struct OvsFilterMemCtrl_t
{
    uint32_t wr_enable : 1;                     /**< Enable writing data from OVS_FILTER_MEMORY_DIN to overshoot filter at selected address
                                                    (addr field), 0  disabled, 1  enabled */
    uint32_t addr : 7;                          /**< Overshoot Filter Coefficient Address */
    uint32_t value : 18;                        /**< Overshoot Filter Coefficient Value */
    uint32_t unused : 6;                        /**< reserved for future use */
} phhalHw_ISO3_OvsFilterMemCtrl_t;

/**
* \32 bit register to control the overshoot filter.
*/
typedef struct OvsFilterCtrl_t
{
    uint32_t ovs_filter_resetn : 1;             /**< Reset Overshoot Filter, 0  reset filter, 1  do nothing */
    uint32_t ovs_filter_enable : 1;             /**< Enable usage of Overshoot Filter in ISO 3 TX, 0  disabled, 1  enabled */
    uint32_t ovs_filter_coeff_enable : 1;       /**< Enable loading of overshoot filter coefficients from memory to overshoot filter.
                                                    Deactivate while writing to overshoot filter memory! 0  disabled, 1  enabled */
    uint32_t ovs_filter_coeff_set_select : 3;  /**< Select set of filter coefficients for the overshoot filter */
    uint32_t unused : 26;                      /**< reserved for future use */
  } phhalHw_ISO3_OvsFilterCtrl_t;

/**
* \brief Status of the ISO3 receiver
*/
typedef struct RxStatRegister_t
{
    uint32_t bRxNumLastBits     : 3; /**< indicates the number of valid bits of the last data byte received */
    uint32_t wNumBytesReceived  : 13;
    uint32_t bRxLenError        : 1; /**< RxLenError */
    uint32_t bRxLatencyError    : 1; /**< RxLatencyError */
    uint32_t bRxBufferOvflError : 1; /**< RxBufferOvflError */
    uint32_t bRxBmaWriteError   : 1; /**< RxBmaWriteError */
    uint32_t bRxEmdError        : 1; /**< EMD event (Minframe, EMD Sup) indication pulse, reset with rx_reset */
    uint32_t bRxCollDet         : 1; /**< collision flag (set to 1 if a collision occures, which is not part of a EOF symbol) */
    uint32_t bRxProtError       : 1; /**< protocol error flag (set to 1 on a protocol error: SOF-, EOF-, Stopbit- or StopConditio-Error) */
    uint32_t bRxDataError       : 1; /**< data integrity error flag (set to 1 on parity or CRC error) */
    uint32_t bEofTrigger        : 1; /**< EOF trigger state */
    uint32_t bSofTrigger        : 1; /**< SOF trigger state */
    uint32_t bRxCollPos         : 6; /**< indicate the bit position of the first collision detected */
} phhalHw_ISO3_RxStatRegister_t;

#define ISO3_REG_RX_STAT (*((phhalHw_ISO3_RxStatRegister_t*)&(pDataParams->dwRxStatReg)))   /**< Makro for rx stat register fields */

/**
* \brief Struct for PHHAL_HW_ISO3_REG_PRESCALER_CTRL_REG.
*/
typedef struct PrescalerCtrlRegStruct_t
{
    uint32_t bModeBitgrid : 1; /**< if mode bitgrid is enabled all bitgrids are adjusted. if disabled the mode for miller min/max is used */
    uint32_t bEnabled     : 1; /**< if enabled the bitgrid values specified with prescalerTiming register will be used instead of prescaler values for the specified datarate */
    uint32_t dwRfu        : 30; /**< RFU */
} phhalHw_ISO3_PrescalerCtrlRegStruct_t;
/**
* \brief Union for PHHAL_HW_ISO3_REG_PRESCALER_CTRL_REG.
*/
typedef union PrescalerCtrlReg_t
{
    phhalHw_ISO3_PrescalerCtrlRegStruct_t Struct;
    uint32_t Value;
} phhalHw_ISO3_PrescalerCtrlReg_t;

/**
* \brief Struct for PHHAL_HW_ISO3_REG_PRESCALER_TIMING_0TO7_REG.
*/
typedef struct PrescalerTiming0to7RegStruct_t
{
    uint32_t bPrescaler0 : 4; /**< Prescaler used for r_br_cnt = 0 */
    uint32_t bPrescaler1 : 4; /**< Prescaler used for r_br_cnt = 1 */
    uint32_t bPrescaler2 : 4; /**< Prescaler used for r_br_cnt = 2 */
    uint32_t bPrescaler3 : 4; /**< Prescaler used for r_br_cnt = 3 */
    uint32_t bPrescaler4 : 4; /**< Prescaler used for r_br_cnt = 4 */
    uint32_t bPrescaler5 : 4; /**< Prescaler used for r_br_cnt = 5 */
    uint32_t bPrescaler6 : 4; /**< Prescaler used for r_br_cnt = 6 */
    uint32_t bPrescaler7 : 4; /**< Prescaler used for r_br_cnt = 7 */
} phhalHw_ISO3_PrescalerTiming0to7RegStruct_t;
/**
* \brief Union for PHHAL_HW_ISO3_REG_PRESCALER_TIMING_0TO7_REG.
*/
typedef union PrescalerTiming0to7Reg_t
{
    phhalHw_ISO3_PrescalerTiming0to7RegStruct_t Struct;
    uint32_t Value;
} phhalHw_ISO3_PrescalerTiming0to7Reg_t;

/**
* \brief Struct for PHHAL_HW_ISO3_REG_PRESCALER_TIMING_8TO15_REG.
*/
typedef struct PrescalerTiming8to15RegStruct_t
{
    uint32_t bPrescaler8 : 4; /**< Prescaler used for r_br_cnt = 8 */
    uint32_t bPrescaler9 : 4; /**< Prescaler used for r_br_cnt = 9 */
    uint32_t bPrescaler10 : 4; /**< Prescaler used for r_br_cnt = 10 */
    uint32_t bPrescaler11 : 4; /**< Prescaler used for r_br_cnt = 11 */
    uint32_t bPrescaler12 : 4; /**< Prescaler used for r_br_cnt = 12 */
    uint32_t bPrescaler13 : 4; /**< Prescaler used for r_br_cnt = 13 */
    uint32_t bPrescaler14 : 4; /**< Prescaler used for r_br_cnt = 14 */
    uint32_t bPrescaler15 : 4; /**< Prescaler used for r_br_cnt = 15 */
} phhalHw_ISO3_PrescalerTiming8to15RegStruct_t;
/**
* \brief Union for PHHAL_HW_ISO3_REG_PRESCALER_TIMING_8TO15_REG.
*/
typedef union PrescalerTiming8to15Reg_t
{
    phhalHw_ISO3_PrescalerTiming8to15RegStruct_t Struct;
    uint32_t Value;
} phhalHw_ISO3_PrescalerTiming8to15Reg_t;

/**
* \brief Status of the ISO3 transmitter
*/
typedef struct TxStatRegister_t
{
    uint32_t dwRxWdtCounter     : 30; /**< Counter value of the Wdt for the reception, if WDT is zero revceiver was stucking in reception mode */
    uint32_t rfu                : 2;
} phhalHw_ISO3_TxStatRegister_t;

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_ISO3_TYPES_H */
