/*
 * Copyright 2013, 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Mp300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_MP300

#include "phhalHw_Mp300_Int.h"
#include <phhalHw_Mp300_Cmd.h>
#include <stdio.h>              /* PRQA S 5124 */

phStatus_t phhalHw_Mp300_Cmd_TriggerOut(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bTriggerPort,
    uint32_t dwConfig,
    uint32_t dwValue
    )
{
    phStatus_t status;

    /* Check parameters */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_Int_CheckTriggerParameter(bTriggerPort, dwConfig, dwValue));

    return phhalHw_Mp300_ConfigureTriggerPort(pDataParams, bTriggerPort, dwConfig, dwValue);
}

phStatus_t phhalHw_Mp300_Cmd_AdjustRXChannel(
    phhalHw_Mp300_DataParams_t * pDataParams
    )
{
    return phhalHw_Mp300_AdjustRXChannel2(pDataParams);
}

phStatus_t phhalHw_Mp300_Cmd_SetModulationShape(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t dwNbPointsFallingEdge,
    uint16_t * pPointsFallingEdge,
    uint32_t dwNbPointsRisingEdge,
    uint16_t * pPointsRisingEdge
    )
{
    /* Check Value */
    if ( dwNbPointsFallingEdge < 2 || dwNbPointsRisingEdge < 2 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    if (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
    {
        if ( dwNbPointsFallingEdge > 1500 || dwNbPointsRisingEdge > 1500 )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }
    else
    {
        if ( dwNbPointsFallingEdge > 256 || dwNbPointsRisingEdge > 256 )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }

    /* Check that the last points of one edge match the first of the other edge */
    if ( pPointsFallingEdge[0] != pPointsRisingEdge[dwNbPointsRisingEdge-1] ||
         pPointsRisingEdge[0] != pPointsFallingEdge[dwNbPointsFallingEdge-1] )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return phhalHw_Mp300_SetModulationShape(pDataParams, dwNbPointsFallingEdge, pPointsFallingEdge, dwNbPointsRisingEdge, pPointsRisingEdge);
}

phStatus_t phhalHw_Mp300_Cmd_OpenScenario(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->dwScenarioID != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Open Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_OPEN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 2));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_OPEN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, (uint32_t * UNALIGNED)&(pDataParams->dwScenarioID)));

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    return status;
}

phStatus_t phhalHw_Mp300_Cmd_ExecuteScenario(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Execute Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_EXECUTE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, pDataParams->dwScenarioID));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_EXECUTE);
}

phStatus_t phhalHw_Mp300_Cmd_CloseScenario(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Clear Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_CLOSE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, pDataParams->dwScenarioID));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_CLOSE));

    pDataParams->dwScenarioID = 0;
    return status;
}

phStatus_t phhalHw_Mp300_Cmd_AddToScenario(
    phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwActionID,                         /**< [In] ID of the action that should be added */
    uint8_t * pParameter,                        /**< [In] string with parameter for the action. */
    uint16_t wParameterLength                    /**< [In] Length of the parameter string. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (dwActionID > 0xFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    if (pParameter == NULL && wParameterLength != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Add */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_ADD));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, pDataParams->dwScenarioID));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, (uint8_t)dwActionID));
    /* If parameter are available add this parameter */
    if (wParameterLength != 0)
    {
        if (pDataParams->wIntBufferLen + 1 + wParameterLength + 1 > pDataParams->wIntBufferSize) /* Space + Param + NULL */
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* First add a space */
        pDataParams->pIntBuffer[pDataParams->wIntBufferLen++] = (uint8_t)' ';
        memcpy(&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], pParameter, wParameterLength);
        pDataParams->wIntBufferLen += wParameterLength;
        pDataParams->pIntBuffer[pDataParams->wIntBufferLen] = 0x00;
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_ADD);
}

phStatus_t phhalHw_Mp300_Cmd_SetupCardEmulationMode(
    phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwAdjustTimeout,                    /**< [In] Timeout used to adjust the input channel */
    uint32_t dwMaxRetryCnt                       /**< [In] number of retries performed if adjust failed */
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t dwRetryCnt;

    phStatus_t PH_MEMLOC_REM statusTimeout;
    uint16_t   PH_MEMLOC_REM wOldBalTimoutMs; /* Store the old Timeout Value */
    uint32_t   PH_MEMLOC_REM dwNewBalTimeoutMs;

    /* Get the old BAL timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));

    /* calculate the new BAL timeout value (in 20sec)*/
    dwNewBalTimeoutMs = (dwAdjustTimeout < 20000) ? 20000 : dwAdjustTimeout;

    /* Check if there would be an overflow */
    if (dwNewBalTimeoutMs > 0xFFFF)
    {
        dwNewBalTimeoutMs = 0xFFFF;
    }
    /* Set the new (temporary) timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

    /* MPC_NfcConfiguration */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_NFC_CONFIGURATION));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_NFC_PASSIVE_MODE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_NFC_TARGET));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, 106));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_NFC_CONFIGURATION);

    if (status == PH_ERR_SUCCESS)
    {
        for (dwRetryCnt = 0; dwRetryCnt <= dwMaxRetryCnt; dwRetryCnt++)
        {
            /* MPC_AdjustLevelRFField */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_ADJUST_LEVEL_RF_FIELD));
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwAdjustTimeout));
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
            status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_ADJUST_LEVEL_RF_FIELD);

            if (status != PH_ERR_SUCCESS)
                continue;

            /* MPC_ChannelOpen */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CHANNEL_OPEN));
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_SIM_MODE));
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
            status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CHANNEL_OPEN);

            if (status == PH_ERR_SUCCESS)
                break;
        }
    }

    /* Restore old timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

    return status;
}

phStatus_t phhalHw_Mp300_Cmd_CloseCardEmulationMode(
    phhalHw_Mp300_DataParams_t * pDataParams     /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    /* Clear Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CHANNEL_CLOSE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CHANNEL_CLOSE);
}

phStatus_t phhalHw_Mp300_Cmd_OpenScenarioPcd(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->dwScenarioPcdId != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Open Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_PCD_OPEN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_PCD_OPEN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, (uint32_t * UNALIGNED)&(pDataParams->dwScenarioPcdId)));

    if (pDataParams->dwScenarioPcdId == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    return status;
}

phStatus_t phhalHw_Mp300_Cmd_ExecuteScenarioPcd(
    phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwTimeout                           /**< [In] Timeout for execution of the scenario in ms. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    phStatus_t PH_MEMLOC_REM statusTimeout;
    uint16_t   PH_MEMLOC_REM wOldBalTimoutMs; /* Store the old Timeout Value */
    uint32_t   PH_MEMLOC_REM dwNewBalTimeoutMs;

    if (pDataParams->dwScenarioPcdId == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Get the old BAL timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));

    /* calculate the new BAL timeout value (5sec on top)*/
    dwNewBalTimeoutMs = dwTimeout + 5000;

    /* Check if there would be an overflow */
    if (dwNewBalTimeoutMs > 0xFFFF)
    {
        dwNewBalTimeoutMs = 0xFFFF;
    }
    /* Set the new (temporary) timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

    /* Execute Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_PCD_EXECUTE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, pDataParams->dwScenarioPcdId));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwTimeout));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_PCD_EXECUTE);

    /* Restore old timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

    return status;
}

phStatus_t phhalHw_Mp300_Cmd_CloseScenarioPcd(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->dwScenarioPcdId == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Clear Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_PCD_CLOSE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, pDataParams->dwScenarioPcdId));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_PCD_CLOSE));

    pDataParams->dwScenarioPcdId = 0;
    return status;
}

phStatus_t phhalHw_Mp300_Cmd_AddToScenarioPcd(
    phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwActionID,                         /**< [In] ID of the action that should be added */
    uint8_t * pParameter,                        /**< [In] string with parameter for the action. */
    uint16_t wParameterLength                    /**< [In] Length of the parameter string. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (dwActionID > 0xFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    if (pParameter == NULL && wParameterLength != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if (pDataParams->dwScenarioPcdId == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Add */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_PCD_ADD));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, pDataParams->dwScenarioPcdId));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, (uint8_t)dwActionID));
    /* If parameter are available add this parameter */
    if (wParameterLength != 0)
    {
        if (pDataParams->wIntBufferLen + 1 + wParameterLength + 1 > pDataParams->wIntBufferSize) /* Space + Param + NULL */
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* First add a space */
        pDataParams->pIntBuffer[pDataParams->wIntBufferLen++] = (uint8_t)' ';
        memcpy(&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], pParameter, wParameterLength);
        pDataParams->wIntBufferLen += wParameterLength;
        pDataParams->pIntBuffer[pDataParams->wIntBufferLen] = 0x00;
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_PCD_ADD);
}

phStatus_t phhalHw_Mp300_Cmd_CouplerCheckLicense(
    phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwLicenseId,                        /**< [In] Specifies the license. */
    int32_t *pValidLicense                       /**< [Out] license status. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_ASSERT_NULL (pDataParams);

    if (pValidLicense == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* MPS_CouplerCheckLicense */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_COUPLER_CHECK_LICENSE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwLicenseId));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_COUPLER_CHECK_LICENSE));
    return phhalHw_Mp300_GetParam_Int32(pDataParams, pValidLicense);
}

phStatus_t phhalHw_Mp300_Cmd_SetBalTimeoutHandler(
    phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    cbBalTimeoutHandling pBalTimeoutHandling,    /**< [In] Pointer to the handler function. */
    void * pContext                              /**< [In] Pointer to the context of the callback function */
    )
{
    pDataParams->pBalTimeoutHandling = pBalTimeoutHandling;
    pDataParams->pBalTimeoutHandlingContext = pContext;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Cmd_EnableSpy(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t dwEventMask,
    uint8_t * pFilename,
    uint16_t wDateFormat
    )
{
    FILE * fh;

    /* Copy filename to pDataParams */
    if(pFilename != NULL)
    {
        memcpy(pDataParams->bSpyFilename, pFilename, strlen((const char *)pFilename) + 1);

        /* try to open file */
        fh = fopen((const char*)pDataParams->bSpyFilename, "ab");

        if(fh == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        fclose(fh);
    }

    return phhalHw_Mp300_Int_EnableSpy(pDataParams, dwEventMask, wDateFormat);
}

phStatus_t phhalHw_Mp300_Cmd_DisableSpy(
    phhalHw_Mp300_DataParams_t * pDataParams
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_Int_DisableSpy(pDataParams));

    /* Reset filename */
    memset(pDataParams->bSpyFilename, 0x00, sizeof(pDataParams->bSpyFilename));
    return status;
}

phStatus_t phhalHw_Mp300_Cmd_GenerateDisturbance(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bDisturbanceOperation,
    uint8_t bDisturbanceType,
    int32_t dwAmplitude,
    int16_t wOffset,
    uint32_t dwDuration,
    uint32_t dwMaintainLevel
    )
{
    phStatus_t  PH_MEMLOC_REM status;

    if(pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP_CMD_GENERATE_DISTURBANCE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bDisturbanceOperation));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bDisturbanceType));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Int32(pDataParams, dwAmplitude));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Int16(pDataParams, wOffset));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwDuration));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwMaintainLevel));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP_CMD_GENERATE_DISTURBANCE);
}

phStatus_t phhalHw_Mp300_Cmd_ResetDisturbance(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bDisturbanceOperation
    )
{
    phStatus_t  PH_MEMLOC_REM status;

    if(pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP_CMD_RESET_DISTURBANCE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bDisturbanceOperation));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP_CMD_RESET_DISTURBANCE);
}

phStatus_t phhalHw_Mp300_Cmd_SetDisturbanceTrigger(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bDisturbanceOperation,
    uint32_t dwBehavior,
    uint32_t dwDelay,
    uint16_t wLoop
    )
{
    phStatus_t  PH_MEMLOC_REM status;

    if(pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
    }

    if (dwDelay > 7732853)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP_CMD_SET_DISTURBANCE_TRIGGER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bDisturbanceOperation));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwBehavior));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwDelay));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wLoop));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP_CMD_SET_DISTURBANCE_TRIGGER);
}

phStatus_t phhalHw_Mp300_Cmd_SetAdvancedTrigger(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bTriggerPort,
    uint32_t dwConfig,
    uint32_t dwValue,
    uint16_t wExchangeCount
    )
{
    phStatus_t status;

    /* Check parameters */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_Int_CheckTriggerParameter(bTriggerPort, dwConfig, dwValue));

    if(wExchangeCount == 0)
    {
        /* Exchange count feature disable therefore set trigger */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_ConfigureTriggerPort(pDataParams, bTriggerPort, dwConfig, dwValue));
    }
    else
    {
        /* Set internal counter which is used in exchange */
        pDataParams->wTriggerExchangeCounters[bTriggerPort - 1] = wExchangeCount;
        pDataParams->dwTriggerConfigs[bTriggerPort - 1] = dwConfig;
        pDataParams->dwTriggerValues[bTriggerPort - 1] = dwValue;

        /* Set trigger to FORCE 1 */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_ConfigureTriggerPort(pDataParams, bTriggerPort, PHHAL_HW_MP300_TRIG_FORCE, 1));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_MP300 */
