/*
 * Copyright 2013, 2015, 2017, 2019 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MP300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_MP300

#include "phhalHw_Mp300.h"
#include "phhalHw_Mp300_Int.h"
#include <phTools.h>
#include <phToolsMicropross.h>
#include <stdio.h>              /* PRQA S 5124 */
#include <stdarg.h>
#include <WinSock2.h>

#ifdef NXPBUILD__PHBAL_REG_MP300USB
#include "../../../phbalReg/src/Mp300Usb/external/lusb0_usb.h"
#endif /* NXPBUILD__PHBAL_REG_MP300USB */

phStatus_t phhalHw_Mp300_SendCommand(
                                     phhalHw_Mp300_DataParams_t * pDataParams
                                     )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t   PH_MEMLOC_REM wNumCharPrint;
    uint8_t *pTransmitBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wTransmitBufferSize = 0;
    uint8_t *pReceiveBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wReceiveBufferSize = 0;
    uint16_t   PH_MEMLOC_REM wReceiveBufferLength = 0;
    uint8_t PH_MEMLOC_REM bRetryCnt = 0;

#ifdef _DEBUG
/* FIXXME */
    FILE * fh = NULL;
#endif

    /* 3 byte for CR LF NULL */
    if (3 > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
    {
        status = PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Add CR LF */
    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], "\r\n");
    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

#ifdef _DEBUG
/* FIXXME */
    fh = fopen(".\\mp300.log","a");
    fprintf(fh,"%s",pDataParams->pIntBuffer);
#endif

    /* Use temporary buffer */
    pTransmitBuffer = pDataParams->pIntBuffer;
    wTransmitBufferSize = pDataParams->wIntBufferLen;
    pReceiveBuffer = pDataParams->pIntBuffer;
    wReceiveBufferSize = pDataParams->wIntBufferSize - 1; /* To have place for the NULL sign */
    pDataParams->wIntBufferLen = 0;

    do
    {
        /* Perform actual exchange */
        status = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            pTransmitBuffer,
            wTransmitBufferSize,
            wReceiveBufferSize,
            pReceiveBuffer,
            &wReceiveBufferLength);

        /* If it is a BAL IO Timeout give it a retry in case we are socket win */
        if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT && ((phbalReg_SocketWin_DataParams_t *)pDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_SOCKETWIN_ID))
        {
            bRetryCnt++;
            /* Perform retry only 2 times */
            if (bRetryCnt <= 2)
            {
#ifdef _DEBUG /* FIXXME */
                fprintf(fh,"BAL timeout -> retry\r\n");
#endif
                phTools_Sleep(5000);

                /* Clear transmit buffer and retry */
                pTransmitBuffer = NULL;
                wTransmitBufferSize = 0;
                continue;
            }

            if (pDataParams->pBalTimeoutHandling != NULL)
            {
                pDataParams->pBalTimeoutHandling(pDataParams->pBalTimeoutHandlingContext, pDataParams->bCouplerNumber);

                /* Flush buffer */
                phbalReg_Exchange(
                    pDataParams->pBalDataParams,
                    PH_EXCHANGE_DEFAULT,
                    NULL,
                    0,
                    wReceiveBufferSize,
                    pReceiveBuffer,
                    &wReceiveBufferLength);
            }
        }

        /* Since we provide the buffer, it's a HAL buffer overflow */
        if ((status & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
        {
#ifdef _DEBUG
/* FIXXME */
            fclose(fh);
#endif
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Add NULL Sign */
        pReceiveBuffer[wReceiveBufferLength] = 0;

#ifdef _DEBUG /* FIXXME */
        fprintf(fh,"%s",pReceiveBuffer);
#endif

        /* Check that there is a valid exchange on BAL layer */
        if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
#ifdef _DEBUG
/* FIXXME */
            fclose(fh);
#endif
            return status;
        }

        /* Reception of one block done -> clear send buffer and adjused next receive buffer */
        pTransmitBuffer = NULL;
        wTransmitBufferSize = 0;
        pReceiveBuffer += wReceiveBufferLength;
        wReceiveBufferSize -= wReceiveBufferLength;
        pDataParams->wIntBufferLen += wReceiveBufferLength;

        if (pDataParams->wIntBufferLen < 2)
        {
#ifdef _DEBUG
/* FIXXME */
            fclose(fh);
#endif
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
    }
    while(pDataParams->pIntBuffer[pDataParams->wIntBufferLen-2] != '\r' ||
        (pDataParams->pIntBuffer[pDataParams->wIntBufferLen-1] != '\n' &&
         pDataParams->pIntBuffer[pDataParams->wIntBufferLen-1] != 0)
        );

    pDataParams->wIntBufferLen-=2;
#ifdef _DEBUG
/* FIXXME */
    fclose(fh);
#endif
    return status;
}

phStatus_t phhalHw_Mp300_SetCommand(
                                    phhalHw_Mp300_DataParams_t * pDataParams,
                                    uint8_t * pCmd
                                    )
{
    /* Reset the scratch buffer len */
    pDataParams->wIntBufferLen = 0;

    /* Buffer overflow check */
    if (strlen((char*)pCmd) + 1 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    strcpy( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], (char*)pCmd); /* PRQA S 3200 */
    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + (uint16_t)strlen((char*)pCmd);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Uint8(
                                        phhalHw_Mp300_DataParams_t * pDataParams,
                                        uint8_t bData
                                        )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* Buffer overflow check */
    /* 4 is the max len of a byte + space */
    if (pDataParams->wIntBufferLen + 4 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], " %u", bData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Uint16(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint16_t wData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 7 is the max len of a word + space */
    if (pDataParams->wIntBufferLen + 7 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %u",wData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Uint32(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t dwData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 11 is the max len of a double word + space */
    if (pDataParams->wIntBufferLen + 11 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %lu",dwData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Int16(
    phhalHw_Mp300_DataParams_t * pDataParams,
    int16_t wData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 8 is the max len of a word + sign + space */
    if (pDataParams->wIntBufferLen + 7 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %d",wData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Int32(
    phhalHw_Mp300_DataParams_t * pDataParams,
    int32_t dwData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 12 is the max len of a double word + sign + space */
    if (pDataParams->wIntBufferLen + 12 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %ld",dwData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Uint8Hex(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t wData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 5 is the max len of a double word + space */
    if (pDataParams->wIntBufferLen + 3 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %02X",wData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Uint16Hex(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint16_t wData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 5 is the max len of a double word + space */
    if (pDataParams->wIntBufferLen + 5 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %04X",wData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_Uint32Hex(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t dwData
    )
{
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* 9 is the max len of a double word + space */
    if (pDataParams->wIntBufferLen + 9 > pDataParams->wIntBufferSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen]," %08lX",dwData);

    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_String(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t * pData
    )
{
    /* Buffer overflow check */
    if (strlen((char*)pData) + 1 > (uint16_t)(pDataParams->wIntBufferSize-pDataParams->wIntBufferLen))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    pDataParams->pIntBuffer[pDataParams->wIntBufferLen] = ' ';
    pDataParams->wIntBufferLen++;
    strcpy( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],(char*)pData); /* PRQA S 3200 */
    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + (uint16_t)strlen((char*)pData);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_AddParam_HexString(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t * pData,
    uint16_t wLen
    )
{
    uint16_t PH_MEMLOC_REM wIndex;
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* Buffer overflow check */
    if ((wLen*2)+1 > pDataParams->wIntBufferSize-pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    pDataParams->pIntBuffer[pDataParams->wIntBufferLen] = ' ';
    pDataParams->wIntBufferLen++;

    for ( wIndex = 0; wIndex < wLen; wIndex++ )
    {
        wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], "%02X", pData[wIndex]);
        if (wNumCharPrint != 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
        pDataParams->wIntBufferLen += 2;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_CatParam_HexString(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t * pData,
    uint16_t wLen
    )
{
    uint16_t PH_MEMLOC_REM wIndex;
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* Buffer overflow check */
    if ((wLen*2) > pDataParams->wIntBufferSize-pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    for ( wIndex = 0; wIndex < wLen; wIndex++ )
    {
        wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], "%02X", pData[wIndex]);
        if (wNumCharPrint != 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
        pDataParams->wIntBufferLen += 2;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_HexString(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t * pData,
    uint16_t wDataSize,
    uint16_t * pwDataLen
    )
{
    uint32_t PH_MEMLOC_REM dwValue;

    *pwDataLen = 0;
    while (pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen)
    {
        /* Overflow detection */
        if ((*pwDataLen) >= wDataSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        if (sscanf((char *)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%02lX",&dwValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        if (dwValue > 0xFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        pData[(*pwDataLen)++] = (uint8_t)dwValue;
        pDataParams->wIntBufferReadPos += 2;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetStatus(
                                   phhalHw_Mp300_DataParams_t * pDataParams,
                                   uint8_t * pCmd
                                   )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t PH_MEMLOC_REM wStatus;

    /* Reset the scratch buffer read position */
    pDataParams->wIntBufferReadPos = 0;

    /* Buffer overflow check */
    if (strlen((char*)pCmd) + 1 > pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* check if command in the response match the expected */
    if (strncmp((char*)pCmd,(char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], strlen((char*)pCmd)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    pDataParams->wIntBufferReadPos = pDataParams->wIntBufferReadPos + (uint16_t)strlen((char*)pCmd);

    if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    pDataParams->wIntBufferReadPos++;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint16Hex(pDataParams, &wStatus));

    /* Evaluate status */
    return phhalHw_Mp300_EvaluateStatus(pDataParams, wStatus);
}

phStatus_t phhalHw_Mp300_EvaluateStatus(
                                        phhalHw_Mp300_DataParams_t * pDataParams,
                                        uint16_t wStatus
                                        )
{
    phStatus_t PH_MEMLOC_REM status;

    switch (wStatus)
    {
    case PHHAL_HW_MP300_ERR_OK:
        status = PH_ERR_SUCCESS;
        break;

    case PHHAL_HW_MP300_ERR_INVALID_PARAMETER:
        status = PH_ERR_INVALID_PARAMETER;
        break;

    case PHHAL_HW_MP300_ERR_TIMEOUTRX:
        status = PH_ERR_IO_TIMEOUT;
        break;

    case PHHAL_HW_MP300_ERR_PARITY:
    case PHHAL_HW_MP300_ERR_CRC:
        status = PH_ERR_INTEGRITY_ERROR;
        break;

    case PHHAL_HW_MP300_ERR_INCONSISTENT_RECEIVED_FRAME:
    case PHHAL_HW_MP300_ERR_BITCOUNTERR:
        status = PH_ERR_FRAMING_ERROR;
        break;

    case PHHAL_HW_MP300_ERR_ATS_INVALID_LENGTH:
    case PHHAL_HW_MP300_ERR_INVALID_ATS_CRC:
    case PHHAL_HW_MP300_ERR_INVALID_ATS_T0:
    case PHHAL_HW_MP300_ERR_INVALID_ATS_FSCI:
    case PHHAL_HW_MP300_ERR_INVALID_ATS_FWI:
    case PHHAL_HW_MP300_ERR_INVALID_ATS_TC1:
    case PHHAL_HW_MP300_ERR_INVALID_ATS_TA1:
    case PHHAL_HW_MP300_ERR_SELECT_FAILED:
    case PHHAL_HW_MP300_ERR_ANTICOL_FAILED:
    case PHHAL_HW_MP300_ERR_ANSWER_TO_HALT_DETECTED:
        status = PH_ERR_PROTOCOL_ERROR;
        break;

    case PHHAL_HW_MP300_ERR_COLLISION:
        status = PH_ERR_COLLISION_ERROR;
        break;

    case PHHAL_HW_MP300_ERR_DATA_RATE_NOT_SUPPORTED:
        status = PH_ERR_UNSUPPORTED_PARAMETER;
        break;

    case PHHAL_HW_MP300_ERR_ADJUSTRX_CHANNEL_2:
        status = PHHAL_HW_MP300_ERR_ADJUST_RX_CHANNEL_2;
        break;

    case PHHAL_HW_MP300_ERR_INVALID_ANSWER_TO_HALT:
    case PHHAL_HW_MP300_ERR_WRONG_ATTRIB:
    case PHHAL_HW_MP300_ERR_DESELECT_SENT:
    case PHHAL_HW_MP300_ERR_INVALID_ANSWER_TO_DESELECT:
    case PHHAL_HW_MP300_ERR_TIMEOUTRX_CMD1:
    case PHHAL_HW_MP300_ERR_TIMEOUTTX:
    case PHHAL_HW_MP300_ERR_UNEXPECTED_RX:
    case PHHAL_HW_MP300_ERR_FELICA_INVALID_LENGTH:
    case PHHAL_HW_MP300_ERR_FELICA_INVALID_SYNC_CODE:
    case PHHAL_HW_MP300_ERR_PPS_INVALID_PARAMETER:
    case PHHAL_HW_MP300_ERR_PPS_INVALID_ANSWER:
    case PHHAL_HW_MP300_ERR_CALIBRATION_1:
    case PHHAL_HW_MP300_ERR_CALIBRATION_2:
    case PHHAL_HW_MP300_ERR_CALIBRATION_3:
    case PHHAL_HW_MP300_ERR_CALIBRATION_4:
    case PHHAL_HW_MP300_ERR_INVALID_IMPEDANCE_CALIBRATION_TABLE:
    case PHHAL_HW_MP300_ERR_NO_CABLE_INFORMATION:
    case PHHAL_HW_MP300_ERR_NO_HEAD_INFORMATION:
    case PHHAL_HW_MP300_ERR_RESONANCE_FREQUENCY_MEASUREMENT:
    case PHHAL_HW_MP300_ERR_NO_CABLE_PLUS_HEAD_INFORMATION:
    case PHHAL_HW_MP300_ERR_NO_SHAPE_SET:
    case PHHAL_HW_MP300_CRET_ANTITEARING:
        pDataParams->wAdditionalInfo = wStatus;
        status = PHHAL_HW_MP300_ERR_MP300;
        break;

    default:
        pDataParams->wAdditionalInfo = wStatus;
        status = PH_ERR_INTERNAL_ERROR;
        break;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_EvaluateMifareStatus(
    uint32_t dwStatus
    )
{
    phStatus_t PH_MEMLOC_REM status;

    switch (dwStatus)
    {
    case MI_OK:
    case MI_POLLING:
        status = PH_ERR_SUCCESS;
        break;

    case MI_NOTAGERR:
        status = PH_ERR_IO_TIMEOUT;
        break;

    case MI_CRCERR:
    case MI_PARITYERR:
        status = PH_ERR_INTEGRITY_ERROR;
        break;

    case MI_AUTHERR:
        status = PH_ERR_AUTH_ERROR;
        break;

    case MI_CODEERR:
    case MI_SERNRERR:
    case MI_BITCOUNTERR:
    case MI_BYTECOUNTERR:
        status = PH_ERR_PROTOCOL_ERROR;
        break;

    case MI_FRAMINGERR:
        status = PH_ERR_FRAMING_ERROR;
        break;

    case MI_READERR:
    case MI_WRITEERR:
        status = PH_ERR_UNSUPPORTED_COMMAND;
        break;

    case MI_COLLERR:
        status = PH_ERR_COLLISION_ERROR;
        break;

    case MI_OVFLERR:
        status = PH_ERR_BUFFER_OVERFLOW;
        break;

    default:
        status = PH_ERR_INTERNAL_ERROR;
        break;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Uint8(
                                        phhalHw_Mp300_DataParams_t * pDataParams,
                                        uint8_t * pData
                                        )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32(pDataParams, &dwValue));

    if (dwValue > 0xFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint8_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Uint16(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint16_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32(pDataParams, &dwValue));

    if (dwValue > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint16_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Uint32(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t * pData
    )
{
    uint16_t PH_MEMLOC_REM wIndex;

    /* find the next space and terminate the string at this position */
    for ( wIndex = pDataParams->wIntBufferReadPos; wIndex < pDataParams->wIntBufferLen; wIndex++ )
    {
        if (pDataParams->pIntBuffer[wIndex] == ' ')
        {
            pDataParams->pIntBuffer[wIndex] = 0;
            break;
        }
    }

    /* scan the return code */
    if (sscanf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%lu",pData ) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* move the read position after the status */
    pDataParams->wIntBufferReadPos = wIndex+1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Int32(
    phhalHw_Mp300_DataParams_t * pDataParams,
    int32_t * pData
    )
{
    uint16_t PH_MEMLOC_REM wIndex;

    /* find the next space and terminate the string at this position */
    for ( wIndex = pDataParams->wIntBufferReadPos; wIndex < pDataParams->wIntBufferLen; wIndex++ )
    {
        if (pDataParams->pIntBuffer[wIndex] == ' ')
        {
            pDataParams->pIntBuffer[wIndex] = 0;
            break;
        }
    }

    /* scan the return code */
    if (sscanf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%d",pData ) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* move the read position after the status */
    pDataParams->wIntBufferReadPos = wIndex+1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Uint8Hex(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwValue));

    if (dwValue > 0xFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint8_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Uint16Hex(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint16_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwValue));

    if (dwValue > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint16_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetParam_Uint32Hex(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t * pData
    )
{
    uint16_t PH_MEMLOC_REM wIndex;

    /* find the next space and terminate the string at this position */
    for ( wIndex = pDataParams->wIntBufferReadPos; wIndex < pDataParams->wIntBufferLen; wIndex++ )
    {
        if (pDataParams->pIntBuffer[wIndex] == ' ')
        {
            pDataParams->pIntBuffer[wIndex] = 0;
            break;
        }
    }

    /* scan the return code */
    if (sscanf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%lX",pData ) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* move the read position after the status */
    pDataParams->wIntBufferReadPos = wIndex+1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetTxBuffer(
                                     phhalHw_Mp300_DataParams_t * pDataParams,
                                     uint8_t ** pTxBuffer,
                                     uint16_t * pTxBufferLen,
                                     uint16_t * pTxBufferSize
                                     )
{
    /* We need to watch that we do not overwrite content below the RxStartPos though */
    if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        *pTxBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufStartPos];
        *pTxBufferSize = pDataParams->wTxBufSize - pDataParams->wRxBufStartPos;
    }
    /* Else just return the actual Buffer */
    else
    {
        *pTxBuffer = pDataParams->pTxBuffer;
        *pTxBufferSize = pDataParams->wTxBufSize;
    }

    /* Return stored length */
    *pTxBufferLen = pDataParams->wTxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetRxBuffer(
                                     phhalHw_Mp300_DataParams_t * pDataParams,
                                     uint8_t ** pRxBuffer,
                                     uint16_t * pRxBufferLen,
                                     uint16_t * pRxBufferSize
                                     )
{
    *pRxBuffer = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    *pRxBufferSize = pDataParams->wRxBufSize - pDataParams->wRxBufStartPos - 2;
    *pRxBufferLen = pDataParams->wRxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_CalcCrc(
                                 phhalHw_Mp300_DataParams_t * pDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint8_t * pCrc
                                 )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTmpBuffer;
    uint16_t   PH_MEMLOC_REM wTmpBufferLen;
    uint16_t   PH_MEMLOC_REM wTmpBufferSize;
    uint16_t   PH_MEMLOC_REM wCrc;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_GetTxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_ISO14443A;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_ISO14443B;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }

        /* invert output */
        wCrc ^= 0xFFFFU;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_FELICA;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_MSB_FIRST,
                wCrc,
                PH_TOOLS_CRC16_POLY_FELICA,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_MSB_FIRST,
                wCrc,
                PH_TOOLS_CRC16_POLY_FELICA,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }
        /* swap bits */
        wCrc = (uint16_t)(wCrc >> 8)|((wCrc&0xFF) << 8);
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_ISO15693;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO15693,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO15693,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }

        /* invert output */
        wCrc ^= 0xFFFFU;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    pCrc[0] = (uint8_t)(wCrc);
    pCrc[1] = (uint8_t)(wCrc >> 8);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_CheckCrc(
                                  phhalHw_Mp300_DataParams_t * pDataParams
                                  )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTmpBuffer;
    uint16_t   PH_MEMLOC_REM wTmpBufferLen;
    uint16_t   PH_MEMLOC_REM wTmpBufferSize;
    uint16_t   PH_MEMLOC_REM wCrc;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_GetRxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

    /* Calculate actuel received rx buffer len */
    wTmpBufferLen = wTmpBufferLen - pDataParams->wRxBufStartPos;

    if (wTmpBufferLen < 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
    }

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_DEFAULT,
            PH_TOOLS_CRC16_PRESET_ISO14443A,
            PH_TOOLS_CRC16_POLY_ISO14443,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        /* CRC over all data must be zero */
        if (wCrc != 0)
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }

        /* Remove CRC */
        wTmpBufferLen -= 2;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {

        wTmpBufferLen -= 2;

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_ISO14443B,
            PH_TOOLS_CRC16_POLY_ISO14443,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        if (((uint8_t)wCrc != pTmpBuffer[wTmpBufferLen]) || ((uint8_t)(wCrc>>8) != pTmpBuffer[wTmpBufferLen+1]))
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_MSB_FIRST,
            PH_TOOLS_CRC16_PRESET_FELICA,
            PH_TOOLS_CRC16_POLY_FELICA,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        /* CRC over all data must be zero */
        if (wCrc != 0)
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }

        /* Remove CRC */
        wTmpBufferLen -= 2;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {

        wTmpBufferLen -= 2;

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_ISO15693,
            PH_TOOLS_CRC16_POLY_ISO15693,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        if (((uint8_t)wCrc != pTmpBuffer[wTmpBufferLen]) || ((uint8_t)(wCrc>>8) != pTmpBuffer[wTmpBufferLen+1]))
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Correct received data length in dataparams */
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpBufferLen;

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_MfcLoadKey(
                                    phhalHw_Mp300_DataParams_t * pDataParams,
                                    uint8_t bMode,
                                    uint8_t bSecNr,
                                    uint8_t * pKey)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwMifareStatus;
    uint32_t PH_MEMLOC_REM dwLastError;

    if (bMode != MP300_MFC_KEYA && bMode != MP300_MFC_KEYB)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    if (bSecNr >= PHHAL_HW_MP300_MAX_NUM_KEYS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_MIFARE_LOAD_KEY));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8Hex(pDataParams, bMode));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8Hex(pDataParams, bSecNr));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_HexString(pDataParams, pKey, 6));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_MIFARE_LOAD_KEY));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwMifareStatus));

    if (dwMifareStatus == MP300_MFC_ERROR_OK)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_MfcGetLastErrorNumber(pDataParams, &dwLastError));
    return phhalHw_Mp300_EvaluateMifareStatus(dwLastError);
}

phStatus_t phhalHw_Mp300_MfcAuthentication(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bMode,
    uint8_t bKeyNr,
    uint32_t wAddress)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwMifareStatus;
    uint32_t PH_MEMLOC_REM dwLastError;

    if (bMode != MP300_MFC_KEYA && bMode != MP300_MFC_KEYB)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    if (bKeyNr >= PHHAL_HW_MP300_MAX_NUM_KEYS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_MIFARE_AUTHENTICAT));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8Hex(pDataParams, bMode));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8Hex(pDataParams, bKeyNr));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, wAddress));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_MIFARE_AUTHENTICAT));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwMifareStatus));

    if (dwMifareStatus == MP300_MFC_ERROR_OK)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_MfcGetLastErrorNumber(pDataParams, &dwLastError));
    return phhalHw_Mp300_EvaluateMifareStatus(dwLastError);
}

phStatus_t phhalHw_Mp300_MfcGetLastErrorNumber(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t * pErrorNr)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwMifareStatus;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_MIFARE_GET_LAST_ERROR));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_MIFARE_GET_LAST_ERROR));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwMifareStatus));

    if (dwMifareStatus != MP300_MFC_ERROR_OK)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* extract last error */
    return phhalHw_Mp300_GetParam_Uint32Hex(pDataParams,pErrorNr);
}

phStatus_t phhalHw_Mp300_SetTimingParameters(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bCommand)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwMeasuredTime;
    uint32_t PH_MEMLOC_REM dwMeasurementCount;
    uint64_t PH_MEMLOC_REM qwTimingSingle;
    uint8_t PH_MEMLOC_REM bTimingMode;
    float64_t PH_MEMLOC_REM dfConvertTmp;
    uint16_t PH_MEMLOC_REM wTimeoutValue_ms;
    float32_t PH_MEMLOC_REM fDriverVersion;
    uint8_t PH_MEMLOC_REM bTimingMeasurementEnableCommand = MP300_TIMING_ENABLE;
    uint8_t PH_MEMLOC_REM bGetFdtUsed = PH_OFF;

    if (sscanf((const char *)pDataParams->bCouplerDriverVersion, "%f", &fDriverVersion) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* New FDT measurement is used if not Method1 is fored and reader supports it */
    bGetFdtUsed = (pDataParams->bForceFdtMethod1 == PH_OFF &&
        pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3 &&
        fDriverVersion >= 1.11) ? PH_ON : PH_OFF;

    /* For Type A cards or if TCL3 is use with FW larger than 1.11 we use methode 2 for time measurement */
    bTimingMeasurementEnableCommand = (pDataParams->bForceFdtMethod1 == PH_OFF &&
        (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A || bGetFdtUsed == PH_ON)) ? MP300_TIMING_ENABLE_M2 : MP300_TIMING_ENABLE;

    /* Adjust enable command if required */
    if (bCommand == MP300_TIMING_ENABLE)
    {
        bCommand = bTimingMeasurementEnableCommand;
    }

    /* Adjust Get command if required */
    if (bCommand == MP300_TIMING_GET && bGetFdtUsed == PH_ON)
    {
        bCommand = MP300_TIMING_GET_FDT;
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_TIMING));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bCommand));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(pDataParams, PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, &wTimeoutValue_ms));

    /* Only TCL3 support Timing in ns (but use only if Timeout value less than 4 sec because there could be a buffer overflow) */
    bTimingMode = (pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3 || wTimeoutValue_ms > 4000) ? MP300_TIMING_IN_CARRIER : MP300_TIMING_IN_NANO;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bTimingMode));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_TIMING));

    if (bCommand == MP300_TIMING_GET || bCommand == MP300_TIMING_GET_FDT)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32(pDataParams, &dwMeasurementCount));
        if (dwMeasurementCount == 1)
        {
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32(pDataParams, &dwMeasuredTime));
            /* If the mode is carrier convert to ns */
            if (bTimingMode == MP300_TIMING_IN_CARRIER)
            {
                dfConvertTmp = (float64_t)0.5 + ((float64_t)dwMeasuredTime * (float64_t)1000.0 / (float64_t)13.56);
            }
            else
            {
                dfConvertTmp = (float64_t)dwMeasuredTime;
            }

            /* Adjust correction for measurement methode 2 if not FDT mode is used */
            if (bTimingMeasurementEnableCommand == MP300_TIMING_ENABLE_M2 && bCommand != MP300_TIMING_GET_FDT)
            {
                if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106)
                {
                    if (dfConvertTmp >= 94395)
                    {
                        dfConvertTmp -= 94395; /* 10 etu offset */
                    }
                    else
                    {
                        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
                    }
                }
                else
                {
                    if (dfConvertTmp >= 37758)
                    {
                        dfConvertTmp -= 37758; /* 32 subcarrier periodes offset */
                    }
                    else
                    {
                        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
                    }
                }
            }

            if (pDataParams->wTimingMode & PHHAL_HW_TIMING_MODE_FDT)
            {
                qwTimingSingle = (uint64_t)dfConvertTmp;
            }
            else if (pDataParams->wTimingMode & PHHAL_HW_TIMING_MODE_COMM)
            {
                qwTimingSingle = (uint64_t)(dfConvertTmp +
                    ((pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent + pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent) * (float64_t)1000.0 / (float64_t)13.56));
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }

            if (pDataParams->wTimingMode & PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR)
            {
                pDataParams->qwTiming_ns = qwTimingSingle;
            }
            else
            {
                /* Check for overflow */
                if ((0xFFFFFFFFFFFFFFFF - pDataParams->qwTiming_ns) < qwTimingSingle)
                {
                    pDataParams->qwTiming_ns = 0xFFFFFFFFFFFFFFFF;
                }
                /* Update global timing value */
                else
                {
                    pDataParams->qwTiming_ns = pDataParams->qwTiming_ns + qwTimingSingle;
                }
            }
        }
        else if (dwMeasurementCount != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
    }
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_TIMING);
}


phStatus_t phhalHw_Mp300_GetPosColl(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t * pdwPosColl)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_POS_COLL));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_POS_COLL));

    return phhalHw_Mp300_GetParam_Uint32(pDataParams, pdwPosColl);
}


phStatus_t phhalHw_Mp300_SetCardType(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bCardtype)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_TYPE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bCardtype));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_TYPE);
}

phStatus_t phhalHw_Mp300_SetModulationASK(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bPercentage)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_MODULATION_ASK));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bPercentage));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_MODULATION_ASK);
}

phStatus_t phhalHw_Mp300_SetTimeout(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t dwTimeout)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwEtu;
    uint32_t PH_MEMLOC_REM dwTimeoutMs;

    dwEtu = (uint32_t)((float32_t)dwTimeout / PHHAL_HW_MP300_ETU_106);

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_TIMEOUT));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwEtu));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_TIMEOUT));

    /* Adjust also the BAL timeout */
    dwTimeoutMs = dwTimeout / 1000;
    /* Check if there would be an overflow */
    if ((uint32_t)PHHAL_HW_MP300_COM_TIMEOUT_OFFSET_MS + dwTimeoutMs > 0xFFFF)
    {
        return phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 0xFFFF);
    }
    return phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)(PHHAL_HW_MP300_COM_TIMEOUT_OFFSET_MS + dwTimeoutMs));
}

phStatus_t phhalHw_Mp300_SetAntiTearing(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t dwTimeout)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_ANTI_TEARING));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwTimeout));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_ANTI_TEARING);
}

phStatus_t phhalHw_Mp300_SetDatarate(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t bTxBitrate, uint16_t bRxBitrate)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_DATARATE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));

    switch (bTxBitrate)
    {
    case PHHAL_HW_RF_DATARATE_106:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_106));
        break;

    case PHHAL_HW_RF_DATARATE_212:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_212));
        break;

    case PHHAL_HW_RF_DATARATE_424:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_424));
        break;

    case PHHAL_HW_RF_DATARATE_848:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_848));
        break;

    case PHHAL_HW_RF_DATARATE_1695:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_1695));
        break;

    case PHHAL_HW_RF_DATARATE_3390:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_3390));
        break;

    case PHHAL_HW_RF_DATARATE_6780:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_6780));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    switch (bRxBitrate)
    {
    case PHHAL_HW_RF_DATARATE_106:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_106));
        break;

    case PHHAL_HW_RF_DATARATE_212:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_212));
        break;

    case PHHAL_HW_RF_DATARATE_424:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_424));
        break;

    case PHHAL_HW_RF_DATARATE_848:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_848));
        break;

    case PHHAL_HW_RF_DATARATE_1695:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_1695));
        break;

    case PHHAL_HW_RF_DATARATE_3390:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_3390));
        break;

    case PHHAL_HW_RF_DATARATE_6780:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_6780));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_DATARATE);
}

phStatus_t phhalHw_Mp300_SetDatarate_ISO15693(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t bTxBitrate, uint16_t bRxBitrate)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_DATARATE_ISO15693));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));

    switch (bTxBitrate)
    {
    case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_1_OUT_OF_256));
        break;
    case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_1_OUT_OF_4));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    switch (bRxBitrate)
    {
    case PHHAL_HW_RF_RX_DATARATE_LOW:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_LOW));
        break;
    case PHHAL_HW_RF_RX_DATARATE_HIGH:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_HIGH));
        break;
    case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_HIGH_X2));
        break;
    case PHHAL_HW_RF_DATARATE_106:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_HIGH_X4));
        break;
    case PHHAL_HW_RF_DATARATE_212:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_HIGH_X8));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    switch (pDataParams->bSubCarrierMode)
    {
    case PHHAL_HW_SUBCARRIER_SINGLE:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_ONE_SUBCARRIER));
        break;
    case PHHAL_HW_SUBCARRIER_DUAL:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, MP300_DATARATE_ISO15693_TWO_SUBCARRIER));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_DATARATE_ISO15693);
}

phStatus_t phhalHw_Mp300_ConfigRxDeafbits(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t bEtu)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwMicroSeconds;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_DEAFTIME));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));

    dwMicroSeconds = (uint32_t)(((float32_t)bEtu * 128.0f)/13.56f);

    /* Comment from Micropross */
    /* you have this behaviour because our product needs at least 19s (255*74ns) before
      the answer from the card to well-detect it in type A (212,424,848) and in type B. */
    if ( ( pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A &&
           pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] != PHHAL_HW_RF_DATARATE_106 ) ||
         pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B )
    {
        if ( dwMicroSeconds >= 49 )
        {
            dwMicroSeconds = dwMicroSeconds - 19;
        }
        else /*if lower than minimum disable */
        {
            dwMicroSeconds = 0;
        }
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwMicroSeconds));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_DEAFTIME);
}

phStatus_t phhalHw_Mp300_SetFieldStrength(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wPermil)
{
    phStatus_t PH_MEMLOC_REM status;
    uint8_t    PH_MEMLOC_REM bPercent = (uint8_t)((wPermil+5)/10);

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_FIELDSTRENGTH));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));

    if( pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL1 )
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_UNIT_PERCENT));
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bPercent));
    }
    else if( pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL2 || pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3 )
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_UNIT_THOUSAND));
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wPermil));
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_FIELDSTRENGTH);
}

phStatus_t phhalHw_Mp300_OpenResource(phhalHw_Mp300_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwResourceId;
    uint8_t    PH_MEMLOC_REM bOpenResourceId;

    pDataParams->wCouplerResourceId = NULL;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_COUPLER_INFO));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_COUPLER_INFO));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwResourceId));

    switch ( dwResourceId )
    {
    case 0x09:
    case 0x0A:
        bOpenResourceId = PHHAL_HW_MP300_RESSOURCE_TCL3;
        break;
    case 0x78:
    case 0x79:
        bOpenResourceId = PHHAL_HW_MP300_RESSOURCE_TCL1;
        break;
    case 0x7A:
        bOpenResourceId = PHHAL_HW_MP300_RESSOURCE_TCL2;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* Try to open the resource not blocking */
    /* If the reader is allready used by the current task it return RET_SEMAPHORE_ALREADY_LOCK F3D */
    /* If the reader is free it allocate it */
    /* If it is allocated by another thread just force */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_OPEN_RESSOURCE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bOpenResourceId));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_RESSOURCE_NOT_BLOCKING));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_OPEN_RESSOURCE);

    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        if (bOpenResourceId != PHHAL_HW_MP300_RESSOURCE_TCL1)
        {
            /* Close the Spy resource again so it can be used from another thread */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SpyClose(pDataParams));
        }
    }
    else if ((status & PH_ERR_MASK) == PH_ERR_INTERNAL_ERROR && pDataParams->wAdditionalInfo == 0x0F3D)
    {
        /* Ok we have a already opened so we just say this open is OK */
        status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    else if ((status & PH_ERR_MASK) == PH_ERR_INTERNAL_ERROR && pDataParams->wAdditionalInfo != 0x0F3D)
    {
        /* If the resourse is not open: Open it with force */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_OPEN_RESSOURCE));
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bOpenResourceId));
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_RESSOURCE_OVERRIDE));

        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_OPEN_RESSOURCE));

        if (bOpenResourceId != PHHAL_HW_MP300_RESSOURCE_TCL1)
        {
            /* Close the Spy resource again so it can be used from another thread */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SpyClose(pDataParams));
        }
    }
    /* Status must be OK now */
    PH_CHECK_SUCCESS(status);

    pDataParams->wCouplerResourceId = bOpenResourceId;
    return phhalHw_Mp300_GetReaderVersions(pDataParams);
}

phStatus_t phhalHw_Mp300_Sleep(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t dwTime)
{
    phStatus_t PH_MEMLOC_REM status;
    phStatus_t PH_MEMLOC_REM statusTimeout;
    uint16_t   PH_MEMLOC_REM wOldBalTimoutMs; /* Store the old Timeout Value */
    uint32_t   PH_MEMLOC_REM dwNewBalTimeoutMs;

    /* Get the old BAL timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));

    /* calculate the new BAL timeout value */
    dwNewBalTimeoutMs = (dwTime / 1000) + (uint32_t)PHHAL_HW_MP300_COM_TIMEOUT_OFFSET_MS;

    /* Check if there would be an overflow */
    if (dwNewBalTimeoutMs > 0xFFFF)
    {
        dwNewBalTimeoutMs = 0xFFFF;
    }
    /* Set the new (temporary) timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

    /* Perform the pause */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_DO_TEMPO));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwTime));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_DO_TEMPO);

    /* Restore old timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

    return status;
}

phStatus_t phhalHw_Mp300_Int_FieldReset(phhalHw_Mp300_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwScenarioHandel;
    phStatus_t PH_MEMLOC_REM statusTimeout;
    uint16_t   PH_MEMLOC_REM wOldBalTimoutMs; /* Store the old Timeout Value */
    uint32_t   PH_MEMLOC_REM dwNewBalTimeoutMs;

    /* Get the old BAL timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));

    /* calculate the new BAL timeout value */
    dwNewBalTimeoutMs = (uint32_t)pDataParams->wFieldOffTime + (uint32_t)pDataParams->wFieldRecoveryTime + (uint32_t)PHHAL_HW_MP300_COM_TIMEOUT_OFFSET_MS;

    /* Check if there would be an overflow */
    if (dwNewBalTimeoutMs > 0xFFFF)
    {
        dwNewBalTimeoutMs = 0xFFFF;
    }
    /* Set the new (temporary) timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

    /* Open Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_OPEN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 2));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_OPEN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwScenarioHandel));

    /* RF Reset */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_ADD));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, dwScenarioHandel));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_TSCN_DO_RF_RESET));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, (pDataParams->wFieldOffTime*1000)));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_ADD));

    /* Sleep */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_ADD));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, dwScenarioHandel));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_TSCN_DO_TEMPO));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, (pDataParams->wFieldRecoveryTime*1000)));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_ADD));

    /* Execute Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_EXECUTE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, dwScenarioHandel));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_EXECUTE));

    /* Clear Scenario */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SCENARIO_CLOSE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32Hex(pDataParams, dwScenarioHandel));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    status = phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SCENARIO_CLOSE);

    /* Restore old timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

    return status;
}

phStatus_t phhalHw_Mp300_ActivateParity(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bEnable)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CHANGE_PARAMETER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_RX));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_PARAM_SIZE));
    if (bEnable == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_OFF));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_ON));
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CHANGE_PARAMETER));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CHANGE_PARAMETER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_TX));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_PARAM_SIZE));
    if (bEnable == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_OFF));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_PARITY_ON));
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CHANGE_PARAMETER);
}

phStatus_t phhalHw_Mp300_ActivateAntiEMD(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bEnable)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CHANGE_PARAMETER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_ANTI_EMD));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_ANTI_EMD_PARAM_SIZE));
    if (bEnable == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_ANTI_EMD_OFF));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_ANTI_EMD_ON));
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CHANGE_PARAMETER);
}

phStatus_t phhalHw_Mp300_SetCarrierFrequency(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t dwCarrierFrequency)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CARRIER_FREQUENCY));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwCarrierFrequency));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CARRIER_FREQUENCY);
}

phStatus_t phhalHw_Mp300_SetWaveformGenerationMode(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bMode)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_WAVEFORM_GENERATION_MODE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bMode));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_WAVEFORM_GENERATION_MODE);
}

phStatus_t phhalHw_Mp300_SetRxGain(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wRxGain)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_RX_GAIN));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wRxGain));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_RX_GAIN);
}

phStatus_t phhalHw_Mp300_SelectRXChannel(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wRXChannel)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SELECT_RX_CHANNEL));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wRXChannel));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SELECT_RX_CHANNEL);
}

phStatus_t phhalHw_Mp300_ConfigureTriggerPort(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bPortnumber, uint32_t dwConfig, uint32_t dwValue)
{
    phStatus_t PH_MEMLOC_REM status;

    if (bPortnumber == MP300_TRIGGER_PORT_1)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_TRIGGER_1));
    }
    else if (bPortnumber == MP300_TRIGGER_PORT_2)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_TRIGGER_2));
    }
    else if (bPortnumber == MP300_TRIGGER_PORT_3 &&
        (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL2 || pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3) )
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_TRIGGER_3));
    }
    else if (bPortnumber == MP300_TRIGGER_PORT_4 &&
        pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3 )
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_TRIGGER_4));
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwConfig));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwValue));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    if (bPortnumber == MP300_TRIGGER_PORT_1)
    {
        return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_TRIGGER_1);
    }
    else if (bPortnumber == MP300_TRIGGER_PORT_2)
    {
        return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_TRIGGER_2);
    }
    else if (bPortnumber == MP300_TRIGGER_PORT_3)
    {
        return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_TRIGGER_3);
    }
    else if (bPortnumber == MP300_TRIGGER_PORT_4)
    {
        return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_TRIGGER_4);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
}

phStatus_t phhalHw_Mp300_SetRxGainTarget(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wTargetRxGainLevel)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SET_INTERNAL_PARAMETER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, 27));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16Hex(pDataParams, wTargetRxGainLevel));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SET_INTERNAL_PARAMETER);
}

phStatus_t phhalHw_Mp300_AdjustRXChannel2(phhalHw_Mp300_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_ADJUST_RX_CHANNEL_2));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_ADJUST_RX_CHANNEL_2));

    if (pDataParams->bRxGain != 100)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetRxGain(pDataParams, pDataParams->bRxGain));
    }

    return status;
}

phStatus_t phhalHw_Mp300_ConfigureRelayPort(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bRelay)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_RELAY));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, MP300_RELAY_1));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bRelay));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_RELAY);
}

phStatus_t phhalHw_Mp300_SetModulationShape(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint32_t dwNbPointsFallingEdge,
    uint16_t * pPointsFallingEdge,
    uint32_t dwNbPointsRisingEdge,
    uint16_t * pPointsRisingEdge)
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t PH_MEMLOC_REM dwIndexCnt;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_WAVEFORM));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwNbPointsFallingEdge));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwNbPointsRisingEdge));

    for ( dwIndexCnt = 0; dwIndexCnt < dwNbPointsFallingEdge; dwIndexCnt++ )
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, pPointsFallingEdge[dwIndexCnt]));
    }

    for ( dwIndexCnt = 0; dwIndexCnt < dwNbPointsRisingEdge; dwIndexCnt++ )
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, pPointsRisingEdge[dwIndexCnt]));
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_WAVEFORM));

    pDataParams->bModulationMode = (pDataParams->bModulationMode&0x0F) | (PH_ON<<4);

    return status;
}

phStatus_t phhalHw_Mp300_SelectPauseWidth(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wPauseWidth)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_PAUSE_WIDTH));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wPauseWidth));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_PAUSE_WIDTH);
}

phStatus_t phhalHw_Mp300_SelectFieldRiseTime(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t dwFieldRiseTime)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_FIELD_RISE_TIME));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, dwFieldRiseTime));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_FIELD_RISE_TIME);
}

phStatus_t phhalHw_Mp300_SelectFallAndRiseTime(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wFallTime, uint16_t wRiseTime)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CONFIG_FALL_AND_RISE_TIME));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wFallTime));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint16(pDataParams, wRiseTime));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CONFIG_FALL_AND_RISE_TIME);
}

phStatus_t phhalHw_Mp300_SetTypeBModulationTimingParameter(phhalHw_Mp300_DataParams_t * pDataParams, uint16_t wValue, uint8_t bIndex)
{
    phStatus_t PH_MEMLOC_REM status;
    uint8_t PH_MEMLOC_REM bIndexCnt;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_CHANGE_PARAMETER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, MP300_FRAME_TYPE_B_CLK));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, MP300_FRAME_TYPE_B_CLK_PARAM_SIZE));

    for ( bIndexCnt = 0; bIndexCnt < 14; bIndexCnt++ )
    {
        if ( bIndexCnt == bIndex )
        {
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, wValue));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, pDataParams->wTypeBTimingParameter[bIndexCnt]));
        }
    }

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_CHANGE_PARAMETER);
}

phStatus_t phhalHw_Mp300_SelectSourceOfCarrier(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bSourceOfCarrier)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SELECT_SOURCE_OF_CARRIER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8Hex(pDataParams, bSourceOfCarrier));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SELECT_SOURCE_OF_CARRIER);
}

phStatus_t phhalHw_Mp300_CalcCommTimePCDToPICC(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint16_t wNumTransmitBits,
    uint16_t wTransmitLen
    )
{
    uint8_t PH_MEMLOC_REM bETUCycles;

    bETUCycles = (uint8_t)(128>>pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE]);

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        /* bits exchanged = SOF + Data Bits (+ Parity) + 2*EOF */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY] == PH_ON)
        {
            pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent = (1 + (wNumTransmitBits>>3)*9 + (wNumTransmitBits&0x07) + 2) * bETUCycles;
        }
        else
        {
            pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent = (1 + wNumTransmitBits + 2) * bETUCycles;
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent = pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_LOW_CYCLES];
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_HIGH_CYCLES];
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EOF_CYCLES];

        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_7_CYCLES]*wTransmitLen;
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_STOP_BIT_CYCLES]*wTransmitLen;

        if (wTransmitLen != 0)
        {   /* If at least on byte was send we also add the EGT */
            pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES]*(wTransmitLen-1);
        }

        /* Type B should be byte exchange but if not bytewise exchange the stopbit of the last byte is missing */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0)
        {
            if (wTransmitLen != 0)
            {   /* If there was a previous byte we must add EGT */
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES];
            }
            pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES];

            switch (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS])
            {
            case 7:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES];
            case 6:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES];
            case 5:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES];
            case 4:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES];
            case 3:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES];
            case 2:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES];
            case 1:
                pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent += pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES];
            default:
                break;
            }
        }
    }
    else
    {
        /* if not supported card type measure no communication timing */
        pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent = 0;
    }
    /* Sum up the overall communication time */
    pDataParams->qwCommunicationTimePCDtoPICCCyclesAll += pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_CalcCommTimePICCToPCD(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint16_t wNumReceivedBits
    )
{
    uint8_t  PH_MEMLOC_REM bETUCycles;
    uint16_t PH_MEMLOC_REM wReceivedLen;

    /* Number of full bytes received */
    wReceivedLen = (wNumReceivedBits>>3);

    bETUCycles = (uint8_t)(128>>pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE]);

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        /* bits exchanged = SOF + Data Bits (+ Parity) + EOF */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY] == PH_ON)
        {
            pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = (1 + wNumReceivedBits + ((wNumReceivedBits + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN])>>3) + 1) * bETUCycles;
        }
        else
        {
            pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = (1 + wNumReceivedBits + 1) * bETUCycles;
        }
        /* If datarate not is 106 add 32*16 Cycles run in sequence */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] != PHHAL_HW_RF_DATARATE_106)
        {
            pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 32*16;
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {
        /* Estimate TR0 as 94.4 usec */
        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = 10*128; /* TR1 minimum */
        /* Estimate the SOF and EOF time as 10+2 and 10 ETU */
        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 10*bETUCycles; /* SOF Low */
        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 2*bETUCycles; /* SOF High */
        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 10*bETUCycles; /* EOF */

        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 10*wReceivedLen*bETUCycles; /* Start D0..D7 End */

        if (wReceivedLen != 0)
        {
            /* EGT estimated as zero */
            pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 0*(wReceivedLen-1)*bETUCycles; /* EGT */
        }

        /* Type B should be byte exchange but if not bytewise exchange the stopbit of the last byte is missing */
        if ((wNumReceivedBits&0x07) != 0)
        {
            if (wReceivedLen != 0)
            {
                pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += 0*bETUCycles; /* EGT */
            }
            pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent += (1+(wNumReceivedBits&0x07))*bETUCycles; /* Start + N*Data Bits */
        }
    }
    else
    {
        /* if not supported card type measure no communication timing */
        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = 0;
    }
    /* Sum up the overall communication time */
    pDataParams->qwCommunicationTimePICCtoPCDCyclesAll += pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetInternalParameter(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bParamId, uint32_t * pdwParamValue)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_INTERNAL_PARAMETER));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bParamId));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, 0)); /* RFU */

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_INTERNAL_PARAMETER));

    return phhalHw_Mp300_GetParam_Uint32(pDataParams, pdwParamValue);
}

phStatus_t phhalHw_Mp300_GetEepromSys(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bParamId, uint8_t * pbParamValue, uint16_t wParamSize)
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t PH_MEMLOC_REM dummy = 0;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_EEPROM_SYS));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bParamId));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_EEPROM_SYS));

    return phhalHw_Mp300_GetParam_HexString(pDataParams, pbParamValue, wParamSize, &dummy);
}

phStatus_t phhalHw_Mp300_GetResourceID(phhalHw_Mp300_DataParams_t * pDataParams, uint8_t bCouplerNumber, uint32_t * pdwResourceID)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_RESOURCE_ID));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_RESOURCE_ID));

    return phhalHw_Mp300_GetParam_Uint32(pDataParams, pdwResourceID);
}

phStatus_t phhalHw_Mp300_GetReaderVersions(phhalHw_Mp300_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwResourceId;
    uint8_t PH_MEMLOC_BUF bDeviceFamily[50];
    uint8_t PH_MEMLOC_BUF bDeviceReferenceWithinFamily[50];
    uint8_t PH_MEMLOC_BUF bMotherboardBootVersion[50];
    uint8_t PH_MEMLOC_BUF bMotherboardSystemVersion[50];
    uint32_t PH_MEMLOC_REM dwHWConfigCoupler;
    uint8_t PH_MEMLOC_BUF bCouplerDriverVersion[50];
    uint8_t PH_MEMLOC_BUF bCouplerSerialNumber[50];
    uint8_t PH_MEMLOC_BUF bCouplerRevision[50];
    uint16_t PH_MEMLOC_REM wDummy;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetEepromSys(pDataParams, MP300_PARAMETER_GET_EEPROM_SYS_EEP_SERIAL_NUMBER, pDataParams->bSerialNumber, PHHAL_HW_MP300_SERIAL_NUMBER_SIZE));

    /* Get Version */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_VERSION));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_VERSION));
    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%s %s %s %s",
        bDeviceFamily, bDeviceReferenceWithinFamily, bMotherboardBootVersion, bMotherboardSystemVersion) != 4)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bDeviceFamily) >= PHHAL_HW_MP300_VERSION_STR_SIZE ||
        strlen((char*)bDeviceReferenceWithinFamily) >= PHHAL_HW_MP300_VERSION_STR_SIZE ||
        strlen((char*)bMotherboardBootVersion) >= PHHAL_HW_MP300_VERSION_STR_SIZE ||
        strlen((char*)bMotherboardSystemVersion) >= PHHAL_HW_MP300_VERSION_STR_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bDeviceFamily, (char*)bDeviceFamily, PHHAL_HW_MP300_VERSION_STR_SIZE); /* MP */
    strncpy((char*)pDataParams->bDeviceReferenceWithinFamily, (char*)bDeviceReferenceWithinFamily, PHHAL_HW_MP300_VERSION_STR_SIZE); /* 500 */
    strncpy((char*)pDataParams->bMotherboardBootVersion, (char*)bMotherboardBootVersion, PHHAL_HW_MP300_VERSION_STR_SIZE); /* 1.07 */
    strncpy((char*)pDataParams->bMotherboardSystemVersion, (char*)bMotherboardSystemVersion, PHHAL_HW_MP300_VERSION_STR_SIZE); /* 5.08 */

    /* Coupler Info & CouplerFPGAVersion */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_GET_COUPLER_INFO));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_GET_COUPLER_INFO));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_Uint32Hex(pDataParams, &dwResourceId));

    switch ( dwResourceId )
    {
    case 0x09: strncpy((char*)pDataParams->bCouplerName, "TCL3", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x0A: strncpy((char*)pDataParams->bCouplerName, "TCL3", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x41: strncpy((char*)pDataParams->bCouplerName, "C1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x42: strncpy((char*)pDataParams->bCouplerName, "C2", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x43: strncpy((char*)pDataParams->bCouplerName, "C3", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xE0: strncpy((char*)pDataParams->bCouplerName, "CSWP", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xE4: strncpy((char*)pDataParams->bCouplerName, "MC1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xD3: strncpy((char*)pDataParams->bCouplerName, "MX1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x81: strncpy((char*)pDataParams->bCouplerName, "TC1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x82: strncpy((char*)pDataParams->bCouplerName, "TC2", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x83: strncpy((char*)pDataParams->bCouplerName, "TC3", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x61:
    case 0x62: strncpy((char*)pDataParams->bCouplerName, "CL1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x63: strncpy((char*)pDataParams->bCouplerName, "CL2", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x78:
    case 0x79: strncpy((char*)pDataParams->bCouplerName, "TCL1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x7A: strncpy((char*)pDataParams->bCouplerName, "TCL2", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xC0: strncpy((char*)pDataParams->bCouplerName, "MHC6", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xE8: strncpy((char*)pDataParams->bCouplerName, "SWPSPY", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    /*case 0xE8: This is also defined in the documentation but then it is used twice */
    case 0xE9:
    case 0xEA: strncpy((char*)pDataParams->bCouplerName, "SC1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0x70: strncpy((char*)pDataParams->bCouplerName, "MT1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xC8: strncpy((char*)pDataParams->bCouplerName, "SCL1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    case 0xD0: strncpy((char*)pDataParams->bCouplerName, "TPC1", PHHAL_HW_MP300_COUPLER_NAME_SIZE); break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%lX %s",
        &dwHWConfigCoupler, bCouplerDriverVersion) != 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bCouplerDriverVersion) >= PHHAL_HW_MP300_VERSION_STR_SIZE || dwHWConfigCoupler > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bCouplerDriverVersion, (char*)bCouplerDriverVersion, PHHAL_HW_MP300_VERSION_STR_SIZE); /* 1.11 */
    pDataParams->wCouplerFPGAVersion = (uint16_t)dwHWConfigCoupler;

    /* bCouplerSerialNumber */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_COUPLER_EEPROM_SYS));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, MP300_PARAMETER_COUPLER_EEPROM_SYS_EEP_COUPLER_IDENT));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_COUPLER_EEPROM_SYS));
    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%s", bCouplerSerialNumber) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bCouplerSerialNumber) >= PHHAL_HW_MP300_COUPLER_SERIAL_NUMBER_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bCouplerSerialNumber, (char*)bCouplerSerialNumber, PHHAL_HW_MP300_COUPLER_SERIAL_NUMBER_SIZE); /* 0x21270215 : Module ID=2127, Unique number : 0215 */

    /* bCouplerRevision */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_COUPLER_EEPROM_SYS));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, MP300_PARAMETER_COUPLER_EEPROM_SYS_EEP_COUPLER_REVISION));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_COUPLER_EEPROM_SYS));
    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%s", bCouplerRevision) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bCouplerRevision) >= PHHAL_HW_MP300_COUPLER_REVISION_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bCouplerRevision, (char*)bCouplerRevision, PHHAL_HW_MP300_COUPLER_REVISION_SIZE); /* 0x3045 : Original version E (no upgrade) */

    /* Calibration Date */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_EEPROM_C_READ));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 10)); /* Offset */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint32(pDataParams, 2)); /* Size in WORD */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_EEPROM_C_READ));
    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetParam_HexString(pDataParams, pDataParams->bCouplerCalibrationDate, PHHAL_HW_MP300_COUPLER_CALC_DATE_SIZE, &wDummy)); /* EECR 0000 0A060B1B  10.06.11 Checksum */
    if (wDummy != PHHAL_HW_MP300_COUPLER_CALC_DATE_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* There is also an additional information that can be obtained: */
    /* --->Remote command HEHW
       <---HEHW 0000 23 3 6 8000000 */

    return status;
}

phStatus_t phhalHw_Mp300_SpyClose(phhalHw_Mp300_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_SPY_CLOSE));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));

    return phhalHw_Mp300_GetStatus(pDataParams, MP300_CMD_SPY_CLOSE);
}

phStatus_t phhalHw_Mp300_Int_EnableSpy(phhalHw_Mp300_DataParams_t * pDataParams, uint32_t dwEventMask, uint16_t wDateFormat)
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    phStatus_t PH_MEMLOC_REM status;
    DWORD PH_MEMLOC_REM GThreadId;
    uint8_t EventMask[4];
    uint8_t localIp[30];
    uint32_t networkaddress;
    uint32_t subnetmask;

    if(pDataParams->SpyThreadHandle != NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Get EventMask from DataParams */
    EventMask[0] = (dwEventMask >> 24) & 0xFF;
    EventMask[1] = (dwEventMask >> 16) & 0xFF;
    EventMask[2] = (dwEventMask >> 8) & 0xFF;
    EventMask[3] = dwEventMask & 0xFF;

    /* Open and Log with given Event Mask and start downloading the Trace */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u",
        MP300_CMD_SPY_OPEN, pDataParams->bCouplerNumber));

    /* Set Dating of events */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u %u",
        MP300_CMD_SPY_DATE_FORMAT, pDataParams->bCouplerNumber, wDateFormat)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u %02X%02X%02X%02X 00000000",
        MP300_CMD_OPEN_LOG, pDataParams->bCouplerNumber, EventMask[0], EventMask[1], EventMask[2], EventMask[3])));

    /* Initialize events */
    pDataParams->GEvtReady = CreateEvent (NULL, 0, 0, "Ready");
    pDataParams->GEvtFinished = CreateEvent (NULL, 0, 0, "Finished");

    /* Create Thread to read data from socket */
    pDataParams->SpyThreadHandle = CreateThread (NULL, 0, phhalHw_Mp300_Int_ReadSpyDataThread, pDataParams, NULL, &GThreadId);

    /* Set highest priority for thread */
    SetThreadPriority(pDataParams->SpyThreadHandle, THREAD_PRIORITY_HIGHEST);

    /* Now waiting for thread to be ready */
    WaitForSingleObject (pDataParams->GEvtReady, 0xFFFFFFFF);

#ifdef NXPBUILD__PHBAL_REG_MP300USB
    if (((phbalReg_Mp300Usb_DataParams_t *)pDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_MP300USB_ID))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u 00000001 %s",
            MP300_CMD_LAUNCH_DOWNLOAD, pDataParams->bCouplerNumber, MP300_USB_PORT)));
    }
    else
#endif /* NXPBUILD__PHBAL_REG_MP300USB */
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams,
            phhalHw_Mp300_Int_GetMpNetworkAddress(pDataParams, &networkaddress, &subnetmask)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_GetLocalIp(networkaddress, subnetmask, localIp)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u 00000001 %s:%s",
            MP300_CMD_LAUNCH_DOWNLOAD, pDataParams->bCouplerNumber, MP300_TCPIP_PORT, localIp)));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_DisableSpy(phhalHw_Mp300_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;

    /* Close and Flush Log, End Download and Close Spy */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u",
        MP300_CMD_CLOSE_LOG, pDataParams->bCouplerNumber)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u",
        MP300_CMD_FLUSH_LOG, pDataParams->bCouplerNumber)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u",
        MP300_CMD_END_DOWNLOAD, pDataParams->bCouplerNumber)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u",
        MP300_CMD_SPY_CLOSE, pDataParams->bCouplerNumber)));

    /* Now waiting for EOF in SpyThread */
    WaitForSingleObject(pDataParams->GEvtFinished, MP300_THREAD_TIMEOUT);

    if (WaitForSingleObject(pDataParams->SpyThreadHandle, MP300_THREAD_TIMEOUT) == WAIT_OBJECT_0)
    {
        /* the thread handle is signaled - the thread has terminated */
        pDataParams->SpyThreadHandle = NULL;
    }
    else
    {
        /* the thread handle is not signaled - the thread is still alive so kill it */
        TerminateThread(pDataParams->SpyThreadHandle, 0);
        pDataParams->SpyThreadHandle = NULL;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_SendAndCheckCmd(phhalHw_Mp300_DataParams_t * pDataParams, const char *format, ...)
{
    phStatus_t      PH_MEMLOC_REM status;
    uint16_t        PH_MEMLOC_REM nbytes;

    va_list args;
    va_start(args, format);

    /* Reset the scratch buffer len and add string to buffer */
    pDataParams->wIntBufferLen = 0;

    /* Buffer overflow check +1 for the '\0' */
    nbytes = (uint16_t)vsnprintf(NULL, 0, format, args) + 1;
    if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    pDataParams->wIntBufferLen += (uint16_t)vsnprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
        pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, format, args);

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_GetStatus(pDataParams, va_arg(args, unsigned char *)));

    va_end(args);
    return status;
}

phStatus_t phhalHw_Mp300_Int_CheckSuccessCloseSpy(phhalHw_Mp300_DataParams_t * pDataParams, phStatus_t status)
{
    if(status != PH_ERR_SUCCESS)
    {
        if(pDataParams->SpyThreadHandle != NULL)
        {
            TerminateThread(pDataParams->SpyThreadHandle, 0);
        }

        phhalHw_Mp300_Int_SendAndCheckCmd(pDataParams, "%s %u", MP300_CMD_SPY_CLOSE, pDataParams->bCouplerNumber);
        return status;
    }
    else
    {
        return status;
    }
}

unsigned long __stdcall phhalHw_Mp300_Int_ReadSpyDataThread(void *voidParams)
{
#ifdef WIN64_BUILD
    /* This is a macro Define for X64 build only
    * Dummy return since usb_find_** interfaces throw Reference not found error in case of WIN64 Build
    */
    PH_UNUSED_VARIABLE(voidParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#else
    phStatus_t statusTmp;
    int32_t dwError;
    int32_t wActRxBufSize = 512;
    uint8_t Buffer[512];
    uint8_t pData[MP300_SPY_BUFFER_SIZE];
    uint32_t bDataOffset = 0, remainingSize = 0;
    uint8_t *pRxBuffer = Buffer;
    SpyHeaderContactless gHeader;
    uint8_t finished = 0;
#ifdef NXPBUILD__PHBAL_REG_MP300USB
    phbalReg_Mp300Usb_DataParams_t *pBalUsbDataParams = NULL;
    uint8_t setRdyFlag = 1;
#endif /* NXPBUILD__PHBAL_REG_MP300USB */
    phbalReg_SocketWin_DataParams_t * pBalWinSocketDataParams;
    uint32_t DataType = 0, Size = 0;
    uint8_t bDataTypeReceived = 0;
    uint8_t bSizeReceived = 0;
    SOCKET ClientSocket = NULL, ServerSocket = NULL;
    uint32_t gHeaderSize;
#ifdef NXPBUILD__PHBAL_REG_MP300USB
    int32_t timeout = 1;
#endif /* NXPBUILD__PHBAL_REG_MP300USB */

    phhalHw_Mp300_DataParams_t *pHalDataParams = (phhalHw_Mp300_DataParams_t *)voidParams;

#ifdef NXPBUILD__PHBAL_REG_MP300USB
    if (((phbalReg_Mp300Usb_DataParams_t *)pHalDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_MP300USB_ID))
    {
        pBalUsbDataParams = (phbalReg_Mp300Usb_DataParams_t*)pHalDataParams->pBalDataParams;
    }
    else
#endif /* NXPBUILD__PHBAL_REG_MP300USB */
    {
        pBalWinSocketDataParams = (phbalReg_SocketWin_DataParams_t*)pHalDataParams->pBalDataParams;
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CreateServerSocket(pHalDataParams->GEvtReady, ((phbalReg_SocketWin_DataParams_t*)(pHalDataParams->pBalDataParams))->pSocketHostName, &ClientSocket, &ServerSocket));
    }

    memset(&gHeader, 0x00, sizeof(gHeader));

    /* Erase old File */
    if(strstr((char *)pHalDataParams->bSpyFilename, (const char *)MP300_TRACEFILE_EXTENSION))
    {
        remove((const char *)pHalDataParams->bSpyFilename);
    }

    while(!finished)
    {
        /* Always read 512 Byte */
#ifdef NXPBUILD__PHBAL_REG_MP300USB
        if (((phbalReg_Mp300Usb_DataParams_t *)pHalDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_MP300USB_ID))
        {
            if (pBalUsbDataParams->pUsbDevHandle != INVALID_HANDLE_VALUE)
            {
                dwError = pBalUsbDataParams->pfBulkRead((usb_dev_handle *)pBalUsbDataParams->pUsbDevHandle,
                    0x82,
                    (int8_t *)pRxBuffer,
                    wActRxBufSize,
                    timeout);

                /* Read once before LGBD to say we are ready */
                if(setRdyFlag == 1)
                {
                    /* Thread ready */
                    SetEvent (pHalDataParams->GEvtReady);
                    setRdyFlag = 0;
                    timeout = 5000;
                }

                /* We always get Usb Packets of specific size (8 Byte Header or up to 512 Byte Data */
                if(dwError > 0)
                {
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_EvaluateData(pHalDataParams, dwError, (uint32_t *)&gHeader, &gHeaderSize, pRxBuffer, pData, &bDataOffset, &remainingSize, &DataType, &Size, &bDataTypeReceived, &bSizeReceived, &finished));
                }
            }
        }
        else
#endif /* NXPBUILD__PHBAL_REG_MP300USB */
        {
            /* We always get 512 Byte so we have to parse Header and Data */
            dwError = recv(ClientSocket, (char *)pRxBuffer, wActRxBufSize, 0);

            /* connection closed or socket error */
            if(dwError != 0 || dwError != SOCKET_ERROR)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_EvaluateData(pHalDataParams, dwError, (uint32_t *)&gHeader, &gHeaderSize, pRxBuffer, pData, &bDataOffset, &remainingSize, &DataType, &Size, &bDataTypeReceived, &bSizeReceived, &finished));
            }
        }
    }

    if (((phbalReg_SocketWin_DataParams_t *)pHalDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_SOCKETWIN_ID))
    {
        closesocket(ServerSocket);
    }

    if(pHalDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
    {
        /* For contactless MP300 we have now got the RAW file so we have to convert to .mplog file */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_ConvertRawToMplog(pHalDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
#endif
}

phStatus_t phhalHw_Mp300_Int_ProcessData(
                                         phhalHw_Mp300_DataParams_t * pDataParams,
                                         uint32_t DataType,
                                         uint32_t Size,
                                         uint8_t *pData,
                                         uint32_t *gHeader,
                                         uint32_t *gHeaderSize,
                                         uint8_t *pbFinished
                                         )
{
    uint32_t counter, i;
    uint32_t eventsNumber;
    SpyHeaderContactless *spyHeader = (SpyHeaderContactless *)gHeader;

    *pbFinished = 0;

    switch (DataType)
    {
    case 1: /* EOF */
        *pbFinished = 1;
        break;

    case 2: /* Header */
        memcpy(gHeader, pData, Size);
        *gHeaderSize = Size;

        if (spyHeader->DataFormat == 0x4D4D)
        {
            phToolsMicropross_SwapHeaderContactless(spyHeader);
            memcpy(pData, gHeader, Size);
        }

        if(strstr((char *)pDataParams->bSpyFilename, (const char *)MP300_TRACEFILE_EXTENSION))
        {
            FILE * fh = fopen((const char*)pDataParams->bSpyFilename, "ab");

            if(fh == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
            }

            for(counter = 0; counter < Size; counter++)
            {
                fprintf(fh, "%c", pData[counter]);
            }

            fclose(fh);
        }
        break;

    case 3: /* Events */
        {
            if(pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
            {
                eventsNumber = Size / sizeof(uint64_t);

                if (spyHeader->DataFormat == 0x4D4D)
                {
                    phToolsMicropross_SwapEvents64((uint64_t *)pData, eventsNumber);
                }
            }
            else
            {
                eventsNumber = Size / sizeof(uint32_t);

                if (spyHeader->DataFormat == 0x4D4D)
                {
                    phToolsMicropross_SwapEvents32((uint32_t *)pData, eventsNumber);
                }
            }

            if(strstr((char *)pDataParams->bSpyFilename, (const char *)MP300_TRACEFILE_EXTENSION))
            {
                FILE * fh = fopen((const char*)pDataParams->bSpyFilename,"ab");

                if(fh == NULL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
                }

                for(counter = 0; counter < Size; counter++)
                {
                    fprintf(fh, "%c", pData[counter]);
                }

                fclose(fh);
            }

            /* Add number of events to header */
            spyHeader->NumberOfEvents += eventsNumber;
        }
        break;

    case 7: /* Burst */
        {
            uint32_t burstNumber = Size / sizeof(uint16_t);
            uint16_t *pBurst = (uint16_t *)pData;

            if (spyHeader->DataFormat == 0x4D4D)
            {
                phToolsMicropross_SwapEvents16(pBurst, burstNumber);
            }

            for (i = 0; i < burstNumber; i++)
            {
                /* printf("\t\tBurst 0x%04X\n", pBurst[i]); */
            }
        }
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}



phStatus_t phhalHw_Mp300_Int_EvaluateData(
                                          phhalHw_Mp300_DataParams_t *pDataParams,
                                          int32_t dwReceivedBytes,
                                          uint32_t *gHeader,
                                          uint32_t *gHeaderSize,
                                          uint8_t *pRxBuffer,
                                          uint8_t *pData,
                                          uint32_t *dwDataOffset,
                                          uint32_t *remainingSize,
                                          uint32_t *DataType,
                                          uint32_t *Size,
                                          uint8_t *bDataTypeReceived,
                                          uint8_t *bSizeReceived,
                                          uint8_t *bFinished
                                          )
{
    phStatus_t statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    uint32_t counter = 0;

    if((uint32_t)dwReceivedBytes == 0)
    {
        *bFinished = 0;
        return statusTmp;
    }

    /* Parse Data */
    if(*bDataTypeReceived && *bSizeReceived)
    {
        /* We got all the data for current DataType */
        if(*remainingSize <= (uint32_t)dwReceivedBytes)
        {
            memcpy(pData + *dwDataOffset, pRxBuffer, *remainingSize);

            /* 4 Bytes DataType - 4 Bytes Size of Data - Actual Data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_ProcessData(pDataParams, *DataType, *Size, pData, gHeader, gHeaderSize, bFinished));

            /* All bytes received */
            *bDataTypeReceived = 0;
            *bSizeReceived = 0;
            *dwDataOffset = 0;

            dwReceivedBytes -= *remainingSize;

            /* Still bytes in buffer */
            if(dwReceivedBytes)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_EvaluateData(pDataParams, dwReceivedBytes, gHeader, gHeaderSize, pRxBuffer + *remainingSize,
                    pData, dwDataOffset, remainingSize, DataType, Size, bDataTypeReceived, bSizeReceived, bFinished));
            }
        }
        else
        {
            /* we got events so we can already write to spy log file in order to clear interal buffer - only for MP500 cause MP300 needs conversion in the end */
            if(*DataType == 3 && pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
            {
                /* write events to file */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_ProcessData(pDataParams, *DataType, dwReceivedBytes, pRxBuffer, gHeader, gHeaderSize, bFinished));

                /* adjust general size of data packet because it was already written to file */
                *Size -= dwReceivedBytes;
            }
            else
            {
                memcpy(pData + *dwDataOffset, pRxBuffer, dwReceivedBytes);
                *dwDataOffset += dwReceivedBytes;
            }

            *remainingSize -= dwReceivedBytes;
        }
    }
    else if(*bDataTypeReceived)
    {
        if(dwReceivedBytes >= 4)
        {
            *Size = (pRxBuffer[0] << 24) | (pRxBuffer[1] << 16) | (pRxBuffer[2] << 8) | pRxBuffer[3];
            *remainingSize = *Size;
            *bSizeReceived = 1;
            dwReceivedBytes -= 4;
        }

        /* Receive remaining bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_EvaluateData(pDataParams, dwReceivedBytes, gHeader, gHeaderSize, pRxBuffer + *bSizeReceived * 4,
            pData, dwDataOffset, remainingSize, DataType, Size, bDataTypeReceived, bSizeReceived, bFinished));
    }
    else
    {
        /* Parse Data Type */
        if(dwReceivedBytes >= 4)
        {
            *DataType = (pRxBuffer[0] << 24) | (pRxBuffer[1] << 16) | (pRxBuffer[2] << 8) | pRxBuffer[3];
            *bDataTypeReceived = 1;
            dwReceivedBytes -= 4;
        }
        if(dwReceivedBytes >= 4)
        {
            *Size = (pRxBuffer[4] << 24) | (pRxBuffer[5] << 16) | (pRxBuffer[6] << 8) | pRxBuffer[7];
            *remainingSize = *Size;
            *bSizeReceived = 1;
            dwReceivedBytes -= 4;
        }

        if(*DataType < 1 || *DataType > 5)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        /* EOF */
        if(*DataType == 1 && *Size == 0)
        {
            *bFinished = 1;

            /* Add the Header once again at the very bottom of the file */
            if(strstr((char *)pDataParams->bSpyFilename, (const char *)MP300_TRACEFILE_EXTENSION))
            {
                FILE * fh = fopen((const char*)pDataParams->bSpyFilename, "ab");

                if(fh == NULL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
                }

                memcpy(pData, gHeader, *gHeaderSize);

                /* set number of events to zero */
                memset(pData + 80, 0x00, 4);

                for(counter = 0; counter < *gHeaderSize; counter++)
                {
                    fprintf(fh, "%c", pData[counter]);
                }

                fclose(fh);
            }

            return phhalHw_Mp300_Int_FinishSpying(pDataParams, gHeader);
        }

        /* Receive remaining bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_EvaluateData(pDataParams, dwReceivedBytes, gHeader, gHeaderSize, pRxBuffer + (*bDataTypeReceived * 4 + *bSizeReceived * 4),
            pData, dwDataOffset, remainingSize, DataType, Size, bDataTypeReceived, bSizeReceived, bFinished));
    }

    return statusTmp;
}

phStatus_t phhalHw_Mp300_Int_FinishSpying(
                                          phhalHw_Mp300_DataParams_t *pDataParams,
                                          uint32_t *gHeader
                                          )
{
    phStatus_t status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    uint32_t counter = 0;
    FILE * fh;
    uint8_t littleEndian = 0x49;
    SpyHeaderContactless *spyHeader = (SpyHeaderContactless *)gHeader;

    if(strstr((char *)pDataParams->bSpyFilename, (const char *)MP300_TRACEFILE_EXTENSION))
    {
        /* write new number of events into file header */
        fh = fopen((const char*)(pDataParams->bSpyFilename),"r+b");

        fseek(fh, 80, SEEK_SET);
        for(counter = 0; counter < 4; counter ++)
        {
            fprintf(fh, "%c", ((spyHeader->NumberOfEvents >> (8 * counter)) & 0xFF));
        }

        fseek(fh, 4, SEEK_SET);
        fwrite(&littleEndian, 1, 1, fh);
        fseek(fh, 5, SEEK_SET);
        fwrite(&littleEndian, 1, 1, fh);
        fclose(fh);
    }

    /* Thread finished */
    SetEvent (pDataParams->GEvtFinished);

    return status;
}

phStatus_t phhalHw_Mp300_Int_ConvertRawToMplog(
                                               phhalHw_Mp300_DataParams_t *pDataParams
                                               )
{
    phStatus_t status;
    int32_t convertStatus;
    uint8_t bResult;
    void *pLibraryHandle;
    MP300_TRANSLATE_MPCLOG TranslateMPCLog2;
    uint8_t bSpyLogFilename[260];

    strcpy((char*)bSpyLogFilename, (char*)pDataParams->bSpyFilename);
    strcat((char*)bSpyLogFilename, ".log");

    /* Load DLL and call convert function */
    pLibraryHandle = LoadLibrary(MP300_MPCLOG_DLL_NAME);

    /* check if loading succeded */
    if (pLibraryHandle == NULL)
    {
        /* failed to load the MPCLog.dll */
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* get function pointer */
    TranslateMPCLog2 = (MP300_TRANSLATE_MPCLOG) GetProcAddress((HMODULE)pLibraryHandle, MP300_TRANSLATE_MPCLOG_FNAME);
    if (TranslateMPCLog2 == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Rename RAW file */
    rename((const char*)pDataParams->bSpyFilename, (const char*)bSpyLogFilename);

    /* Convert */
    convertStatus = TranslateMPCLog2((int8_t *)bSpyLogFilename, (int8_t *)pDataParams->bSpyFilename, 0, 0, 0, 0);

    /* Delte RAW file */
    remove((const char*)bSpyLogFilename);

    if(convertStatus)
    {
        status = PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }
    else
    {
        status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* unload the library */
    bResult = (uint8_t)FreeLibrary((HMODULE)pLibraryHandle);

    /* check the success of the command */
    if (bResult == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    return status;
}

phStatus_t phhalHw_Mp300_Int_GetMpNetworkAddress(
                                                 phhalHw_Mp300_DataParams_t * pDataParams,
                                                 uint32_t * NetworkAddress,
                                                 uint32_t * SubnetMask
                                                 )
{
    phStatus_t status;
    uint32_t subnetmask[4];
    uint32_t ipaddress[4];
    uint32_t dwip;

    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_Int_CheckSuccessCloseSpy(pDataParams, phhalHw_Mp300_Int_SendAndCheckCmd(
        pDataParams, "%s", MP300_CMD_GET_NETWORK_INFO)));

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }

    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%d.%d.%d.%d %d.%d.%d.%d",
        &ipaddress[0], &ipaddress[1], &ipaddress[2], &ipaddress[3], &subnetmask[0], &subnetmask[1], &subnetmask[2], &subnetmask[3]) != 8)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    dwip = (ipaddress[0] << 24) + (ipaddress[1] << 16) + (ipaddress[2] << 8) + ipaddress[3];
    *SubnetMask = (subnetmask[0] << 24) + (subnetmask[1] << 16) + (subnetmask[2] << 8) + subnetmask[3];

    *NetworkAddress = dwip & *SubnetMask;

    return status;
}

phStatus_t phhalHw_Mp300_Int_CheckTriggerParameter(
                                                   uint8_t bTriggerPort,
                                                   uint32_t dwConfig,
                                                   uint32_t dwValue
                                                   )
{
   /* Check Trigger Port */
    if( bTriggerPort != MP300_TRIGGER_PORT_1 && bTriggerPort != MP300_TRIGGER_PORT_2 && bTriggerPort != MP300_TRIGGER_PORT_3 && bTriggerPort != MP300_TRIGGER_PORT_4 )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check Value */
    if ((dwConfig != PHHAL_HW_MP300_TRIG_FORCE) &&
        (dwConfig != PHHAL_HW_MP300_TRIG_TX_ON) &&
        (dwConfig != PHHAL_HW_MP300_TRIG_TX_OUT) &&
        (dwConfig != PHHAL_HW_MP300_TRIG_RX_ON) &&
        (dwConfig != PHHAL_HW_MP300_TRIG_DELAY_AFTER_TX) &&
        (dwConfig != PHHAL_HW_MP300_TRIG_CARRIER) &&
        (dwConfig != PHHAL_HW_MP300_TRIG_EMD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if ((dwConfig == PHHAL_HW_MP300_TRIG_FORCE) &&
        (dwValue != 0 && dwValue != 1))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if ((dwConfig == PHHAL_HW_MP300_TRIG_DELAY_AFTER_TX) &&
        (dwValue > 335000000 || dwValue%20 != 0))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_EnableTrigger(
                                           phhalHw_Mp300_DataParams_t * pDataParams
                                           )
{
    phStatus_t status;
    uint8_t triggerPort;

    /* If ExchangeCount is reached enabled desired Trigger with set config */
    for(triggerPort = 0; triggerPort < MP300_TRIGGER_PORT_4; triggerPort++)
    {
        if(pDataParams->wTriggerExchangeCounters[triggerPort] == 1)
        {
            PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_ConfigureTriggerPort(
                pDataParams,
                triggerPort + 1,
                pDataParams->dwTriggerConfigs[triggerPort],
                pDataParams->dwTriggerValues[triggerPort])
                );
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_DisableTrigger(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            )
{
    phStatus_t status;
    uint8_t triggerPort;

    /* If ExchangeCount is reached disable desired Trigger -> set to FORCE 1 */
    for(triggerPort = 0; triggerPort < MP300_TRIGGER_PORT_4; triggerPort++)
    {
        if(pDataParams->wTriggerExchangeCounters[triggerPort] == 1)
        {
            /* Set trigger to FORCE 1 */
            PH_CHECK_SUCCESS_FCT_CLEANUP(status, phhalHw_Mp300_ConfigureTriggerPort(pDataParams, triggerPort + 1, PHHAL_HW_MP300_TRIG_FORCE, 1), phhalHw_Mp300_Int_UpdateTriggerCounter(pDataParams));
        }
    }
    PH_CHECK_SUCCESS_FCT(status, phhalHw_Mp300_Int_UpdateTriggerCounter(pDataParams));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_UpdateTriggerCounter(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            )
{
    uint8_t triggerPort;
    for(triggerPort = 0; triggerPort < MP300_TRIGGER_PORT_4; triggerPort++)
    {
        if(pDataParams->wTriggerExchangeCounters[triggerPort] > 0)
        {
            /* Decrease counter for current exchange */
            pDataParams->wTriggerExchangeCounters[triggerPort]--;
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_EnableTearing(
                                           phhalHw_Mp300_DataParams_t * pDataParams
                                           )
{
    phStatus_t statusTmp;

    /* Anti Tearing functionality is supported with TCL2/TCL3 */
    /* Tearing is either enabled if wTearingExchangeCounter is zero (always enabled) or if the counter reach 1 */
    if ( (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL2 || pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3) &&
        pDataParams->bRfResetAfterTo != PH_OFF && pDataParams->wTearingExchangeCounter <= 1)
    {
        /* Calculate values for Microsecond values */
        if ( pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS )
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetAntiTearing(pDataParams, (uint32_t)(13.56*pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US])));
        }
        /* Calculate values for Millisecond values */
        else if ( pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS )
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetAntiTearing(pDataParams, (uint32_t)(13560*pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS])));
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_DisableTearing(
                                            phhalHw_Mp300_DataParams_t * pDataParams,
                                            phStatus_t * statusExchange
                                            )
{
    phStatus_t statusTmp;

    /* Anti Tearing functionality is supported with TCL2/TCL3 */
    /* Tearing is either enabled if wTearingExchangeCounter is zero (always enabled) or if the counter reach 1 */
    if (pDataParams->bRfResetAfterTo != PH_OFF && pDataParams->wTearingExchangeCounter <= 1)
    {
        if (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL2 || pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            /* Deactivate Anti tearing */
            PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_SetAntiTearing(pDataParams, 0), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));

            /* If tearing occured or field should be switched off always */
            if ( ((*statusExchange & PH_ERR_MASK) == PHHAL_HW_MP300_ERR_MP300 && pDataParams->wAdditionalInfo == PHHAL_HW_MP300_CRET_ANTITEARING) ||
                pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_OFF ||
                pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_ON)
            {
                /* Field Off to switch RF off cleanly */
                PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_FieldOff(pDataParams), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));

                /* Pause */
                PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, pDataParams->wFieldOffTime), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));

                if(pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_ON || pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_ON_AFTER_TEARING)
                {
                    /* Field On again */
                    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_FieldOn(pDataParams), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));

                    /* Pause */
                    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, pDataParams->wFieldRecoveryTime), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));
                }

            }

            /* There was an antitearing event */
            if ( (*statusExchange & PH_ERR_MASK) == PHHAL_HW_MP300_ERR_MP300 && pDataParams->wAdditionalInfo == PHHAL_HW_MP300_CRET_ANTITEARING )
            {
                *statusExchange = PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
            }
        }
        else /* TCL 1 just do a field reset */
        {
            /* If tearing (in TCL1 case timeout) occured or field should be switched off always */
            if ( (*statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT ||
                pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_OFF ||
                pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_ON)
            {
                if(pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_ON || pDataParams->bRfAfterTearing == PHHAL_HW_MP300_TEARING_FIELD_ON_AFTER_TEARING)
                {
                    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_Int_FieldReset(pDataParams), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));
                }
                else
                {
                    /* Field Off to switch RF off cleanly */
                    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_FieldOff(pDataParams), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));

                    /* Pause */
                    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, pDataParams->wFieldOffTime), phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));
                }
            }
        }
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_UpdateTearingCounter(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_UpdateTearingCounter(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            )
{
    /* Count down exchange counter */
    if(pDataParams->wTearingExchangeCounter != 0)
    {
        /* disable tearing for next commands when counter hit */
        if (pDataParams->wTearingExchangeCounter == 1)
        {
            pDataParams->bRfResetAfterTo = PH_OFF;
        }
        /* decrease tearing index for current exchange */
        pDataParams->wTearingExchangeCounter--;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Int_CleanupTearingAndTrigger(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            )
{
    phStatus_t statusTrigger;
    phStatus_t statusTearing;
    phStatus_t statusExchange = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    statusTrigger = phhalHw_Mp300_Int_DisableTrigger(pDataParams);
    statusTearing = phhalHw_Mp300_Int_DisableTearing(pDataParams, &statusExchange);
    PH_CHECK_SUCCESS(statusTrigger);
    PH_CHECK_SUCCESS(statusTearing);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_MP300 */
