/*
 * Copyright 2013, 2015, 2017, 2019 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal definitions for Mp300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_MP300_INT_H
#define PHHALHW_MP300_INT_H

#include <ph_Status.h>

#define PHHAL_HW_MP300_ETU_106              (1000.0f / 106.0f)  /**< Duration of one ETU at 106 kBit/s in [us]. */

#define PHHAL_HW_MP300_MODINDEX_I14443B     0x0AU   /**< Default modulation index for ISO14443B. */
#define PHHAL_HW_MP300_MODINDEX_FELICA      0x0CU   /**< Default modulation index for FELICA. */
#define PHHAL_HW_MP300_MAX_NUM_KEYS         16U     /**< Maximum possible number of keys in the EEPROM. */

#define PHHAL_HW_MP300_COM_TIMEOUT_OFFSET_MS  5000U    /**< Additional Timeout used for the BAL. */

#define PHHAL_HW_MP300_ERR_OK                           0x000U      /**< Everything OK. */

#define PHHAL_HW_MP300_ERR_INVALID_PARAMETER            0x003U      /**< invalid parameters. */

#define PHHAL_HW_MP300_CRET_ANTITEARING                 0xF0U      /**< Command aborted due to Antitearing function. */

#define PHHAL_HW_MP300_ERR_TIMEOUTRX                    0x190U      /**< Receive timeout has been reached. */
#define PHHAL_HW_MP300_ERR_COLLISION                    0x191U      /**< Collision TYPE A detected. */
#define PHHAL_HW_MP300_ERR_ANTICOL_FAILED               0x192U      /**< Anticollision type A has failed. */
#define PHHAL_HW_MP300_ERR_PARITY                       0x193U      /**< Parity error detected (type A). */
#define PHHAL_HW_MP300_ERR_BITCOUNTERR                  0x195U      /**< Received invalid number of bits (type A). */
#define PHHAL_HW_MP300_ERR_SELECT_FAILED                0x196U      /**< Select command failed (type A). */
#define PHHAL_HW_MP300_ERR_CRC                          0x197U      /**< CRC received failed. */
#define PHHAL_HW_MP300_ERR_ANSWER_TO_HALT_DETECTED      0x199U      /**< The Picc type A responds to HALT command. */

#define PHHAL_HW_MP300_ERR_ATS_INVALID_LENGTH      0x19AU
#define PHHAL_HW_MP300_ERR_INVALID_ATS_CRC         0x19BU
#define PHHAL_HW_MP300_ERR_INVALID_ATS_T0          0x19CU
#define PHHAL_HW_MP300_ERR_INVALID_ATS_FSCI        0x19DU
#define PHHAL_HW_MP300_ERR_INVALID_ATS_FWI         0x19EU
#define PHHAL_HW_MP300_ERR_INVALID_ATS_TC1         0x19FU
#define PHHAL_HW_MP300_ERR_INVALID_ATS_TA1         0x1A0U

#define PHHAL_HW_MP300_ERR_INCONSISTENT_RECEIVED_FRAME      0x1A1U     /**< Type B. */
#define PHHAL_HW_MP300_ERR_INVALID_ANSWER_TO_HALT           0x1A2U     /**< Answer to halt command is not valid. */
#define PHHAL_HW_MP300_ERR_WRONG_ATTRIB                     0x1A3U     /**< Attrib content is wrong or not supported (MPC_SendATTRIB). */
#define PHHAL_HW_MP300_ERR_DESELECT_SENT                    0x1A4U     /**< DESELECT command sent due to invalid exchange in the protocol 14443-4. */
#define PHHAL_HW_MP300_ERR_DATA_RATE_NOT_SUPPORTED          0x1A5U     /**< Data rate not supported by the CLASS (API MPC_SelectDataRate). */
#define PHHAL_HW_MP300_ERR_INVALID_ANSWER_TO_DESELECT       0x1A6U     /**< DESELECT command return error. */
#define PHHAL_HW_MP300_ERR_TIMEOUTRX_CMD1                   0x1A9U     /**< Fo answer after the first command (API MPC_TEST ( TEST_FDT_PICCPCD_A/B). */
#define PHHAL_HW_MP300_ERR_TIMEOUTTX                        0x1AAU     /**< End of transmission not detected (API MPC_TEST). */
#define PHHAL_HW_MP300_ERR_UNEXPECTED_RX                    0x1ABU     /**< Unexpected RX frame received. */

#define PHHAL_HW_MP300_ERR_FELICA_INVALID_LENGTH                430    /**< Received invalid length (type Felica). */
#define PHHAL_HW_MP300_ERR_FELICA_INVALID_SYNC_CODE             431    /**< Received invalid sync code (type Felica). */

#define PHHAL_HW_MP300_ERR_PPS_INVALID_PARAMETER                432    /**< Invalid parameter passed to the API MPC_SendPPS. */
#define PHHAL_HW_MP300_ERR_PPS_INVALID_ANSWER                   433    /**< Received invalid answer from the PPS command. */

#define PHHAL_HW_MP300_ERR_CALIBRATION_1                        500
#define PHHAL_HW_MP300_ERR_CALIBRATION_2                        501
#define PHHAL_HW_MP300_ERR_CALIBRATION_3                        502
#define PHHAL_HW_MP300_ERR_CALIBRATION_4                        503
#define PHHAL_HW_MP300_ERR_INVALID_IMPEDANCE_CALIBRATION_TABLE  504
#define PHHAL_HW_MP300_ERR_NO_CABLE_INFORMATION                 505
#define PHHAL_HW_MP300_ERR_NO_HEAD_INFORMATION                  506

#define PHHAL_HW_MP300_ERR_RESONANCE_FREQUENCY_MEASUREMENT      507       /**< system can not determine the resonance frequency. */
#define PHHAL_HW_MP300_ERR_ADJUSTRX_CHANNEL_2                   508       /**< System can not adjust Rx channel 2. */

#define PHHAL_HW_MP300_ERR_NO_CABLE_PLUS_HEAD_INFORMATION       509
#define PHHAL_HW_MP300_ERR_NO_SHAPE_SET                         510

#define PHHAL_HW_MP300_ERR_PROTOCOL_ERROR_DETECTED              513
#define PHHAL_HW_MP300_ERR_PROTOCOL_STOP_TIMEOUT_DETECTED       514

#define MI_OK                           (0)
#define MI_NOTAGERR                     (1)
#define MI_CRCERR                       (2)
#define MI_EMPTY                        (3)
#define MI_AUTHERR                      (4)
#define MI_PARITYERR                    (5)
#define MI_CODEERR                      (6)
#define MI_SERNRERR                     (8)
#define MI_KEYERR                       (9)
#define MI_NOTAUTHERR                   (10)
#define MI_BITCOUNTERR                  (11)
#define MI_BYTECOUNTERR                 (12)
#define MI_TRANSERR                     (14)
#define MI_WRITEERR                     (15)
#define MI_INCRERR                      (16)
#define MI_DECRERR                      (17)
#define MI_READERR                      (18)
#define MI_OVFLERR                      (19)
#define MI_POLLING                      (20)
#define MI_FRAMINGERR                   (21)
#define MI_COLLERR                      (24)
#define MI_INTERFACEERR                 (26)
#define MI_ACCESSTIMEOUT                (27)
#define MI_VALERR                       (124)

/* Resource Commands */
#define MP300_CMD_OPEN_RESSOURCE            (uint8_t *)"RMOP"
#define MP300_CMD_CLOSE_RESSOURCE           (uint8_t *)"RMCL"
/* Info Commands */
#define MP300_CMD_GET_COUPLER_INFO          (uint8_t *)"HEHM"
#define MP300_CMD_GET_RESOURCE_ID           (uint8_t *)"GTRI"
/* Config Commands */
#define MP300_CMD_TYPE                      (uint8_t *)"CLTY"
#define MP300_CMD_CHANGE_PARAMETER          (uint8_t *)"CLCP"
#define MP300_CMD_FIELDSTRENGTH             (uint8_t *)"CLFS"
#define MP300_CMD_DATARATE                  (uint8_t *)"CLDR"
#define MP300_CMD_DATARATE_ISO15693         (uint8_t *)"VCSC"
#define MP300_CMD_DEAFTIME                  (uint8_t *)"CLDT"
#define MP300_CMD_MODULATION_ASK            (uint8_t *)"CLAS"
#define MP300_CMD_TIMEOUT                   (uint8_t *)"FWTE"
#define MP300_CMD_CARRIER_FREQUENCY         (uint8_t *)"CLFQ"
#define MP300_CMD_RX_GAIN                   (uint8_t *)"CLRG"
#define MP300_CMD_SELECT_RX_CHANNEL         (uint8_t *)"SLRX"
#define MP300_CMD_SELECT_SOURCE_OF_CARRIER  (uint8_t *)"SLSC"
#define MP300_CMD_CONFIG_TRIGGER_1          (uint8_t *)"CLTO"
#define MP300_CMD_CONFIG_TRIGGER_2          (uint8_t *)"CLT2"
#define MP300_CMD_CONFIG_TRIGGER_3          (uint8_t *)"CLT3"
#define MP300_CMD_CONFIG_TRIGGER_4          (uint8_t *)"CLT4"
#define MP300_CMD_CONFIG_RELAY              (uint8_t *)"CLRL"
/* Waveshape Commands */

/* Exchange + measurement */
#define MP300_CMD_EXCHANGE                  (uint8_t *)"XBCD"
#define MP300_CMD_EXCHANGE_ISO15693         (uint8_t *)"VCEX"
#define MP300_CMD_TIMING                    (uint8_t *)"CLG2"
/* Scenario Commands */
#define MP300_CMD_SCENARIO_OPEN             (uint8_t *)"SCOP"
#define MP300_CMD_SCENARIO_ADD              (uint8_t *)"SCAD"
#define MP300_CMD_SCENARIO_EXECUTE          (uint8_t *)"SCEX"
#define MP300_CMD_SCENARIO_CLOSE            (uint8_t *)"SCCL"

/* Card Emulation Commands */
#define MP300_CMD_NFC_CONFIGURATION         (uint8_t *)"NF01"
#define MP300_CMD_CHANNEL_OPEN              (uint8_t *)"CHOP"
#define MP300_CMD_ADJUST_LEVEL_RF_FIELD     (uint8_t *)"ADRF"
#define MP300_CMD_CHANNEL_CLOSE             (uint8_t *)"CHCL"

/* Scenario PCD Commands */
#define MP300_CMD_SCENARIO_PCD_OPEN         (uint8_t *)"SCO2"
#define MP300_CMD_SCENARIO_PCD_ADD          (uint8_t *)"SCA2"
#define MP300_CMD_SCENARIO_PCD_EXECUTE      (uint8_t *)"SCE2"
#define MP300_CMD_SCENARIO_PCD_CLOSE        (uint8_t *)"SCC2"

/* Mifare Commands */
#define MP300_CMD_MIFARE_LOAD_KEY           (uint8_t *)"MI16"
#define MP300_CMD_MIFARE_AUTHENTICAT        (uint8_t *)"MI07"
#define MP300_CMD_MIFARE_GET_LAST_ERROR     (uint8_t *)"MI32"

/* Disturbance Commands */
#define MP_CMD_GENERATE_DISTURBANCE         (uint8_t *)"DGGD"
#define MP_CMD_RESET_DISTURBANCE            (uint8_t *)"DGDR"
#define MP_CMD_SET_DISTURBANCE_TRIGGER      (uint8_t *)"DGTG"

/* Other */
#define MP300_CMD_DO_TEMPO                  (uint8_t *)"DOTP"
#define MP300_CMD_SPY_OPEN                  (uint8_t *)"LGOP"
#define MP300_CMD_SPY_CLOSE                 (uint8_t *)"LGCL"
#define MP300_CMD_SPY_DATE_FORMAT           (uint8_t *)"LGCM"
#define MP300_CMD_OPEN_LOG                  (uint8_t *)"CPOL"
#define MP300_CMD_CLOSE_LOG                 (uint8_t *)"CPCL"
#define MP300_CMD_END_DOWNLOAD              (uint8_t *)"LGED"
#define MP300_CMD_FLUSH_LOG                 (uint8_t *)"CPFL"
#define MP300_CMD_LAUNCH_DOWNLOAD           (uint8_t *)"LGBD"
#define MP300_CMD_GET_NETWORK_INFO          (uint8_t *)"NMGA"


/* Not used/Not tested */
#define MP300_CMD_LEDON                     (uint8_t *)"LDON"
#define MP300_CMD_LEDOFF                    (uint8_t *)"LDOF"
#define MP300_CMD_BATCH_OPEN                (uint8_t *)"BAOP"
#define MP300_CMD_BATCH_CLOSE               (uint8_t *)"BACL"
#define MP300_CMD_BATCH_ADD_2               (uint8_t *)"BAAD"
#define MP300_CMD_BATCH_EXECUTE             (uint8_t *)"BAEX"

#define MP300_CMD_CONFIG_PAUSE_WIDTH        (uint8_t *)"CLPS"
#define MP300_CMD_CONFIG_WAVEFORM           (uint8_t *)"CLSH"
#define MP300_CMD_CONFIG_WAVEFORM_GENERATION_MODE  (uint8_t *)"CLMG"
#define MP300_CMD_CONFIG_FIELD_RISE_TIME    (uint8_t *)"SRFS"
#define MP300_CMD_CONFIG_FALL_AND_RISE_TIME (uint8_t *)"SFRT"
#define MP300_CMD_GET_POS_COLL              (uint8_t *)"CLPC"
#define MP300_CMD_ADJUST_RX_CHANNEL_2       (uint8_t *)"ADRX"
#define MP300_CMD_SET_INTERNAL_PARAMETER    (uint8_t *)"XX02"
#define MP300_CMD_ANTI_TEARING              (uint8_t *)"TETR"
#define MP300_CMD_GET_INTERNAL_PARAMETER    (uint8_t *)"SVGP"
#define MP300_CMD_GET_EEPROM_SYS            (uint8_t *)"EEGS"
#define MP300_CMD_GET_VERSION               (uint8_t *)"HELO"
#define MP300_CMD_COUPLER_EEPROM_SYS        (uint8_t *)"EECS"
#define MP300_CMD_COUPLER_CHECK_LICENSE     (uint8_t *)"EECL"
#define MP300_CMD_EEPROM_C_READ             (uint8_t *)"EECR"
#define MP300_CMD_SPY_CLOSE                 (uint8_t *)"LGCL"

/* Spy Param */

typedef int32_t (__stdcall *MP300_TRANSLATE_MPCLOG) (int8_t * SourceFile, int8_t *DestinationFile, int32_t rfu1, uint32_t rfu2, uint32_t rfu3, int32_t rfu4);
#define MP300_MPCLOG_DLL_NAME               "MPCLog.dll"
#define MP300_TRANSLATE_MPCLOG_FNAME        "_TranslateMPCLog2@24"
#define MP300_USB_PORT                      (uint8_t *)"USB:102"
#define MP300_TCPIP_PORT                    (uint8_t *)"61641"
#define MP300_TRACEFILE_EXTENSION           (uint8_t *)".mplog"
#define MP300_SPY_BUFFER_SIZE               4096   /**< Size of Buffers in Spy Thread */
#define MP300_THREAD_TIMEOUT                50000

/* Scenario Param */
#define MP300_TSCN_PARAM_CARD_TYPE      1
#define MP300_TSCN_PARAM_FWT            17
#define MP300_TSCN_PARAM_FDT_PCD        18
#define MP300_TSCN_DO_TEMPO             19
#define MP300_TSCN_DO_RF_FIELD_STRENGTH 20
#define MP300_TSCN_DO_RF_RESET          21
#define MP300_TSCN_DO_EXCHANGE          22
#define MP300_EXCHANGE_NOPCDCRC         1
#define MP300_EXCHANGE_PCDCRC           2
#define MP300_EXCHANGE_WAIT_RX          1
#define MP300_EXCHANGE_NO_WAIT_RX       2
#define MP300_TSCN_DO_PARITY_ERROR      23
#define MP300_TSCN_DO_CHANGE_DATA_RATE  24
#define MP300_TSCN_DO_USER_EVENT        25
#define MP300_TSCN_DO_TRIGGER_OUT       26
#define MP300_TRIGGER_OUT1              1
#define MP300_TRIGGER_OUT2              2
#define MP300_TRIGGER_OFF               0
#define MP300_TRIGGER_ON                1

/* MPC_NfcConfiguration parameter */
/* define for parameter Mode */
#define MP300_NFC_PASSIVE_MODE          1
#define MP300_NFC_PASSIVE_MODE2         3   /* In Target passive mode 2, the TCL2 can also generate the carrier */
#define MP300_NFC_ACTIVE_MODE           2

/* define for parameter Initiator Or Target */
#define MP300_NFC_INITIATOR             1
#define MP300_NFC_TARGET                2

/* MPC_ChannelOpen parameter */
#define MP300_SIM_MODE                  2

/* Mifare Param */
#define MP300_MFC_KEYB              0x04U
#define MP300_MFC_KEYA              0x00U
#define MP300_MFC_ERROR_OK             1U
#define MP300_MFC_ERROR_FAIL           0U

#define MP_COUPLER_0                0x00
#define MP_COUPLER_1                0x01
#define MP_COUPLER_2                0x02
#define MP_COUPLER_3                0x03

#define MP300_TIMING_GET            0x00
#define MP300_TIMING_GET_TR1        0x03
#define MP300_TIMING_GET_FDT        0x07 /* New in version 1.11: Duration of FDT */
#define MP300_TIMING_ENABLE         0x01
#define MP300_TIMING_ENABLE_M2      0x05
#define MP300_TIMING_DISABLE        0x02
#define MP300_TIMING_CLEAR          0x04
#define MP300_TIMING_LAST_APDU      0x06

#define MP300_TIMING_IN_MICRO       0x03
#define MP300_TIMING_IN_NANO        0x04
#define MP300_TIMING_IN_CARRIER     0x06

#define MP300_TYPE_A                0x01
#define MP300_TYPE_B                0x02
#define MP300_TYPE_MIFARE           0x03
#define MP300_TYPE_ISO15693         0x04
#define MP300_TYPE_FELICA           0x05

#define MP300_UNIT_PERCENT          0x01
#define MP300_UNIT_THOUSAND         0x04
#define MP300_POWER_ON              0x64
#define MP300_POWER_OFF             0x00

#define MP300_FRAME_TYPE_B_CLK      0x4
#define MP300_FRAME_TYPE_B_CLK_PARAM_SIZE 14

#define MP300_PARITY_TX             0xA
#define MP300_PARITY_RX             0xB
#define MP300_PARITY_PARAM_SIZE     0x4
#define MP300_PARITY_ON             0x1
#define MP300_PARITY_OFF            0x0

#define MP300_ANTI_EMD              20
#define MP300_ANTI_EMD_PARAM_SIZE   0x4
#define MP300_ANTI_EMD_ON           0x1
#define MP300_ANTI_EMD_OFF          0x0

#define MP300_RESSOURCE_BLOCKING     0x1
#define MP300_RESSOURCE_NOT_BLOCKING 0x0
#define MP300_RESSOURCE_CONCURRENT   0x2
#define MP300_RESSOURCE_OVERRIDE     0x3

#define MP300_DATARATE_106          106
#define MP300_DATARATE_212          212
#define MP300_DATARATE_424          424
#define MP300_DATARATE_848          848
#define MP300_DATARATE_1695         1695
#define MP300_DATARATE_3390         3390
#define MP300_DATARATE_6780         6780

#define MP300_DATARATE_ISO15693_1_OUT_OF_4      1
#define MP300_DATARATE_ISO15693_1_OUT_OF_256    2
#define MP300_DATARATE_ISO15693_LOW             0
#define MP300_DATARATE_ISO15693_HIGH            1
#define MP300_DATARATE_ISO15693_HIGH_X2         2
#define MP300_DATARATE_ISO15693_HIGH_X4         3
#define MP300_DATARATE_ISO15693_HIGH_X8         4
#define MP300_DATARATE_ISO15693_ONE_SUBCARRIER  1
#define MP300_DATARATE_ISO15693_TWO_SUBCARRIER  2

#define MP300_RXCHANNEL_NORMAL      0x01
#define MP300_RXCHANNEL_EXTERNAL    0x02

#define MP300_TRIGGER_PORT_1        0x01
#define MP300_TRIGGER_PORT_2        0x02
#define MP300_TRIGGER_PORT_3        0x03
#define MP300_TRIGGER_PORT_4        0x04
#define MP300_TRIGGER_FORCE         0x01

#define MP300_RELAY_1               0x01

#define MP300_BATCH_EXECUTE_STOP_ON_ERROR 0
#define MP300_BATCH_EXECUTE_IGNORE_ERRORS 1

#define MP300_TYPE_B_SOF_LOW_CYCLES                     0        /**< Storage position for the number of Cycles used for Type B SOF Low. */
#define MP300_TYPE_B_SOF_HIGH_CYCLES                    1        /**< Storage position for the number of Cycles used for Type B SOF High. */
#define MP300_TYPE_B_EGT_CYCLES                         2        /**< Storage position for the number of Cycles used for Type B EGT. */
#define MP300_TYPE_B_EOF_CYCLES                         3        /**< Storage position for the number of Cycles used for Type B EOF. */
#define MP300_TYPE_B_START_BIT_CYCLES                   4        /**< Storage position for the number of Cycles used for Type B Start Bit. */
#define MP300_TYPE_B_BIT_0_CYCLES                       5        /**< Storage position for the number of Cycles used for Type B Bit 0. */
#define MP300_TYPE_B_BIT_1_CYCLES                       6        /**< Storage position for the number of Cycles used for Type B Bit 1. */
#define MP300_TYPE_B_BIT_2_CYCLES                       7        /**< Storage position for the number of Cycles used for Type B Bit 2. */
#define MP300_TYPE_B_BIT_3_CYCLES                       8        /**< Storage position for the number of Cycles used for Type B Bit 3. */
#define MP300_TYPE_B_BIT_4_CYCLES                       9        /**< Storage position for the number of Cycles used for Type B Bit 4. */
#define MP300_TYPE_B_BIT_5_CYCLES                       10       /**< Storage position for the number of Cycles used for Type B Bit 5. */
#define MP300_TYPE_B_BIT_6_CYCLES                       11       /**< Storage position for the number of Cycles used for Type B Bit 6. */
#define MP300_TYPE_B_BIT_7_CYCLES                       12       /**< Storage position for the number of Cycles used for Type B Bit 7. */
#define MP300_TYPE_B_STOP_BIT_CYCLES                    13       /**< Storage position for the number of Cycles used for Type B Stop Bit. */

/* Internal Parameter IDs */
#define MP300_PARAMETER_INTERNAL_FIELD_STRENGTH         100      /**< Parameter ID to get the current FS. */
#define MP300_PARAMETER_INTERNAL_MODULATION_ASK         101      /**< Parameter ID to get the current Modulation Index. */

/* GET_EEPROM_SYS Parameter IDs */
#define MP300_PARAMETER_GET_EEPROM_SYS_EEP_SERIAL_NUMBER  1      /**< Parameter ID to get the serial number. */
#define MP300_PARAMETER_GET_EEPROM_SYS_EEP_ETHERNET_ADDR  2      /**< Parameter ID to get the ethernet addess. */
#define MP300_PARAMETER_GET_EEPROM_SYS_EEP_UC_IDENT       3      /**< Parameter ID to get the uc ident. */

/* CouplerEepromSys Parameter IDs */
#define MP300_PARAMETER_COUPLER_EEPROM_SYS_EEP_COUPLER_IDENT    1      /**< Parameter ID to get the coupler serial number. */
#define MP300_PARAMETER_COUPLER_EEPROM_SYS_EEP_COUPLER_REVISION 2      /**< Parameter ID to get the coupler revision. */

/* Maximum gains */
#define MP300_MAX_FIELDSTRENGTH_THOUSAND    1500                /**< Mp300 TCl1 and TCL2 has 150% max output */
#define MP500_MAX_FIELDSTRENGTH_THOUSAND    2000                /**< Mp500 TCl3 has 200% max output */

phStatus_t phhalHw_Mp300_SendCommand(
                                     phhalHw_Mp300_DataParams_t * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
                                     );

phStatus_t phhalHw_Mp300_SetCommand(
                              phhalHw_Mp300_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                              uint8_t * pCmd                                /**< [In] Command code. */
                              );

phStatus_t phhalHw_Mp300_AddParam_Uint8(
                                  phhalHw_Mp300_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                                  uint8_t bData                                 /**< [In] Data. */
                                  );

phStatus_t phhalHw_Mp300_AddParam_Uint16(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wData                               /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_Uint32(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint32_t dwData                              /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_Int16(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   int16_t wData                                /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_Int32(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   int32_t dwData                               /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_Uint8Hex(
                                  phhalHw_Mp300_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                                  uint8_t bData                                 /**< [In] Data. */
                                  );

phStatus_t phhalHw_Mp300_AddParam_Uint16Hex(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wData                               /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_Uint32Hex(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint32_t dwData                              /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_String(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t * pData                              /**< [In] Data. */
                                   );

phStatus_t phhalHw_Mp300_AddParam_HexString(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t * pData,                             /**< [In] Data. */
                                   uint16_t wLen                                /**< [In] Len of Data. */
                                   );

phStatus_t phhalHw_Mp300_CatParam_HexString(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t * pData,                             /**< [In] Data. */
                                   uint16_t wLen                                /**< [In] Len of Data. */
                                   );

phStatus_t phhalHw_Mp300_GetParam_Uint8(
                                  phhalHw_Mp300_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                                  uint8_t * pData                               /**< [Out] Data. */
                                  );

phStatus_t phhalHw_Mp300_GetParam_Uint16(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t * pData                             /**< [Out] Data. */
                                   );

phStatus_t phhalHw_Mp300_GetParam_Uint32(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint32_t * pData                             /**< [Out] Data. */
                                   );

phStatus_t phhalHw_Mp300_GetParam_Int32(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   int32_t * pData                              /**< [Out] Data. */
                                   );

phStatus_t phhalHw_Mp300_GetParam_Uint8Hex(
                                  phhalHw_Mp300_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                                  uint8_t * pData                               /**< [Out] Data. */
                                  );

phStatus_t phhalHw_Mp300_GetParam_Uint16Hex(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t * pData                             /**< [Out] Data. */
                                   );

phStatus_t phhalHw_Mp300_GetParam_Uint32Hex(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint32_t * pData                             /**< [Out] Data. */
                                   );

phStatus_t phhalHw_Mp300_GetParam_HexString(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t * pData,                             /**< [Out] Data. */
                                   uint16_t wDataSize,                          /**< [In] Size of Data Buffer. */
                                   uint16_t * pwDataLen                         /**< [Out] Len of Data. */
                                   );

phStatus_t phhalHw_Mp300_GetStatus(
                                   phhalHw_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                   uint8_t * pCmd                               /**< [In] Command code do compare. */
                                   );

phStatus_t phhalHw_Mp300_EvaluateStatus(
                                        phhalHw_Mp300_DataParams_t * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                        uint16_t wStatus                        /**< [In] MP300 status. */
                                        );

phStatus_t phhalHw_Mp300_EvaluateMifareStatus(
                                        uint32_t dwStatus                       /**< [In] MP300 mifare status. */
                                        );

phStatus_t phhalHw_Mp300_GetTxBuffer(
                                     phhalHw_Mp300_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t ** pTxBuffer,                      /**< [Out] Pointer to the TxBuffer. */
                                     uint16_t * pTxBufferLen,                   /**< [Out] Number of bytes already in the TxBuffer. */
                                     uint16_t * pTxBufferSize                   /**< [Out] Size of the TxBuffer. */
                                     );

phStatus_t phhalHw_Mp300_GetRxBuffer(
                                     phhalHw_Mp300_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t ** pRxBuffer,                      /**< [Out] Pointer to the RxBuffer. */
                                     uint16_t * pRxBufferLen,                   /**< [Out] Number of bytes already in the RxBuffer. */
                                     uint16_t * pRxBufferSize                   /**< [Out] Size of the RxBuffer. */
                                     );

phStatus_t phhalHw_Mp300_CalcCrc(
                                 phhalHw_Mp300_DataParams_t * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                 uint8_t * pTxBuffer,                           /**< [In] Pointer to the transmit buffer. */
                                 uint16_t wTxLength,                            /**< [In] Len of the transmit buffer. */
                                 uint8_t * pCrc                                 /**< [Out] Calculated Crc. */
                                 );

phStatus_t phhalHw_Mp300_CheckCrc(
                                 phhalHw_Mp300_DataParams_t * pDataParams       /**< [In] Pointer to this layer's parameter structure. */
                                 );

phStatus_t phhalHw_Mp300_MfcLoadKey(
                                    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
                                    uint8_t bMode,                              /**< [In] Selects key mode #MP300_MFC_KEYA or #MP300_MFC_KEYB. */
                                    uint8_t bSecNr,                             /**< [In] Address where to store the key in the PCD, range [0,15] */
                                    uint8_t * pKey);                            /**< [In] Pointer to a 6-byte buffer holding the key. */

phStatus_t phhalHw_Mp300_MfcAuthentication(
                                    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
                                    uint8_t bMode,                              /**< [In] Selects key mode #MP300_MFC_KEYA or #MP300_MFC_KEYB. */
                                    uint8_t bKeyNr,                             /**< [In] Address where the key to use is stored in the PCD, range [0,15] */
                                    uint32_t wAddress);                         /**< [In] Address of PICC block defining the sector that should be authenticated. */

phStatus_t phhalHw_Mp300_MfcGetLastErrorNumber(
                                    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
                                    uint32_t * pErrorNr);                       /**< [Out] Error code of the last Mifare command. */

phStatus_t phhalHw_Mp300_SetTimingParameters(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bCommand                                /**< [In] The Task to do for the timemeasurement(Enable, Disable, Clear, Get). */
    );

phStatus_t phhalHw_Mp300_GetPosColl(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t * pdwPosColl                           /**< [Out] Position of the Last Collision */
    );

phStatus_t phhalHw_Mp300_SetCardType(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bCardtype                               /**< [In] Sets the cardtype for following communication at the reader. */
    );

phStatus_t phhalHw_Mp300_SetModulationASK(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bPercentage                             /**< [In] The Modulation index in percent. */
    );

phStatus_t phhalHw_Mp300_SetTimeout(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwTimeout                              /**< [In] Frame waiting time in microseconds. */
    );

phStatus_t phhalHw_Mp300_SetAntiTearing(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwTimeout                              /**< [In] Time to wait after first modulation in clock cycles. */
    );

phStatus_t phhalHw_Mp300_SetDatarate(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t bTxBitrate,                            /**< [In] PCD to Picc bitrate. */
    uint16_t bRxBitrate                             /**< [In] Picc to PCD bitrate. */
    );

phStatus_t phhalHw_Mp300_SetDatarate_ISO15693(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t bTxBitrate,                            /**< [In] PCD to Picc bitrate. */
    uint16_t bRxBitrate);                           /**< [In] Picc to PCD bitrate. */

phStatus_t phhalHw_Mp300_ConfigRxDeafbits(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t bEtu                                   /**< [In] deafbits in etu */
    );

phStatus_t phhalHw_Mp300_SetFieldStrength(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wPermil                                /**< [In] The fieldstrength defined in permil of the maximum available at the Mp300 */
    );

phStatus_t phhalHw_Mp300_OpenResource(
    phhalHw_Mp300_DataParams_t * pDataParams        /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_Int_FieldReset(
    phhalHw_Mp300_DataParams_t * pDataParams        /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_Sleep(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwTime                                 /**< [In] time to wait in usec. */
    );

phStatus_t phhalHw_Mp300_ActivateParity(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bEnable                                 /**< [In] PH_OFF disables and PH_ON enables the parity bits. */
    );

phStatus_t phhalHw_Mp300_ActivateAntiEMD(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bEnable                                 /**< [In] PH_OFF disables and PH_ON enables the EMD protection. */
    );

phStatus_t phhalHw_Mp300_SetCarrierFrequency(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwCarrierFrequency                     /**< [In] The new Carrier frequency which shall be used. */
    );

phStatus_t phhalHw_Mp300_SetWaveformGenerationMode(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bMode                                   /**< [In] The new waveform generation mode. */
    );

phStatus_t phhalHw_Mp300_SetRxGain(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wRxGain                                /**< [In] The new Rx Gain Value which shall be used. */
    );

phStatus_t phhalHw_Mp300_SelectRXChannel(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wRXChannel                             /**< [In] MP300_RXCHANNEL_NORMAL or MP300_RXCHANNEL_EXTERNAL for external RX-antenna. */
    );

phStatus_t phhalHw_Mp300_ConfigureTriggerPort(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bPortnumber,                            /**< [In] Which Triggerport to configure MP300_TRIGGER_OUT1 or MP300_TRIGGER_OUT2. */
    uint32_t dwConfig,                              /**< [In] Which mode to set for the chosen port. */
    uint32_t dwValue                                /**< [In] The Value 0x00/0x01 PH_ON/PH_OFF needed for Trigger Force. PH_ON(0x00) otherwise). */
    );

phStatus_t phhalHw_Mp300_ConfigureRelayPort(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bRelay                                  /**< [In] Relayport PH_OFF or PH_ON. */
    );

phStatus_t phhalHw_Mp300_SelectPauseWidth(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wPauseWidth                            /**< [In] Pause width in the range 100 to 4500 in ns. */
    );

phStatus_t phhalHw_Mp300_SelectFieldRiseTime(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwFieldRiseTime                        /**< [In] Time in us (range 0 to 5000) the field rises after following fieldresets. */
    );

phStatus_t phhalHw_Mp300_SelectFallAndRiseTime(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wFallTime,                             /**< [In] Fall time in ns (range 0 to 10000) of the modulation. */
    uint16_t wRiseTime                              /**< [In] Rise time in ns (range 0 to 10000) of the modulation. */
    );

phStatus_t phhalHw_Mp300_SetTypeBModulationTimingParameter(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wValue,                                /**< [In] new value for the specified index. */
    uint8_t bIndex                                  /**< [In] index for the new value. */
    );

phStatus_t phhalHw_Mp300_SelectSourceOfCarrier(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bSourceOfCarrier                        /**< [In] one element from SourceOfCarrier enum to select the correct source. */
    );

phStatus_t phhalHw_Mp300_SetRxGainTarget(
    phhalHw_Mp300_DataParams_t * pDataParams,       /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wTargetRxGainLevel                     /**< [In] target value to set. */
    );

phStatus_t phhalHw_Mp300_AdjustRXChannel2(
    phhalHw_Mp300_DataParams_t * pDataParams        /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_SetModulationShape(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwNbPointsFallingEdge,             /**< [In] Number of points of falling edge. */
    uint16_t * pPointsFallingEdge,              /**< [In] points of falling edge. In a range of 0..1000 */
    uint32_t dwNbPointsRisingEdge,              /**< [In] Number of points of rising edge. */
    uint16_t * pPointsRisingEdge                /**< [In] points of rising edge. In a range of 0..1000 */
    );

phStatus_t phhalHw_Mp300_CalcCommTimePCDToPICC(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wNumTransmitBits,                  /**< [In] Number of bits transmitted from the PCD. */
    uint16_t wTransmitLen                       /**< [In] Number of full bytes transmitted from the PCD. */
    );

phStatus_t phhalHw_Mp300_CalcCommTimePICCToPCD(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint16_t wNumReceivedBits                   /**< [In] Number of bits received from the PCD. */
    );

phStatus_t phhalHw_Mp300_GetInternalParameter(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bParamId,                           /**< [In] ParameterID. */
    uint32_t * pdwParamValue                    /**< [out] Received value of the paramter. */
    );

phStatus_t phhalHw_Mp300_GetEepromSys(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bParamId,                           /**< [In] ParameterID. */
    uint8_t * pbParamValue,                     /**< [out] Received value of the paramter. */
    uint16_t wParamSize                         /**< [In] Size of the param buffer. */
    );

phStatus_t phhalHw_Mp300_GetResourceID(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bCouplerNumber,                     /**< [In] Coupler Number to check. */
    uint32_t * pdwResourceID                    /**< [out] Received Resource ID of the coupler. */
    );

phStatus_t phhalHw_Mp300_GetReaderVersions(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_SpyClose(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_Int_EnableSpy(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwEventMask,                       /**< [In] Event Mask used for Tracefile. */
    uint16_t wDateFormat                        /**< [In] Date Format used for Tracefile (nanoseconds or clk cycles). */
    );

phStatus_t phhalHw_Mp300_Int_DisableSpy(
    phhalHw_Mp300_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_Int_SendAndCheckCmd(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    const char *format, ...                     /**< [In] Format of the command to be sent. */
    );

phStatus_t phhalHw_Mp300_Int_CheckSuccessCloseSpy(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    phStatus_t status                           /**< [In] Status to check. */
    );

unsigned long __stdcall phhalHw_Mp300_Int_ReadSpyDataThread(
    void *voidParams
    );

phStatus_t phhalHw_Mp300_Int_ProcessData(
    phhalHw_Mp300_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t DataType,
    uint32_t Size,
    uint8_t *pData,
    uint32_t *gHeader,
    uint32_t *gHeaderSize,
    uint8_t *pbFinished
    );

phStatus_t phhalHw_Mp300_Int_EvaluateData(
   phhalHw_Mp300_DataParams_t *pDataParams,     /**< [In] Pointer to an initialized HAL parameter structure. */
   int32_t dwReceivedBytes,
   uint32_t *gHeader,
   uint32_t *gHeaderSize,
   uint8_t *pRxBuffer,
   uint8_t *pData,
   uint32_t *dwDataOffset,
   uint32_t *remainingSize,
   uint32_t *DataType,
   uint32_t *Size,
   uint8_t *bDataTypeReceived,
   uint8_t *bSizeReceived,
   uint8_t *bFinished
   );

phStatus_t phhalHw_Mp300_Int_FinishSpying(
   phhalHw_Mp300_DataParams_t *pDataParams,     /**< [In] Pointer to an initialized HAL parameter structure. */
   uint32_t *gHeader
   );

phStatus_t phhalHw_Mp300_Int_ConvertRawToMplog(
    phhalHw_Mp300_DataParams_t *pDataParams     /**< [In] Pointer to an initialized HAL parameter structure. */
    );

phStatus_t phhalHw_Mp300_Int_GetMpNetworkAddress(
                                                 phhalHw_Mp300_DataParams_t * pDataParams,
                                                 uint32_t * NetworkAddress,
                                                 uint32_t * SubnetMask
                                                 );

phStatus_t phhalHw_Mp300_Int_CheckTriggerParameter(
                                                   uint8_t bTriggerPort,
                                                   uint32_t dwConfig,
                                                   uint32_t dwValue
                                                   );

phStatus_t phhalHw_Mp300_Int_EnableTrigger(
                                           phhalHw_Mp300_DataParams_t * pDataParams
                                           );

phStatus_t phhalHw_Mp300_Int_DisableTrigger(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            );

phStatus_t phhalHw_Mp300_Int_UpdateTriggerCounter(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            );

phStatus_t phhalHw_Mp300_Int_EnableTearing(
                                           phhalHw_Mp300_DataParams_t * pDataParams
                                           );

phStatus_t phhalHw_Mp300_Int_DisableTearing(
                                            phhalHw_Mp300_DataParams_t * pDataParams,
                                            phStatus_t * statusExchange
                                            );

phStatus_t phhalHw_Mp300_Int_UpdateTearingCounter(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            );

phStatus_t phhalHw_Mp300_Int_CleanupTearingAndTrigger(
                                            phhalHw_Mp300_DataParams_t * pDataParams
                                            );

#endif /* PHHALHW_MP300_INT_H */
