/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Host Communication command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_HC.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_LockUnlock(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bLockType, uint16_t wRdKeyNo,
    uint16_t wRdKeyVer, uint8_t bSamKeyNo, uint8_t bSamKeyVer, uint8_t bUnlockKeyNo, uint8_t bUnlockKeyVer,
    uint32_t dwMaxChainBlocks)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint16_t   PH_MEMLOC_REM wKeyType = 0;
    uint8_t    PH_MEMLOC_REM aRnd1[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND];
    uint8_t    PH_MEMLOC_REM aRnd2[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND];
    uint8_t    PH_MEMLOC_REM aRndAB[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U];
    uint8_t    PH_MEMLOC_REM aKxeKey[PH_KEYSTORE_KEY_TYPE_AES256_SIZE];
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t    PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES256_SIZE];
    uint8_t    PH_MEMLOC_REM bKeyLen = 0;

    uint8_t    PH_MEMLOC_REM aMac[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_MAC];
    uint8_t    PH_MEMLOC_REM bMacLen = 0;

    uint8_t    PH_MEMLOC_REM aOptionalData[5U];
    uint8_t    PH_MEMLOC_REM bOptDataLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Clear buffers */
    PHHAL_HW_SAM_CLEAR_BUFFER(aRnd1);
    PHHAL_HW_SAM_CLEAR_BUFFER(aRnd2);
    PHHAL_HW_SAM_CLEAR_BUFFER(aRndAB);
    PHHAL_HW_SAM_CLEAR_BUFFER(aKey);
    PHHAL_HW_SAM_CLEAR_BUFFER(aKxeKey);
    PHHAL_HW_SAM_CLEAR_BUFFER(aMac);
    PHHAL_HW_SAM_CLEAR_BUFFER(aOptionalData);
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Get Key from Software KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wRdKeyNo,
        wRdKeyVer,
        sizeof(aKey),
        aKey,
        &wKeyType));

    /* Check for valid key type. */
    if((wKeyType != PH_KEYSTORE_KEY_TYPE_AES128) && (wKeyType != PH_KEYSTORE_KEY_TYPE_AES192) &&
        (wKeyType != PH_KEYSTORE_KEY_TYPE_AES256))
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_HAL);
    }

    /* Frame first part of SAM_LockUnlock information. --------------------------------------------------------------------------
     * CMD => CLA || INS || P1 || P2 || LC || KeyNo || KeyVer || [MaxChainBlocks] || [Unlock KeyNo] || [Unlock KeyVer] || LE
     * RSP <= Rnd2 || 0x90AF
     */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_LOCK_UNLOCK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bLockType;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSamKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSamKeyVer;

    /* Save the key number and version information according to bLockType value. */
    switch(bLockType)
    {
        case PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK:
        case PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK_PL:
        case PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_NO_KEY:
            /* bLc is already OK*/
            break;

        case PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY:
            aOptionalData[bOptDataLen++] = bUnlockKeyNo;
            aOptionalData[bOptDataLen++] = bUnlockKeyVer;
            break;

        case PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_ACTIVATE_SAM:
            aOptionalData[bOptDataLen++] = (uint8_t) (dwMaxChainBlocks);
            aOptionalData[bOptDataLen++] = (uint8_t) (dwMaxChainBlocks >> 8U);
            aOptionalData[bOptDataLen++] = (uint8_t) (dwMaxChainBlocks >> 16U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Add option data to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aOptionalData, bOptDataLen);
    bCmdLen += bOptDataLen;

    /* Add LE to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange first part of SAM_LockUnlock information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* We expect chaining as the status from Sam hardware. */
    if(wStatus != (PH_COMP_HAL | PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE))
        return wStatus;

    /* Check if response if of not the expected size. */
    if(wRespLen != PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1)
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);

    /* Calculate MACHost = MAC(Kx, Rnd2 || P1 || (MaxChainBlocks or Unlock Key number and version or Zero padded)  */

    /* Prepare the payload for second part of SAM_LockUnlock command. */
    memcpy(aRnd2, pResponse, PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1);
    aRnd2[12U] = bLockType;
    memcpy(&aRnd2[13U], aOptionalData, bOptDataLen);

    /* Load the key to CryptoSym. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pMACCryptoDataParams,
        aKey,
        wKeyType));

    /* Compute the Mac. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pMACCryptoDataParams,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aRnd2,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aMac,
        &bMacLen));

    /* Truncate the MAC and add to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(aMac, &bMacLen));

    /* Generate the Random number to be sent to Sam hardware and add it to Command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams,
        PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1, aRnd1));

    /* Frame second part of SAM_LockUnlock information. -------------------------------------------------------------------------
     * CMD => CLA || INS || P1 || P2 || LC || MACHost || Rnd1 || LE
     * RSP <= MACSAM || EncSAM1 || 0x90AF
     */
    /* Reset P1 and LC information byte to default value in command buffer. */
    bCmdLen = PHHAL_HW_SAM_ISO7816_HEADER_LENGTH;
    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAMAV3_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add MAC to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aMac, bMacLen);
    bCmdLen += bMacLen;

    /* Add Rnd1 to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aRnd1, PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1);
    bCmdLen += PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1;

    /* Add LE to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange second part of SAM_LockUnlock information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* We expect chaining as the status from Sam hardware. */
    if(wStatus != (PH_COMP_HAL | PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE))
        return wStatus;

    /* Check if response if of not the expected size. */
    if(wRespLen != PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART2)
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);

    /* Prepare the payload for MAC calculation. */
    aRnd1[12U] = bLockType;
    memcpy(&aRnd1[13U], aOptionalData, bOptDataLen);

    /* Calculate the Mac to verify it with the received data from Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pMACCryptoDataParams,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aRnd1,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aMac,
        &bMacLen));

    /* Truncate the MAC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(aMac, &bMacLen));

    /* Check if the received MAC and calculated MAC are same. */
    if(memcmp(aMac, pResponse, bMacLen))
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Derive the Kxe key from kx using Rnd1 and Rnd2 - note: Secret key needs to be loaded into MAC data params*/
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GenerateHostAuthSessionKey(
        pDataParams,
        (uint8_t) wKeyType,
        aRnd1,
        aRnd2,
        aKxeKey,
        &bKeyLen));

    /* Load the generated Session key to crypto params. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pENCCryptoDataParams,
        aKxeKey,
        wKeyType));

    /* Load default initialization vector. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pENCCryptoDataParams,
        (PH_CRYPTOSYM_CIPHER_MODE_CBC),
        &pResponse[8],
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aRnd2));

    /* Now we start with part 3 exchange */
    /* Initialize the RndA array*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, aRnd1));
    memcpy(aRndAB, aRnd1, PH_CRYPTOSYM_AES_BLOCK_SIZE);
    /* calculate RndB'*/
    memcpy(&aRndAB[16U], &aRnd2[2U], 14U);
    aRndAB[30U] = aRnd2[0];
    aRndAB[31U] = aRnd2[1U];

    /* Load default initialization vector. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Key is already loaded */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pENCCryptoDataParams,
        PH_CRYPTOSYM_CIPHER_MODE_CBC,
        aRndAB,
        (uint16_t) (PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U),
        aRndAB));

    /* Frame third part of SAM_LockUnlock information. --------------------------------------------------------------------------
     * CMD => CLA || INS || P1 || P2 || LC || EncHost || LE
     * RSP <= EncSAM2 || 0x90AF
     */
    /* Reset P1 and LC information byte to default value in command buffer. */
    bCmdLen = PHHAL_HW_SAM_ISO7816_HEADER_LENGTH;
    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAMAV3_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add EncHost information to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aRndAB, (uint16_t) (PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U));
    bCmdLen += (uint16_t) (PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U);

    /* Add LE to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange third part of SAM_LockUnlock information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    if(wRespLen != PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART3)
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);

    /* Load default init vector */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Key is already loaded */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pENCCryptoDataParams,
        PH_CRYPTOSYM_CIPHER_MODE_CBC,
        pResponse,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aRndAB));

    /* The response for RndA is not equal to sent RndA, Authentication failed PH_ERR_AUTH_ERROR. */
    if(memcmp(aRndAB, &aRnd1[2U], 14U) != 0)
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);

    /* The response for RndA is not equal to sent RndA, Authentication failed PH_ERR_AUTH_ERROR. */
    if((aRnd1[0] != aRndAB[14U]) || (aRnd1[1U] != aRndAB[15U]))
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);

    /* SAM resets all the Authentication state when LockUnlock command is called for Lock variant.
     * Refer the document section, 6.2.2.3 SAM Unlocked State, Tab. 6.7, description of ActLock.
     */
    if((bLockType == PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_NO_KEY) ||
        (bLockType == PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY))
    {
        pDataParams->bAuthType = PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN;
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateHost(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bHostMode, uint16_t wRdKeyNo,
    uint16_t wRdKeyV, uint8_t bSamKeyNo, uint8_t bSamKeyVer)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint16_t   PH_MEMLOC_REM wKeyType = 0;
    uint8_t    PH_MEMLOC_REM aRnd1[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND];
    uint8_t    PH_MEMLOC_REM aRnd2[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND];
    uint8_t    PH_MEMLOC_REM aRndAB[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U];
    uint8_t    PH_MEMLOC_REM aKxeKey[PH_KEYSTORE_KEY_TYPE_AES256_SIZE];
    uint8_t    PH_MEMLOC_REM aSessionEncKey[PH_KEYSTORE_KEY_TYPE_AES256_SIZE];
    uint8_t    PH_MEMLOC_REM aSessionMacKey[PH_KEYSTORE_KEY_TYPE_AES256_SIZE];
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t    PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES256_SIZE];
    uint8_t    PH_MEMLOC_REM bKeyLen = 0;

    uint8_t    PH_MEMLOC_REM aMac[PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_MAC];
    uint8_t    PH_MEMLOC_REM bMacLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    PH_LOG_HELPER_ALLOCATE_PARAMNAME(SessKey_Enc_SAM_HOST);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(SessKey_Mac_SAM_HOST);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Clear buffers */
    PHHAL_HW_SAM_CLEAR_BUFFER(aRnd1);
    PHHAL_HW_SAM_CLEAR_BUFFER(aRnd2);
    PHHAL_HW_SAM_CLEAR_BUFFER(aRndAB);
    PHHAL_HW_SAM_CLEAR_BUFFER(aKey);
    PHHAL_HW_SAM_CLEAR_BUFFER(aKxeKey);
    PHHAL_HW_SAM_CLEAR_BUFFER(aMac);
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Get Key from software KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wRdKeyNo,
        wRdKeyV,
        sizeof(aKey),
        aKey,
        &wKeyType));

    /* Check for valid key type. */
    if((wKeyType != PH_KEYSTORE_KEY_TYPE_AES128) && (wKeyType != PH_KEYSTORE_KEY_TYPE_AES192) &&
        (wKeyType != PH_KEYSTORE_KEY_TYPE_AES256))
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_HAL);

    /* Update the AuthType member. */
    pDataParams->bAuthType = 0;

    /* Frame first part of SAM_AuthenticateHost information. --------------------------------------------------------------------
     * CMD => CLA || INS || P1 || P2 || LC || KeyNo || KeyVer || HostMode || LE
     * RSP <= Rnd2 || 0x90AF
     */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_HOST;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSamKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSamKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHostMode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange first part of SAM_LockUnlock information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* We expect chaining as the status from Sam hardware. */
    if(wStatus != (PH_COMP_HAL | PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE))
        return wStatus;

    /* Check if response if of not the expected size. */
    if(wRespLen != PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1)
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);

    /* Calculate MACHost = MAC(Kx, Rnd2 || HostMode || 0x00 || 0x00 || 0x00)  */

    /* Prepare the payload for second part of SAM_AuthenticateHost command. */
    memcpy(aRnd2, pResponse, PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1);
    aRnd2[12] = bHostMode;

    /* Load the key to CryptoSym. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pMACCryptoDataParams,
        aKey,
        wKeyType));

    /* Compute the Mac. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pMACCryptoDataParams,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aRnd2,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aMac,
        &bMacLen));

    /* Truncate the MAC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(aMac, &bMacLen));

    /* Generate the Random number to be sent to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams,
        PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1, aRnd1));

    /* Frame second part of SAM_AuthenticateHost information. -------------------------------------------------------------------
     * CMD => CLA || INS || P1 || P2 || LC || MACHost || Rnd1 || LE
     * RSP <= MACSAM || EncSAM1 || 0x90AF
     */
    bCmdLen = 0;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_HOST;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Mac to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aMac, bMacLen);
    bCmdLen += bMacLen;

    /* Add Rnd1 to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aRnd1, PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1);
    bCmdLen += PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange second part of SAM_AuthenticateHost information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* We expect chaining as the status frm Sam hardware. */
    if(wStatus != (PH_COMP_HAL | PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE))
        return wStatus;

    /* Check if response if of not the expected size. */
    if(wRespLen != PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART2)
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);

    /* Prepare the payload for MAC calculation. */
    aRnd1[12U] = bHostMode;

    /* Calculate the Mac to verify it with the received data from Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pMACCryptoDataParams,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aRnd1,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aMac,
        &bMacLen));

    /* Truncate the MAC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(aMac, &bMacLen));

    /* Check if the received MAC and calculated MAC are same. */
    if(memcmp(aMac, pResponse, bMacLen))
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Derive the Kxe key from kx using Rnd1 and Rnd2 - note: Secret key needs to be loaded into MAC data params*/
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GenerateHostAuthSessionKey(
        pDataParams,
        (uint8_t) wKeyType,
        aRnd1,
        aRnd2,
        aKxeKey,
        &bKeyLen));

    /* Load the generated Session key to crypto params. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pENCCryptoDataParams,
        aKxeKey,
        wKeyType));

    /* Load default initialization vector. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Get the Encrypted(Kxe, RndB) in Rnd2*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pENCCryptoDataParams,
        (PH_CRYPTOSYM_CIPHER_MODE_CBC),
        &pResponse[8],
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aRnd2));

    /* Now we start with part 3 exchange */
    /* Initialize the RndA array*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, aRnd1));
    memcpy(aRndAB, aRnd1, PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* calculate RndB'*/
    memcpy(&aRndAB[16U], &aRnd2[2U], 14U);
    aRndAB[30U] = aRnd2[0];
    aRndAB[31U] = aRnd2[1U];

    /* Encrypt aRndAB*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pENCCryptoDataParams,
        PH_CRYPTOSYM_CIPHER_MODE_CBC,
        aRndAB,
        (uint16_t) (PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U),
        aRndAB));



    /* Frame third part of Cmd.SAM_AuthenticateHost information. ----------------------------------------------------------------
     * CMD => CLA || INS || P1 || P2 || LC || EncHost || LE
     * RSP <= EncSAM2 || 0x90AF
     */
    bCmdLen = 0;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_HOST;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add EncHost information to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aRndAB, (uint16_t) (PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U));
    bCmdLen += (uint16_t) (PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND * 2U);

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange third part of SAM_AuthenticateHost information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    if(wRespLen != PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART3)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Load initial IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* decrypt the Response3 and check RndA'*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pENCCryptoDataParams,
        (PH_CRYPTOSYM_CIPHER_MODE_CBC),
        pResponse,
        16,
        aRndAB));

    /* The response for RndA is not equal to sent RndA, Authentication failed PH_ERR_AUTH_ERROR */
    if(memcmp(aRndAB, &aRnd1[2U], 14U) != 0)
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);

    /* The response for RndA is not equal to sent RndA, Authentication failed PH_ERR_AUTH_ERROR */
    if((aRnd1[0] != aRndAB[14U]) || (aRnd1[1U] != aRndAB[15U]))
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);

    /* Host Authentication is successfully completed */
    /* Generate the current SessionKey for this HostAuthentication*/
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GenerateSessionKey(
        pDataParams,
        (uint8_t) wKeyType,
        aRnd1,
        aRnd2,
        aSessionEncKey,
        aSessionMacKey,
        &bKeyLen));

    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(SessKey_Enc_SAM_HOST), aSessionEncKey, bKeyLen);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(SessKey_Mac_SAM_HOST), aSessionMacKey, bKeyLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* reset both pENCCryptoDataParams and pMACCryptoDataParams with Session key*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pENCCryptoDataParams,
        aSessionEncKey,
        wKeyType));

    /* Load initial IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pMACCryptoDataParams,
        aSessionMacKey,
        wKeyType));

    /* Load initial IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pMACCryptoDataParams,
        phhalHw_Sam_Cmd_ZeroIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Reset the Command counter */
    pDataParams->wCmd_Ctr = 0;
    pDataParams->bAuthType = bHostMode;
    pDataParams->bKeyNo = bSamKeyNo;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
