/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Host Communication command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_HC_H
#define PHHALHW_SAM_CMD_HC_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_RND                                         16U /* Buffer size for Random number */
#define PHHAL_HW_SAM_CMD_HC_BUFFER_SIZE_MAC                                         16U /* Buffer size for MAC buffer */

#define PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART1                          12U /* Size of Random number received from SAM for part 1 */
#define PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART2                          24U /* Size of Random number received from SAM for part 2 */
#define PHHAL_HW_SAM_CMD_HC_RESPONSE_BUFFER_SIZE_RND_PART3                          16U /* Size of Random number received from SAM for part 3 */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_HC 01_HostCommunication
 * \brief Implementation of SAM Host Communication commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines Defines
 * \brief Definitions for SAM Host Authentication commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_LOCK_UNLOCK                                    0x10U   /**< Sam Instruction code for SAM_LockUnlock command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_HOST                              0xA4U   /**< Sam Instruction code for SAM_AuthenticateHost command. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_LockType LockType
 * \brief Definitions of Lock types. To be used with \ref phhalHw_Sam_Cmd_SAM_LockUnlock "SAM_LockUnlock" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK                            0x00U   /**< Option mask for Sub-command type as unlock. */
#define PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_NO_KEY                       0x01U   /**< Option mask for Sub-command type as lock without specifying unlock key. */
#define PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY                          0x02U   /**< Option mask for Sub-command type as lock with specifying unlock key. */
#define PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_ACTIVATE_SAM                      0x03U   /**< Option mask for Sub-command type as activate MIFARE SAM. */
#define PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK_PL                         0x04U   /**< Option mask for Sub-command type as unlock PL. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_LockType
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_HostMode HostMode
 * \brief Definitions of Host Protection modes.
 * To be used with \ref phhalHw_Sam_Cmd_SAM_AuthenticateHost "SAM_AuthenticateHost" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN                       0x00U   /**< Option mask for protection mode as plain. */
#define PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_MAC                         0x01U   /**< Option mask for protection mode as MAC protection. */
#define PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_FULL                        0x02U   /**< Option mask for protection mode as Full protection. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_HostMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_HC_Defines
 * @}
 */

/**
 * \brief Activate, Lock or Unlock the SAM.
 *  - The SAM_LockUnlock command is used to modify the SAM Global State for all active LC's, and to activate
 *    the SAM.
 *  - It is implemented by mean of a mutual authentication between the SAM and host system.
 *  - Such authentication proves that both the SAM and the host contain the same secret, namely the AES
 *    authentication key Kx.
 *  - Only authorized Host systems are allowed to change the SAM Global State.
 *  - Starting SAM_LockUnlock command invalidates any other existing authentication (incl. Offline Key Activation)
 *    on any of the logical channels.
 *  - If key entry (SAMKey) is not of AES_128, AES_192 or AES_256 key, no authentication is possible with this
 *    key entry.
 *  - If key entry (SAMKey != MasterKey) is not a LockKey, no authentication is possible with this key entry.
 *  - During the first part of the authentication, the host indicates whether the SAM shall execute the SAM Activation,
 *    the lock or the unlock operation.
 *  - The activation is used to protect the SAM against tampering before delivery to and to provide a way to configure\
 *    some settings of the SAM.
 *
 * \note
 *  - It's important to perform Host Authentication after successful LockUnlock operation to set the new session keys.
 *  - Host Authentication is required because, LockUnlock interface will utilize the same CryptoParams which was utilized
 *    by Host Authentication interface.
 *  - Since the CryptoParams are common, the keys will be updated by LockUnlock interface and the existing Host
 *    Authentication session keys will not be available.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_KEY                  If \b wRdKeyNo is not one of AES128, AES192 or AES256
 * \retval #PH_ERR_LENGTH_ERROR         If response length received is not as expected for part1, part2 and part3.
 * \retval #PH_ERR_AUTH_ERROR           The response for RndA is not equal to sent RndA
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_LockUnlock(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bLockType,                                                              /**< [In] Sub Command type.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK
                                                                                         *            "Unlock"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_NO_KEY
                                                                                         *            "Lock without specifying unlock key"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY
                                                                                         *            "Lock with specifying unlock key"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_ACTIVATE_SAM
                                                                                         *            "Activate MIFARE SAM"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK_PL
                                                                                         *            "Unlock Programmable Logic"
                                                                                         */
        uint16_t wRdKeyNo,                                                              /**< [In] Key reference number to be used in Software KeyStore. */
        uint16_t wRdKeyVer,                                                             /**< [In] Key version to be used in Software KeyStore. */
        uint8_t bSamKeyNo,                                                              /**< [In] Key reference number in SAM.  One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x07
                                                                                         *          - PL Unlock Key: 0xF0
                                                                                         */
        uint8_t bSamKeyVer,                                                             /**< [In] Key version to be used in SAM. */
        uint8_t bUnlockKeyNo,                                                           /**< [In] Unlock Key Number to be used in SAM (only used
                                                                                         *        when \b bLockType = \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY
                                                                                         *        "Lock with specifying unlock key").
                                                                                         */
        uint8_t bUnlockKeyVer,                                                          /**< [In] Unlock Key Version to be used in SAM (only used
                                                                                         *        when \b bLockType = \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY
                                                                                         *        "Lock with specifying unlock key").
                                                                                         */
        uint32_t dwMaxChainBlocks                                                       /**< [In] Maximal message size under command chaining in MAC or Full Protection (only used
                                                                                         *        when \b bLockType = \ref PHHAL_HW_SAM_CMD_SAM_LOCK_UNLOCK_TYPE_ACTIVATE_SAM
                                                                                         *        "Activate MIFARE SAM").
                                                                                         */
    );

/**
 * \brief Mutual 3-pass authentication between Host and SAM.
 *  - The SAM_AuthenticateHost will perform a mutual authentication for a specific LC between the SAM and host.
 *    system. Such an authentication proves that both the SAM and the host contain the same secret, namely the
 *    AES authentication key Kx.
 *  - Starting SAM_AuthenticateHost command invalidates any other existing authentication (incl. Offline Key
 *    Activation) in the LC.
 *  - If the authentication key entry (SAMKey) is not of AES_128, AES_192 or AES_256, no authentication is
 *    possible with this key entry.
 *  - If the authentication key entry (SAMKey != MasterKey) is not a AuthKey, no authentication is possible
 *    with this key entry.
 *  - SAM_AuthenticateHost is executed in three parts.
 *      - During the first part of the authentication, the host indicates the AES authentication key (Kx) and
 *        the selected protection mode (HostMode) to be used by the established SAC: Plain, MAC or Full mode.
 *      - If an invalid key entry and/or key version are referenced, or if the key to be used is not an AuthKey
 *        (or an AuthLockKey in the SAM Locked State), an error is returned.
 *      - If the provided HostMode is not one of the expected values, an error is returned and the protocol is
 *        aborted.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_KEY                  If \b wRdKeyNo is not one of AES128, AES192 or AES256
 * \retval #PH_ERR_LENGTH_ERROR         If response length received is not as expected for part1, part2 and part3.
 * \retval #PH_ERR_AUTH_ERROR           The response for RndA is not equal to sent RndA
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateHost(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bHostMode,                                                              /**< [In] Type of Protection mode to be applied.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN
                                                                                         *            "Protection Mode as PLAIN" (Data in PLAIN)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_MAC
                                                                                         *            "Protection Mode as MAC" (Data in PLAIN and MAC Available)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_FULL
                                                                                         *            "Protection Mode as FULL" (Data Crypted and MAC available)
                                                                                         */
        uint16_t wRdKeyNo,                                                              /**< [In] Key reference number to be used in Software KeyStore. */
        uint16_t wRdKeyV,                                                               /**< [In] Key version to be used in Software KeyStore. */
        uint8_t bSamKeyNo,                                                              /**< [In] Key reference number in SAM. */
        uint8_t bSamKeyV                                                                /**< [In] Key version to be used in SAM. */
    );

/**
 * end of group phhalHw_Sam_Cmd_HC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_HC_H */
