/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Security and Configuration command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_SC.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_GetVersion(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pVersion, uint8_t * pVerLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pVersion, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pVerLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_GetVersion command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GET_VERSION;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange SAM_GetVersion information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Copy the length to the actual parameter. */
    memcpy(pVersion, pResponse, wRespLen);
    *pVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_DisableCrypto(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wProMas)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_DisableCrypto command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_CRYPTO;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add ProMas information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wProMas);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wProMas);

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange SAM_DisableCrypto information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ActivateOfflineKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDivInputLen != 0)
    {
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    }

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ActivateOfflineKey command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_ACTIVATE_OFFLINE_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (bOption | (bDivInputLen != 0));
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the key number and version to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Add Diversification input to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pDivInput, bDivInputLen);
    bCmdLen += bDivInputLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange SAM_DisableCrypto information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_LoadInitVector(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_LoadInitVector command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_LOAD_INIT_VECTOR;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Diversification input to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
    bCmdLen += bDataLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange SAM_LoadInitVector information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_KillAuthentication(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_KillAuthentication command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_KILL_AUTHENTICATION;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchange SAM_KillAuthentication information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Reset Crypto in case of success and P1 == 0x00 */
    if(bOption == PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_FULL)
    {
        /* Reset the session keys data params. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));

        /* Reset the secure messaging members.  */
        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;

        /* Reset the Authentication host mode to plain. */
        pDataParams->bAuthType = 0x00;

        /* Reset the MAC and ENC buffers and length. */
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingCmdIv);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingCmdMac);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingEncCmdData);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingMacCmdData);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingMacRespData);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingRespData);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingRespIv);
        PHHAL_HW_SAM_CLEAR_BUFFER(pDataParams->bPendingRespMac);

        pDataParams->bPendingEncCmdDataLength = 0;
        pDataParams->bPendingMacCmdDataLength = 0;
        pDataParams->bPendingMacRespDataLength = 0;
        pDataParams->bPendingRespDataLength = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_SelectApplication(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pDF_Aid)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pDF_Aid, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_SelectApplication command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_SELECT_APPLICATION;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Diversification input to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pDF_Aid, 3U);
    bCmdLen += 3U;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange SAM_SelectApplication information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_GetRandom(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bExpLen, uint8_t * pRnd)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRnd, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_GetRandom command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GET_RANDOM;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bExpLen;

    /* Exchange SAM_GetRandom information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Check if required number of bytes are returned by Sam hardware. */
    if(wRespLen != bExpLen)
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);

    /* Copy the response to parameter.  */
    memcpy(pRnd, pResponse, bExpLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_Sleep(phhalHw_Sam_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t * PH_MEMLOC_REM pResponse = NULL;
    uint16_t  PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_Sleep command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_SLEEP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchange SAM_Sleep information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_SetConfiguration(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_SetConfiguration command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_SET_CONFIGURATION;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Add LC and Data information to command buffer if Option <= FULL_ATR */
    if(bOption <= PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_FULL_ATR)
    {
        /* Add LC to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add Data to command buffer */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
        bCmdLen += bDataLen;

        /* Update LC */
        phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);
    }

    /* Exchange SAM_SetConfiguration information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
