/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) File Management command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7290 $
 * $Date: 2025-05-06 17:52:50 +0530 (Tue, 06 May 2025) $
 */

#include "phhalHw_Sam_Cmd_FM.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_CreateFile(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t bFileType,
    uint8_t bKeyNoAEK_Read, uint8_t bKeyVAEK_Read, uint8_t bKeyNoAEK_Write, uint8_t bKeyVAEK_Write, uint8_t * pFileSize,
    uint8_t bCRLOptions, uint8_t bCSNSize, uint8_t bCSNSigKey)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pFileSize, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_CreateFile command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CREATE_FILE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFileNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Access Entry Key for Reading to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNoAEK_Read;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVAEK_Read;

    /* Add Access Entry Key for Writing to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNoAEK_Write;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVAEK_Write;

    /* Add File Size to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pFileSize, 3U);
    bCmdLen += 3U;

    /* Add CRLOptions, CSNSize and CSNSignatureKey to command buffer */
    if(bFileType == PHHAL_HW_SAM_FILE_TYPE_CRL)
    {
        /* Add CRLOptions to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCRLOptions;

        /* Add CSNSize to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCSNSize;

        /* Add Signature Key to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCSNSigKey;
    }

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Exchange the buffered information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ReadFile(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pLength, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ReadFile command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_READ_FILE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Update information based on bOption data. */
    if(bOption != PH_EXCHANGE_RXCHAINING)
    {
        /* Add File Number to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = bFileNo;

        /* Add LC to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add Offset to command buffer. */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pOffset, 3U);
        bCmdLen += 3U;

        /* Add Length to command buffer. */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pLength, 3U);
        bCmdLen += 3U;

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }
    else
    {
        /* Do Nothing */
    }

    /* Buffer initial information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL);

    /* Exchange the buffered information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppResponse,
        pRespLen);

    /* Check status. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_WriteFile(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint16_t wCRLVer, uint8_t * pOffset, uint8_t * pData, uint8_t * pLength)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint32_t   PH_MEMLOC_REM dwTotDataLen = 0;
    uint32_t   PH_MEMLOC_REM dwRem = 0;
    uint32_t   PH_MEMLOC_REM dwOffset = 0;
    uint8_t    PH_MEMLOC_REM bExchangeLen = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bCmdDataLen = 0;
    uint8_t    PH_MEMLOC_REM bIsFirstFrame = 1U;
    uint8_t    PH_MEMLOC_REM bFinished = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pLength, PH_COMP_HAL);

    /* Get the data length from pointer */
    dwTotDataLen = (uint16_t) ((pLength[2U] << 16U) | (pLength[1U] << 8U) | (pLength[0]));

    do
    {
        /* Reset the command buffer and its length variable. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_WriteFile command. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_WRITE_FILE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add information to command buffer. */
        if(bIsFirstFrame == 1U)
        {
            /* Update P1 byte to File Number */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = bFileNo;

            /* Add CRLVersion to command buffer. */
            if(bOption == PH_ON)
            {
                memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], (uint8_t *) &wCRLVer, 2U);
                bCmdLen = bCmdDataLen += 2U;
            }

            /* Add Offset to command buffer. */
            memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pOffset, 3U);
            bCmdLen = bCmdDataLen += 3U;

            /* Add Length to command buffer. */
            memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pLength, 3U);
            bCmdLen = bCmdDataLen += 3U;

            /* Clear the flag */
            bIsFirstFrame = PH_OFF;

            /* Calculate the Remaining bytes to be added in SAM command frame */
            dwRem = (uint32_t) (dwTotDataLen + bCmdDataLen);
        }

        /* Update P2 information byte */
        if(dwRem <= PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK)
        {
            /* Update Exchange Length with remaining data to be exchanged. */
            bExchangeLen = (uint8_t) dwRem;

            /* Set the flag to end the loop */
            bFinished = PH_ON;

            /* Update P2 information byte to Chained Frame */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        }
        else
        {
            /* Calculate the Data bytes to be exchanged */
            bExchangeLen = (uint8_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - bCmdDataLen);

            /* Clear Command Data Length */
            bCmdDataLen = 0;

            /* Update P2 information byte to Last Frame */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Add command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Update LC */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Add Certificate to command buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            &pData[dwOffset],
            bExchangeLen,
            NULL,
            NULL));

        /* Add Certificate to command buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0,
            &pResponse,
            &wRespLen));

        /* Calculate Next Offset and length to be exchanged */
        dwOffset += bExchangeLen;
        dwRem -= (uint32_t) bExchangeLen;

    } while(bFinished != 0);

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_WriteFileOffline(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bFileNo, uint16_t wChangeCtr,
    uint8_t * pOfflineCrypto, uint32_t dwOfflineCryptoLen, uint8_t bEnableOfflineAck, uint8_t ** ppOfflineAck,
    uint16_t * pOfflineAckLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint32_t   PH_MEMLOC_REM dwRem = 0;
    uint32_t   PH_MEMLOC_REM dwOffset = 0;
    uint8_t    PH_MEMLOC_REM bExchangeLen = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bCmdDataLen = 0;
    uint8_t    PH_MEMLOC_REM bIsFirstFrame = 1U;
    uint8_t    PH_MEMLOC_REM bFinished = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineCrypto, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineAckLen, PH_COMP_HAL);

    do
    {
        /* Reset the command buffer and its length variable. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_WriteFile command. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_WRITE_FILE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add information to command buffer. */
        if(bIsFirstFrame == 1U)
        {
            /* Update P1 byte to File Number */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = bFileNo;

            /* Add Change Counter to command buffer. */
            memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], (uint8_t *) &wChangeCtr, 2U);
            bCmdLen = bCmdDataLen += 2U;

            /* Clear the flag */
            bIsFirstFrame = PH_OFF;

            /* Calculate the Remaining bytes to be added in SAM command frame */
            dwRem = (uint32_t) (bCmdDataLen + dwOfflineCryptoLen);
        }

        /* Update P2 information byte */
        if(dwRem <= PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK)
        {
            /* Update Exchange Length with remaining data to be exchanged. */
            bExchangeLen = (uint8_t) dwRem;

            /* Set the flag to end the loop */
            bFinished = PH_ON;

            /* Update P2 information byte to Chained Frame */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        }
        else
        {
            /* Calculate the Data bytes to be exchanged */
            bExchangeLen = (uint8_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - bCmdDataLen);

            /* Clear Command Data Length */
            bCmdDataLen = 0;

            /* Update P2 information byte to Last Frame */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Add command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Add Certificate to command buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            &pOfflineCrypto[dwOffset],
            bExchangeLen,
            NULL,
            NULL));

        /* Add Certificate to command buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            bEnableOfflineAck,
            ppOfflineAck,
            pOfflineAckLen));

        /* Update LC */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Calculate Next Offset and length to be exchanged */
        dwOffset += bExchangeLen;
        dwRem -= (uint32_t) bExchangeLen;

    } while(bFinished != 0);

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_GetFileSettings(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bFileNo,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_GetFileSettings command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GET_FILE_SETTINGS;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFileNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange the command information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_Format(phhalHw_Sam_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_Format command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GET_FILE_SETTINGS;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchange the command information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
