/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) File Management command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7290 $
 * $Date: 2025-05-06 17:52:50 +0530 (Tue, 06 May 2025) $
 */


#ifndef PHHALHW_SAM_CMD_FM_H
#define PHHALHW_SAM_CMD_FM_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_FM 04_FileManagement
 * \brief Implementation of SAM File Management commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_FM_Defines Defines
 * \brief Definitions for SAM File Management commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_FM_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_CREATE_FILE                                    0xF0U   /**< Sam Instruction code for SAM_CreateFile command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_READ_FILE                                      0xF1U   /**< Sam Instruction code for SAM_ReadFile command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_WRITE_FILE                                     0xF2U   /**< Sam Instruction code for SAM_WriteFile command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_GET_FILE_SETTINGS                              0xF3U   /**< Sam Instruction code for SAM_GetFileSettings command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_FORMAT                                         0xF4U   /**< Sam Instruction code for SAM_Format command. */
/**
 * end of group phhalHw_Sam_Cmd_FM_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_FM_Defines_FileType FileType
 * \brief Definitions for File Type. To be used with below interfaces
 *  - \ref phhalHw_Sam_Cmd_SAM_CreateFile "SAM_CreateFile"
 * @{
 */
#define PHHAL_HW_SAM_FILE_TYPE_CERTIFICATE                                      0x00U   /**< Option mask for file type as Certificate File
                                                                                         *   (pure binary data file).
                                                                                         */
#define PHHAL_HW_SAM_FILE_TYPE_CRL                                              0x01U   /**< Option mask for file type as CRL File. */
/**
 * end of group phhalHw_Sam_Cmd_FM_Defines_FileType
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_FM_Defines
 * @}
 */

/**
 * \brief Performs creation of file. The FileType defines if the created file is a regular binary
 * data file (typically used to store certificates), or a CRLFile. A CRLFile is used for certificate
 * revocation. If a CRLFile is targeted, the following additional parameters are present: CRLOptions,
 * CSNSize and , depending on CRLOptions value CSNSigKey. The CRLVersion is initialized to zero.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pFileSize and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_CreateFile(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFileNo,                                                                /**< [In] File number to be created. Supported numbers are 00 - 0F */
        uint8_t bFileType,                                                              /**< [In] Type of file to create. One of the following,
                                                                                         *          - \ref PHHAL_HW_SAM_FILE_TYPE_CERTIFICATE "Certificate File"
                                                                                         *          - \ref PHHAL_HW_SAM_FILE_TYPE_CRL "CRL File"
                                                                                         */
        uint8_t bKeyNoAEK_Read,                                                         /**< [In] Key reference number of KST access entry key for reading.
                                                                                         *          - 0xFE       : No access restriction
                                                                                         *          - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bKeyVAEK_Read,                                                          /**< [In] Key version of KST access entry key for reading. */
        uint8_t bKeyNoAEK_Write,                                                        /**< [In] Key reference number of KST access entry key for writing.
                                                                                         *          - 0xFE       : No access restriction
                                                                                         *          - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bKeyVAEK_Write,                                                         /**< [In] Key version of KST access entry key for writing. */
        uint8_t * pFileSize,                                                            /**< [In] File size in bytes for the file to be created. Will be of 3 bytes
                                                                                         *        with LSB first.
                                                                                         *          - If size 0x10 need to be created, then the FileSize will be 10 00 00.
                                                                                         *          - Supported size ranges from 0x000001 - 0x008000
                                                                                         */
        uint8_t bCRLOptions,                                                            /**< [In] CRLVersion required to be incremented by 1. Will be exchange if
                                                                                         *        \b bFileType = \ref PHHAL_HW_SAM_FILE_TYPE_CRL "CRL File"
                                                                                         */
        uint8_t bCSNSize,                                                               /**< [In] CSN size to use. Supported values are 0x04, 0x07, 0x0A. Will
                                                                                         *        be exchange if \b bFileType = \ref PHHAL_HW_SAM_FILE_TYPE_CRL
                                                                                         *        "CRL File"
                                                                                         */
        uint8_t bCSNSigKey                                                               /**< [In] Targeted ECC Key Entry for CRLSignature validation. Supported
                                                                                          *        values are 0x00 - 0x07. Will be exchange if \b bFileType =
                                                                                          *        \ref PHHAL_HW_SAM_FILE_TYPE_CRL "CRL File"
                                                                                          */
    );

/**
 * \brief Performs reading of data.
 *  - The data to be read is defined by the file number of the targeted file, the offset
 *    in the data file where to start the reading and its size in bytes. The file number
 *    specifying the file where to read the data from is given by FileNo encoded in P1.
 *  - The position byte-wise in the data file where to start to read data is given by Offset.
 *    Its valid range is from 0x000000 to FileSize - 1. The data size to be read is given by
 *    Length specifying the number of bytes. If Length is equal to 0x000000 then the entire
 *    data file has to be read starting from the position specified by the Offset value.
 *    Length valid range is 0x000000 to FileSize - Offset.
 *  - If the number of bytes to read does not fit into one single APDU response,
 *    \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is provided to user and user should call
 *    this interface again with \b bOption set to \ref PH_EXCHANGE_RXCHAINING
 *    "Receive Remaining Data".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with more response data remaining.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOffset, \b pLength and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ReadFile(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option of reception of data from SAM. One of the
                                                                                         *        following,
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *            "Exchange command and receive the response". This
                                                                                         *            option means there is no further data to be read from SAM.
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_RXCHAINING "Receive Remaining Data"
                                                                                         *            from SAM. Should be used only when interface returns
                                                                                         *            \ref PH_ERR_SUCCESS_CHAINING "Chaining Response". Here
                                                                                         *            \b bFileNo, \b pOffset and \b pLength information will
                                                                                         *            not be exchanged to SAM.
                                                                                         */
        uint8_t bFileNo,                                                                /**< [In] File number of the file to be read.
                                                                                         *          - 00 - 0F: User File
                                                                                         *          - F0     : Originality Certificate File
                                                                                         */
        uint8_t * pOffset,                                                              /**< [In] The offset from where the data should be read. Will be of 3 bytes.
                                                                                         *        with LSB first. If 0x10 need to be offset, then it will be 10 00 00.
                                                                                         *          - 0 to (FileSize - 1): Starting position of Read operation
                                                                                         *          - 0xFFFFFFFF         : Return CRLFile Meta-Data
                                                                                         */
        uint8_t * pLength,                                                              /**< [In] The number of bytes to be read. Will be of 3 bytes with LSB first.
                                                                                         *          - If 0x10 bytes need to be read, then it will be 10 00 00.
                                                                                         *          - If complete file need to be read, then it will be 00 00 00.
                                                                                         */
        uint8_t ** ppResponse,                                                          /**< [Out] The contents of the File returned by SAM. */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Performs writing of data.
 *  - The location of data to be written is defined by the file number of the targeted file,
 *    the offset in the data file where to start the writing and its size in bytes. The file
 *    number specifying the file where to write to is given by FileNo encoded in P1.
 *  - The position byte-wise in the data file where to start to write data is given by Offset.
 *    Its valid range is from 0x000000 to FileSize - 1. The data size to be written is given
 *    by Length specifying the number of bytes.
 *  - If the number of bytes to send does not fit into one single APDU response, chaining is
 *    applied internally.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOffset, \b pLength
 *                                      and \b pData are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_WriteFile(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to exchange CRL Version, One of the following,
                                                                                         *          - \ref PH_OFF "Do Not Exchange CRL Version"
                                                                                         *          - \ref PH_ON "Exchange CRL Version"
                                                                                         */
        uint8_t bFileNo,                                                                /**< [In] File number of the file to be read.
                                                                                         *          - Bit[7]  : Secondary Application Indicator
                                                                                         *          - Bit[4-0]: File Number
                                                                                         */
        uint16_t wCRLVer,                                                               /**< [In] CRLVersion is a 16-bit value encoding the current version
                                                                                         *        of the CRLFile. Valid if targeting CRL File.
                                                                                         */
        uint8_t * pOffset,                                                              /**< [In] The offset from where the data should be written. Will be
                                                                                         *        of 3 bytes. with LSB first. If 0x10 need to be offset, then
                                                                                         *        it will be 10 00 00.
                                                                                         *          - 0 to (FileSize - 1): Starting position of Write operation
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The data to be written to the PICC.
                                                                                         *          - Complete Data to be provided if not targeting CRL File
                                                                                         *          - Complete Data including CRLSignature should be provided if
                                                                                         *            targeting CRF File.
                                                                                         */
        uint8_t * pLength                                                               /**< [In] The number of bytes to be written. Will be of 3 bytes with
                                                                                         *        LSB first. If 0x10 bytes need to be written, then it will be
                                                                                         *        10 00 00.
                                                                                         *
                                                                                         *        \note
                                                                                         *          - If not targeting CRF File, the length will be as mentioned above.
                                                                                         *          - If targeting CRL file, then the length will be Length of Data +
                                                                                         *            Length of CRL Signature In this case if Data Length is 10 bytes
                                                                                         *            and CRL Signature Length is 64 bytes, then \b pLength will be
                                                                                         *            (10 + 40) 00 00 => 50 00 00
                                                                                         */
    );

/**
 * \brief Performs writing of data offline.
 *  - User should complete the Offline cryptogram and provide the computed information to
 *    this interface.
 *  - If the number of bytes to send does not fit into one single APDU response, chaining is
 *    applied internally.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOffset, \b pLength
 *                                      and \b pData are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_WriteFileOffline(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFileNo,                                                                /**< [In] File number of the file to be read.
                                                                                         *          - Bit[7]  : Secondary Application Indicator
                                                                                         *          - Bit[4-0]: File Number
                                                                                         */
        uint16_t wChangeCtr,                                                            /**< [In] Change Counter to avoid replay attacks. */
        uint8_t * pOfflineCrypto,                                                       /**< [In] Offline Cryptogram exchanged to SAM (EncKeyEntry + OfflineMAC)
                                                                                         *          - Encrypted Write File data (Kce, WriteFileData [|| SAMUID])
                                                                                         *            where WriteFileData is the command data field parameters,
                                                                                         *            i.e. CRLVersion until CRLSignature (if applicable) in plain.
                                                                                         *          - Integrity MAC protection as MACt (Kcm, CLA || INS || P1 ||
                                                                                         *            P2 || LC || Change_Ctr || E(Kce, WriteFileData [|| SAMUID]))
                                                                                         */
        uint32_t dwOfflineCryptoLen,                                                    /**< [In] Length of bytes available in \b pOfflineCrypto buffer */
        uint8_t bEnableOfflineAck,                                                      /**< [In] To Enable reception of Offline Acknowledge
                                                                                         *          - \ref PH_OFF "Disable reception of Offline ACK"
                                                                                         *            (LE will not be exchanged)
                                                                                         *          - \ref PH_ON "Enable reception of Offline ACK"
                                                                                         *            (LE will be exchanged)
                                                                                         */
        uint8_t ** ppOfflineAck,                                                        /**< [Out] Offline Acknowledge information received from SAM. Will be based on
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_OFF "Disable reception of Offline ACK",
                                                                                         *             buffer will contain NULL.
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_ON "Enable reception of Offline ACK",
                                                                                         *             buffer will contain Offline change acknowledge as MACt (Kcm, 0x90 || 0x00
                                                                                         *             || INS || Change_Ctr || FileNo || SAMUID).
                                                                                         *
                                                                                         */
        uint16_t * pOfflineAckLen                                                       /**< [Out] Length of bytes available in \b ppOfflineAck buffer */
    );

/**
 * \brief Performs reading the meta-data of a file.
 *  - FileType, Access Rights and File Size of the targeted file encoded in P1 will be returned.
 *  - Additionally, if a CRLFile is targeted the CRLOptions, CSNSize and, if applicable,
 *    CSNSigKey will be returned.
 *  - If P1 is set to 0xFE, will return the remaining fee user memory that is available for file
 *    creation.
 *  - If P1 is set to 0xFF, a list of the already created file identifiers will be returned.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_GetFileSettings(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFileNo,                                                                /**< [In] File number of the file to be read.
                                                                                         *          - 00 - 0F: User File
                                                                                         *          - F0     : Originality Certificate File
                                                                                         *          - FE     : Free Memory
                                                                                         *          - FF     : List of file IDs
                                                                                         */
        uint8_t ** ppResponse,                                                          /**< [Out] The settings of the File returned by SAM. */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );


/**
 * \brief Performs deletion of all files. After successful execution the full user memory
 * for files is again available for file creation.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_Format(
        phhalHw_Sam_DataParams_t * pDataParams                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * end of group phhalHw_Sam_Cmd_FM
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_FM_H */
