/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Data Processing command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_DP.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_ApplySM(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bCommMode,
    uint8_t bOffset, uint8_t bCmdCtrIncr, uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);
    if(bDataLen != 0U)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer and its length variable. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_ApplySM command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_APPLY_SM;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCommMode;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDataLen;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Add Offset to command buffer. */
        if((bCommMode == PHHAL_HW_CMD_SAM_COMM_MODE_FULL) &&
            (wOption & PHHAL_HW_SAM_CMD_INCLUDE_OFFSET))
        {
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOffset;
        }

        /* Add Command Counter to command buffer. */
        if(bCommMode == PHHAL_HW_CMD_SAM_COMM_MODE_PLAIN)
        {
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCmdCtrIncr;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));
    }

    /* Buffer data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            ppResponse,
            pRespLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_RemoveSM(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bCommMode,
    uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);
    if(bDataLen != 0U)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer and its length variable. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_RemoveSM command information. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_SAM_REMOVE_SM;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = bCommMode;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = bDataLen;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));
    }

    /* Buffer data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            1U,
            ppResponse,
            pRespLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_VerifyMAC(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bNum,
    uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_Verify_MAC command information. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_SAM_VERIFY_MAC;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = bNum;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = bDataLen;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));
    }

    /* Buffer data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0,
            &pResponse,
            &wRespLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_GenerateMAC(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bNum,
    uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
        PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_Generate_MAC command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GENERATE_MAC;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNum;

        /* Append LC only if data is available. */
        if((wOption & PHHAL_HW_SAM_GENERATE_MAC_INCLUDE_LC) || (bDataLen != 0))
        {
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDataLen;
        }

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));
    }

    /* Buffer data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Update LC. */
    if((wOption & PHHAL_HW_SAM_GENERATE_MAC_INCLUDE_LC) || (bDataLen != 0))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Add LE to exchange buffer if its the last frame. */
        if(pDataParams->pTxBuffer[PHHAL_HW_SAM_ISO7816_P1_POS] != PHHAL_HW_SAM_ISO7816_CHAINED_FRAME)
        {
            wStatus = phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                phhalHw_Sam_Cmd_DefaultLe,
                1,
                NULL,
                NULL);
        }

        /* Exchange the buffered information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0,
            ppResponse,
            pRespLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_DecipherData(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pEncData,
    uint8_t bEncDataLen, uint8_t * pLength, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(wOption & PHHAL_HW_SAM_DECIPHER_LENGTH_INCLUDE)
        PH_ASSERT_NULL_PARAM(pLength, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPlainDataLen, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_Decipher_Data command information. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_DATA;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));

        /* Buffer length information to exchange buffer. */
        if(wOption & PHHAL_HW_SAM_DECIPHER_LENGTH_INCLUDE)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pLength,
                3U,
                NULL,
                NULL));
        }
    }

    /* Buffer data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pEncData,
        bEncDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Buffer LE and exchange the buffered information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            1,
            ppPlainData,
            pPlainDataLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_EncipherData(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pPlainData,
    uint8_t bPlainDataLen, uint8_t bOffset, uint8_t ** ppEncData, uint16_t * pEncDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPlainDataLen != 0)
        PH_ASSERT_NULL_PARAM(pPlainData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncDataLen, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_Encipher_Data command information. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_DATA;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = bOffset;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));
    }

    /* Exchange second part with lower layer */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPlainData,
        bPlainDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Buffer LE and exchange the buffered information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            1U,
            ppEncData,
            pEncDataLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_DecipherOfflineData(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pEncData,
    uint8_t bEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPlainDataLen, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_DecipherOffline_Data command information. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_OFFLINE_DATA;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));
    }

    /* Buffer data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pEncData,
        bEncDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Buffer LE and exchange the buffered information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            1,
            ppPlainData,
            pPlainDataLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_EncipherOfflineData(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pPlainData,
    uint8_t bPlainDataLen, uint8_t ** ppEncData, uint16_t * pEncDataLen)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPlainData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncDataLen, PH_COMP_HAL);

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_EncipherOffline_Data command information. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_OFFLINE_DATA;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 byte. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));
    }

    /* Exchange second part with lower layer */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPlainData,
        bPlainDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Buffer LE and exchange the buffered information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            1,
            ppEncData,
            pEncDataLen);

        /* Check status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
