/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Data Processing command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_DP_H
#define PHHALHW_SAM_CMD_DP_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DP 04_DataProcessing
 * \brief Implementation of SAM Data Processing commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DP_Defines Defines
 * \brief Definitions for SAM Data Processing commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DP_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_APPLY_SM                                       0xAEU   /**< Sam Instruction code for SAM_Apply_SM command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_REMOVE_SM                                      0xADU   /**< Sam Instruction code for SAM_Remove_SM command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_VERIFY_MAC                                     0x5CU   /**< Sam Instruction code for SAM_Verify_MAC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_GENERATE_MAC                                   0x7CU   /**< Sam Instruction code for SAM_Generate_MAC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_DATA                                  0xDDU   /**< Sam Instruction code for SAM_Decipher_Data command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_DATA                                  0xEDU   /**< Sam Instruction code for SAM_Encipher_Data command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_OFFLINE_DATA                          0x0DU   /**< Sam Instruction code for SAM_DecipherOffline_Data command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_OFFLINE_DATA                          0x0EU   /**< Sam Instruction code for SAM_EncipherOffline_Data command. */
/**
 * end of group phhalHw_Sam_Cmd_DP_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DP_Defines_IncludeOffset IncludeOffset
 * \brief Definitions for Offset inclusion. To be used with below interfaces
 *  - \ref phhalHw_Sam_Cmd_SAM_ApplySM "SAM_Apply_SM"
 * @{
 */
#define PHHAL_HW_SAM_CMD_EXCLUDE_OFFSET                                         0x00U   /**< Option mask for not exchanging the Offset information for
                                                                                         *   \ref phhalHw_Sam_Cmd_SAM_ApplySM "SAM_Apply_SM" interface.
                                                                                         */
#define PHHAL_HW_SAM_CMD_INCLUDE_OFFSET                                         0x80U   /**< Option mask for exchanging the Offset information for
                                                                                         *   \ref phhalHw_Sam_Cmd_SAM_ApplySM "SAM_Apply_SM" interface.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DP_Defines_ProtMode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DP_Defines_TruncMode TruncationMode
 * \brief Definitions for Protection Mode. To be used with \ref phhalHw_Sam_Cmd_SAM_GenerateMAC
 * "SAM_Generate_MAC" and \ref phhalHw_Sam_Cmd_SAM_VerifyMAC "SAM_Verify_MAC" interface.
 * @{
 */
#define PHHAL_HW_SAM_TRUNCATION_MODE_STANDARD                                   0x00U   /**< Option mask for truncation mode as standard truncation. */
#define PHHAL_HW_SAM_TRUNCATION_MODE_MFP                                        0x80U   /**< Option mask for truncation mode as MFP truncation. */
#define PHHAL_HW_SAM_GENERATE_MAC_INCLUDE_LC                                    0x80U   /**< Option mask for inclusion of LC in the command frame.
                                                                                         *   To be used with \ref phhalHw_Sam_Cmd_SAM_GenerateMAC
                                                                                         *   "SAM_Generate_MAC"
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DP_Defines_TruncMode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DP_Defines_Length Length
 * \brief Definitions for exchanging Length information to SAM. To be used with \ref phhalHw_Sam_Cmd_SAM_DecipherData
 * "SAM_Decipher_Data" interface.
 * @{
 */
#define PHHAL_HW_SAM_DECIPHER_LENGTH_EXCLUDE                                    0x00U   /**< Option mask for excluding the length information in the
                                                                                         *   command frame.
                                                                                         */
#define PHHAL_HW_SAM_DECIPHER_LENGTH_INCLUDE                                    0x80U   /**< Option mask for including the length information in the
                                                                                         *   command frame.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DP_Defines_Length
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DP_Defines
 * @}
 */

/**
 * \brief Apply the DESFire EVx (EV2 or higher) Secure Messaging in S-mode on the provided DESFire EVx (EV2 or higher)
 * command according to the required mode and the currently activated session keys. The required protection mode is
 * selected via the command parameter \b bCommMode.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ApplySM(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Refer below for Option to be used.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_EXCLUDE_OFFSET
                                                                                         *            "Offset" (\b bOffset is not exchanged to SAM)
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_INCLUDE_OFFSET
                                                                                         *            "Offset" (\b bOffset is exchanged to SAM)
                                                                                         *
                                                                                         *          - Buffering options. Ored with above options.
                                                                                         *              - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                     "Exchange" information to SAM and Receive
                                                                                         *                     the response
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                     "Buffer first" set of information.
                                                                                         *                     No exchange is performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                     "Buffer intermediate" set of information.
                                                                                         *                     No exchange is performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                     "Buffer last" set of information.
                                                                                         *                     Exchange information to SAM and Receive
                                                                                         *                     the response
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                     "Should be used to exchange chunks" of data
                                                                                         *                     and receive response
                                                                                         */
        uint8_t bCommMode,                                                              /**< [In] Communication mode to be used.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_PLAIN
                                                                                         *                 "Communication mode as Plain"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_MAC
                                                                                         *                 "Communication mode as MAc"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL
                                                                                         *                 "Communication mode as Full"
                                                                                         */
        uint8_t bOffset,                                                                /**< [In] Command offset. Index of the first byte in data field of the
                                                                                         *        PICC command data.
                                                                                         */
        uint8_t bCmdCtrIncr,                                                            /**< [In] Command counter increment value. Value by which to increase
                                                                                         *        the CmdCtr.
                                                                                         */
        uint8_t * pData,                                                                /**< [In] Plain data to be protected according to the communication
                                                                                         *        mode specified.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] The protected data returned by Sam according to communication
                                                                                         *         mode specified.
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Removes the DESFire EVx (EV2 or higher) Secure Messaging in S-mode from the provided PICC response
 * payload according to the required mode and the currently activated session keys. The required protection
 * mode is selected via the command parameter \b bCommMode.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_RemoveSM(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options to exchange data to SAM and receive
                                                                                         *        data from SAM.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange" information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information.
                                                                                         *                 Exchange information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "Should be used to exchange chunks" of data
                                                                                         *                 and receive response
                                                                                         */
        uint8_t bCommMode,                                                              /**< [In] Communication mode to be used.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_MAC
                                                                                         *                 "Communication mode as MAc"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL
                                                                                         *                 "Communication mode as Full"
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The complete data received form the PICC including the
                                                                                         *        status code.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] The plain data returned by Sam according to communication
                                                                                         *         mode specified.
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Verifies the MAC which was sent by the PICC or any other system based on the given MACed plain text data
 * and the currently valid cryptographic key.
 *  - The valid key has been activated using a valid PICC authentication (SAM_AuthenticatePICC , SAM_IsoAuthenticatePICC)
 *    in case of a PICC key or using a valid key activation (SAM_ActivateOfflineKey) in case of an OfflineCrypto Key
 *  - The applied MAC algorithm depends on the key type. The command can be used for verifying only a part, up to the full
 *    MAC. The number of MAC bytes to be verified is defined by parameter \b bNum. By default the standard truncation is
 *    expected. For AES key types (including KeyType.AES_128LRP), also the truncation as used by a.o. the MIFARE Plus
 *    Secure Messaging can be applied. This option for MFP Truncation is not allowed with a key that has the option
 *    'Keep IV' (SET bit 2) set.
 *  - In OfflineCrypto PICC State, if the targeted key is of KeyType.AES_128LRP, SAM_Verify_MAC shall apply
 *    KeyID.LRPUpdate or KeyID.LRPMACUpdate, if respectively one or two updated keys were generated with
 *    SAM_ActivateOfflineKey.
 *  - In OfflineCrypto PICC State, if activated key from KST is referencing a KUC, SAM_Verify_MAC increments the
 *    KUC associated to the targeted key before any crypto processing.
 *  - Note that SAM_Verify_MAC supports the verification of a MAC over zero-byte input length messages. In that
 *    case the Data field of the APDU will just contain the MAC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_VerifyMAC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options to exchange data to SAM and receive
                                                                                         *        data from SAM.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange" information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information.
                                                                                         *                 Exchange information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "Should be used to exchange chunks" of data
                                                                                         *                 and receive response
                                                                                         */
        uint8_t bNum,                                                                   /**< [In] Type of truncation mode to be applied if AES key type is
                                                                                         *        used.
                                                                                         *          - \ref PHHAL_HW_SAM_TRUNCATION_MODE_STANDARD
                                                                                         *            "Standard truncation"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_TRUNCATION_MODE_MFP
                                                                                         *            "MFP truncation"
                                                                                         *
                                                                                         *          - Number of MAC bytes (\b bNum) to be verified.
                                                                                         *            Should be combined with one of the above options.
                                                                                         *              - 0x01 - 0x08: (3)DES key types
                                                                                         *              - 0x01 - 0x10: AES key types
                                                                                         *              - 0x00: Default number of bytes (depending on key type)
                                                                                         */
        uint8_t * pData,                                                                /**< [In] Plain data including the MAC to be checked. */
        uint8_t bDataLen                                                                /**< [In] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Generates a MAC which is meant to be sent to the PICC or any other system based on the given plain text data
 * and the currently valid cryptographic key.
 *  - The valid key has been activated using a valid PICC authentication (SAM_AuthenticatePICC , SAM_IsoAuthenticatePICC)
 *    in case of a PICC key or using a valid key activation (SAM_ActivateOfflineKey) in case of an OfflineCrypto Key.
 *  - The applied MAC algorithm depends on the key type. The command can be used for creating only a part, up to the full
 *    MAC. The number of MAC bytes to be generated is defined by parameter \b bNum. By default the standard truncation is
 *    applied. For AES key types (including KeyType.AES_128LRP), also the truncation as used by a.o. the MIFARE Plus Secure
 *    Messaging can be applied. This option for MFP Truncation is not allowed with a key that has the option 'Keep IV' (SET
 *    bit 2) set.
 *  - In OfflineCrypto PICC State, if the targeted key is of KeyType.AES_128LRP, SAM_Generate_MAC shall apply KeyID.LRPUpdate
 *    or KeyID.LRPMACUpdate, if respectively one or two updated keys were generated with SAM_ActivateOfflineKey.
 *  - In OfflineCrypto PICC State, if activated key from KST is referencing a KUC, SAM_Generate_MAC increments the KUC associated
 *    to the targeted key before any crypto processing.
 *  - Note that SAM_Generate_MAC supports the generation of a MAC over zero-byte input length messages. In that case the LC and
 *    Data fields of the command APDU will be absent.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_GenerateMAC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange" information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information.
                                                                                         *                 Exchange information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "Should be used to exchange chunks" of data
                                                                                         *                 and receive response
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_GENERATE_MAC_INCLUDE_LC "Include LC" should be used
                                                                                         *                 for inclusion of LC when \ref PH_EXCHANGE_BUFFER_FIRST "First Data Chunk"
                                                                                         *                 is used and \b bDataLen is not available but \b bDataLen is part of
                                                                                         *                 \ref PH_EXCHANGE_BUFFER_CONT "Intermediate Data Chunk" or
                                                                                         *                 \ref PH_EXCHANGE_BUFFER_LAST "Last Data Chunk".
                                                                                         */
        uint8_t bNum,                                                                   /**< [In] Type of truncation mode to be applied if AES key type is used.
                                                                                         *        used.
                                                                                         *          - \ref PHHAL_HW_SAM_TRUNCATION_MODE_STANDARD
                                                                                         *            "Standard truncation"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_TRUNCATION_MODE_MFP
                                                                                         *            "MFP truncation"
                                                                                         *
                                                                                         *          - Number of MAC bytes (\b bNum) to be processed.
                                                                                         *            Should be combined with one of the above options.
                                                                                         *              - 0x01 - 0x08: (3)DES key types
                                                                                         *              - 0x01 - 0x10: AES key types
                                                                                         *              - 0x00: Default number of bytes (depending on key type)
                                                                                         */
        uint8_t * pData,                                                                /**< [In] Plain data to be maced. Can be null if there is no data. */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] The generated MAC returned by SAM. */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Deciphers data packages sent by a PICC, any other system or a MIFARE card based on the currently valid
 * cryptographic key and returns plain data to the host.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pEncData and \b pPlainDataLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DecipherData(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option for including the length information in command frame.
                                                                                         *          - \ref PHHAL_HW_SAM_DECIPHER_LENGTH_EXCLUDE
                                                                                         *                 "Length information (\b pLength) will not be exchanged"
                                                                                         *          - \ref PHHAL_HW_SAM_DECIPHER_LENGTH_INCLUDE
                                                                                         *                 "Length information (\b pLength) will be exchanged"
                                                                                         *
                                                                                         *          - Buffering options. Ored with above options.
                                                                                         *              - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                     "Exchange" information to SAM and Receive
                                                                                         *                     the response
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                     "Buffer first" set of information.
                                                                                         *                     No exchange is performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                     "Buffer intermediate" set of information.
                                                                                         *                     No exchange is performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                     "Buffer last" set of information.
                                                                                         *                     Exchange information to SAM and Receive
                                                                                         *                     the response
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                     "Should be used to exchange chunks" of data
                                                                                         *                     and receive response
                                                                                         */
        uint8_t * pEncData,                                                             /**< [In] Encrypted data to be deciphered. */
        uint8_t bEncDataLen,                                                            /**< [In] Length of bytes available in \b pEncData buffer. */
        uint8_t * pLength,                                                              /**< [In] Overall length of encrypted input data. This 3 byte value is
                                                                                         *        only used if \b wOption has \ref PHHAL_HW_SAM_DECIPHER_LENGTH_INCLUDE
                                                                                         *        "Length information" (\b pLength) will be exchanged.
                                                                                         */
        uint8_t ** ppPlainData,                                                         /**< [Out] Deciphered data returned by SAM. */
        uint16_t * pPlainDataLen                                                        /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Enciphers data packages which are meant to be sent to a PICC or any other system based on the given
 * plain text data and the currently valid cryptographic key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b bPlainDataLen and \b pEncDataLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_EncipherData(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange" information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information.
                                                                                         *                 Exchange information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "Should be used to exchange chunks" of data
                                                                                         *                 and receive response
                                                                                         */
        uint8_t * pPlainData,                                                           /**< [In] Plain Data to be enciphered. */
        uint8_t bPlainDataLen,                                                          /**< [In] Length of bytes available in \b pPlainData buffer. */
        uint8_t bOffset,                                                                /**< [In] Offset into the input data indicating the first data
                                                                                         *        byte to be enciphered.
                                                                                         */
        uint8_t ** ppEncData,                                                           /**< [Out] Enciphered data returned by SAM. */
        uint16_t * pEncDataLen                                                          /**< [Out] Length of bytes available in \b ppEncData buffer. */
    );

/**
 * \brief Decrypts data received from any other system based on the given cipher text data and the currently valid
 * cryptographic OfflineCrypto Key. The valid key has been activated using a valid key activation (SAM_ActivateOfflineKey).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pEncData and \b pPlainDataLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DecipherOfflineData(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange" information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information.
                                                                                         *                 Exchange information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "Should be used to exchange chunks" of data
                                                                                         *                 and receive response
                                                                                         */
        uint8_t * pEncData,                                                             /**< [In] Encrypted data to be deciphered. */
        uint8_t bEncDataLen,                                                            /**< [In] Length of bytes available in \b pEncData buffer. */
        uint8_t ** ppPlainData,                                                         /**< [Out Deciphered data returned by SAM. */
        uint16_t * pPlainDataLen                                                        /**< [Out] Length of bytes available in \b ppPlainData buffer. */
    );

/**
 * \brief Encrypts data received from any other system based on the given cipher text data and the currently valid
 * cryptographic OfflineCrypto Key. The valid key has been activated using a valid key activation (SAM_ActivateOfflineKey).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pEncData and \b pPlainDataLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_EncipherOfflineData(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange" information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information.
                                                                                         *                 No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information.
                                                                                         *                 Exchange information to SAM and Receive
                                                                                         *                 the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "Should be used to exchange chunks" of data
                                                                                         *                 and receive response
                                                                                         */
        uint8_t * pPlainData,                                                           /**< [In] Plain data to be enciphered. */
        uint8_t bPlainDataLen,                                                          /**< [In] Length of bytes available in \b pPlainData buffer. */
        uint8_t ** ppEncData,                                                           /**< [Out] Enciphered data returned by SAM. */
        uint16_t * pEncDataLen                                                          /**< [Out] Length of bytes available in \b ppEncData buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_DP
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_DP_H */
