/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Public Key Infrastructure command implementation of Reader
 * Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_PKI_H
#define PHHALHW_SAM_CMD_PKI_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM


#define PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_LC_MIN                           10U     /* Minimum Length of LC data for PKI_GenerateKeyPair command. */
#define PHHAL_HW_SAM_CMD_PKI_IMPORT_KEY_LC_MIN                                  6U      /* Minimum Length of LC data for PKI_ImportKey command. */
#define PHHAL_HW_SAM_CMD_PKI_E_LEN                                              228U    /* Maximum PKI_e information that can be exchanged
                                                                                         * to SAM in one frame
                                                                                         */

#define PHHAL_HW_SAM_CMD_PKI_IMPORT_ECC_KEY_LC_MIN                              8U      /* Minimum Length of LC data for PKI_ImportEccKey command. */
#define PHHAL_HW_SAM_CMD_PKI_IMPORT_ECC_CURVE_LC_MIN                            3U      /* Minimum Length of LC data for PKI_ImportEccCurve command. */
#define PHHAL_HW_SAM_CMD_PKI_VERIFY_ECC_SIGNATURE_LC_MIN                        3U      /* Minimum Length of LC data for PKI_VerifyEccSignature command. */

#define PHHAL_HW_SAM_CMD_PKI_REFNO_CURVE_PRESENT                                0x0002U /* Option to indicate Reference of ECC Curve is present. */
#define PHHAL_HW_SAM_CMD_PKI_CRL_FILE_ID_PRESENT                                0x1000U /* Option to indicate CRL File ID is present. */
#define PHHAL_HW_SAM_CMD_PKI_ISSUER_PRESENT                                     0x2000U /* Option to indicate Issuer information is present. */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI 05_PublicKeyInfrastructure
 * \brief Implementation of SAM Public Key Infrastructure (PKI) commands.
 * Here RSA (Rivest-Shamir-Adleman) and ECC (Elliptic Curve Cryptography) command implementation are available.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_Defines Defines
 * \brief Definitions for SAM Public Key Infrastructure's RSA (Rivest-Shamir-Adleman)
 *  and ECC (Elliptic Curve Cryptography) commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_Defines_UpdateMode UpdateMode
 * \brief Definitions for Update settings and value or settings only.
 * To be used with below mentioned interface(s)
 *  - \ref phhalHw_Sam_Cmd_PKI_ImportKey "PKI_ImportKey"
 *  - \ref phhalHw_Sam_Cmd_PKI_ImportEccKey "PKI_ImportEccKey"
 *  - \ref phhalHw_Sam_Cmd_PKI_ImportEccCurve "PKI_ImportEccCurve"
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_VALUE                              0x00U   /**< Option mask for updating settings and values. */
#define PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY                               0x01U   /**< Option mask for updating settings only. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_Defines_UpdateMode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_Defines_FirstFrame FirsFrame
 * \brief Definitions for indicating the interface call for first frame and add the
 * required information to command buffer for exchange. For the next do not add the
 * previously exchanged information.
 * To be used with below mentioned interface(s)
 *  - \ref phhalHw_Sam_Cmd_PKI_GenerateHash "PKI_GenerateHash"
 *  - \ref phhalHw_Sam_Cmd_PKI_DecipherData "PKI_DecipherData"
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_FIRST_FRAME                                        0x80U   /**< Option mask for a framing the first frame of Decipher Data command. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_Defines_FirstFrame
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_Defines_Hash HashAlgo
 * \brief Definitions for Hashing algorithm. To be used with below mentioned interface(s)
 *  - \ref phhalHw_Sam_Cmd_PKI_UpdateKeyEntries "PKI_UpdateKeyEntries"
 *  - \ref phhalHw_Sam_Cmd_PKI_EncipherKeyEntries "PKI_EncipherKeyEntries"
 *  - \ref phhalHw_Sam_Cmd_PKI_GenerateHash "PKI_GenerateHash"
 *  - \ref phhalHw_Sam_Cmd_PKI_GenerateSignature "PKI_GenerateSignature"
 *  - \ref phhalHw_Sam_Cmd_PKI_VerifySignature "PKI_VerifySignature"
 *  - \ref phhalHw_Sam_Cmd_PKI_EncipherData "PKI_EncipherData"
 *  - \ref phhalHw_Sam_Cmd_PKI_DecipherData "PKI_DecipherData"
 *  - \ref phhalHw_Sam_Cmd_PKI_GenerateEccSignature "PKI_GenerateEccSignature"
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1                               0x00U   /**< Option mask for SHA 1 hashing algorithm to be used. */
#define PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224                             0x01U   /**< Option mask for SHA 224 hashing algorithm to be used. */
#define PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256                             0x03U   /**< Option mask for SHA 256 hashing algorithm to be used. */
#define PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_384                             0x04U   /**< Option mask for SHA 384 hashing algorithm to be used. */
#define PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_512                             0x05U   /**< Option mask for SHA 512 hashing algorithm to be used. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_Defines_Hash
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_PKI_Defines
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_PKI_RSA RSA
 * \brief SAM commands used for asymmetric RSA key management, signature handling and symmetric
 * key updates based on PKI.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines Defines
 * \brief Definitions for SAM PKI RSA commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines_CommandCodes CommandCodes
 * \brief Definitions for SAM PKI RSA command codes.
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_KEY_PAIR                              0x15U   /**< Sam Instruction code for PKI_GenerateKeyPair command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_KEY                                     0x19U   /**< Sam Instruction code for PKI_ImportKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_PRIVATE_KEY                             0x1FU   /**< Sam Instruction code for PKI_ExportPrivateKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_PUBLIC_KEY                              0x18U   /**< Sam Instruction code for PKI_ExportPublicKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_UPDATE_KEY_ENTRIES                             0x1DU   /**< Sam Instruction code for PKI_UpdateKeyEntries command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_ENCIPHER_KEY_ENTRIES                           0x12U   /**< Sam Instruction code for PKI_EncipherKeyEntries command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_HASH                                  0x17U   /**< Sam Instruction code for PKI_GenerateHsh command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_SIGNATURE                             0x16U   /**< Sam Instruction code for PKI_GenerateSignature command */
#define PHHAL_HW_SAM_CMD_INS_PKI_SEND_SIGNATURE                                 0x1AU   /**< Sam Instruction code for PKI_SendSignature command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_VERIFY_SIGNATURE                               0x1BU   /**< Sam Instruction code for PKI_VerifySignature command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_ENCIPHER_DATA                                  0x13U   /**< Sam Instruction code for PKI_EncipherData command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_DECIPHER_DATA                                  0x14U   /**< Sam Instruction code for PKI_Decipher command. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines_GenerationMode GenerationMode
 * \brief Definitions for Random number generation mode.
 * To be used with \ref phhalHw_Sam_Cmd_PKI_GenerateKeyPair "PKI_GenerateKeyPair" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_RANDOM_E                         0x00U   /**< Option mask for a key generation with a randomly selected exponent e. */
#define PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E                           0x01U   /**< Option mask for a key generation with a given exponent e. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines_GenerationMode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines_AEK Include_AccessEntryKey
 * \brief Definitions for Access Entry Key exchange to SAM.
 * To be used with below mentioned interface(s)
 *  - \ref phhalHw_Sam_Cmd_PKI_GenerateKeyPair "PKI_GenerateKeyPair"
 *  - \ref phhalHw_Sam_Cmd_PKI_ImportKey "PKI_ImportKey"
 *  -
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK                                        0x02U   /**< Option mask for exchanging Access Entry Key information to SAM. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines_AEK
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines_Export_AEK Export_AccessEntryKey
 * \brief Definitions for Access Entry Key retrieval from SAM.
 * To be used with below mentioned interface(s)
 *  - \ref phhalHw_Sam_Cmd_PKI_ExportPrivateKey "PKI_ExportPrivateKey"
 *  - \ref phhalHw_Sam_Cmd_PKI_ExportPublicKey "PKI_ExportPublicKey"
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_RECEIVE_AEK                                        0x80U   /**< Option mask for receiving Access Entry Key information from SAM. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines_Export_AEK
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines_ACK Acknowledge
 * \brief Definitions to include Le byte and Acknowledge key number to command frame for receiving
 * the UpdateAck data from SAM. To be used with \ref phhalHw_Sam_Cmd_PKI_UpdateKeyEntries
 * "PKI_UpdateKeyEntries" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_RECEIVE_ACK                                        0x80U   /**< Option mask for receiving Update Acknowledgment information from SAM. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines_ACK
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_RSA_Defines_Div Diversification
 * \name Definitions for SAM Public Key Infrastructure key diversification.To be used
 * with \ref phhalHw_Sam_Cmd_PKI_EncipherKeyEntries "EncipherKeyEntries" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_DIVERSIFICATION_ON                                 0x10U   /**< Option mask enabling the key diversification. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines_Div
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA_Defines
 * @}
 */

/**
 * \brief Create a pair of a public and private RSA key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPKI_e is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateKeyPair(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option for P1 information byte.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_RANDOM_E
                                                                                         *                "PKI_e is randomly generated"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E
                                                                                         *                "PKI_e is provided by the host"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK "Access Entry Key"
                                                                                         *                (PKI_KeyNoAEK, PKI_KeyVAEK) exchanged to SAM
                                                                                         */
        uint8_t bPKI_KeyNo,                                                             /**< [In] Key reference number of the PKI Key Entry (00h to 01h). */
        uint16_t wPKI_Set,                                                              /**< [In] Configuration settings of the created key entry. */
        uint8_t bPKI_KeyNoCEK,                                                          /**< [In] Key reference number of KST change entry key.
                                                                                         *          - 0xFE       : No change restriction
                                                                                         *          - 0xFF       : Entry locked
                                                                                         *          - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bPKI_KeyVCEK,                                                           /**< [In] Key version of KST change entry key. */
        uint8_t bPKI_RefNoKUC,                                                          /**< [In] Reference number of key usage counter (00h - 0Fh, FFh). */
        uint8_t bPKI_KeyNoAEK,                                                          /**< [In] Key reference number of KST access entry key.
                                                                                         *          - 0xFE         : No Access Restrictions
                                                                                         *          - 0xFF         : Entry Disabled
                                                                                         *          - 0x00 - 0x7F  : Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bPKI_KeyVAEK,                                                           /**< [In] Key version of KST access entry key. */
        uint16_t wPKI_NLen,                                                             /**< [In] RSA key length size in bytes: between 64 and 256 bytes, a multiple
                                                                                         *        of 8 byte.
                                                                                         */
        uint16_t wPKI_eLen,                                                             /**< [In] Public exponent length size in bytes between 4 and 256 bytes, a
                                                                                         *        multiple of 4 byte, and shall not be greater than PKI_NLen:
                                                                                         *        between 4 and 256 bytes, a multiple of 4 byte
                                                                                         *
                                                                                         */
        uint8_t * pPKI_e                                                                /**< [In] Public exponent e where eLen is PKI_eLen. It must be an odd integer.
                                                                                         *          - Exchanged if \b bOption = \ref PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E
                                                                                         *            "PKI_e is provided by the host".
                                                                                         *          - NULL if \b bOption = \ref PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_RANDOM_E
                                                                                         *            "PKI_e is randomly generated".
                                                                                         */
    );

/**
 * \brief Import a public or private RSA key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    - \b pPKI_N and \b pPKI_e are NULL if \b bOption =
 *                                        \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY
 *                                        "Update only key settings"
 *                                      - \b pPKI_p, \b pPKI_q, \b pPKI_dP, \b pPKI_dQ and \b pPKI_ipq
 *                                        are NULL if \b bOption = \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY
 *                                        "Update only key settings" and Private Key is included.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ImportKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option for P1 information byte.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_VALUE
                                                                                         *              "Update key settings and key value"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY
                                                                                         *              "Update only key settings" (PKI_KeyNo, PKI_SET, PKI_
                                                                                         *               KeyNoCEK, PKI_KeyVCEK, PKI_RefNoKUC, and, if
                                                                                         *               present, PKI_KeyNoAEK and PKI_KeyVAEK)
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK "Access Entry Key"
                                                                                         *                (PKI_KeyNoAEK, PKI_KeyVAEK) exchanged to SAM
                                                                                         */
        uint8_t bPKI_KeyNo,                                                             /**< [In] Key reference number of the PKI Key Entry to update.
                                                                                         *          - 0x00 - 0x02: Without private key
                                                                                         *          - 0x00 - 0x01: With private key included
                                                                                         */
        uint16_t wPKI_Set,                                                              /**< [In] Configuration settings of the imported key entry. It indicates
                                                                                         *        whether a private of public key shall be imported.
                                                                                         */
        uint8_t bPKI_KeyNoCEK,                                                          /**< [In] Key reference number of KST change entry key.
                                                                                         *          - 0xFE       : No change restriction
                                                                                         *          - 0xFF       : Entry Locked
                                                                                         *          - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bPKI_KeyVCEK,                                                           /**< [In] Key version of KST change entry key. */
        uint8_t bPKI_RefNoKUC,                                                          /**< [In] Reference number of key usage counter (00h - 0Fh, FFh). */
        uint8_t bPKI_KeyNoAEK,                                                          /**< [In] Key reference number of KST access entry key
                                                                                         *          - 0xFE         : No Access Restrictions
                                                                                         *          - 0xFF         : Entry Disabled
                                                                                         *          - 0x00 - 0x7F  : Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bPKI_KeyVAEK,                                                           /**< [In] Key version of KST access entry key. */
        uint16_t wPKI_NLen,                                                             /**< [In] RSA key length size in bytes: between 64 and 256 bytes, a multiple
                                                                                         *        of 8 byte.
                                                                                         */
        uint16_t wPKI_eLen,                                                             /**< [In] Public exponent length size in bytes between 4 and 256 bytes, a
                                                                                         *        multiple of 4 byte, and shall not be greater than PKI_NLen:
                                                                                         *        between 4 and 256 bytes, a multiple of 4 byte.
                                                                                         */
        uint16_t wPKI_PLen,                                                             /**< [In] If private key included, the prime p length size in bytes
                                                                                         *        ceil(PKI_pLen / 4) + 2 <= ceil(PKI_NLen / 4).
                                                                                         */
        uint16_t wPKI_QLen,                                                             /**< [In] If private key included, the prime q length size in bytes
                                                                                         *        ceil(PKI_qLen/4) + 2 <= ceil(PKI_NLen/4).
                                                                                         */
        uint8_t * pPKI_N,                                                               /**< [In] If private key included, Modulus N where NLen is PKI_NLen. The most
                                                                                         *        significant 32-bit word of N shall not be equal to zero.
                                                                                         */
        uint8_t * pPKI_e,                                                               /**< [In] If private key included, Public exponent e where eLen is PKI_eLen. It
                                                                                         *        must be an odd integer.
                                                                                         */
        uint8_t * pPKI_p,                                                               /**< [In] If private key included, Prime p where pLen is PKI_pLen. The MSB of p
                                                                                         *        shall not be equal to zero.
                                                                                         */
        uint8_t * pPKI_q,                                                               /**< [In] If private key included, Prime q where pLen is PKI_qLen. The MSB of q
                                                                                         *        shall not be equal to zero.
                                                                                         */
        uint8_t * pPKI_dP,                                                              /**< [In] If private key included, Private exponent d_p where pLen is PKI_pLen. */
        uint8_t * pPKI_dQ,                                                              /**< [In] If private key included, Private exponent d_q where qLen is PKI_qLen. */
        uint8_t * pPKI_ipq                                                              /**< [In] If private key included, Inverse p-1mod(q) where qLen is PKI_qLen. */
    );

/**
 * \brief Export a full RSA key entry (i.e including the private key if present).
 * A successful host authentication in the LC using SAM_AuthenticateHost with a Host Key is required to
 * execute the PKI_ExportPrivateKey command. This needs to be done with the Host Key referenced by the
 * PKI_KeyNoCEK and PKI_KeyVCEK of the targeted key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful. More data available to receive from SAM.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyDataLen is NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ExportPrivateKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option for AEK selection and differentiating between first part
                                                                                         *        and last part of data.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_RECEIVE_AEK "Access Entry Key"
                                                                                         *                (PKI_KeyNoAEK, PKI_KeyVAEK) received from SAM
                                                                                         *
                                                                                         *         - Should be combined with the above option
                                                                                         *             - \ref PH_EXCHANGE_DEFAULT "Receiving the first part of data"
                                                                                         *             - \ref PH_EXCHANGE_RXCHAINING "Receiving the intermediate or
                                                                                         *                    final part of data"
                                                                                         */
        uint8_t bPKI_KeyNo,                                                             /**< [In] Key reference number of the PKI Key Entry (00h to 01h). */
        uint8_t ** ppKeyData,                                                           /**< [Out] Received Private Key information from SAM. */
        uint16_t * pKeyDataLen                                                          /**< [Out] Length of bytes available in \b ppKeyData buffer. */
    );

/**
 * \brief Export the public part of an RSA key pair.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful. More data available to receive from SAM.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyDataLen is NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ExportPublicKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option for AEK selection and differentiating between first part
                                                                                         *        and last part of data.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_RECEIVE_AEK "Access Entry Key"
                                                                                         *                (PKI_KeyNoAEK, PKI_KeyVAEK) received from SAM
                                                                                         *
                                                                                         *         - Should be combined with the above option
                                                                                         *             - \ref PH_EXCHANGE_DEFAULT "Receiving" the first part of data
                                                                                         *             - \ref PH_EXCHANGE_RXCHAINING "Receiving" the intermediate or
                                                                                         *                    final part of data
                                                                                         */
        uint8_t bPKI_KeyNo,                                                             /**< [In] Key reference number of the PKI Key Entry (00h to 02h). */
        uint8_t ** ppKeyData,                                                           /**< [Out] Received Public Key information from SAM. */
        uint16_t * pKeyDataLen                                                          /**< [Out] Length of bytes available in \b ppKeyData buffer. */
    );

/**
 * \brief The PKI_UpdateKeyEntries can be used to change key entries of the symmetric key storage (KST).
 *  - Executing this command does not use the protection based on the Key Access Control, hence the addressed KST Key Entrys
 *    change entry key (KeyNoCEK) is ignored. Instead the commands execution is protected by asymmetric techniques
 *    using the PKI support of the SAM.
 *  - The SAM supports the update of up to three key entries with PKI_UpdateKeyEntries. The SAM will limit the number of key
 *    entries according to the key size and the hashing function used for OAEP padding.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyFrame and \b pUpdateACKLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_UpdateKeyEntries(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option for AEK selection and differentiating between
                                                                                         *        first part and last part of data.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_RECEIVE_ACK "Acknowledge"
                                                                                         *                key number. Option to include Le byte and
                                                                                         *                Acknowledge key number to command frame for
                                                                                         *                receiving the UpdateAck data from Sam hardware.
                                                                                         *
                                                                                         *         - Should be combined with the above option
                                                                                         *             - \ref PH_EXCHANGE_DEFAULT "Exchange and receive"
                                                                                         *                    information from SAM
                                                                                         *             - \ref PH_EXCHANGE_RXCHAINING "To be used if"
                                                                                         *                    \ref PH_ERR_SUCCESS_CHAINING "Chaining"
                                                                                         *                    status is provided by Library
                                                                                         */
        uint8_t bNoOfKeyEntries,                                                        /**< [In] Number of key entries to include in the cryptogram
                                                                                         *         - 0x00: RFU
                                                                                         *         - 0x01: 1 Key Entry
                                                                                         *         - 0x02: 2 Key Entry
                                                                                         *         - 0x03: 3 Key Entry
                                                                                         */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint8_t bPKI_KeyNo_Enc,                                                         /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for decryption (00h to 01h).
                                                                                         */
        uint8_t bPKI_KeyNo_Sign,                                                        /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for signature verification (00h to 02h).
                                                                                         */
        uint8_t bPKI_KeyNo_Ack,                                                         /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for acknowledge signature generation (00h to 01h).
                                                                                         */
        uint8_t * pKeyFrame,                                                            /**< [In] Buffer containing the RSA encrypted key entries and the
                                                                                         *        signature (EncKeyFrame || Signature).
                                                                                         *          - EncKeyFrame: RSA encrypted key frame as
                                                                                         *            RSA_E ( KPKI_KeyNo_Enc , Change_Ctr || KeyNo1 || ProMas1
                                                                                         *            || NewEntry1 [|| KeyNo2 || ProMas2 || NewEntry2 [|| KeyNo3
                                                                                         *            || ProMas3 || New Entry3]]) where LenE is the PKI_NLen of
                                                                                         *            the PKI_KeyNo_Enc key entry.
                                                                                         *          - Signature: RSA digital signature as
                                                                                         *            RSA_S ( KPKI_KeyNo_Sign , PKI_KeyNo_Enc || PKI_KeyNo_Sign ||
                                                                                         *            EncKeyFrame) where LenS is the PKI_NLen of the PKI_KeyNo_Sign
                                                                                         *            key entry
                                                                                         */
        uint16_t wKeyFrameLen,                                                          /**< [In] Length of bytes available in \b wKeyFrameLen buffer. */
        uint8_t ** ppUpdateACK,                                                         /**< [Out] Buffer containing the RSA encrypted Acknowledge signature. */
        uint16_t * pUpdateACKLen                                                        /**< [Out] Length of bytes available in \b ppUpdateACK buffer. */
    );

/**
 * \brief Prepare a cryptogram (according to Asymmetric Offline Change Cryptogram) for the PKI offline update of
 * KST key entries on a target SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyEntries, \b pDivInput and \b pEncKeyFrame_Sign_Len are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_EncipherKeyEntries(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option to differentiate between the command frame to be exchanged.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Receive" first part of
                                                                                         *                 information from SAM
                                                                                         *          - \ref PH_EXCHANGE_RXCHAINING "Receive" intermediate or final part
                                                                                         *                 of information from SAM
                                                                                         *
                                                                                         *          - For enabling or disabling of key diversification. Should be
                                                                                         *            combined with the above options
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_PKI_DIVERSIFICATION_ON "Diversification"
                                                                                         *                enabled
                                                                                         */
        uint8_t bNoOfKeyEntries,                                                        /**< [In] Number of key entries to include in the cryptogram
                                                                                         *         - 0x00: RFU
                                                                                         *         - 0x01: 1 Key Entry
                                                                                         *         - 0x02: 2 Key Entry
                                                                                         *         - 0x03: 3 Key Entry
                                                                                         */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint8_t bPKI_KeyNo_Enc,                                                         /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for the cryptogram encryption (00h to 02h).
                                                                                         */
        uint8_t bPKI_KeyNo_Sign,                                                        /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for the cryptogram signature generation (00h to 01h).
                                                                                         */
        uint8_t bPKI_KeyNo_Dec,                                                         /**< [In] Key reference number of the PKI Key Entry that will be
                                                                                         *        used for decryption in the target SAM (00h to 01h).
                                                                                         */
        uint8_t bPKI_KeyNo_Verif,                                                       /**< [In] Key reference number of the PKI Key Entry that will be
                                                                                         *        used for signature verification in the target SAM (00h to 02h).
                                                                                         */
        uint16_t wPerso_Ctr,                                                            /**< [In] Targeted offline change counter data. */
        uint8_t * pKeyEntries,                                                          /**< [In] Key entry descriptions
                                                                                         *         - PersoKeyNo: Key reference number of the KST Key Entry
                                                                                         *           to include in the cryptogram
                                                                                         *              - NVM Key: 0x00 - 0x7F
                                                                                         *              - RAM Key: 0x#0 - 0xE3
                                                                                         *         - KeyNo: Key reference number of the KST Key Entry in the
                                                                                         *           target SAM (0x00 - 0x7F)
                                                                                         *
                                                                                         */
        uint8_t bKeyEntriesLen,                                                         /**< [In] Length of bytes available in \b pKeyEntries buffer. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification input for key diversification. (1 to 31 byte(s) input). */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppEncKeyFrame_Sign,                                                  /**< [Out] The Encrypted Key frame and Signature as returned by Sam hardware.
                                                                                         *          - EncKeyFrame: RSA encrypted key frame as RSA_E ( KPKI_KeyNo_Enc,
                                                                                         *            Perso_Ctr || KeyNo1 || ProMas1 || NewEntry1 [|| KeyNo2 || ProMas2
                                                                                         *            || NewEntry2 [|| KeyNo3 || ProMas3 || New Entry3]]) where LenE is
                                                                                         *            the PKI_NLen of the PKI_KeyNo_Enc key entry
                                                                                         *          - Signature: RSA digital signature as RSA_S ( KPKI_KeyNo_Sign,
                                                                                         *            PKI_KeyNo_Dec || PKI_KeyNo_Verif || EncKeyFrame) where LenS is
                                                                                         *            the PKI_NLen of the PKI_KeyNo_Sign key entry
                                                                                         */
        uint16_t * pEncKeyFrame_Sign_Len                                                /**< [Out] Length of bytes available in \b ppEncKeyFrame_Sign buffer. */
    );

/**
 * \brief Generate Hash DataFrame from Data.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMessage and \b pHashLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateHash(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering Options:
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Exchange Message and Receive
                                                                                         *                 Hash from SAM"
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Fist part of message".
                                                                                         *                 Information is not exchanged to SAM.
                                                                                         *          - \ref #PH_EXCHANGE_BUFFER_CONT "Buffer intermediate message".
                                                                                         *                 Information is not exchanged to SAM.
                                                                                         *          - \ref #PH_EXCHANGE_BUFFER_LAST "Buffer last message".
                                                                                         *                 Information is exchanged to SAM and Hash is received.
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING "The LFI is set to AFh if the flag"
                                                                                         *
                                                                                         *          - On the first frame of the command chain the flag
                                                                                         *            \ref PHHAL_HW_SAM_CMD_PKI_FIRST_FRAME "Exchange overall length"
                                                                                         *            has to be set to exchange HashAlgo and length of the overall message.
                                                                                         */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint32_t dwMLen,                                                                /**< [In] Overall message length (4 byte). */
        uint8_t * pMessage,                                                             /**< [In] Message chunk to be hashed. */
        uint16_t wMsgLen,                                                               /**< [In] Length of bytes available in \b pMessage buffer. */
        uint8_t ** ppHash,                                                              /**< [Out] Buffer containing the hash after sending the last message chunk. */
        uint16_t * pHashLen                                                             /**< [Out] Length of bytes available in \b pHashLen buffer. */
);

/**
 * \brief Generate a signature with a given RSA key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pHash is NULL
 */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateSignature(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint8_t bPKI_KeyNo_Sign,                                                        /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for the cryptogram signature generation (00h to 01h).
                                                                                         */
        uint8_t * pHash,                                                                /**< [In] Hash message to be signed. */
        uint8_t bHashLen                                                                /**< [In] Length of bytes available in \b pHash buffer. */
    );

/**
 * \brief Get a previously generated signature.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pHash is NULL
 */
phStatus_t phhalHw_Sam_Cmd_PKI_SendSignature(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t ** ppSignature,                                                         /**< [Out] The signature received from Sam hardware. */
        uint16_t * pSignatureLen                                                        /**< [Out] Length of bytes available in \b ppSignature buffer. */
    );

/**
 * \brief Verify a hash / signature pair with a given RSA key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pHash and \b pSignature are NULL
 */
phStatus_t phhalHw_Sam_Cmd_PKI_VerifySignature(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPKI_KeyNo_Verif,                                                       /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for the cryptogram signature verification (00h to 02h)
                                                                                         */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint8_t * pHash,                                                                /**< [In] Hash message to be signed. */
        uint8_t bHashLen,                                                               /**< [In] Length of bytes available in \b pHash buffer. */
        uint8_t * pSignature,                                                           /**< [In] RSA digital signature where NLen is the PKI_NLen of
                                                                                         *        the PKI_KeyNo key entry
                                                                                         */
        uint16_t wSignatureLen                                                          /**< [In] Length of bytes available in \b pSignature buffer. */
    );

/**
 * \brief Performs the offline encryption of plain RSA data.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPlainData and \b pEncDataLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_EncipherData(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint8_t bPKI_KeyNo_Enc,                                                         /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for encryption (00h to 02h).
                                                                                         */
        uint8_t * pPlainData,                                                           /**< [In] RSA Plain Data to be encrypted where mLen (\b wPlainDataLen)
                                                                                         *        is restricted by the PKI_NLen of the PKI_KeyNo_Enc key entry
                                                                                         *        and the output length of the hash function
                                                                                         */
        uint8_t bPlainDataLen,                                                          /**< [In] Length of bytes available in \b pPlainData buffer. */
        uint8_t ** ppEncData,                                                           /**< [Out] RSA encrypted data as
                                                                                         *         EncData = RSA_E ( KPKI_KeyNo_Enc , PlainData)
                                                                                         *         where LenE is the PKI_NLen of the PKI_KeyNo_Enc
                                                                                         *         key entry.
                                                                                         */
        uint16_t * pEncDataLen                                                          /**< [Out] Length of bytes available in \b ppEncData buffer. */
    );

/**
 * \brief Performs the offline decryption of encrypted RSA data.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pEncData and \b pPlainDataLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_DecipherData(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option parameter:
                                                                                         *          - Local buffering is implemented by the flags
                                                                                         *              - \ref PH_EXCHANGE_DEFAULT "Exchange information to SAM"
                                                                                         *              - \ref PH_EXCHANGE_TXCHAINING "The LFI is set to AFh if the flag"
                                                                                         *              - Usage:
                                                                                         *                  - For chained data
                                                                                         *                      - For First Frame: Use flags
                                                                                         *                        \ref PHHAL_HW_SAM_CMD_PKI_FIRST_FRAME "First Frame" |
                                                                                         *                        \ref PH_EXCHANGE_TXCHAINING "Chained Frame"
                                                                                         *                      - For Next N Frame(s): Use flag \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                        "Chained Frame"
                                                                                         *                      - For Last Frame: Use flags \ref PH_EXCHANGE_DEFAULT "Last Frame"
                                                                                         *
                                                                                         *                  - For non chained data i.e. only single frame use
                                                                                         *                    \ref PHHAL_HW_SAM_CMD_PKI_FIRST_FRAME "First Frame" |
                                                                                         *                    \ref PH_EXCHANGE_DEFAULT "Exchange information to SAM"
                                                                                         */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         */
        uint8_t bPKI_KeyNo_Dec,                                                         /**< [In] Key reference number of the PKI Key Entry to be used
                                                                                         *        for decryption (00h to 01h).
                                                                                         */
        uint8_t * pEncData,                                                             /**< [In] RSA encrypted data as
                                                                                         *        EncData = RSA_E ( KPKI_KeyNo_Enc , PlainData) where
                                                                                         *        LenE is the PKI_NLen of the PKI_KeyNo_Enc key entry
                                                                                         */
        uint16_t wEncDataLen,                                                           /**< [In] Length of bytes available in \b pEncData buffer. */
        uint8_t ** ppPlainData,                                                         /**< [In] RSA decrypted data as
                                                                                         *        PlainData= RSA_D ( KPKI_KeyNo_Enc , EncData)
                                                                                         *        where mLen is limited by the PKI_NLen of the PKI_Key
                                                                                         *        No_Enc key entry and the output length of the hash
                                                                                         *        function
                                                                                         */
        uint16_t * pPlainDataLen                                                        /**< [In] Length of bytes available in \b ppPlainData buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_PKI_RSA
 * @}
 */






/** \defgroup phhalHw_Sam_Cmd_PKI_ECC ECC
 * \brief SAM commands used for asymmetric ECC key management, signature handling and verification.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_ECC_Defines Defines
 * \brief Definitions for SAM PKI ECC commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_ECC_Defines_CommandCodes CommandCodes
 * \brief Definitions for SAM PKI ECC command codes.
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_ECC_KEY                               0x3EU   /**< Sam Instruction code for PKI_GenerateEccKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_KEY                                 0x21U   /**< Sam Instruction code for PKI_ImportEccKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_CURVE                               0x22U   /**< Sam Instruction code for PKI_ImportEccCurve command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_ECC_PRIVATE_KEY                         0x3FU   /**< Sam Instruction code for PKI_ExportEccPrivateKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_ECC_PUBLIC_KEY                          0x23U   /**< Sam Instruction code for PKI_ExportEccPublicKey command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_ECC_SIGNATURE                         0x39U   /**< Sam Instruction code for PKI_GenerateEccSignature command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_VERIFY_ECC_SIGNATURE                           0x20U   /**< Sam Instruction code for PKI_VerifyEccSignature command. */
#define PHHAL_HW_SAM_CMD_INS_PKI_VALIDATE_ECC_CERTIFICATE                       0x3BU   /**< Sam Instruction code for PKI_ValidateEccCert command. */
/**
 * end of group phhalHw_Sam_Cmd_PKI_ECC_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_PKI_ECC_Defines_CertFormat CertificateFormat
 * \name Definitions for SAM Public Key Infrastructure Certificate Format. To be used
 * with \ref phhalHw_Sam_Cmd_PKI_ValidateEccCert "Validate ECC Certificate" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_PKI_CERT_FORMAT_X509                                   0x00U   /**< Option mask for certificate format as X.509 certificate */
#define PHHAL_HW_SAM_CMD_PKI_CERT_FORMAT_EV_CHARGING                            0x01U   /**< Option mask for certificate format as EV charging GP
                                                                                         *   certificate
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_PKI_ECC_Defines_CertFormat
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_PKI_ECC_Defines
 * @}
 */

/**
 * \brief PKI_GenerateEccKey creates a pair of a public and a private ECC key. To make the key usable,
 * Bit 0 of ECC_SET must be unset (that is disable key entry is false). ECC_SET Bit 1 must be set,
 * indicating a private key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pECC_xy is NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateEccKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number of the ECC key entry to be
                                                                                         *        update.
                                                                                         *          - NVM key: 0x00 - 0x0F
                                                                                         *          - RAM key: 0xE0 - 0xE3
                                                                                         */
        uint16_t wECC_Set,                                                              /**< [In] Configuration settings of the key entry. */
        uint8_t bECC_KeyNoCEK,                                                          /**< [In] Key reference number of change entry key.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bECC_KeyVCEK,                                                           /**< [In] Key version of change entry key. */
        uint8_t bECC_RefNoKUC,                                                          /**< [In] Reference number of key usage counter. */
        uint8_t bECC_KeyNoAEK,                                                          /**< [In] Key reference number of access entry key.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bECC_KeyVAEK,                                                           /**< [In] Key version of access entry key. */
        uint8_t bECC_RefNoCurve,                                                        /**< [In] Reference of ECC Curve. */
        uint8_t ** ppECC_xy,                                                            /**< [Out] Public key point coordinate. */
        uint16_t * pECC_xyLen                                                           /**< [Out] Length of bytes available in \b ppECC_xy buffer. */
    );

/**
 * \brief The PKI_ImportEccKey imports an ECC private or public key in the Key Storage. The command can also
 * be used to only update the key settings (that is ECC_SET, ECC_KeyNoCEK, ECC_KeyVCEK, ECC_RefNoKUC,
 * ECC_KeyNoAEK and ECC_KeyVAEK and (optionally for public keys) CRLFile and/or Issuer). This can be
 * done for example to temporarily disable a key entry. This is indicated by P1.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pECC_xy is NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ImportEccKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option for P1 information byte.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_VALUE
                                                                                         *              "Update key settings and key value"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY
                                                                                         *              "Update only key settings" (ECC_KeyNo, ECC_SET,
                                                                                         *               ECC_KeyNoCEK, ECC_KeyVCEK, ECC_RefNoKUC,
                                                                                         *               ECC_KeyNoAEK, ECC_KeyVAEK and Optionally
                                                                                         *               Issuer)
                                                                                         */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number of the ECC key entry to be
                                                                                         *        update.
                                                                                         *          - NVM key: 0x00 - 0x0F
                                                                                         *          - RAM key: 0xE0 - 0xE3
                                                                                         */
        uint16_t wECC_Set,                                                              /**< [In] Configuration settings of the key entry. */
        uint8_t bECC_KeyNoCEK,                                                          /**< [In] Key reference number of change entry key.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bECC_KeyVCEK,                                                           /**< [In] Key version of change entry key. */
        uint8_t bECC_RefNoKUC,                                                          /**< [In] Reference number of key usage counter. */
        uint8_t bECC_KeyNoAEK,                                                          /**< [In] Key reference number of access entry key.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bECC_KeyVAEK,                                                           /**< [In] Key version of access entry key. */
        uint16_t wECC_Len,                                                              /**< [In] ECC bit field size in bytes. */
        uint8_t * pECC_KeyValue,                                                        /**< [In] Private or Public key information.
                                                                                         *          - Exchanged if P1 = 0x00 and ECC_SET Bit 1 is not set,
                                                                                         *            Public key (ECC_xy) point coordinates should be
                                                                                         *            provided.
                                                                                         *          - Exchanged if P1 = 0x00 and ECC_SET Bit 1 is set,
                                                                                         *            Private key (ECC_priv) scalar should be provided.
                                                                                         */
        uint8_t bECC_KeyValueLen,                                                       /**< [In] Length of bytes available in \b pECC_KeyValue buffer. */
        uint8_t bECC_RefNoCurve,                                                        /**< [In] Reference of ECC Curve. Exchanged if P1 = 0x00 and
                                                                                         *        ECC_SET Bit 1 is set
                                                                                         */
        uint8_t bCRLFile,                                                               /**< [Out] CRL File. Exchanged if ECC_SET Bit 12 is set and value
                                                                                         *         will be from 00 - 0F
                                                                                         */
        uint8_t * pIssuer,                                                              /**< [In] Trusted issuer name. Exchanged if ECC_SET Bit 12 is set */
        uint8_t bIssuerLen                                                              /**< [In] Length of bytes available in \b pIssuer buffer. */
    );

/**
 * \brief The PKI_ImportEccCurve imports a full ECC Curve description in the ECC Curve Storage Table.
 * The command can also be used to only update the curve settings (that is ECC_KeyNoCCK and
 * ECC_KeyVCCK). This is indicated by P1.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pECC_Prime, \b pECC_ParamA, \b pECC_ParamB,
 *                                      \b pECC_Px, \b pECC_Py and \b pECC_Order are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ImportEccCurve(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option for P1 information byte.
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_VALUE
                                                                                         *              "Update curve settings and curve value"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY
                                                                                         *              "Update only curve settings" (ECC_CurveNo,
                                                                                         *               ECC_KeyNoCCK and ECC_KeyVCCK)
                                                                                         */
        uint8_t bECC_CurveNo,                                                           /**< [In] Curve reference number of the ECC curve entry to
                                                                                         *        update (00h to 03h).
                                                                                         */
        uint8_t bECC_KeyNoCCK,                                                          /**< [In] Key reference number to change the curve entry.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t bECC_KeyVCCK,                                                           /**< [In] Key version to change curve entry. */
        uint8_t bECC_N,                                                                 /**< [In] Size of the field in bytes. Ranges from 16 to 32 bytes. */
        uint8_t bECC_M,                                                                 /**< [In] Size of the order in bytes. Ranges from 16 to 32 bytes. */
        uint8_t * pECC_Prime,                                                           /**< [In] Prime, field definition: ECC_N bytes. */
        uint8_t * pECC_ParamA,                                                          /**< [In] Curve parameter (a): ECC_N bytes. */
        uint8_t * pECC_ParamB,                                                          /**< [In] Curve parameter (b): ECC_N bytes. */
        uint8_t * pECC_Px,                                                              /**< [In] x-coordinate of base point: ECC_N bytes. */
        uint8_t * pECC_Py,                                                              /**< [In] y-coordinate of base point: ECC_N bytes. */
        uint8_t * pECC_Order                                                            /**< [In] Order of base point: ECC_M bytes. */
    );

/**
 * \brief The PKI_ExportEccPrivateKey exports a private ECC key entry, including the private key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pECC_Set, \b pECC_KeyNoCEK, \b pECC_KeyVCEK,
 *                                      \b pECC_RefNoKUC, \b pECC_KeyNoAEK, \b pECC_KeyVAEK
 *                                      \b pECC_Len, \b pECC_RefNoCurve, and \b pECC_PrivLen
 *                                      are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ExportEccPrivateKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number (ECC_KeyNo) of the ECC
                                                                                         *        key entry to be exported (00h to 07h).
                                                                                         */
        uint16_t * pECC_Set,                                                            /**< [Out] Configuration settings of the key entry. */
        uint8_t * pECC_KeyNoCEK,                                                        /**< [Out] Key reference number of change entry key.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t * pECC_KeyVCEK,                                                         /**< [Out] Key version of change entry key. */
        uint8_t * pECC_RefNoKUC,                                                        /**< [Out] Reference number of key usage counter. */
        uint8_t * pECC_KeyNoAEK,                                                        /**< [Out] Key reference number of access entry key.
                                                                                         *          - 0xFE       : No Access Restrictions
                                                                                         *          - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t * pECC_KeyVAEK,                                                         /**< [Out] Key version of access entry key. */
        uint16_t * pECC_Len,                                                            /**< [Out] ECC bit field size in bytes. */
        uint8_t * pECC_RefNoCurve,                                                      /**< [In] Reference of ECC Curve. */
        uint8_t ** ppECC_Priv,                                                          /**< [Out] Private key scalar. Ranges from 16 - 80 bytes. */
        uint8_t * pECC_PrivLen                                                          /**< [Out] Length of bytes available in \b ppECC_Priv buffer. */
    );

/**
 * \brief The PKI_ExportEccPublicKey exports the public part of ECC key.
 *
 *  - If a private key entry is targeted, the public key is computed by the SAM on the fly.
 *  - If a public key entry is targeted, the CRLFile number is returned, if any CRLFile is
 *    associated with the targeted key entry. If the targeted key entry is associated with
 *    a trusted Issuer, this is never returned as it is only stored hashed internally in
 *    the SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pECC_Set, \b pECC_KeyNoCEK, \b pECC_KeyVCEK,
 *                                      \b pECC_RefNoKUC, \b pECC_KeyNoAEK, \b pECC_KeyVAEK
 *                                      \b pECC_Len, \b pCRLFile and \b pECC_xyLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ExportEccPublicKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number (ECC_KeyNo) of the ECC
                                                                                         *        key entry to be exported (00h to 07h).
                                                                                         */
        uint16_t * pECC_Set,                                                            /**< [Out] Configuration settings of the key entry. */
        uint8_t * pECC_KeyNoCEK,                                                        /**< [Out] Key reference number of change entry key.
                                                                                         *         - 0xFE       : No Restrictions
                                                                                         *         - 0xFF       : Entry Locked
                                                                                         *         - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t * pECC_KeyVCEK,                                                         /**< [Out] Key version of change entry key. */
        uint8_t * pECC_RefNoKUC,                                                        /**< [Out] Reference number of key usage counter. */
        uint8_t * pECC_KeyNoAEK,                                                        /**< [Out] Key reference number of access entry key.
                                                                                         *          - 0xFE       : No Access Restrictions
                                                                                         *          - 0x00 - 0x7F: Restricted to specific permanent KST Key Entry
                                                                                         */
        uint8_t * pECC_KeyVAEK,                                                         /**< [Out] Key version of access entry key. */
        uint16_t * pECC_Len,                                                            /**< [Out] ECC bit field size in bytes. */
        uint8_t ** ppECC_xy,                                                            /**< [Out] Public key point coordinate. Ranges from 33 - 65 bytes. */
        uint8_t * pECC_xyLen,                                                           /**< [Out] Length of bytes available in \b ppECC_xy buffer. */
        uint8_t * pCRLFile                                                              /**< [Out] CRL File.
                                                                                         *          - Exchanged if ECC_SET Bit 12 is set and value will be
                                                                                         *            from 00 - 0F
                                                                                         *          - Will be 0xFF otherwise
                                                                                         */
    );

/**
 * \brief Generate ECC Signature generates a signature on a hash given as input using a
 * private key stored in the ECC Key Entry. The signature is immediately returned in the
 * response as the concatenation of r and s. This response has a length of 2 * ECC_Len of
 * the targeted ECC Key Entry, i.e. twice the curve size.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pHash and \b pSigLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateEccSignature(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bHashingAlg,                                                            /**< [In] Hashing algorithm selection (for padding MGFs and
                                                                                         *        digital signature).
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_1
                                                                                         *                "SHA-1"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_224
                                                                                         *                "SHA-224"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_256
                                                                                         *                "SHA-256"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_384
                                                                                         *                "SHA-384"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_HASH_ALGORITHM_SHA_512
                                                                                         *                "SHA-512"
                                                                                         */
        uint8_t bECC_KeyNo_Sign,                                                        /**< [In] The key reference number of the ECC key entry to be used for
                                                                                         *        signature generation.
                                                                                         *          - NVM key: 0x00 - 0x0F
                                                                                         *          - RAM key: 0xE0 - 0xE3
                                                                                         */
        uint8_t * pHash,                                                                /**< [In] Hash message to be signed. */
        uint8_t bHashLen,                                                               /**< [In] Length of bytes available in \b pHash buffer. */
        uint8_t ** ppSignature,                                                         /**< [In] Generated signature (r, s) with an actual length of
                                                                                         *        2 * ECC_Len of the targeted ECC key entry.
                                                                                         */
        uint16_t * pSigLen                                                              /**< [In] Length of bytes available in \b ppSignature buffer. */
    );

/**
 * \brief The PKI_VerifyEccSignature is used to support the originality check architecture. The originality
 * check allows verification of the genuineness of NXP chips after manufacturing. The command verifies the
 * correctness of an ECC signature (for example: NXPOriginalitySignature) obtained from the product to verify.
 * The signature is computed according to Elliptic Curve DSA (ECDSA). In case of originality checking, the
 * PKI_VerifyEccSignature cryptographic parameters, such as the input message (M), the selected ECC curve
 * and the required public key depend on the type of product for which the verification is performed.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMessage and \b pSignature are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_VerifyEccSignature(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number of the ECC key entry to be used for
                                                                                         *        signature verification (00h to 07h).
                                                                                         */
        uint8_t bECC_CurveNo,                                                           /**< [In] The curve reference number of the ECC curve entry to be used
                                                                                         *        for signature verification (00h to 03h).
                                                                                         */
        uint8_t bLen,                                                                   /**< [In] Length in bytes of the message to verify. */
        uint8_t * pMessage,                                                             /**< [In] Signed input data. */
        uint8_t * pSignature,                                                           /**< [In] The ECC digital signature where N is 2 * ECC_Len of ECC_KeyNo
                                                                                         *        key entry.
                                                                                         */
        uint16_t wSignatureLen                                                          /**< [In] Length of bytes available in \b pSignature buffer. */
    );

/**
 * \brief The PKI_ValidateEccCert command supports certificate validation in the context of the following
 * non-exhaustive list of use cases
 *  - ECC-based Card-Unilateral Authentication supporting the MIFARE DUOX ISOInternalAuthenticate
 *  - EV charging protocol supporting the MIFARE DUOX VDE_ECDSA_Sign
 *  - Secure Dynamic Messaging, i.e. the SUN-feature, with ECDSA signature as supported by MIFARE DUOX
 *  - Transaction Signature with ECDSA signature as supported by MIFARE DUOX
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pCertificate is NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PKI_ValidateEccCert(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCertFormat,                                                            /**< [In] Format of the Certificate. This will be used for P1
                                                                                         *        parameter information
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_CERT_FORMAT_X509
                                                                                         *                "X.509 certificate"
                                                                                         *         - \ref PHHAL_HW_SAM_CMD_PKI_CERT_FORMAT_EV_CHARGING
                                                                                         *                "EV charging GP certificate"
                                                                                         */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number of the ECC key entry to be used for
                                                                                         *        signature verification.
                                                                                         *          - NVM key: 0x00 - 0x0F
                                                                                         *          - RAM key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bECC_CurveNo,                                                           /**< [In] Curve reference number of the ECC curve entry to
                                                                                         *        update (00h to 03h).
                                                                                         */
        uint8_t bECC_Target,                                                            /**< [In] ECC key entry targeted to store the extracted public key.
                                                                                         *          - NVM key: 0x00 - 0x0F
                                                                                         *          - RAM key: 0xE0 - 0xE3
                                                                                         */
        uint8_t * pCertificate,                                                         /**< [In] Certificate to be validated */
        uint16_t wCertLen                                                               /**< [In] Length of bytes available in \b pCertificate buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_PKI_ECC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_PKI
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_PKI_H */
