/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE DESFire command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_DESFire.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

/* S - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t bAuthMode, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pCardResponse, uint8_t bCardRespLen,
    uint8_t ** ppSamResponse, uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOption & PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticatePICC part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PICC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Append the Key number and Key version. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Add Auth Mode if EV2 Authentication. */
    if(bOption & PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bAuthMode;
    }

    /* Buffer the first part of command data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer the cards response to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pCardResponse,
        bCardRespLen,
        NULL,
        NULL));

    /* Buffer diversification input informationto exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to Sam hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppSamResponse,
        pSamRespLen);

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Return the response received from Sam hardware. */
    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPiccErrorCode,
    uint8_t * pCardResponse, uint8_t bCardRespLen, uint8_t * pPDcap2, uint8_t * pPCDcap2, uint8_t * pStatusCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(!bPiccErrorCode) PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pStatusCode, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticatePICC part 2 command  information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PICC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* LC can be of 1 byte only in case of PICC error */
    if(bPiccErrorCode != PHHAL_HW_SAM_DESFIRE_PICC_SUCCESS_STATUS_CODE)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPiccErrorCode;
    }

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer card response to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pCardResponse,
        bCardRespLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Append LE in either of below 2 cases,
     * Case-1 : In case of error response code from PICC. In this case SAM echoes the DESFire error Status code.
     * Case-2 : In case of EV2 First Authentication is performed. 12 bytes of PCdCap and PDCap data is returned
     */
    if((bPiccErrorCode != PHHAL_HW_SAM_DESFIRE_PICC_SUCCESS_STATUS_CODE) ||
        (bCardRespLen == PHHAL_HW_SAM_DESFIRE_PICC_RESPONSE_LEN))
    {
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            &pResponse,
            &wRespLen);

        /* Extract the PICC error code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
        {
            *pStatusCode = pResponse[0];
        }

        /* Extract PCD and PD capabilities for EV2 authentication type. */
        if(((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS) &&
            (wRespLen == PHHAL_HW_SAM_DESFIRE_SAM_RESPONSE_LEN))
        {
            /* Assign the PCD and PD capabilities to the parameter. */
            memcpy(pPDcap2, &pResponse[0], PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN);
            memcpy(pPCDcap2, &pResponse[PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN],
                PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN);
        }
    }
    else
    {
        /*  No Le byte, Exchange the rest buffered data */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0x00,
            &pResponse,
            &wRespLen));
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pCardResponse, uint8_t bCardRespLen,
    uint8_t ** ppSamResponse, uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOption & PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_IsoAuthenticatePICC part 1 command  information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_ISO_AUTHENTICATE_PICC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Append the Key number and Key version. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Buffer command informationto exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer card response to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pCardResponse,
        bCardRespLen,
        NULL,
        NULL));

    /* Buffer diversification input information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered inforamtionto SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppSamResponse,
        pSamRespLen);

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Return the response received from Sam hardware. */
    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pCardResponse,
    uint8_t bCardRespLen)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_IsoAuthenticatePICC part 2 command inforamtion. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_ISO_AUTHENTICATE_PICC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer card response to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pCardResponse,
        bCardRespLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Exchange the buffered informationto Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        NULL,
        0,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyPICC(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCryptoMethod, uint8_t bConfig,
    uint8_t bKeySetNo, uint8_t bDFKeyNo, uint8_t bCurrKeyNo, uint8_t bCurrKeyVer, uint8_t bNewKeyNo, uint8_t bNewKeyVer,
    uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t ** ppSamResponse, uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDivInputLen != 0)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ChangeKeyPICC command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_PICC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCryptoMethod;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bConfig;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add KeySetNo and DFKeyno for command type as ChangeKeyEV2. */
    if(bConfig & PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeySetNo;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDFKeyNo;
    }
    else
    {
        /* Append the key number. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] |= bDFKeyNo;
    }

    /* Add key information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCurrKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCurrKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNewKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNewKeyVer;

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer diversification information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppSamResponse,
        pSamRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_CreateTMFilePICC(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pTMIExclFileMap,
    uint8_t bTMKeyOptions, uint8_t bTSIGKeyNo, uint8_t * pDivInput, uint8_t bDivInputLen,
    uint8_t ** ppSamResponse, uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pAccessRights, PH_COMP_HAL);
    if(bFileOption & 0x40U)
        PH_ASSERT_NULL_PARAM(pTMIExclFileMap, PH_COMP_HAL);
    if(bOption != 0)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_CreateTMFilePICC command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CREATE_TM_FILE_PICC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the payload information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFileNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFileOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAccessRights[0];
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAccessRights[1U];

    /* Add TMI Exclusion File Map to command buffer. */
    if(bFileOption & 0x40U)
    {
        (void) memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pTMIExclFileMap, 4U);
        bCmdLen += 4U;
    }

    /* Add TMI Options to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bTMKeyOptions;

    /* Add TSigKeyNo to command buffer. */
    if(bTMKeyOptions & 0x80U)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bTSIGKeyNo;
    }

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer diversification input to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppSamResponse,
        pSamRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}





/* X - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_DESFire_AuthenticatePICC(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bISOMode,
    uint8_t bDFKeyNo, uint8_t bKeyNo, uint8_t bKeyVer, uint8_t bPCDcap2InLen, uint8_t * pPCDcap2In, uint8_t * pDivInput,
    uint8_t  bDivInputLen, uint8_t * pPDcap2, uint8_t * pPCDcap2, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOption & PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH)
    {
        if(bPCDcap2InLen && (bPCDcap2InLen != PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_USAGE))
            PH_ASSERT_NULL_PARAM(pPCDcap2In, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pPDcap2, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pPCDcap2, PH_COMP_HAL);
    }
    if(bOption & PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DESFire_AuthenticatePICC command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DESFIRE_AUTHENTICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bISOMode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add key information to comand buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDFKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Add PCDCap length for EV2 authentication type. */
    if(bOption & PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPCDcap2InLen;
    }

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PCDcap information to exchange buffer for EV2 authentication type. */
    if((bOption & PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH) &&
        (bPCDcap2InLen != PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_USAGE))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pPCDcap2In,
            bPCDcap2InLen,
            NULL,
            NULL));
    }

    /* Buffer diversification input information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter in case of Native command set. */
        if(bISOMode == PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE)
        {
            *pPiccReturnCode = pResponse[0];
        }

        /* Assign two byte PICC response code the parameter in case of Native command set. */
        else
        {
            memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN);
        }
    }

    /* Extract PCD and PD capabilities for EV2 authentication type. */
    if(((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS) &&
        (bOption & PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH))
    {
        /* Assign the PCD and PD capabilities to the parameter. */
        memcpy(pPDcap2, &pResponse[0], PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN);
        memcpy(pPCDcap2, &pResponse[PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN],
            PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyCompMeth, uint8_t bConfig,
    uint8_t bKeySetNo, uint8_t bDFKeyNo, uint8_t bCurrKeyNo, uint8_t bCurrKeyVer, uint8_t bNewKeyNo, uint8_t bNewKeyVer,
    uint8_t * pDivInput, uint8_t  bDivInputLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDivInputLen != 0)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DESFire_ChangeKeyPICC command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DESFIRE_CHANGE_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyCompMeth;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bConfig;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add KeySetNo and DFKeyno for command type as ChangeKeyEV2. */
    if(bConfig & PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeySetNo;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDFKeyNo;
    }

    /* Add key information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCurrKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCurrKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNewKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNewKeyVer;

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer diversification information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter in case of Native command set. */
        if(bConfig & PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816)
        {
            memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN);
        }

        /* Assign two byte PICC response code the parameter in case of Native command set. */
        else
        {
            *pPiccReturnCode = pResponse[0];
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DESFire_WriteX(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bCrypto, uint8_t * pData,
    uint8_t bDataLen, uint8_t * pPiccReturnCode, uint8_t * pErrLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST))
    {
        PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pErrLen, PH_COMP_HAL);
    }

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Buffer initial information to exchange buffer. */
    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Frame DESFire_WriteX comamnd information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DESFIRE_WRITE_X;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCrypto;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 information if chaining of data is performed. */
        if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_TXCHAINING)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        }

        /* Buffer command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));
    }

    /* Buffer intermediate data to exchange buffer. */
    if(wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer final information to exchange buffer. */
    else
    {
        /* Buffer data to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Buffer LE and exchange the buffered information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            &pResponse,
            &wRespLen);

        /* Return the chaining code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }

        /* Extract the PICC error code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
        {
            /* Assign one byte PICC response code the parameter in case of Native command set. */
            if(wRespLen == PHHAL_HW_SAM_DESFIRE_PICC_NATIVE_STATUS_LEN)
            {
                *pPiccReturnCode = pResponse[0];
                *pErrLen = PHHAL_HW_SAM_DESFIRE_PICC_NATIVE_STATUS_LEN;
            }

            /* Assign two byte PICC response code the parameter in case of Native command set. */
            else
            {
                memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN);
                *pErrLen = PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN;
            }
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DESFire_ReadX(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bCrypto, uint8_t * pAppData,
    uint8_t bAppDataLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccReturnCode, uint8_t * pErrLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;
    uint8_t    PH_MEMLOC_REM bAppDataOffset = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pErrLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DESFire_ReadX comamnd information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DESFIRE_READ_X;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCrypto;

    /* Append LC in cases of Native Chaining  or First frame. */
    if(!(wOption & PHHAL_HW_CMD_SAM_ISO_CHAINING) || !(wOption & PH_EXCHANGE_RXCHAINING))
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;
    }

    /* If Communication mode is FULL and more data is expected. */
    if((bCrypto == PHHAL_HW_CMD_SAM_COMM_MODE_FULL) &&
        !(wOption & PH_EXCHANGE_RXCHAINING))
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAppData[bAppDataOffset++];
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAppData[bAppDataOffset++];
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAppData[bAppDataOffset++];
    }

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        &pAppData[bAppDataOffset],
        (uint16_t) (bAppDataLen - bAppDataOffset),
        NULL,
        NULL));

    /* Update LC in case of Native Chaining or First frame. */
    if(!(wOption & PHHAL_HW_CMD_SAM_ISO_CHAINING) || !(wOption & PH_EXCHANGE_RXCHAINING))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }

    /* Send Le byte and perform actual exchange */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Return the chaining code. */
    if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE) ||
        ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Update the response buffer with actual data for Success or Success Chaining. */
        *ppResponse = pResponse;
        *pRespLen = wRespLen;

        /* Return chaining status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter in case of Native command set. */
        if(wRespLen == PHHAL_HW_SAM_DESFIRE_PICC_NATIVE_STATUS_LEN)
        {
            *pPiccReturnCode = pResponse[0];
            *pErrLen = PHHAL_HW_SAM_DESFIRE_PICC_NATIVE_STATUS_LEN;
        }

        /* Assign two byte PICC response code the parameter in case of Native command set. */
        else
        {
            memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN);
            *pErrLen = PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN;
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bISOMode,
    uint8_t bKeyNo, uint8_t bKeyVer, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pTMIExclFileMap,
    uint8_t bTMKeyOptions, uint8_t bTSIGKeyNo, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pAccessRights, PH_COMP_HAL);
    if(bFileOption & 0x40U)
        PH_ASSERT_NULL_PARAM(pTMIExclFileMap, PH_COMP_HAL);
    if(bOption != 0)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DESFire_CreateTMFilePICC command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DESFIRE_CREATE_TM_FILE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bISOMode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the payload information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFileNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFileOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAccessRights[0];
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAccessRights[1];

    /* Add TMI Exclusion File Map to command buffer. */
    if(bFileOption & 0x40U)
    {
        (void) memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pTMIExclFileMap, 4U);
        bCmdLen += 4U;
    }

    /* Add TMI Options to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bTMKeyOptions;

    /* Add TSigKeyNo to command buffer. */
    if(bTMKeyOptions & 0x80U)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bTSIGKeyNo;
    }

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer diversification input to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter in case of Native command set. */
        if(bISOMode & PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816)
        {
            memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN);
        }

        /* Assign two byte PICC response code the parameter in case of Native command set. */
        else
        {
            *pPiccReturnCode = pResponse[0];
        }
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
