/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE DUOX command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7360 $
 * $Date: 2025-06-26 16:00:10 +0530 (Thu, 26 Jun 2025) $
 */

#include "phhalHw_Sam_Cmd_DUOX.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

 /* S - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t bECCKeyNo_Priv, uint8_t bCertA_FileNo, uint8_t bCertB_Options, uint8_t bECCKeyNo_CA,
    uint8_t ** ppSamResponse, uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_MutualAuthECC part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_MUTUAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Append the Private Key number. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;

    /* Add CertA_FileNo if OptsA.AuthMethod is Mutual (0x80) or Reader-UniLateral (0x40) with Cert.A */
    if(!(bOption & PHHAL_HW_CMD_SAM_AUTH_METHOD_NO_CERT_A_MASK))
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCertA_FileNo;
    }

    /* Add CertB_Options and CA KeyNo if OptsA.AuthMethod is Mutual (0x80) with or without Cert.A */
    if(!(bOption & PHHAL_HW_CMD_SAM_AUTH_METHOD_CERT_B_OPT_MASK))
    {
        /* Add CertB_Optins */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCertB_Options;

        /* Add ECC_KeyNoCA */
        if(bCertB_Options != PHHAL_HW_CMD_SAM_CERTB_OPTION_MSB_B_PLAIN)
        {
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_CA;
        }
    }

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_MutualAuthECC part 1 information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppSamResponse,
        pSamRespLen);

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Return the response received from Sam hardware. */
    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t * pPiccErrorCode, uint8_t * pCardResponse, uint8_t bCardRespLen, uint8_t ** ppSamResponse,
    uint16_t * pSamRespLen, uint8_t * pPiccRetCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t    PH_MEMLOC_REM *pData = NULL;
    uint8_t    PH_MEMLOC_REM bDataLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);
    if(bCardRespLen == 0)
    {
        PH_ASSERT_NULL_PARAM(pPiccErrorCode, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pPiccRetCode, PH_COMP_HAL);

        pData = pPiccErrorCode;
        bDataLen = PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN;
    }
    else
    {
        PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);

        pData = pCardResponse;
        bDataLen = bCardRespLen;
    }

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_MutualAuthECC part 2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_MUTUAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer SAM_MutualAuthECC part 2 information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Update information based on bOption data. */
    if(bOption != PH_EXCHANGE_RXCHAINING)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Update LC */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Add LE and exchange information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppSamResponse,
        pSamRespLen);

    /* Return Auth Error in case of 0x901E status code */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_CRYPTO)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    }

    /* Return Chaining status */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign PICC response code the parameter. */
        memcpy(pPiccRetCode, ppSamResponse[0], *pSamRespLen);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part3(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t * pPiccErrorCode, uint8_t * pCardResponse, uint16_t wCardRespLen, uint8_t ** ppSamResponse,
    uint16_t * pSamRespLen, uint8_t * pPiccRetCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint16_t   PH_MEMLOC_REM wOffset = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bExchLen = 0;
    uint8_t    PH_MEMLOC_REM bEndLoop = 0;

    uint8_t    PH_MEMLOC_REM *pData = NULL;
    uint16_t   PH_MEMLOC_REM wDataLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);
    if(wCardRespLen == 0)
    {
        PH_ASSERT_NULL_PARAM(pPiccErrorCode, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pPiccRetCode, PH_COMP_HAL);

        pData = pPiccErrorCode;
        wDataLen = PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN;
    }
    else
    {
        PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);

        pData = pCardResponse;
        wDataLen = wCardRespLen;
    }

    /* Reset the command buffer and length. */
    bCmdLen = 0x00;
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_MutualAuthECC part 3 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_MUTUAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchange Msg.B.enc or PICC Error to SAM */
    if(!(bOption & PH_EXCHANGE_RXCHAINING))
    {
        /* Append LC to command frame */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        do
        {
            /* Calculate Data bytes to be exchanged */
            if(wDataLen <= PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK)
            {
                /* Indicate last frame */
                PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;

                /* Update Exchange Length with remaining data to be exchanged. */
                bExchLen = (uint8_t) wDataLen;

                /* Set the flag to end the loop */
                bEndLoop = PH_ON;
            }
            else
            {
                bExchLen = PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK;
            }

            /* Buffer SAM_MutualAuthECC part 3 information. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                PHHAL_HW_SAM_CMD_BUFFER,
                bCmdLen,
                NULL,
                NULL));

            /* Add Data to command buffer */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                &pData[wOffset],
                bExchLen,
                NULL,
                NULL));

            /* Update LC */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

            /* Buffer LE and exchange the buffered information to SAM.*/
            wStatus = phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                phhalHw_Sam_Cmd_DefaultLe,
                PHHAL_HW_CMD_SAM_LE_LENGTH,
                ppSamResponse,
                pSamRespLen);

            /* In case of error. */
            if(((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE) &&
                ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
            {
                /* Return Auth Error in case of 0x901E status code */
                if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_CRYPTO)
                {
                    wStatus = PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
                }

                /* Extract the PICC error code. */
                if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
                {
                    /* Assign PICC response code the parameter. */
                    memcpy(pPiccRetCode, ppSamResponse[0], *pSamRespLen);
                }
            }

            /* Calculate Next Offset and length to be exchanged */
            wOffset += bExchLen;
            wDataLen -= bExchLen;

        } while(bEndLoop != 0);
    }
    else
    {
        /* Append LE to command frame */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

        /* Buffer LE and exchange the buffered information to SAM.*/
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            ppSamResponse,
            pSamRespLen);
    }

    /* Return chaining status. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t bECCKeyNo_Priv, uint8_t bECC_CurveNo, uint8_t ** ppSamResponse, uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_UnilatAuthEcc part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_UNILATERAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Append the Private Key number. and CurveNo */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_CurveNo;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_UnilatAuthEcc part 1 information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppSamResponse,
        pSamRespLen);

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Return the response received from Sam hardware. */
    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pPiccErrorCode,
    uint8_t * pCardResponse, uint8_t bCardRespLen, uint8_t * pPiccRetCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bCardRespLen == 0)
    {
        PH_ASSERT_NULL_PARAM(pPiccErrorCode, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pPiccRetCode, PH_COMP_HAL);
    }
    else
    {
        PH_ASSERT_NULL_PARAM(pCardResponse, PH_COMP_HAL);
    }

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_UnilatAuthEcc part 2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_UNILATERAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer SAM_UnilatAuthEcc part 2 command information */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Add PICC Response or PICC Error code */
    if(bCardRespLen != 0x00)
    {
        /* Buffer PICC Response */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pCardResponse,
            bCardRespLen,
            NULL,
            NULL));
    }
    else
    {
        /* Buffer PICC Status */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pPiccErrorCode,
            PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN,
            NULL,
            NULL));
    }

    /* Update LC */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Add LE and exchange information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign PICC response code the parameter. */
        memcpy(pPiccRetCode, pResponse, PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_BindCertificate_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t * pMKPParams, uint8_t bMKPParamsLen, uint8_t bECCKeyNo_Priv, uint8_t ** ppSamResponse,
    uint16_t * pSamRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pMKPParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_BindCertificate_Part1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_BIND_CERTIFICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add MKPParams to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pMKPParams, bMKPParamsLen);
    bCmdLen += bMKPParamsLen;

    /* Add ECCKeyNo_Priv to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_BindCertificate_Part1 information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppSamResponse,
        pSamRespLen);

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Return the response received from Sam hardware. */
    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_BindCertificate_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t * pData, uint16_t wDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint16_t   PH_MEMLOC_REM wRemLen = wDataLen;
    uint16_t   PH_MEMLOC_REM wOffset = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bExchLen = 0;
    uint8_t    PH_MEMLOC_REM bEndLoop = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    bCmdLen = 0x00;
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_BindCertificate_Part2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_BIND_CERTIFICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;

    /* Exchange Msg.B.enc or PICC Error to SAM */
    if(!(bOption & PH_EXCHANGE_RXCHAINING))
    {
        /* Append LC to command frame */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Configure the remaining length to be exchanged */
        wRemLen = wDataLen;

        do
        {
            /* Calculate Data bytes to be exchanged */
            if(wRemLen <= PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK)
            {
                /* Indicate last frame */
                PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;

                /* Update Exchange Length with remaining data to be exchanged. */
                bExchLen = (uint8_t) wRemLen;

                /* Set the flag to end the loop */
                bEndLoop = PH_ON;
            }
            else
            {
                bExchLen = PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK;
            }

            /* Buffer SAM_MutualAuthECC part 3 information. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                PHHAL_HW_SAM_CMD_BUFFER,
                bCmdLen,
                NULL,
                NULL));

            /* Add Data to command buffer */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                &pData[wOffset],
                bExchLen,
                NULL,
                NULL));

            /* Update LC */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

            /* Buffer LE and exchange the buffered information to SAM.*/
            wStatus = phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                phhalHw_Sam_Cmd_DefaultLe,
                PHHAL_HW_CMD_SAM_LE_LENGTH,
                ppResponse,
                pRespLen);

            /* In case of error. */
            if(((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE) &&
                ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
            {
                /* Set the flag to end the loop */
                bEndLoop = PH_ON;
            }

            /* Return the chaining code. */
            else
            {
                /* Return chaining status. */
                if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
                {
                    wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
                }
                else
                {
                    /* Set the flag to end the loop */
                    bEndLoop = PH_ON;
                }
            }

            /* Calculate Next Offset and length to be exchanged */
            wOffset += bExchLen;
            wRemLen -= bExchLen;

        } while(bEndLoop != 0);
    }
    else
    {
        /* Append LE to command frame */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

        /* Buffer LE and exchange the buffered information to SAM.*/
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            ppResponse,
            pRespLen);

        /* Return chaining status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_ImportEccKeyDUOX(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bECCKeyNo_Priv,
    uint8_t * pMKPParams, uint8_t bMKPParamsLen, uint8_t ** ppMKPCrypto, uint16_t * pMKPCryptoLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pMKPParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pMKPCryptoLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ImportEccKeyDUOX command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_IMPORT_ECC_KEY_DUOX;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add MKPParams to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pMKPParams, bMKPParamsLen);
    bCmdLen += bMKPParamsLen;

    /* Add ECCKeyNo_Priv to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange DUOX_MutualAuthEcc information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppMKPCrypto,
        pMKPCryptoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}





/* X - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_DUOX_MutualAuthEcc(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pOptsA,
    uint8_t bOptsALen, uint8_t bDUOX_P2, uint8_t bECCKeyNo_Priv, uint8_t bCertA_FileNo, uint8_t bCertB_Options,
    uint8_t bECCKeyNo_CA, uint8_t ** ppSamResponse, uint16_t * pSamRespLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOptsA, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DUOX_MutualAuthEcc command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DUOX_MUTUAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (bOption & 0x01U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Update the P1 information with default value. */
    if(bOption & PH_EXCHANGE_RXCHAINING)
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    }
    else
    {
        /* Copy OptsA information to command buffer */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pOptsA, bOptsALen);
        bCmdLen += bOptsALen;

        /* Add DUOX_P2 and ECC_KeyNoPrivto command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDUOX_P2;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;

        /* Add remaining information to command buffer */
        if((pOptsA[2U] == 0x80U /* Mutual Auth with Cert.A */) ||
            (pOptsA[2U] == 0xA0U /* Mutual Auth without Cert.A */))
        {
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCertA_FileNo;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCertB_Options;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_CA;
        }
    }

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange DUOX_MutualAuthEcc information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Return the chaining code. */
    if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE) ||
        ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Update the response buffer with actual data for Success or Success Chaining. */
        *ppSamResponse = pResponse;
        *pSamRespLen = wRespLen;

        /* Return chaining status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter. */
        memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DUOX_UnilatAuthEcc(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bECC_KeyNo,
    uint8_t bECC_CurveNo, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DUOX_MutualAuthEcc command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DUOX_UNILATERAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add ECC_KeyNo and ECC_CurveNo command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_CurveNo;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange DUOX_MutualAuthEcc information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter. */
        memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DUOX_BindCertificate(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pMKPParams,
    uint8_t bMKPParamsLen, uint8_t bECCKeyNo_Priv, uint8_t * pTBSCertificate, uint16_t wTBSCertLen, uint8_t ** ppSamResponse,
    uint16_t * pSamRespLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pMKPParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSamRespLen, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DUOX_MutualAuthEcc command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DUOX_BIND_CERTIFICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Update the P1 information with default value. */
    if(bOption & PH_EXCHANGE_RXCHAINING)
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    }
    else
    {
        /* Copy MKPParams information to command buffer */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pMKPParams, bMKPParamsLen);
        bCmdLen += bMKPParamsLen;

        /* Add ECC_KeyNoPriv to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;
    }

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer TBSCertificate information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pTBSCertificate,
        wTBSCertLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Return the chaining code. */
    if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE) ||
        ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Update the response buffer with actual data for Success or Success Chaining. */
        *ppSamResponse = pResponse;
        *pSamRespLen = wRespLen;

        /* Return chaining status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter in case of Native command set. */
        if(wRespLen == PHHAL_HW_SAM_DUOX_PICC_NATIVE_STATUS_LEN)
        {
            *pPiccReturnCode = pResponse[0];
        }

        /* Assign two byte PICC response code the parameter in case of ISO7816-4 Wrapped command set. */
        else
        {
            memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_DUOX_ImportEccKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pMKPParams,
    uint8_t bMKPParamsLen, uint8_t bECCKeyNo_Priv, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame DUOX_ImportEccKey command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_DUOX_UNILATERAL_AUTH_ECC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add ECC_KeyNo command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECCKeyNo_Priv;

    /* Add MKPParams command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pMKPParams, bMKPParamsLen);
    bCmdLen += bMKPParamsLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange DUOX_MutualAuthEcc information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_DESFIRE_GEN)
    {
        /* Assign one byte PICC response code the parameter in case of Native command set. */
        if(wRespLen == PHHAL_HW_SAM_DUOX_PICC_NATIVE_STATUS_LEN)
        {
            *pPiccReturnCode = pResponse[0];
        }

        /* Assign two byte PICC response code the parameter in case of ISO7816-4 Wrapped command set. */
        else
        {
            memcpy(pPiccReturnCode, pResponse, PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN);
        }
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
