/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE Plus command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_Plus.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

/* S - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pPDChal, uint8_t bPDChalLen, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t ** ppPCDChalResp, uint16_t * pPCDChalRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPDChalLen != 0)
        PH_ASSERT_NULL_PARAM(pPDChal, PH_COMP_HAL);
    if(bDivInputLen != 0)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPCDChalRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticateMFP part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MFP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Append Key No */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;

    /* Append Key Ver */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PDChal information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPDChal,
        bPDChalLen,
        NULL,
        NULL));

    /* Buffer Diversification information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppPCDChalResp,
        pPCDChalRespLen);

    /* Check for the Chaining active */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPiccErrCode,
    uint8_t * pPDResp, uint8_t bPDRespLen, uint8_t ** ppPDCap2, uint8_t ** ppPCDCap2, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPDRespLen != 0)
        PH_ASSERT_NULL_PARAM(pPDResp, PH_COMP_HAL);

    /* Reset the command buffer.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticateMFP part 2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MFP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
        NULL,
        NULL));

    /* Buffer PICC error code to exchange buffer. */
    if(bPiccErrCode != PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            &bPiccErrCode,
            PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE_LEN,
            NULL,
            NULL));
    }

    /* Buffer PDResp to exchange buffer. */
    if(bPiccErrCode == PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pPDResp,
            bPDRespLen,
            NULL,
            NULL));
    }

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardwre. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    /* Extract PCD and PD capabilities. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Add the received response to reference parameters. */
        if(wRespLen)
        {
            /*
             * Check if memroy is created because from AL the pointer has a memory allocated from wrapper interface
             * but from HAL the wrapper thinks that the memory will be passed on from C library.
             *
             * Also when the memory is alocated by AL the data is available in the internal interface only and not available
             * in the C generic or wrapper interface. The data returned by C interface is always zero.
             */
            if((ppPDCap2[0] != NULL) && (ppPCDCap2[0] != NULL))
            {
                memcpy(*ppPDCap2, &pResponse[0], PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN);
                memcpy(*ppPCDCap2, &pResponse[PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN],
                    PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN);
            }
            else
            {
                *ppPDCap2 = &pResponse[0];
                *ppPCDCap2 = &pResponse[PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN];
            }
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t * pPDChal, uint8_t bPDChalLen, uint16_t wSSKeyBNr, uint8_t bSSKeyNo, uint8_t bSSKeyVer, uint8_t bMSKeyNo,
    uint8_t bMSKeyVer, uint8_t bSectorCount, uint8_t *pKeyBlocks, uint8_t bKeyBlocksLen, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t ** ppPCDChalResp, uint16_t * pPCDChalRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPDChalLen != 0)
        PH_ASSERT_NULL_PARAM(pPDChal, PH_COMP_HAL);
    if(bKeyBlocksLen != 0)
        PH_ASSERT_NULL_PARAM(pKeyBlocks, PH_COMP_HAL);
    if(bDivInputLen != 0)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPCDChalRespLen, PH_COMP_HAL);

    /* Reset the command buffer.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthSectorSwitchMFP part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTH_SECTOR_SWITCH_MFP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PDChal information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPDChal,
        bPDChalLen,
        NULL,
        NULL));

    /* Reset the command buffer and its length variable.*/
    bCmdLen = 0;
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the initial Payload information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wSSKeyBNr);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wSSKeyBNr);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSSKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSSKeyVer;

    /* Add Master Key number and version if set in P1.*/
    if((bOption & PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON) == PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMSKeyNo;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMSKeyVer;
    }

    /* Add sector count to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSectorCount;

    /* Buffer intial payload information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer KeyBlocks information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pKeyBlocks,
        bKeyBlocksLen,
        NULL,
        NULL));

    /* Buffer Diversification Input information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppPCDChalResp,
        pPCDChalRespLen);

    /* Check for the Chaining active */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPiccErrCode,
    uint8_t * pPDResp, uint8_t bPDRespLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPDRespLen != 0)
        PH_ASSERT_NULL_PARAM(pPDResp, PH_COMP_HAL);

    /* Reset the command buffer.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command SAM_AuthSectorSwitchMFP part 2. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTH_SECTOR_SWITCH_MFP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PICC error code to exchange buffer. */
    if(bPiccErrCode != PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            &bPiccErrCode,
            PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE_LEN,
            NULL,
            NULL));
    }

    /* Buffer PDResp to exchange buffer. */
    if(bPiccErrCode == PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pPDResp,
            bPDRespLen,
            NULL,
            NULL));
    }

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pPDChal, uint8_t bPDChalLen, uint8_t * pUpgradeInfo, uint8_t bLen,
    uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t ** ppPCDChalResp, uint16_t * pPCDChalRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPDChalLen != 0)
        PH_ASSERT_NULL_PARAM(pPDChal, PH_COMP_HAL);
    if(bLen != 0)
        PH_ASSERT_NULL_PARAM(pUpgradeInfo, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPCDChalRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length variable.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticatePDC part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PDC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add KeyNo and KeyVer to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PDChal information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPDChal,
        bPDChalLen,
        NULL,
        NULL));

    /* Buffer Len information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        &bLen,
        1,
        NULL,
        NULL));

    /* Buffer UpgradeInfo information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pUpgradeInfo,
        bLen,
        NULL,
        NULL));

    /* Buffer Diversification input information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppPCDChalResp,
        pPCDChalRespLen);

    /* Check for the Chaining active */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPiccErrCode,
    uint8_t * pPDResp, uint8_t bPDRespLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bPDRespLen != 0)
        PH_ASSERT_NULL_PARAM(pPDResp, PH_COMP_HAL);

    /* Reset the command buffer.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command SAM_AuthenticatePDC part 2. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PDC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PICC error code to exchange buffer. */
    if(bPiccErrCode != PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            &bPiccErrCode,
            PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE_LEN,
            NULL,
            NULL));
    }

    /* Buffer PDResp to exchange buffer. */
    if(bPiccErrCode == PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pPDResp,
            bPDRespLen,
            NULL,
            NULL));
    }

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the buffered information to SAM hardwre. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_CombinedReadMFP(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bLFI, uint16_t wOption,
    uint8_t * pData, uint8_t bDataLen, uint8_t ** ppOutput, uint16_t * pOutputLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    if(((wOption & PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE) == PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE) ||
        ((wOption & PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_BOTH) == PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_BOTH))
    {
        if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT) ||
            ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST))
        {
            PH_ASSERT_NULL_PARAM(pOutputLen, PH_COMP_HAL);
            PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
        }
    }

    /* Buffer the command information. */
    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_FIRST) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame  SAM_CombinedReadMFP command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_READ_MFP;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wOption & PHHAL_HW_SAM_PLUS_COMBINED_READ_MASK);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bLFI;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Buffer the data information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer the intermediate Data information to exchange buffer. */
    if((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_CONT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer the last payload information and exchange bufferred information to SAM hardware. */
    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT))
    {
        /* Buffer the final data. */
        if((wOption & PH_EXCHANGE_BUFFER_MASK) != PH_EXCHANGE_DEFAULT)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pData,
                bDataLen,
                NULL,
                NULL));
        }

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Buffer LE and exchange the bufferred information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            ppOutput,
            pOutputLen);

        /* Extract the PICC error code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
        {
            /* Assign the PICC response code the parameter. */
            *pPiccReturnCode = *ppOutput[0];
        }

        /* Set the Status for Chaiining. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_CombinedWriteMFP(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption,
    uint8_t * pData, uint8_t bDataLen, uint8_t ** ppOutput, uint16_t * pOutputLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen) PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST))
    {
        if((wOption & PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE) == PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE)
        {
            PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
        }
    }

    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_FIRST))
    {
        /* Reset the command buffer.*/
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame SAM_CombinedWriteMFP command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_WRITE_MFP;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 information byte.  */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] |= (uint8_t) (wOption & PHHAL_HW_SAM_PLUS_COMBINED_WRITE_MASK);
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] |= (uint8_t) (wOption & PHHAL_HW_SAM_PLUS_COMBINED_WRITE_PLIAN_RESPONSE_MASK);

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Add the data information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer intermediate data information to exchange buffer. */
    if((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_CONT)
    {
        /* Add the data information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer final data information to exchange buffer and exchange the bufferred information to Sam hardware. */
    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST))
    {
        if((wOption & PH_EXCHANGE_BUFFER_MASK) != PH_EXCHANGE_DEFAULT)
        {
            /* Add the data information to exchange buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pData,
                bDataLen,
                NULL,
                NULL));
        }

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Add LE to and exchange the bufferred information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            ppOutput,
            pOutputLen);

        /* Extract the PICC error code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
        {
            /* Assign the PICC response code the parameter. */
            *pPiccReturnCode = *ppOutput[0];
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyMFP(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData,
    uint8_t bDataLen, uint8_t ** ppProtectedData, uint16_t * pProtectedDataLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pProtectedDataLen, PH_COMP_HAL);
    if((bOption & PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE) == PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE)
    {
        PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
    }

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ChangeKeyMFP command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_MFP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Buffer commnad informration to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer commnad informration to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Add LE to exchange buffer and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppProtectedData,
        pProtectedDataLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = *ppProtectedData[0];
    }

    return wStatus;
}





/* X - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_MFP_Authenticate(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint16_t wBlockNo, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t * pPcdCapsOut, uint8_t * pPdCaps, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPcdCapsOut, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPdCaps, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer and its length.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame MFP_Authenticate command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_MFP_AUTHENTICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE + bPcdCapsInLen +
        bDivInputLen + 5U /* KeyNo, KeyVer, BNR, PcdCapInLen */;

    /* Add the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wBlockNo);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wBlockNo);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPcdCapsInLen;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PCD capabilites to exchange buffer. */
    if(pPcdCapsIn != NULL)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pPcdCapsIn,
            bPcdCapsInLen,
            NULL,
            NULL));
    }

    /* Buffer diversification to exchange buffer. */
    if(pDivInput != NULL)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));
    }

    /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    /* Extract PCD and PD capabilities. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Assign the PCD and PD capabilities to the parameter. */
        memcpy(pPdCaps, &pResponse[0], PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN);
        memcpy(pPcdCapsOut, &pResponse[PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN],
            PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_MFP_AuthSectorSwitch(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint16_t wSSKeyBNr,
    uint8_t bSSKeyNo, uint8_t bSSKeyVer, uint8_t bMSKeyNo, uint8_t bMSKeyVer, uint8_t bSectorCount, uint8_t * pKeyBlocks,
    uint8_t bKeyBlocksLen, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST))
        PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Buffer the command information. */
    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_FIRST) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT))
    {
        /* Reset the command buffer and its length.*/
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame MFP_AuthSectorSwitch header. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_MFP_AUTH_SECTOR_SWITCH;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wOption & ~PH_EXCHANGE_BUFFER_MASK);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add the payload information to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wSSKeyBNr);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wSSKeyBNr);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSSKeyNo;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSSKeyVer;

        /* Add Master Key number and version if set in P1.*/
        if((wOption & PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON) == PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON)
        {
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMSKeyNo;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMSKeyVer;
        }

        /* Add Sector count to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSectorCount;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Buffer the KeyBlocks information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pKeyBlocks,
            bKeyBlocksLen,
            NULL,
            NULL));

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }

    /* Buffer the intermediate KeyBlocks or DivInput information to exchange buffer. */
    if((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_CONT)
    {
        /* Buffer KeyBlocks to exchange buffer. */
        if((wOption & PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_KEY_BLOCKS) == PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_KEY_BLOCKS)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pKeyBlocks,
                bKeyBlocksLen,
                NULL,
                NULL));
        }

        /* Buffer DivInput to exchange buffer. */
        if((wOption & PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_DIV_INPUT) == PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_DIV_INPUT)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pDivInput,
                bDivInputLen,
                NULL,
                NULL));
        }

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }

    /* Buffer the last payload information and exchange bufferred information to SAM hardware. */
    if(((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_BUFFER_LAST) ||
        ((wOption & PH_EXCHANGE_BUFFER_MASK) == PH_EXCHANGE_DEFAULT))
    {
        /* Buffer the final data. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            &pResponse,
            &wRespLen);

        /* Extract the PICC error code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
        {
            /* Assign the PICC response code the parameter. */
            *pPiccReturnCode = pResponse[0];
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PDC_Authenticate(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint16_t wUpgradeKey, uint8_t * pUpgradeInfo, uint8_t bLen, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset the command buffer and its length.*/
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PDC_Authenticate information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PDC_AUTHENTICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = 5U /* KeyNo, KeyVer, UpgradeKey, Len */ + bLen + bDivInputLen;

    /* Add payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wUpgradeKey);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wUpgradeKey);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bLen;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer UpgradeInfo to exchange buffer */
    if(pUpgradeInfo != NULL)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pUpgradeInfo,
            bLen,
            NULL,
            NULL));
    }

    /* Buffer DivInput to exchage buffer */
    if(pDivInput != NULL)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));
    }

    /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_MFP_CombinedRead(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pReadCmd,
    uint8_t bReadCmdLen, uint8_t ** ppData, uint16_t * pDataLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(wOption != PH_EXCHANGE_RXCHAINING) PH_ASSERT_NULL_PARAM(pReadCmd, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame MFP_CombinedRead command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_MFP_COMBINED_READ;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Add LC is the option is not chaining. */
    if(wOption != PH_EXCHANGE_RXCHAINING)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bReadCmdLen;
    }

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer the ReadCmd information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pReadCmd,
        bReadCmdLen,
        NULL,
        NULL));

    /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppData,
        pDataLen);

    /* Extract the PICC error code. */
    if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN) ||
        ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE) ||
        ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Update the pointer address and length if First Frame. */
        if(wOption != PH_EXCHANGE_RXCHAINING)
        {
            /* Assign the PICC response code the parameter. */
            *pPiccReturnCode = *ppData[0];

            ppData[0]++;
            *pDataLen = *pDataLen - 1U;
        }
    }

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_MFP_CombinedWrite(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pData,
    uint8_t bDataLen, uint8_t * pTMC, uint8_t * pTMV, uint8_t * pPiccReturnCode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen) PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

    if((wOption == PH_EXCHANGE_BUFFER_LAST) || (wOption == PH_EXCHANGE_DEFAULT))
    {
        PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
    }

    /* Buffer the command information. */
    if((wOption == PH_EXCHANGE_BUFFER_FIRST) || (wOption == PH_EXCHANGE_DEFAULT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame MFP_CombinedWrite command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_MFP_COMBINED_WRITE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Buffer the data information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer the intermediate payload information to exchange buffer. */
    if(wOption == PH_EXCHANGE_BUFFER_CONT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            bDataLen,
            NULL,
            NULL));
    }

    /* Buffer the last payload information and exchange bufferred information to SAM hardware. */
    if((wOption == PH_EXCHANGE_BUFFER_LAST) || (wOption == PH_EXCHANGE_DEFAULT))
    {
        if(wOption != PH_EXCHANGE_DEFAULT)
        {
            /* Buffer the final data. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pData,
                bDataLen,
                NULL,
                NULL));
        }

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            &pResponse,
            &wRespLen);

        /* Extract the PICC error code. */
        if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN) ||
            ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
        {
            if(wRespLen)
            {
                /* Assign the PICC response code the parameter. */
                *pPiccReturnCode = pResponse[0];
            }
        }

        /* Extract TMC and TMV information. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Assign the TMC and TMV to the parameter if the block is TM protected block. */
            if(wRespLen > 1U)
            {
                if(pTMC != NULL)
                {
                    memcpy(pTMC, &pResponse[1U], 4U);
                }

                if(pTMV != NULL)
                {
                    memcpy(pTMV, &pResponse[5U], 8U);
                }
            }
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_MFP_ChangeKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCmdCode,
    uint16_t wBlockNo, uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen,
    uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame MFP_ChangeKey command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_MFP_CHANGE_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (bOption >> 1);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = 5 /* CmdCode, BlockNo, KeyNo, KeyVer */ + bDivInputLen;

    /* Add payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCmdCode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wBlockNo);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wBlockNo);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer DivInput to exchange buffer. */
    if(pDivInput != NULL)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));
    }

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_MFP_WritePerso(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pBlocks,
    uint8_t bBlocksLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameter. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if((wOption == PH_EXCHANGE_BUFFER_LAST))
    {
        PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);
    }

    /* Buffer the command information. */
    if((wOption == PH_EXCHANGE_BUFFER_FIRST) || (wOption == PH_EXCHANGE_DEFAULT))
    {
        /* Reset the command buffer. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame MFP_WritePerso command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_MFP_WRITE_PERSO;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Buffer the data information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pBlocks,
            bBlocksLen,
            NULL,
            NULL));
    }

    /* Buffer the intermediate payload information to exchange buffer. */
    if(wOption == PH_EXCHANGE_BUFFER_CONT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pBlocks,
            bBlocksLen,
            NULL,
            NULL));
    }

    /* Buffer the last payload information and exchange bufferred information to SAM hardware. */
    if((wOption == PH_EXCHANGE_BUFFER_LAST) || (wOption == PH_EXCHANGE_DEFAULT))
    {
        if(wOption != PH_EXCHANGE_DEFAULT)
        {
            /* Buffer the final data. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pBlocks,
                bBlocksLen,
                NULL,
                NULL));
        }

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Buffer Le to exchange buffer and exchange the bufferred information to SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            &pResponse,
            &wRespLen);

        /* Extract the PICC error code. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) &&
            ((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN))
        {
            return wStatus;
        }

        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
