/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE Common command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_Common.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

/* S - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_SAM_CommitReaderID_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bState, uint16_t wBlockNr,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame Cmd.SAM_CommitReaderID_Part1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_COMMIT_READER_ID;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Append LC and Block number if the state is MFP. */
    if(bState == PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP)
    {
        /* Append LC to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = 2U /* Two bytes of block number. */;

        /* Append block number to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wBlockNr);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wBlockNr);
    }

    /* Append Command data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Append LE to exchange buffer and perform final exchange with SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppResponse,
        pRespLen);

    /* Check for the Chaining active. */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_CommitReaderID_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPiccErrCode,
    uint8_t * pData, uint8_t bDataLen, uint8_t * pPiccReturnCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccReturnCode, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_CommitReaderID_Part2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_COMMIT_READER_ID;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Response code to comamnd buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPiccErrCode;

    /* Add Data to command buffer */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
    bCmdLen += bDataLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_CommitReaderID_Part2 command information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccReturnCode = pResponse[0];
    }

    return wStatus;
}





/* X - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_TMRI_CommitReaderID(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bISOMode, uint8_t bState,
    uint16_t wBlockNr, uint8_t ** ppEncTMRI, uint16_t * pEncTMRILen, uint8_t * pStatusCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncTMRILen, PH_COMP_HAL);

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command Cmd.SAM_PLExec. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_TMRI_COMMIT_READER_ID;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bISOMode;

    /* Add block number to command buffer is MFP state. */
    if(bState == PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP)
    {
        /* Add LC to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = 2U;

        /* Add Block number to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wBlockNr);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wBlockNr);
    }

    /* Buffer the command information to exchnage buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer LE and exchange the bufferred information to Sam hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
    {
        /* Assign the PICC response code the parameter. */
        memcpy(pStatusCode, pResponse, (bISOMode ? 2U : 1U));
    }

    /* Update the response to the parameters if success. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        *ppEncTMRI = pResponse;
        *pEncTMRILen = wRespLen;
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
