/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ICODE and UCODE command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_iCode.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateTAM1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t * pData, uint8_t bDataLen, uint8_t ** ppIChallange, uint16_t * pIChallangeLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Clear all the local variables. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command to be sent to SAM and receive the TAM1 random number. --------------------------------------- */
    if(bOption == PHHAL_HW_SAM_CMD_TAM_GET_RND)
    {
        /* Validate the parameters. */
        if(bDataLen) PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pIChallangeLen, PH_COMP_HAL);

        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_TAM;

        /* Custom data flag not required for TAM1 Request.*/
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) PHHAL_HW_SAM_CMD_TAM_CLEAR ;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Set the Bit 0 of P1 if diversification data is present. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] |= (uint8_t) (bDataLen != 0);

        /* Add Key information to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

        /* Add Diversification Input to command buffer. */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
        bCmdLen += bDataLen;

        /* Add LE to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

        /* Update LC */
        phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

        /* Buffer command information exchange buffer. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            ppIChallange,
            pIChallangeLen);

        /* Check for the Chaining active. */
        if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return wStatus;
        }

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Frame the command to be sent to SAM for decryption, verification. --------------------------------------------- */
    if(bOption == PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE)
    {
        /* Validate the parameters. */
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);

        /* Clear all the local variables. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_TAM;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add Data information to command buffer. */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
        bCmdLen += bDataLen;

        /* Update LC */
        phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

        /* Buffer command information exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            ppIChallange,
            pIChallangeLen));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateTAM2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t * pData, uint8_t bDataLen, uint8_t bBlockSize, uint8_t bBlockCount, uint8_t bProtMode,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus;
    uint8_t    PH_MEMLOC_REM bCustDataLen = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Clear all the local variables. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command to be sent to SAM and receive the TAM1 random number. --------------------------------------- */
    if(bOption == PHHAL_HW_SAM_CMD_TAM_GET_RND)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_TAM;

        /* Custom data flag required for TAM2 Request.*/
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) ((bProtMode << 4U) | PHHAL_HW_SAM_CMD_TAM_SET);

        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Set the Bit 0 of P1 if diversification data is present. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] |= (uint8_t) (bDataLen != 0);

        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

        /* Add Diversification Input to command buffer. */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
        bCmdLen += bDataLen;

        /* Add LE to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

        /* Update LC */
        phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

        /* Buffer command information exchange buffer. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            ppResponse,
            pRespLen);

        /* Check for the Chaining active */
        if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return wStatus;
        }

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Frame the command to be sent to SAM for decryption, verification. --------------------------------------------- */
    if(bOption == PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE)
    {
        /* Clear all the local variables. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /*
         * Calculate the Custom Data Length. For calculation D refer
         *      DocumentID      : ISO-IEC_29167-10(E)_AMD1_2015-05-07_red.pdf
         *      Document Name   : Information technology  Automatic identification and data capture techniques 
         *                        Part 10: Crypto suite AES-128 security services for air interface communications - AMD1
         *      PgNo            : 16,
         *      Section         : 10.2 Adding custom data to authentication process
         */
        if(bBlockSize)
        {
            /*
             * The formula for calculating D for 64 bit block size is as follows.
             * D = ( n + 1 ) / 2 + ( n + 1 ) % 2, Where n = BlockCount ranging from 0 - 15.
             * Because the current BlockCount value range is from 1 - 16, to match it with ISO 29167 protocol, the incrementation
             * of BlockCount in the actual formual is removed as its already incremented. So the new formula to calculate D is
             * D = n / 2 + n % 2, Where n = BlockCount ranging from 1 - 16.
             * Finally mutiplying by 16 because the CustomData is represented as D * 128 => D * 16
             */
            bCustDataLen = (uint8_t) (((bBlockCount / 2U) + (bBlockCount % 2U)) * 16U);
        }
        else
        {
            /*
             * The formula for calculating D for 16 bit block size is as follows.
             * D = ( n + 8 ) / 8, Where n = BlockCount ranging from 0 - 15.
             * Because the current BlockCount value range is from 1 - 16, to match it with ISO 29167 protocol, the BlockCount
             * is decremented by 1 in the actual formual. So the new formula to calculate D is
             * D = ( ( n - 1) + 8) / 8, Where n = BlockCount ranging from 1 - 16.
             * Finally mutiplying by 16 because the CustomData is represented as D * 128 => D * 16
             */
            bCustDataLen = (uint8_t) ((((bBlockCount - 1U) + 8U) / 8U) * 16U);
        }

        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_TAM;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add Custom data Length to command buffer */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) bCustDataLen;

        /* Append the TResponse available in pData to command buffer. */
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
        bCmdLen += bDataLen;

        /* Append LE byte. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

        /* Update LC */
        phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

        /* Exchange the command to SAM hardware. SAM should send 90 00 for successfull execution. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            ppResponse,
            pRespLen));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMAM1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t * pData, uint8_t bDataLen, uint8_t bPurposeMAM2, uint8_t ** ppIChallange, uint16_t * pIChallangeLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_DATA_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_DATA_PARAM(pIChallangeLen, PH_COMP_HAL);

    /* Clear all the local variables. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticateMAM part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MAM;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (bPurposeMAM2 << 4U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Set the Bit 0 of P1 if diversification data is present. */
    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] |= (uint8_t) (bDataLen != 0);

    /* Add Key information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Buffer diversification information exchange buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
    bCmdLen += bDataLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_AuthenticateMAM part 1 information to SAM. */
    wStatus =  phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppIChallange,
        pIChallangeLen);

    /* Check for the Chaining active. */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMAM2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pData, uint8_t bDataLen,
    uint8_t ** ppIResponse, uint16_t * pIResponseLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bDataLen != 0)
        PH_ASSERT_NULL_DATA_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_DATA_PARAM(pIResponseLen, PH_COMP_HAL);

    /* Clear all the local variables. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_AuthenticateMAM part 2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MAM;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add TResponse information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
    bCmdLen += bDataLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_AuthenticateMAM part 2 to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppIResponse,
        pIResponseLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
