/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ICODE and UCODE command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_SAM_CMD_ICODE_H
#define PHHALHW_SAM_CMD_ICODE_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_SAM_CMD_TAM_CLEAR                                              0x00U   /**< Macro to represent the custom data value
                                                                                         *   for Tag authentication cleared.
                                                                                         */
#define PHHAL_HW_SAM_CMD_TAM_SET                                                0x02U   /**< Macro to represent the custom data value
                                                                                         *   for Tag authentication set.
                                                                                         */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ICODE_UCODE 12_ICODE_UCODE
 * \brief Implementation of SAM commands used for ICODE and UCODE communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Common communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_TAM                               0xB0U   /**< SAM Insturction code for SAM_AuthenticateTAM command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MAM                               0xB2U   /**< SAM Insturction code for SAM_AuthenticateMAM command. */
/**
 * end of group phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines_Mode Mode
 * \brief Definitions to Get IChallange or Exchange TResponse.
 * To be used with below mentioned interfaces.
 -  - \ref phhalHw_Sam_Cmd_SAM_AuthenticateTAM1 "SAM_Authenticate TAM1"
 -  - \ref phhalHw_Sam_Cmd_SAM_AuthenticateTAM2 "SAM_Authenticate TAM2"
 * @{
 */
#define PHHAL_HW_SAM_CMD_TAM_GET_RND                                            0x00U   /**< Option for exchanging Key and Div input and receive
                                                                                         *   the IChallange as response.
                                                                                         */
#define PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE                                   0x01U   /**< Option for exchanging TResponse and validating it. */
/**
 * end of group phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines_Mode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines_PurposeMAM2 PurposeMAM2
 * \brief Definitions for updating the PurposeMAM2.
 * To be used with \ref phhalHw_Sam_Cmd_SAM_AuthenticateMAM1 "SAM_Authenticte MAM1" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_MAM_NONE                                               0x00U   /**< Option to indicate PurposeMAM2 value as None.
                                                                                         *   Here only authentication will be performed.
                                                                                         */
#define PHHAL_HW_SAM_CMD_MAM_DISABLE_PRIVACY_HF_RESET                           0x08U   /**< Option to indicate PurposeMAM2 value as
                                                                                         *   Privacy disable until HF reset.
                                                                                         */
#define PHHAL_HW_SAM_CMD_MAM_ENABLE_PRIVACY                                     0x09U   /**< Option to indicate PurposeMAM2 value as
                                                                                         *    Privacy enable.
                                                                                         */
#define PHHAL_HW_SAM_CMD_MAM_DISABLE_PRIVACY                                    0x0AU   /**< Option to indicate PurposeMAM2 value as
                                                                                         *   Privacy disable.
                                                                                         */
#define PHHAL_HW_SAM_CMD_MAM_DESTROY                                            0x0BU   /**< Option to indicate PurposeMAM2 value as
                                                                                         *   Destroy.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines_PurposeMAM2
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_ICODE_UCODE_SMode_Defines
 * @}
 */

/**
 * \brief Generates 10 bytes of random challange to be given to Tag. Also Decrypts the TResponse received from
 * Tag and verifies the decrypted data for authenticity.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful when \b bOption is
 *                                      \ref PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE "Exchange TResponse".
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with chaining status when \b bOption is
 *                                      \ref PHHAL_HW_SAM_CMD_TAM_GET_RND "Get IChallange".
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pIChallangeLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateTAM1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] The command to be framed for SAM.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_TAM_GET_RND "Get IChallange"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE
                                                                                         *            "Exchange TResponse"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pData,                                                                /**< [In] Diversification Input or TResponse received from Tag. */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer */
        uint8_t ** ppIChallange,                                                        /**< [Out] The IChallange to be sent to Tag. This will contain
                                                                                         *         10 bytes of random challange data.
                                                                                         */
        uint16_t * pIChallangeLen                                                       /**< [Out] Length of bytes available in \b ppIChallange buffer. */
    );

/**
 * \brief Generates 10 bytes of random challange to be given to Tag. Also Decrypts the TResponse received from
 * Tag and verifies the decrypted data for authenticity and provides the custom data received form Tag.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful when \b bOption is
 *                                      \ref PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE "Exchange TResponse".
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with chaining status when \b bOption is
 *                                      \ref PHHAL_HW_SAM_CMD_TAM_GET_RND "Get IChallange".
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pIChallangeLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateTAM2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] The command to be framed for SAM.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_TAM_GET_RND "Get IChallange"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_TAM_PROCESS_TRESPONE
                                                                                         *            "Exchange TResponse"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pData,                                                                /**< [In] Diversification Input or TResponse received from Tag. */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t bBlockSize,                                                             /**< [In] To select the size of custom data block to be used.
                                                                                         *        The value should either be 0x00 for 16 bit block size or
                                                                                         *        0x01 for 64 bit block size. As per ISO 29167
                                                                                         */
        uint8_t bBlockCount,                                                            /**< [In] To select the custom data block to be used from the offset
                                                                                         *        specified. The BlockCount range is from 1 - 16.
                                                                                         */
        uint8_t bProtMode,                                                              /**< [In] To specify the mode of operation to be used for encryption
                                                                                         *        /decryption. The ProtMode ranges form 0 - 3. As per ISO 29167
                                                                                         */
        uint8_t ** ppResponse,                                                          /**< [Out] The IChallange to be sent to Tag. This will contain 10 bytes
                                                                                         *         of random challange data. Or the Custom Data received from Tag.
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Generates 10 bytes of random challange to be given to Tag.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with chaining status
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pIChallangeLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMAM1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pData,                                                                /**< [In] Diversification Input. */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t bPurposeMAM2,                                                           /**< [In] Purpose MAM 2 data. A 4 bit value.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MAM_NONE "None"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MAM_DISABLE_PRIVACY_HF_RESET
                                                                                         *                 "Privacy disable until HF reset"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MAM_ENABLE_PRIVACY
                                                                                         *                 "Privacy enable"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MAM_DISABLE_PRIVACY
                                                                                         *                 "Privacy disable"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MAM_DESTROY
                                                                                         *                 "Destroy"
                                                                                         */
        uint8_t ** ppIChallange,                                                        /**< [Out] The IChallange to be sent to Tag. This will contain
                                                                                         *         10 bytes of random challange data.
                                                                                         */
        uint16_t * pIChallangeLen                                                       /**< [Out] Length of bytes available in \b ppIChallange buffer. */
    );

/**
 * \brief Decrypts the TResponse received from Tag and verifies the decrypted data for authenticity. Also
 * frames the IResponse with will be sent to the Tag.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pIResponseLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMAM2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pData,                                                                /**< [In] TResponse received from Tag. */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer */
        uint8_t ** ppIResponse,                                                         /**< [Out] The IResponse generated by SAM that will be sent
                                                                                         *         to Tag. This will contain 16 bytes of data.
                                                                                         */
        uint16_t * pIResponseLen                                                        /**< [Out] Length of bytes available in \b ppIResponse buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_ICODE_UCODE
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_ICODE_H */
