/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Programmable Logic command implementation of Reader
 * Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_PL.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include <phCryptoSym.h>
#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_PLExec(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bLFI, uint8_t * pPLData,
    uint8_t bPLDataLen, uint8_t ** ppPLResp, uint16_t * pPLRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Vaildate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPLData, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command SAM_PLExec. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_CMD_INS_SAM_PL_EXEC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bLFI;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPLDataLen;

    /* Append Command data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Append PLData to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPLData,
        bPLDataLen,
        NULL,
        NULL));

    /* Append LE to exchange buffer. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppPLResp,
        pPLRespLen);

    /* Return the chaining code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Return success response. */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_PLUpload(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bIsFirstFrame,
    uint8_t bIsFinalFrame, uint16_t wUploadCtr, uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pPLCode,
    uint16_t wPLCodeLen, uint8_t * pPLReKey, uint8_t bPLReKeyLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    phStatus_t PH_MEMLOC_REM wStatus1 = 0;
    uint16_t   PH_MEMLOC_REM wPLCodeOffset = 0;
    uint16_t   PH_MEMLOC_REM wPLCodeChunks = 0;
    uint16_t   PH_MEMLOC_REM wRemPLCodeLen = 0;
    uint16_t   PH_MEMLOC_REM wFinalPaddLen = 0;
    uint16_t   PH_MEMLOC_REM wPaddOutLen = 0;
    uint16_t   PH_MEMLOC_REM wIteration = 0;
    uint8_t    PH_MEMLOC_REM bCmdBuffOffset = 0;
    uint8_t    PH_MEMLOC_REM bKeyType = 0;
    uint8_t    PH_MEMLOC_REM bMacLen = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t    PH_MEMLOC_REM aSessionKey[PH_CRYPTOSYM_AES256_KEY_SIZE];
    uint8_t    PH_MEMLOC_REM aPLUploadReKey[PH_CRYPTOSYM_AES256_KEY_SIZE * PH_CRYPTOSYM_KEY_TYPE_AES256];
    uint8_t    PH_MEMLOC_REM aPLUploadMAC[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t    PH_MEMLOC_REM aRespMAC[PH_CRYPTOSYM_AES128_KEY_SIZE];

    uint8_t * PH_MEMLOC_REM pResponse = NULL;
    uint16_t  PH_MEMLOC_REM wRespLen = 0;

    /* Check memory is created for pPLCode. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPLCode, PH_COMP_HAL);
    if(bIsFinalFrame == (uint8_t) PH_ON)
        PH_ASSERT_NULL_PARAM(pPLReKey, PH_COMP_HAL);

    /* Compute the initial Session Keys and load them to crypto params. */
    if(bIsFirstFrame == (uint8_t) PH_ON)
    {
        /* Get Session Upload ENC key. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GetSessionUploadKey(
            pDataParams,
            PHHAL_HW_CMD_SAM_SESSION_KEY_ENC,
            wUploadCtr,
            bKeyNo,
            bKeyVer,
            aSessionKey,
            &bKeyType));

        /* Load the Session Upload ENC key to Crypto. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pPLUpload_ENCCryptoDataParams,
            aSessionKey,
            bKeyType));

        /* Get Session Upload MAC key. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GetSessionUploadKey(
            pDataParams,
            PHHAL_HW_CMD_SAM_SESSION_KEY_MAC,
            wUploadCtr,
            bKeyNo,
            bKeyVer,
            aSessionKey,
            &bKeyType));

        /* Load the Session Upload MAC key to Crypto. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pPLUpload_MACCryptoDataParams,
            aSessionKey,
            bKeyType));

        /* Load the initial Session MAC key to internal dataparams. */
        memcpy(pDataParams->aPLUploadSessMAC0, aSessionKey, ((bKeyType == PH_CRYPTOSYM_KEY_TYPE_AES128) ? 16U :
            (bKeyType == PH_CRYPTOSYM_KEY_TYPE_AES192) ? 24U : 32U));

        /* Load Zero IV for both the Crypto dataparams. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pPLUpload_ENCCryptoDataParams,
            phhalHw_Sam_Cmd_ZeroIV,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pPLUpload_MACCryptoDataParams,
            phhalHw_Sam_Cmd_ZeroIV,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        /* Set the KeepIV flag for both the dataparams. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
            pDataParams->pPLUpload_ENCCryptoDataParams,
            PH_CRYPTOSYM_CONFIG_KEEP_IV,
            PH_CRYPTOSYM_VALUE_KEEP_IV_ON));

        /* Update the KeyType. */
        pDataParams->bPLUploadKeyType = bKeyType;
    }

    /* Evaluate the PLCode chunks to be exchanged. */
    wPLCodeChunks = (uint16_t) (wPLCodeLen / PHHAL_HW_SAM_PL_CHUNK_SIZE);

    /* Set the remaining code length. */
    wRemPLCodeLen = wPLCodeLen;

    /* Update chunk count if PLCodeChunk is not 1 and not mulitple of 128. */
    if(!wPLCodeChunks || ((uint16_t) (wPLCodeLen % PHHAL_HW_SAM_PL_CHUNK_SIZE)))
        wPLCodeChunks++;

    /* Set the remaining code length to chunk size. */
    if(wPLCodeLen > PHHAL_HW_SAM_PL_CHUNK_SIZE)
        wRemPLCodeLen = PHHAL_HW_SAM_PL_CHUNK_SIZE;

    /* Exchagne the information. */
    for(wIteration = 0; wIteration < wPLCodeChunks; wIteration++)
    {
        /* Reset the command buffer and its length. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame the command SAM_PLUpload. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_CMD_INS_SAM_PL_UPLOAD;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = 0x01U; /* Crypto On-Going. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Append the Upload_Ctr to command buffer if its first frame. */
        if(bIsFirstFrame && !wIteration)
        {
            /* Add Upload Counter value to command buffer. */
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);
        }

        /* Apply padding for the last chunk. */
        if((wIteration + 1) == wPLCodeChunks)
        {
            /* Update the remaining PLCode data. */
            wRemPLCodeLen = (uint16_t) (wPLCodeLen - wPLCodeOffset);

            /* Compute the output padding length. */
            wPaddOutLen = (uint16_t) (wRemPLCodeLen + (16U - wRemPLCodeLen % 16U));

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_ApplyPadding(
                PH_CRYPTOSYM_PADDING_MODE_2,
                &pPLCode[wPLCodeOffset],
                wRemPLCodeLen,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                wPaddOutLen,
                pPLCode,
                &wFinalPaddLen));

                /* Reset the Offset pointer. */
            wPLCodeOffset = 0;

            /* Update the remaining code length. */
            wRemPLCodeLen = (uint8_t) wFinalPaddLen;
        }

        /* Encrypt the PLUpload code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
            pDataParams->pPLUpload_ENCCryptoDataParams,
            (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
            &pPLCode[wPLCodeOffset],
            wRemPLCodeLen,
            &PHHAL_HW_SAM_CMD_BUFFER[bCmdLen]));

        /* Update the command buffer length. */
        bCmdLen += (uint8_t) wRemPLCodeLen;

        /* Compute PLUploadReKeys if its the last chunk and not the final segment. */
        if(((wIteration + 1U) == wPLCodeChunks) && !bIsFinalFrame)
        {
            /* Encrypt the PLUpload code. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pPLUpload_ENCCryptoDataParams,
                (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
                pPLReKey,
                bPLReKeyLen,
                aPLUploadReKey));

            /* Buffer the aPLUploadReKey to Command buffer. */
            memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], aPLUploadReKey, bPLReKeyLen);
            bCmdLen += (uint8_t) bPLReKeyLen;
        }

        /* Update the Offset */
        wPLCodeOffset += PHHAL_HW_SAM_PL_CHUNK_SIZE;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Copy the command information to PLUpload internal buffer. */
        memcpy(&pDataParams->pPLUploadBuf[pDataParams->wPLUploadBufLen], &PHHAL_HW_SAM_CMD_BUFFER[bCmdBuffOffset],
            bCmdLen - bCmdBuffOffset);

        pDataParams->wPLUploadBufLen += (uint16_t) (bCmdLen - bCmdBuffOffset);

        /* Update the command buffer offset. */
        bCmdBuffOffset = PHHAL_HW_SAM_ISO7816_HEADER_LENGTH;

        /* Exchange the command information if not the last frame. */
        if((wIteration + 1U) < wPLCodeChunks)
        {
            /* Update LC value. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

            wStatus = phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                NULL,
                bMacLen,
                &pResponse,
                &wRespLen);

            /* Check if SUCCESS_CHAINING is returned. */
            if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
            {
                /* Reset the internal PLUpload length variable. */
                pDataParams->wPLUploadBufLen = 0;

                return wStatus;
            }
        }
    }

    /* Buffer LE to command buffer. */
    if(bIsFinalFrame)
        pDataParams->pPLUploadBuf[pDataParams->wPLUploadBufLen++] = 0x00;

    /* Reset the header bytes. */
    pDataParams->pPLUploadBuf[PHHAL_HW_SAM_ISO7816_P1_POS] = 0x00;
    pDataParams->pPLUploadBuf[PHHAL_HW_SAM_ISO7816_P2_POS] = 0x00;
    pDataParams->pPLUploadBuf[PHHAL_HW_SAM_ISO7816_LC_POS] = 0x00;

    /* Buffer LE for PLUploadMAC computation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pPLUpload_MACCryptoDataParams,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pDataParams->pPLUploadBuf,
        pDataParams->wPLUploadBufLen,
        aPLUploadMAC,
        &bMacLen));

    /* Truncate the MAC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(aPLUploadMAC, &bMacLen));

    /* Buffer the PLUpload MAC data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        aPLUploadMAC,
        bMacLen,
        NULL,
        NULL));

    /* Update P1 information byte for Last frame. */
    if(bIsFinalFrame)
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateP1(pDataParams, 0x00));

    /* Update P2 information to Crypto finalization and re-key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateP2(pDataParams, 0x00));

    /* Update LC value. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Append LE if required and perform exchange to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        bIsFinalFrame,
        &pResponse,
        &wRespLen);

    /* Clear the internal buffer and length variable. */
    memset(pDataParams->pPLUploadBuf, 0x00, pDataParams->wPLUploadBufLen);
    pDataParams->wPLUploadBufLen = 0;

    /* Update the session keys with the new keys. */
    if(!bIsFinalFrame)
    {
        /* Load the new Session Upload ENC key to Crypto. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_LoadKeyDirect(
            pDataParams->pPLUpload_ENCCryptoDataParams,
            pPLReKey,
            pDataParams->bPLUploadKeyType));

        /* Load the new Session Upload MAC key to Crypto. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_LoadKeyDirect(
            pDataParams->pPLUpload_MACCryptoDataParams,
            &pPLReKey[bPLReKeyLen / 2],
            pDataParams->bPLUploadKeyType));
    }

    /* Check if SUCCESS_CHAINING is returned. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);

    /* Compute MAC on Response and verify with the received one. */
    if(wStatus == PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL) && bIsFinalFrame)
    {
        /* Reset the MAC response buffer and its length. */
        bMacLen = 0;
        PHHAL_HW_SAM_CLEAR_BUFFER(aRespMAC);

        /* Load the constant values and INS. */
        aRespMAC[bMacLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(PHHAL_HW_SAM_RET_CODE_OK);
        aRespMAC[bMacLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(PHHAL_HW_SAM_RET_CODE_OK);
        aRespMAC[bMacLen++] = PHHAL_HW_CMD_INS_SAM_PL_UPLOAD;

        aRespMAC[bMacLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
        aRespMAC[bMacLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);

        /* Append SAM UID to MAC computation buffer. */
        memcpy(&aRespMAC[bMacLen], pDataParams->bUid, 7U);
        bMacLen += (uint8_t) 7U /* Sam UID Length. */;

        /* Load the Session Upload MAC key to Crypto. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pPLUpload_MACCryptoDataParams,
            pDataParams->aPLUploadSessMAC0,
            pDataParams->bPLUploadKeyType));

        /* Buffer the Command buffer and UploadCtr (if available) for macing. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pPLUpload_MACCryptoDataParams,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            aRespMAC,
            bMacLen,
            aRespMAC,
            &bMacLen));

        /* Truncate the computed MAC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(aRespMAC, &bMacLen));

        /* Compare the computed MAC with the one received in the response. */
        if(memcmp(aRespMAC, pResponse, PHHAL_HW_CMD_SAM_TRUNCATED_MAC_SIZE) != 0x00)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
