/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-3 command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_ISO14443_3.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCmdCode,
    uint8_t * pAtqa)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pAtqa, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-3_Request_Wakeup command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_REQUEST_WAKEUP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the ISO14443-3 command code to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCmdCode;

    /* Add LE to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange ISO14443-3_Request_Wakeup command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Finally we can copy the ATQA */
    memcpy(pAtqa, pResponse, 2U);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_AnticollisionSelect(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pSelCodes,
    uint8_t bSelCodesLen, uint8_t * pSak, uint8_t * pUid, uint8_t * pUidLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSelCodes, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSak, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-3_Anticollision_Select command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_ANTICOLLSION_SELECT;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add selection codes to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pSelCodes, bSelCodesLen);
    bCmdLen += bSelCodesLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange ISO14443-3_Anticollision_Select command information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_ISO_UID_INCOMPLETE &&
        (wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        return wStatus;
    }

    /* Extract the SAK information. */
    *pSak = pResponse[0];

    /* Extract the UID information and its length. */
    *pUidLen = (uint8_t) (wRespLen - 1U);
    memcpy(pUid, &pResponse[1U], *pUidLen);

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_ActivateIdle(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t bNumCards, uint16_t wTime, uint8_t * pAtqaIn, uint8_t * pSakIn, uint8_t ** ppResponse,
    uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOption & PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_ATQA)
        PH_ASSERT_NULL_PARAM(pAtqaIn, PH_COMP_HAL);
    if(bOption & PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_SAK)
        PH_ASSERT_NULL_PARAM(pSakIn, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-3_ActivateIdle command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_ACTIVATE_IDLE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNumCards;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (bOption & 0x03U);

    /* Add default LC byte if there is payload data. */
    if(bOption != 0)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;
    }

    /* Add Time information to command buffer. */
    if(bOption & PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_TIME)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wTime);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wTime);
    }

    /* Add AtqA filter information to command buffer. */
    if(bOption & PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_ATQA)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAtqaIn[0U];
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAtqaIn[1U];
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAtqaIn[2U];
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pAtqaIn[3U];
    }

    /* Add SAK filter information to command buffer. */
    if(bOption & PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_SAK)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pSakIn[0U];
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = pSakIn[1U];
    }

    /* Buffer command information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Recalculate the LC information. */
    if(bOption != 0)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }

    /* Exchange command information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_ActivateWakeUp(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pUid,
    uint8_t bUidLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-3_ActivateWakeup command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_ACTIVATE_WAKEUP;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bUidLen;

    /* Add UID to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pUid, bUidLen);
    bCmdLen += bUidLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchagne ISO14443-3_ActivateWakeup command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_HaltA(phhalHw_Sam_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-3_HaltA command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_HALTA;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchagne ISO14443-3_HaltA command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_TransparentExchange(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pTxBuf,
    uint8_t bTxLen, uint8_t bTxBitLen, uint8_t ** ppRxBuf, uint16_t * pRxLen, uint8_t * pRxBitLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pTxBuf, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRxLen, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRxBitLen, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Reset response length */
    *pRxLen = 0;

    /* Frame ISO14443-3_TransparentExchange command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bTxBitLen;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Data information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pTxBuf, bTxLen);
    bCmdLen += bTxLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchagne ISO14443-3_TransparentExchange command information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppRxBuf,
        pRxLen);

    /* Check for incomplete byte status code. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
    {
        PH_CHECK_SUCCESS(wStatus);
        *pRxBitLen = 0;
    }
    else
    {
        *pRxBitLen = (uint8_t) pDataParams->wAdditionalInfo;
    }

    /* Check length */
    if(*pRxLen > PHHAL_HW_SAM_ISO7816_EXCHANGE_RESPONSE_MAX)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
