/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-4 command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_ISO14443_4.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCidIn, uint8_t bDsiIn,
    uint8_t bDriIn, uint8_t * pCidOut, uint8_t * pDsiOut, uint8_t * pDriOut, uint8_t * pAts)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCidOut, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pDsiOut, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pDriOut, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pAts, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-4_RATS_PPS command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_4_RATS_PPS;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCidIn;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDriIn;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDsiIn;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange ISO14443-4_RATS_PPS command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    *pCidOut = pResponse[0U];
    *pDriOut = pResponse[1U];
    *pDsiOut = pResponse[2U];

    /* Copy the ATS information. */
    memcpy(pAts, &pResponse[3U], wRespLen - 3U);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_Init(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCid, uint8_t bDri, uint8_t bDsi,
    uint8_t bFwi, uint8_t bFsci)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-4_Init command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_4_INIT;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bCid;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDri;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDsi;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFwi;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFsci;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange ISO14443-4_Init command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_Exchange(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pAppDataIn,
    uint8_t bLenAppData, uint8_t ** ppAppDataOut, uint16_t * pAppDataOutLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bLEPresent = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pAppDataIn, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pAppDataOutLen, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Check if LE should be added. */
    bLEPresent = (uint8_t) ((wOption & PH_EXCHANGE_TXCHAINING) ? PH_OFF : PH_ON);

    /* Frame ISO14443-4_Exchange command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_4_EXCHANGE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Update command information with chaining frame. */
    if(wOption & PH_EXCHANGE_TXCHAINING)
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
    }

    /* Add Application data to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pAppDataIn, bLenAppData);
    bCmdLen += bLenAppData;

    /* Add LE to command buffer */
    if(bLEPresent == (uint8_t) PH_ON)
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, bLEPresent);

    /* Exchange ISO14443-4_Exchange command information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Return received data */
    *pAppDataOutLen = wRespLen;
    *ppAppDataOut = pResponse;

    /* Check status */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    /* Check length */
    if(wRespLen > (PHHAL_HW_SAM_ISO7816_EXCHANGE_RESPONSE_MAX))
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_PresenceCheck(phhalHw_Sam_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-4_PresenceCheck command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_4_PRESENCE_CHECK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchange ISO14443-4_PresenceCheck command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    if(wRespLen != 0x00)
    {
        /* Remapping of return values */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_Deselect(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bFreeCid)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-4_Deselect command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_4_DESELECT;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bFreeCid;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Exchange ISO14443-4_Deselect command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    if(wRespLen != 0x00)
    {
        /* Remapping of return values */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_FreeCid(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pCid, uint8_t bCidLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCid, PH_COMP_HAL);

    /* Reset command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame ISO14443-4_FreeCID command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO14443_4_FREE_CID;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add CID information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pCid, bCidLen);
    bCmdLen += bCidLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange ISO14443-4_FreeCID command information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    if(wRespLen != 0x00)
    {
        /* Remapping of return values */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
