/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-4 command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_SAM_CMD_ISO14443_4_H
#define PHHALHW_SAM_CMD_ISO14443_4_H

#include <ph_Status.h>
#include <phhalHw.h>

#include "../../phhalHw_Sam.h"

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_4 16_ISO14443_4
 * \brief SAM commands used for ISO14443 layer 4 communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_4_Defines Defines
 * \brief Definitions for ISO14443 layer 4 commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_4_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_4_RATS_PPS                                0xE0U   /**< SAM Insturction code for Cmd.ISO14443-4_RATS_PPS command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_4_INIT                                    0x11U   /**< SAM Insturction code for Cmd.ISO14443-4_Init command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_4_EXCHANGE                                0xECU   /**< SAM Insturction code for Cmd.ISO14443-4_Exchange command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_4_PRESENCE_CHECK                          0x4CU   /**< SAM Insturction code for Cmd.ISO14443-4_Init command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_4_DESELECT                                0xD4U   /**< SAM Insturction code for Cmd.ISO14443-4_Deselect command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_4_FREE_CID                                0xFCU   /**< SAM Insturction code for Cmd.ISO14443-4_Init command. */
/**
 * end of group phhalHw_Sam_Cmd_ISO14443_4_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_4_Defines_Deselect DeSelect
 * \brief Definitions for ISO14443 - 4 DeSelect.
 * To be used with \ref phhalHw_Sam_Cmd_X_ISO14443_4_Deselect
 * "ISO14443_4_Deselect" interface
 * @{
 */
#define PHHAL_HW_SAM_CMD_ISO14443_4_DESELECT_DO_NOT_FREE_CID                    0x00U   /**< Option to not free CID is deselect fails. */
#define PHHAL_HW_SAM_CMD_ISO14443_4_DESELECT_FORCE_FREE_CID                     0x01U   /**< Option to forcefully free CID in any case. */
/**
 * end of group phhalHw_Sam_Cmd_ISO14443_4_Defines_Deselect
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_ISO14443_4_Defines
 * @}
 */

/**
 * \brief Perform a combined RATS and PPS sequence to prepare a PICC for T = CL data exchange.
 *  - The CID assigned to the PICC will be assigned to the current logical channel. This means,
 *    every further ISO14443-4 command issued in this logical channel will be executed using
 *    this CID automatically
 *  - If a CID value of FFh is passed to the MIFARE SAM, the MIFARE SAM will automatically assign a
 *    CID to the PICC. Therefore it takes already assigned CIDs of other logical channels into account.
 *    If a PICC is activated in any other logical channel with an assigned CID of 00h, or not supporting
 *    a CID, the PICC will not be activated and the command returns an error. Also all user-defined CIDs
 *    will be checked against the already activated PICCs and rejected if the PICC cannot be activated
 *    following the ISO/IEC 14443-4 standard.
 *  - For bit rate selection, the values in the data field define the maximum bit rates. If a PICC does
 *    not support these bit rates, the MIFARE SAM will automatically decide the bit rate to apply.
 *  - The MIFARE SAM returns the actual CID, DRI, DSI and the ATS. If a PICC was activated, which does not
 *    support a CID, a CID value of 0 is returned.
 *  - The MIFARE SAM automatically configures the frame sizes to apply for T=CL data exchanges. As for a
 *    bit rate of 848 kbit/s FSD is limited to 96 byte, it is recommended to not indicate a DSI value of
 *    3 in case the PICC is not able to support it, if already known in advance. The activation procedure
 *    will succeed since the MIFARE SAM automatically chooses the next applicable bit rate, but since the
 *    FSDI value has already to be sent to the PICC in the RATS command, the MIFARE SAM will indicate an
 *    FSD of 96 byte supposing that 848 kbit/s will be used for communication. The actually supported bit
 *    rate of the PICC is not known to the MIFARE SAM at this time (returned in ATS afterwards) so also
 *    if a lower bit rate is selected later on, FSD will be at the non-optimum value of 96 byte.
 *  - It could happen that the command indicates an error although a PICC has been activated correctly by
 *    the RATS command. This is the case if the subsequently sent PPS failed.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pCidOut, \b pDsiOut, \b pDriOut
 *                                      and \b pAts are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS (
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCidIn,                                                                 /**< [In] CID to be sent. */
        uint8_t bDsiIn,                                                                 /**< [In] DSI to be sent. One of the following,
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_106
                                                                                         *                 "106 kbit/s Bitrate, PCD to PICC"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_212
                                                                                         *                 "212 kbit/s Bitrate, PCD to PICC"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_424
                                                                                         *                 "424 kbit/s Bitrate, PCD to PICC"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_848
                                                                                         *                 "848 kbit/s Bitrate, PCD to PICC"
                                                                                         */
        uint8_t bDriIn,                                                                 /**< [In] DRI to be sent. One of the following,
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_106
                                                                                         *                 "106 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_212
                                                                                         *                 "212 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_424
                                                                                         *                 "424 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_848
                                                                                         *                 "848 kbit/s Bitrate, PICC to PCD"
                                                                                         */
        uint8_t * pCidOut,                                                              /**< [Out] CID used (1 byte). */
        uint8_t * pDsiOut,                                                              /**< [Out] DSI used (1 byte). One of the following,
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_106
                                                                                         *                 "106 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_212
                                                                                         *                 "212 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_424
                                                                                         *                 "424 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_848
                                                                                         *                 "848 kbit/s Bitrate, PICC to PCD"
                                                                                         */
        uint8_t * pDriOut,                                                              /**< [Out] DRI used (1 byte). One of the following,
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_106
                                                                                         *                 "106 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_212
                                                                                         *                 "212 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_424
                                                                                         *                 "424 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_848
                                                                                         *                 "848 kbit/s Bitrate, PICC to PCD"
                                                                                         */
        uint8_t * pAts                                                                  /**< [Out] Buffer containing the received ATS (Answer to Select).
                                                                                         *         The length of the ATS can be found on the first position.
                                                                                         */
    );

/**
 * \brief Perform a init of ISO-14443-4 layer (init T = CL protocol). The intent of this command
 * is to configure the protocol for data exchanges. This is necessary if a PICC was already activated
 * and configured for doing data exchanges without using the \ref phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS
 * "ISO14443-4_RATS_PPS" command.
 *  - The CID assigned to the PICC will be assigned to the current logical channel. This means, every
 *    further ISO14443-4 command issued in this logical channel, will be executed using this CID
 *    automatically. If in any other logical channel a PICC is activated with an assigned CID of 00h or
 *    not supporting a CID, the PICC will not be activated and the command returns an error. Also, all
 *    user-defined CIDs will be checked against the already activated PICCs and rejected if the PICC
 *    cannot be activated following the ISO/IEC 14443-4 standard. If a PICC does not support a CID, this
 *    has to be indicated by a CID value of FFh.
 *  - The MIFARE SAM will automatically adjust FSD according to the DSI input value. The user driving the
 *    activation sequence is responsible for indicating the correct FSDI value to the PICC. For details on
 *    applied frame sizes pleas refer to ISO14443-4_Exchange
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_Init(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCid,                                                                   /**< [In] CID to apply. One of the following
                                                                                         *          - CID to apply: 0x00 - 0x0E
                                                                                         *          - No CID to apply: 0xFF
                                                                                         */
        uint8_t bDri,                                                                   /**< [In] DSI to be sent. One of the following,
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_106
                                                                                         *                 "106 kbit/s Bitrate, PCD to PICC"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_212
                                                                                         *                 "212 kbit/s Bitrate, PCD to PICC"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_424
                                                                                         *                 "424 kbit/s Bitrate, PCD to PICC"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_848
                                                                                         *                 "848 kbit/s Bitrate, PCD to PICC"
                                                                                         */
        uint8_t bDsi,                                                                   /**< [In] DRI to be sent. One of the following,
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_106
                                                                                         *                 "106 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_212
                                                                                         *                 "212 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_424
                                                                                         *                 "424 kbit/s Bitrate, PICC to PCD"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_RF_DATARATE_848
                                                                                         *                 "848 kbit/s Bitrate, PICC to PCD"
                                                                                         */
        uint8_t bFwi,                                                                   /**< [In] Frame waiting time indicator (0x00 - 0x0F). */
        uint8_t bFsci                                                                   /**< [In] Frame size card indicator (0x00 - 0x08). */
    );

/**
 * \brief Perform an exchange bytes with a PICC according ISO14443-4 T = CL protocol.
 * The MIFARE SAM offers the possibility to send more data to a PICC than the maximum data field size.
 * In this case the last frame indicator shall indicate AFh. The MIFARE SAM will then set the chaining bit
 * of the last sent I-block indicating to the PICC that more data will follow and to continue the chaining
 * process after receiving the next command from the host. If more than 256 bytes are to be received from
 * the PICC, the MIFARE SAM will not acknowledge the last received chained I-block from the PICC, returns
 * the data to the host and waits for the next command to continue the chaining process. The following
 * exchange command from the host shall not contain any data, since more data has to be received from the
 * PICC. However, if there is a data field content in the command, the MIFARE SAM will ignore it and
 * continue reception.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with chaining status.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pAppDataIn and \b pAppDataOutLen are NULL.
 * \retval #PH_ERR_LENGTH_ERROR         If received response length is greater than
 *                                      \ref PHHAL_HW_SAM_ISO7816_EXCHANGE_RESPONSE_MAX
 *                                      "Maximum Exchange Response length"
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_Exchange (
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Bufferring Option.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "To send and receive complete data"
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING
                                                                                         *                 "To exchange chainned data". Here
                                                                                         *                 \ref PHHAL_HW_SAM_ISO7816_CHAINED_FRAME
                                                                                         *                 "Chainned Exchange" flag will be set by the
                                                                                         *                 library internally to P1 information byte.
                                                                                         */
        uint8_t * pAppDataIn,                                                           /**< [In] Buffer containing application data to sent. */
        uint8_t bLenAppData,                                                            /**< [In] Length of bytes available in \b pAppDataIn buffer. */
        uint8_t ** ppAppDataOut,                                                        /**< [Out] Buffer containing the data returned by the PICC. */
        uint16_t * pAppDataOutLen                                                       /**< [Out] Length of bytes available in \b ppAppDataOut buffer. */
    );

/**
 * \brief Check if an activated card is still in the field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_PresenceCheck(
        phhalHw_Sam_DataParams_t * pDataParams                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Perform de-selection of an activated PICC. The CID is freed by this command. If the deselect fails,
 * the CID will not be freed and cannot be used for activating another PICC. This behavior might be overridden
 * by setting a flag in the P1 byte. CIDs can also be freed using the \ref phhalHw_Sam_Cmd_X_ISO14443_4_FreeCid
 * "ISO14443-4_FreeCID" command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_Deselect(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFreeCid                                                                /**< [In] Bitmask for deselect option.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_4_DESELECT_DO_NOT_FREE_CID
                                                                                         *                 "CID will not be freed if deselect fails"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_4_DESELECT_FORCE_FREE_CID
                                                                                         *                 "Force CID to be freed in any case"
                                                                                         */
    );

/**
 * \brief Free up one, more, or all currently assigned CIDs.
 *  - This command might be necessary if several deselect commands failed and the CIDs were not forced to be
 *    freed but the PICC is deactivated or no longer available in the field.
 *  - The command takes at least one CID value in the data field. A CID value of FFh indicates that all CIDs have to
 *    be freed. Sending the same CID several times will not cause an error. The command is not related to a logical
 *    channel since CIDs are managed in a global pool.
 *  - Be careful when calling this command. It may free CIDs in logical channels in which PICCs are still activated.
 *    The activated PICCs can then no longer be accessed. Freeing the CID additionally causes the MIFARE SAM to reset
 *    the state of the logical channel related to the CID, thus the MIFARE SAM clears the indication for an activated
 *    PICC.
 *  - If an incorrect CID (value higher than 14) is detected, the command is aborted. Nevertheless all CIDs which were
 *    contained in the data field previous to the incorrect one are freed. The value FFh will cause all CIDs to be
 *    freed independent from its occurring position in the data field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pCid is NULL.
 * \retval #PH_ERR_LENGTH_ERROR         In case if SAM provides data in response.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_4_FreeCid(
        phhalHw_Sam_DataParams_t * pDataParams,                             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pCid,                                                     /**< [In] Buffer containing all CIDs to be freed.
                                                                             *          - CID: 0x00 - 0x0E
                                                                             *          - Free all CIDs: 0xFF
                                                                             */
        uint8_t bCidLen                                                     /**< [In] Length of bytes available in \b pCid buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_ISO14443_4
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_ISO14443_4_H */
