/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Originality Check command implementation of Reader
 * Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7232 $
 * $Date: 2025-03-14 18:16:11 +0530 (Fri, 14 Mar 2025) $
 */

#ifndef PHHALHW_SAM_CMD_OC_H
#define PHHALHW_SAM_CMD_OC_H

#include <ph_Status.h>
#include <phhalHw.h>

#include "../../phhalHw_Sam.h"

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_RND_CHALLENGE_LENGTH                                       16U     /* Length of Random Challange (RndA) */
#define PHHAL_HW_ORIGINALITY_KEY                                            0xF0U   /* Originality Key number in SAM */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_OC 18_OriginalityCheck
 * \brief SAM commands used for OriginalityCheck communication in S-Mode and X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_OC_Defines Defines
 * \brief Definitions for Originality Check commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_OC_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_ISO_INTERNAL_AUTHENTICATE                          0x88U   /**< SAM Insturction code for Cmd.SAM_ISOInternalAuthenticate
                                                                                         *   command.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_OC_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_OC_Defines_Challenge Challenge
 * \brief Options to be used for RndA usage.
 * @{
 */
#define PHHAL_HW_GENERATE_CHALLENGE_INTERNAL                                    0x00U   /**< Option to generate the random number internally. */
#define PHHAL_HW_GENERATE_CHALLENGE_USER_PROVIDED                               0x01U   /**< Option to use the random number provided by user. */
/**
 * end of group phhalHw_Sam_Cmd_OC_Defines_Challenge
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_OC_Defines
 * @}
 */

/**
 * \brief Performs Originality Check of SAM . The authentication is initiated by ISOInternalAuthenticate.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOptsA, \b pRndA,
 *                                      \b pRndALen and
 *                                      \b pRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ISOInternalAuthenticate (
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Challenge usage (RndA)
                                                                                         *          - \ref PHHAL_HW_GENERATE_CHALLENGE_INTERNAL
                                                                                         *            "Generate Challenge internally and exchange"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_GENERATE_CHALLENGE_USER_PROVIDED
                                                                                         *            "Exchange the Challenge provided by user"
                                                                                         */
        uint8_t * pOptsA,                                                               /**< [In] PCD Option.
                                                                                         *          - Will be exchanged if \b bOptsALen has a value
                                                                                         *            other than zero
                                                                                         *          - User should provide Value field only. Tag and
                                                                                         *            Length will be included internally.
                                                                                         */
        uint8_t bOptsALen,                                                              /**< [In] Length of bytes available in \b pOptsA buffer. */
        uint8_t * pRndA,                                                                /**< [In, Out] Authentication Data Object: random challenge from
                                                                                         *        PCD.
                                                                                         *          - Empty buffer if \ref PHHAL_HW_GENERATE_CHALLENGE_INTERNAL
                                                                                         *            "Generate Challenge internally" is provided in
                                                                                         *            \b bOption parameter. Will be updated with Random Challenge
                                                                                         *            generated internally. The buffer size should be equal to
                                                                                         *            the RndA size that is expected.
                                                                                         *
                                                                                         *          - Random Challenge if \ref PHHAL_HW_GENERATE_CHALLENGE_USER_PROVIDED
                                                                                         *            "Challenge provided by user" is provided in
                                                                                         *            \b bOption parameter. User should provide Value
                                                                                         *            field only. AuthDOHdr (Tag and Length) will be
                                                                                         *            included internally.
                                                                                         */
        uint8_t * pRndALen,                                                             /**< [In, Out] Length of bytes available in \b pRndA buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] RndB and Signature returned by SAM. */
        uint16_t * pRespLen                                                             /**< [Out] Length pf bytes available in \b ppResponse parameter. */
    );

/**
 * end of group phhalHw_Sam_Cmd_OC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_OC_H */
