/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) generic command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_H
#define PHHALHW_SAM_CMD_H


#include <ph_Status.h>
#include <ph_TypeDefs.h>
#include <ph_RefDefs.h>

#include <phCryptoSym.h>

#include <phhalHw.h>
#include "../phhalHw_Sam.h"
#include "../Utils/phhalHw_Sam_HcUtils.h"

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/* Global varaibles */
extern uint8_t phhalHw_Sam_Cmd_DefaultLe[1U];
extern const uint8_t PH_MEMLOC_CONST_ROM phhalHw_Sam_Cmd_ZeroIV[PH_CRYPTOSYM_AES_BLOCK_SIZE];

#define PHHAL_HW_CMD_SAM_LE_LENGTH                                              0x01U   /* LE length. */

#define PHHAL_HW_SAM_CMD_BUFFER                                                 pDataParams->pCmdBuff
#define PHHAL_HW_SAM_CMD_BUFFER_SIZE                                            pDataParams->wCmdBuffSize

#define PHHAL_HW_SAM_CLEAR_BUFFER(Buffer)                                       memset((Buffer), 0x00, sizeof((Buffer)))
#define PHHAL_HW_SAM_CLEAR_CMD_BUFFER()                                         memset(PHHAL_HW_SAM_CMD_BUFFER, 0x00, sizeof(PHHAL_HW_SAM_CMD_BUFFER_SIZE))

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Defines Defines
 * \brief Definitions generic for S and X mode commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Defines_CommMode CommunicationMode
 * \brief Definitions for Communication Modes to be used for Data
 * Processing / PICC commands.
 * To be used for below mentioned interfaces
 *  - \ref phhalHw_Sam_Cmd_SAM_ApplySM "SAM_Apply_SM"
 *  - \ref phhalHw_Sam_Cmd_SAM_RemoveSM "SAM_Remove_SM"
 *  - \ref phhalHw_Sam_Cmd_DESFire_WriteX "DESFire_WriteX"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ReadX "DESFire_ReadX"
 *  - \ref phhalHw_Sam_Cmd_SAM_BindCertificate_Part1 "BindCertificate Part1"
 * @{
 */
#define PHHAL_HW_CMD_SAM_COMM_MODE_PLAIN                                        0x00U   /**< SAM communication mode as Plain. */
#define PHHAL_HW_CMD_SAM_COMM_MODE_MAC                                          0x10U   /**< SAM communication mode as MAC. */
#define PHHAL_HW_CMD_SAM_COMM_MODE_FULL                                         0x30U   /**< SAM communication mode as Full. */
/**
 * end of group phhalHw_Sam_Cmd_Defines_CommMode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Defines_ISOMode ISOMode
 * \brief Definitions for ISO mode selection.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC "DESFire_CreateTMFilePICC"
 *  - \ref phhalHw_Sam_Cmd_DUOX_BindCertificate "DUOX_BindCertificate"
 * @{
 */
#define PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE                                        0x00U   /**< SAM ISO mode selection for Native command set. */
#define PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816                                       0x40U   /**< SAM ISO mode selection for ISO 7816-4 command
                                                                                         *   set.
                                                                                         */
#define PHHAL_HW_CMD_SAM_ISO_MODE_ISO_AUTHENTICATION                            0x80U   /**< SAM ISO mode selection for Iso complaint
                                                                                         *   Authentication.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_Defines_ISOMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Defines
 * @}
 */

/**
 * \brief Exchange commands with the SAM.
 * \note If this interface is called with one of the below mentioned buffering options and if LC needs to
 *  be updated before the final call with \b wOption = \ref PH_EXCHANGE_BUFFER_LAST "Buffer last information",
 *  then \ref phhalHw_Sam_Utils_UpdateLc "Update LC" interface can be used. \ref phhalHw_Sam_Utils_UpdateLc
 *  "Update LC" will update the LC byte which the data that is loaded via \b pData parameter.
 *  - \b wOption = \ref PH_EXCHANGE_BUFFER_FIRST "Buffer first set of information".
 *    Exchange is not performed
 *  - \b wOption = \ref PH_EXCHANGE_BUFFER_CONT "Buffer intermediate information".
 *    Exchange is not performed
 *  - \b wOption = \ref PH_EXCHANGE_BUFFER_LAST "Buffer last set of information".
 *    Exchange and received information from SAM
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                          Operation successful.
 * \retval #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE     Operation successful chaining.
 * \retval #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_EXT Operation successful chaining.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_7816Exchange(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Exchange and Receive"
                                                                                         *                 information from SAM
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first set of information".
                                                                                         *                 Exchange is not performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate information".
                                                                                         *                 Exchange is not performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last information",
                                                                                         *                 exchange and received information from SAM
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING "Exchange chained data"
                                                                                         *          - \ref PH_EXCHANGE_RXCHAINING "Receive chained data"
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The command to be sent to Sam hardware. */
        uint16_t wDataLen,                                                              /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] The data received form SAM hardware. */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse. */
    );

/**
 * \brief Update the LC information for SAM command frame.
 * \note Can be used to update LC of SAM command frame before calling \ref phhalHw_Sam_Cmd_7816Exchange
 * "Cmd_7816Exchange" with \b wOption = \ref PH_EXCHANGE_BUFFER_LAST "Buffer last set of information",
 *  Exchange and received information from SAM
 *
 */
void phhalHw_Sam_Cmd_UpdateLC(
        uint8_t * pData,                                                                /**< [In, Out] Input data having the complete command information
                                                                                         *      - [In]: The buffer having command information with LC as zero.
                                                                                         *      - [Out]: The buffer having command information with LC updated.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer */
        uint8_t bLE_Available                                                           /**< [In] Is LE available in the \b pData buffer.
                                                                                         *          - \ref PH_OFF "LE Not Present"
                                                                                         *          - \ref PH_ON "LE Present"
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_H */
