/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Host Secure Messging implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phKeyStore.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "phhalHw_Sam_HSM_AES.h"

#include "../phhalHw_Sam.h"
#include "../Utils/phhalHw_Sam_HcUtils.h"

#include "../Commands/01_HostCommunication/phhalHw_Sam_Cmd_HC.h"
#include "../Commands/02_SecurityConfiguration/phhalHw_Sam_Cmd_SC.h"
#include "../Commands/03_KeyManagement/phhalHw_Sam_Cmd_KM.h"
#include "../Commands/05_DataProcessing/phhalHw_Sam_Cmd_DP.h"
#include "../Commands/06_PKI/phhalHw_Sam_Cmd_PKI.h"
#include "../Commands/07_VirtualCard/phhalHw_Sam_Cmd_VC.h"
#include "../Commands/08_DESFire/phhalHw_Sam_Cmd_DESFire.h"
#include "../Commands/09_DUOX/phhalHw_Sam_Cmd_DUOX.h"
#include "../Commands/10_Plus/phhalHw_Sam_Cmd_Plus.h"
#include "../Commands/11_Ultralight/phhalHw_Sam_Cmd_UL.h"
#include "../Commands/12_Common/phhalHw_Sam_Cmd_Common.h"
#include "../Commands/13_ICODE/phhalHw_Sam_Cmd_iCode.h"
#include "../Commands/14_ProgrammableLogic/phhalHw_Sam_Cmd_PL.h"
#include "../Commands/15_ReaderIC/phhalHw_Sam_Cmd_RC.h"
#include "../Commands/16_ISO14443_3/phhalHw_Sam_Cmd_ISO14443_3.h"
#include "../Commands/17_ISO14443_4/phhalHw_Sam_Cmd_ISO14443_4.h"
#include "../Commands/18_OriginalityCheck/phhalHw_Sam_Cmd_OC.h"

/*
* Private constants
*/
static const uint8_t PH_MEMLOC_CONST_ROM phhalHw_Sam_HSM_AES_ZeroIv[PH_CRYPTOSYM_AES_BLOCK_SIZE] =
{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

phStatus_t phhalHw_Sam_HSM_AES_Encrypt(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pBuffer, uint16_t wBuffLen, uint16_t wBufferSize,
    uint16_t * pBuffLen, uint8_t bFirst, uint8_t bLast)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wHelper = 0;
    uint16_t    PH_MEMLOC_REM wCurrentEncryptedDataSize = 0;
    uint16_t    PH_MEMLOC_REM wRemainingDataSize = 0;
    uint8_t     PH_MEMLOC_REM bIsLePresent = PH_OFF;
    uint8_t     PH_MEMLOC_REM bIsLcPresent = PH_OFF;
    uint8_t     PH_MEMLOC_REM bLc = 0;
    uint8_t     PH_MEMLOC_REM bLe = 0;

    uint8_t     PH_MEMLOC_REM aTmpBuf[16];

    *pBuffLen = wBuffLen;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GetCheckLcLe(pBuffer, wBuffLen, &bIsLcPresent, &bLc, &bIsLePresent));

    if(bFirst)
    {
        /* load the InitializationVector, because we start a new Encryption.
         * This has to be done even if no data are send to the SAM.
         */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_HSM_AES_InitAndLoadIV(pDataParams, pDataParams->bPendingCmdIv, true));
    }
    else
    {
        /* Load encryption IV */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pENCCryptoDataParams,
            pDataParams->bPendingCmdIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));
    }

    /* Do we need encryption at all? */
    if(!bLc)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Check for overall size */
    if((uint16_t) (wBuffLen) > (uint16_t) (wBufferSize - 15U))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /*/////////////////////////
    // Start Encryption Process
    ///////////////////////////
    // At this point, the encryption process can be started and processed
    // we only have to know if a frame chaining was running*/

    /* Save Le byte */
    bLe = pBuffer[wBuffLen - 1U];

    if(bFirst)
    {
        /* Find all blocks but the last block */
        wRemainingDataSize = (uint16_t) (bLc % 16U);

        wCurrentEncryptedDataSize = ((uint16_t) bLc - wRemainingDataSize);

        if(wCurrentEncryptedDataSize)
        {
            /* Encrypt everything but the last block*/
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pENCCryptoDataParams,
                (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT),
                &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH],
                wCurrentEncryptedDataSize,
                &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH]));
        }

        /* Recopy remaining part */
        memcpy(aTmpBuf, &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wCurrentEncryptedDataSize], wRemainingDataSize);
    }
    else
    {
        /* How much data do we still have? - wHelper contains already consumed data out of pBuffer */
        wHelper = (uint16_t) (16U - pDataParams->bPendingEncCmdDataLength);
        wCurrentEncryptedDataSize = 0;

        /* Do we have sufficient user Payload? */
        if(wHelper > bLc)
        {
            memcpy(aTmpBuf, pDataParams->bPendingEncCmdData, pDataParams->bPendingEncCmdDataLength);
            memcpy(&aTmpBuf[pDataParams->bPendingEncCmdDataLength], &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH], bLc);
            wRemainingDataSize = pDataParams->bPendingEncCmdDataLength + bLc;
        }
        else
        {
            memcpy(&pDataParams->bPendingEncCmdData[pDataParams->bPendingEncCmdDataLength], &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH],
                wHelper);

            /* Encrypt first block*/
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pENCCryptoDataParams,
                (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT),
                pDataParams->bPendingEncCmdData,
                16U,
                pDataParams->bPendingEncCmdData));

            wRemainingDataSize = (uint16_t) ((bLc - wHelper) % 16U);

            /* Next blocks we can now encipher inline */
            wCurrentEncryptedDataSize = (uint16_t) (bLc - wRemainingDataSize - wHelper);

            if(wCurrentEncryptedDataSize)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                    pDataParams->pENCCryptoDataParams,
                    (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT),
                    &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wHelper],
                    wCurrentEncryptedDataSize,
                    &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wHelper]));

                /* Now move the data to the TmpBuf*/
                memcpy(aTmpBuf, &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wHelper + wCurrentEncryptedDataSize], wRemainingDataSize);

                /* Now move encrypted payload to the end */
                memmove(&pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + 16U], &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wHelper],
                    (uint16_t) wCurrentEncryptedDataSize);

            }
            else
            {
                /* Recopy remaining part */
                memcpy(aTmpBuf, &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wHelper], wRemainingDataSize);
            }

            /* Now copy the stuff to the front */
            memcpy(&pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH], pDataParams->bPendingEncCmdData, 16U);
            wCurrentEncryptedDataSize = (uint16_t) (wCurrentEncryptedDataSize + 16U);
        }
    }

    /* Is this the last command in a sequence? */
    if(bLast)
    {
        /* copy temporary buffer to the end of the Tx Buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_ApplyPadding(
            PH_CRYPTOSYM_PADDING_MODE_2,
            aTmpBuf,
            wRemainingDataSize,
            PH_CRYPTOSYM_AES_BLOCK_SIZE,
            16U,
            aTmpBuf,
            &wRemainingDataSize));

        /* now encrypt the data */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
            pDataParams->pENCCryptoDataParams,
            (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_LAST),
            aTmpBuf,
            wRemainingDataSize,
            &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wCurrentEncryptedDataSize]));

        wCurrentEncryptedDataSize = (uint16_t) (wCurrentEncryptedDataSize + 16U);
    }
    else
    {
        if(wCurrentEncryptedDataSize >= 16U)
        {
            /* Recopy last block of the encrypted data into the temporary IV space */
            memcpy(pDataParams->bPendingCmdIv, &pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wCurrentEncryptedDataSize - 16U],
                sizeof(pDataParams->bPendingCmdIv));
        }

        /* Copy data into pending data structure */
        memcpy(pDataParams->bPendingEncCmdData, aTmpBuf, wRemainingDataSize);
        pDataParams->bPendingEncCmdDataLength = (uint8_t) (wRemainingDataSize);
    }

    /* Update Lc */
    pBuffer[PHHAL_HW_SAM_ISO7816_LC_POS] = (uint8_t) wCurrentEncryptedDataSize;

    /* Update overall length */
    if(bLc > wCurrentEncryptedDataSize)
    {
        bLc = (uint8_t) (bLc - wCurrentEncryptedDataSize);
        wBuffLen = (uint16_t) (wBuffLen - bLc);
    }
    else
    {
        bLc = (uint8_t) (wCurrentEncryptedDataSize - bLc);
        wBuffLen = (uint16_t) (wBuffLen + bLc);
    }

    /* Update Le */
    if(bIsLePresent)
    {
        pBuffer[wBuffLen - 1U] = bLe;
    }

    *pBuffLen = wBuffLen;
    /* End Encryption Process */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_HSM_AES_Decrypt(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pBuffer, uint16_t wBuffLen, uint16_t * pBuffLen,
    uint8_t bFirst, uint8_t bLast)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    *pBuffLen = 0;

    if(bFirst)
    {
        /* Load the InitializationVector (phCryptoSym_CryptoPP_LoadIv), because we start a new decryption.
         * This has to be done even if no data is returned by the SAM.
         */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_HSM_AES_InitAndLoadIV(
            pDataParams,
            pDataParams->bPendingRespIv,
            false));
    }
    else
    {
        /* Load decryption IV */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pENCCryptoDataParams,
            pDataParams->bPendingRespIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));
    }

    if(wBuffLen < (16U /* Data */ + PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH))
    {
        *pBuffLen = wBuffLen;

        /* Obviously, no data is available */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    if(((wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH) % 16U) != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    if(!bLast)
    {
        /* Recopy last block of the encrypted data into the temporary IV space */
        memcpy(pDataParams->bPendingRespIv, &pBuffer[wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH - 16U],
            sizeof(pDataParams->bPendingRespIv));
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pENCCryptoDataParams,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT),
        pBuffer,
        (uint16_t) (wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH),
        pBuffer));

    if(bLast)
    {
        /* remove padding in pPlainBuffer and Update the size of decrypted buffer*/
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_RemovePadding(
            PH_CRYPTOSYM_PADDING_MODE_2,
            pBuffer,
            (uint16_t) (wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH),
            PH_CRYPTOSYM_AES_BLOCK_SIZE,
            (uint16_t) (wBuffLen - PHHAL_HW_SAMAV2_ISO7816_SW1SW2_LENGTH),
            pBuffer,
            pBuffLen));

        /* Reorder SW1 SW2 */
        pBuffer[(*pBuffLen)++] = pBuffer[wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH];
        pBuffer[(*pBuffLen)++] = pBuffer[wBuffLen - 1U];
    }
    else
    {
        /* Set response length only */
        *pBuffLen = wBuffLen;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_HSM_AES_AppendMac(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pBuffer, uint16_t wBuffLen, uint16_t wBufferSize,
    uint16_t * pBuffLen, uint8_t bFirst, uint8_t bLast)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wHelper = 0;
    uint16_t    PH_MEMLOC_REM wValidMacData = 0;
    uint8_t     PH_MEMLOC_REM bLcPresent = PH_OFF;
    uint8_t     PH_MEMLOC_REM bLePresent = PH_OFF;
    uint8_t     PH_MEMLOC_REM bLc = 0;
    uint8_t     PH_MEMLOC_REM bLeValue = 0;

    uint8_t     PH_MEMLOC_REM aTmpBuf[16];
    uint8_t     PH_MEMLOC_REM bMacLen = 0;

    *pBuffLen = wBuffLen;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GetCheckLcLe(pBuffer, wBuffLen, &bLcPresent, &bLc, &bLePresent));

    /* Check the buffer size compare to the size of data to MAC*/
    if((wBuffLen) > (wBufferSize - 8))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* In case of non-first command and LC == 0 (response chaining no MAC should be appended */
    if((!bFirst) && (!bLcPresent))
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Remember Le */
    bLeValue = pBuffer[wBuffLen - 1];

    if(bFirst)
    {
        pDataParams->bPendingMacCmdDataLength = 0;

        /* load the initial IV, because we start a new MAC calculation */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pMACCryptoDataParams,
            phhalHw_Sam_HSM_AES_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        /* Also set the pending MAC to 0 */
        memset(pDataParams->bPendingCmdMac, 0, sizeof(pDataParams->bPendingCmdMac));

        /* Build the buffer to MAC */
        aTmpBuf[wValidMacData++] = pBuffer[PHHAL_HW_SAM_ISO7816_CLA_POS];
        aTmpBuf[wValidMacData++] = pBuffer[PHHAL_HW_SAM_ISO7816_INS_POS];
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0xFF000000U) >> 24U);
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x00FF0000U) >> 16U);
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x0000FF00U) >> 8U);
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x000000FFU) >> 0);

        /* In case of chaining detected, we need to load 0x00 for LFI */
        if((!bLast) && (pBuffer[PHHAL_HW_SAM_ISO7816_P1_POS] == 0xAFU))
        {
            aTmpBuf[wValidMacData++] = 0x00;
        }
        else
        {
            aTmpBuf[wValidMacData++] = pBuffer[PHHAL_HW_SAM_ISO7816_P1_POS];
        }

        if((!bLast) && (pBuffer[PHHAL_HW_SAM_ISO7816_P2_POS] == 0xAFU))
        {
            aTmpBuf[wValidMacData++] = 0x00;
        }
        else
        {
            aTmpBuf[wValidMacData++] = pBuffer[PHHAL_HW_SAM_ISO7816_P2_POS];
        }

        /* Chained commands have a LC == 0 */
        if(bLast)
        {
            aTmpBuf[wValidMacData++] = bLc + 8U;
            /* Also set updated LC in original buffer */
            pBuffer[PHHAL_HW_SAM_ISO7816_LC_POS] = bLc + 8U;

            /* As we have updated LC, we also need to update LE... */
            if((!bLcPresent) && (bLePresent))
            {
                aTmpBuf[wValidMacData++] = bLeValue;
            }
        }
        else
        {
            aTmpBuf[wValidMacData++] = 0;
            /* we definitively have had LC in here */
        }

        pDataParams->bPendingMacCmdDataLength = 0;
    }
    else
    {
        /* Update LC in case of Last frame */
        if(bLast)
        {
            pBuffer[PHHAL_HW_SAM_ISO7816_LC_POS] = bLc + 8U;
        }

        /* Copy pending data */
        memcpy(aTmpBuf, pDataParams->bPendingMacCmdData, (uint16_t) pDataParams->bPendingMacCmdDataLength);
        wValidMacData = pDataParams->bPendingMacCmdDataLength;
        pDataParams->bPendingMacCmdDataLength = 0;

        /* Load pending command MAC */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pMACCryptoDataParams,
            pDataParams->bPendingCmdMac,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));
    }

    /* Start MACing Process */
    /* Now recopy the remaining data into the aTmpBuf in case of we have at least 5 bytes in the buffer */
    wHelper = 16U - wValidMacData;

    /* The helper should not be bigger than bLc (also bLc = 0 is covered) */
    if(wHelper > bLc)
    {
        wHelper = bLc;
    }

    memcpy(&aTmpBuf[wValidMacData], &pBuffer[PHHAL_HW_SAM_ISO7816_LC_POS + 1U], (uint16_t) wHelper);
    wValidMacData = (uint16_t) (wValidMacData + wHelper);

    /* If we have a complete pending block, we can always use it. */
    if(wValidMacData == 16U)
    {
        /* Do we have remaining data? */
        if((bLc > wHelper) || (bLePresent))
        {
            /* Switch to CMAC mode without padding */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pMACCryptoDataParams,
                (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_CONT),
                aTmpBuf,
                wValidMacData,
                pDataParams->bPendingCmdMac,
                &bMacLen));

            wValidMacData = 0;

            /* Now we calculate all blocks but the last one*/
            wValidMacData = (uint16_t) (bLc - wHelper);

            /* Calculate pending data of last block */
            bLc = (uint8_t) (wValidMacData % 16U);
            wValidMacData = (wValidMacData - bLc);

            /* Skip MACing of the last block. */
            if(bLc == 0 && !bLePresent)
            {
                if(wValidMacData >= 16U)
                {
                    wValidMacData -= 16U;
                }
                bLc += 16U;
            }

            /* If we have data, we can now MAC it */
            if(wValidMacData)
            {
                /* we have remaining data */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                    pDataParams->pMACCryptoDataParams,
                    (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_CONT),
                    &pBuffer[wHelper + PHHAL_HW_SAM_ISO7816_HEADER_LENGTH],
                    wValidMacData,
                    pDataParams->bPendingCmdMac,
                    &bMacLen));
            }

            /* Recopy the last chunk into the tmp Buffer */
            memcpy(aTmpBuf, &pBuffer[wHelper + PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + wValidMacData], bLc);
            wValidMacData = bLc;
        }
    }

    /* Now let's distinguish, what to do with the pending data */
    if(bLast)
    {
        /* Do we need to append Le?*/
        if((bLcPresent) && (bLePresent))
        {
            if(wValidMacData >= 16U)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
            }
            aTmpBuf[wValidMacData++] = bLeValue;
        }

        /* Switch to CMAC mode with padding*/
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pMACCryptoDataParams,
            (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_LAST),
            aTmpBuf,
            wValidMacData,
            pDataParams->bPendingCmdMac,
            &bMacLen));

        /* we have to truncate the MAC*/
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(pDataParams->bPendingCmdMac, &bMacLen));

        /* Append MAC at end of buffer */
        if((bLcPresent) && (bLePresent))
        {
            /* Le is still on the correct position, so copy mac and move Le */
            memcpy(&pBuffer[wBuffLen - 1U], pDataParams->bPendingCmdMac, bMacLen);
            wBuffLen = (uint16_t) (wBuffLen + bMacLen);
            pBuffer[wBuffLen - 1U] = bLeValue;
        }
        else if(bLcPresent)
        {
            /* Before, there was no Lc byte - this is newly introduced Le needs to be recopied*/
            memcpy(&pBuffer[wBuffLen], pDataParams->bPendingCmdMac, (uint16_t) bMacLen);
            wBuffLen = (uint16_t) (wBuffLen + bMacLen);
        }
        else if(bLePresent)
        {
            /* Le is still on the correct position, so copy mac and move Le */
            memcpy(&pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH], pDataParams->bPendingCmdMac, (uint16_t) bMacLen);
            wBuffLen = (uint16_t) (PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + bMacLen + 1U);
            pBuffer[wBuffLen - 1U] = bLeValue;
        }
        else
        {
            /* We do not have Le or Lc before, now we have Lc */
            memcpy(&pBuffer[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH], pDataParams->bPendingCmdMac, (uint16_t) bMacLen);
            wBuffLen = PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + bMacLen;
        }
    }
    else
    {
        /* Setup pending data*/
        memcpy(pDataParams->bPendingMacCmdData, aTmpBuf, (uint16_t) wValidMacData);
        pDataParams->bPendingMacCmdDataLength = (uint8_t) wValidMacData;
    }

    *pBuffLen = wBuffLen;

    /* End MACing Process */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_HSM_AES_VerifyRemoveMac(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pBuffer, uint16_t wBuffLen, uint16_t * pBuffLen,
    uint8_t bFirst, uint8_t bLast)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wValidMacData = 0;
    uint16_t    PH_MEMLOC_REM wHelper = 0;
    uint16_t    PH_MEMLOC_REM wPayloadLen = 0;
    uint8_t     PH_MEMLOC_REM bMacLen;
    uint8_t     PH_MEMLOC_REM aTmpBuf[16U];
    uint8_t     PH_MEMLOC_REM bOldPendingRespDataLen = 0;

    /* Begin Checks */
    memset(aTmpBuf, 0, 16U);

    /* In case of chaining the last 16 bytes of the rx data could be padding or MAC data. */
    if(wBuffLen > 2U && ((pDataParams->bResponseChaining == PHHAL_HW_SAM_HSM_AES_CHAINING) || (!bFirst && bLast)))
    {
        if(bFirst)
        {
            pDataParams->bPendingRespDataLength = 0;
        }
        else
        {
            memcpy(aTmpBuf, pDataParams->bPendingRespData, pDataParams->bPendingRespDataLength);
            bOldPendingRespDataLen = pDataParams->bPendingRespDataLength;
        }

        if(!bLast)
        {
            if((wBuffLen - 2U) >= 16U)
            {
                pDataParams->bPendingRespDataLength = 16U;
            }
            else
            {
                pDataParams->bPendingRespDataLength = (uint8_t) (wBuffLen - 2U);
            }

            /* If the response is only MACed skip MACing of the last 8 bytes if it is not the last frame */
            memcpy(pDataParams->bPendingRespData, &pBuffer[(wBuffLen - 2U) - pDataParams->bPendingRespDataLength],
                pDataParams->bPendingRespDataLength);

            memcpy(&pBuffer[(wBuffLen - 2U) - pDataParams->bPendingRespDataLength], &pBuffer[(wBuffLen - 2U)],
                PHHAL_HW_SAMAV2_ISO7816_SW1SW2_LENGTH);

            wBuffLen = (uint16_t) (wBuffLen - pDataParams->bPendingRespDataLength);
        }

        if(!bFirst && bOldPendingRespDataLen)
        {
            memmove(&pBuffer[bOldPendingRespDataLen], pBuffer, wBuffLen);
            memcpy(pBuffer, aTmpBuf, bOldPendingRespDataLen);
            wBuffLen = (uint16_t) (wBuffLen + bOldPendingRespDataLen);
        }

        if(bLast)
        {
            pDataParams->bPendingRespDataLength = 0;
        }
    }

    *pBuffLen = 0;

    if(bLast)
    {
        /* Received length needs to be at least 10 bytes or 2 bytes in case of Tx Chaining! */
        if(wBuffLen < (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH + 8U /* MAC */))
        {
            return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
        }
        wPayloadLen = (wBuffLen - (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH + 8U /* MAC */));
    }
    else
    {
        wPayloadLen = (uint16_t) (wBuffLen - (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH));
    }

    /* Start UnMACing Process */
    if(bFirst)
    {
        pDataParams->bPendingMacRespDataLength = 0;

        /* load the InitializationVector, because we start a new MAC calculation */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(pDataParams->pMACCryptoDataParams,
            phhalHw_Sam_HSM_AES_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        /* Also set the pending MAC to 0 */
        memset(pDataParams->bPendingRespMac, 0, sizeof(pDataParams->bPendingRespMac));

        /* Calculate the MAC according to the response pMacedBuffer */
        aTmpBuf[wValidMacData++] = pBuffer[wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH];

        /* In case of chaining detected, we need to load 0x00 */
        if((pBuffer[wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH] == 0x90U) && (pBuffer[wBuffLen - 1U] == 0xAFU))
        {
            aTmpBuf[wValidMacData++] = 0x00;
        }
        else
        {
            aTmpBuf[wValidMacData++] = pBuffer[wBuffLen - 1U];
        }
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0xFF000000U) >> 24U);
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x00FF0000U) >> 16U);
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x0000FF00U) >> 8U);
        aTmpBuf[wValidMacData++] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x000000FFU) >> 0);
    }
    else
    {
        /* Get pending data*/
        memcpy(aTmpBuf, pDataParams->bPendingMacRespData, (uint16_t) pDataParams->bPendingMacRespDataLength);
        wValidMacData = pDataParams->bPendingMacRespDataLength;
        pDataParams->bPendingMacRespDataLength = 0;

        /* Load pending response MAC */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pMACCryptoDataParams,
            pDataParams->bPendingRespMac,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));
    }

    /* Now recopy the remaining data into the aTmpBuf in case of we have user data */
    wHelper = (uint16_t) (16U - wValidMacData);
    if(wPayloadLen)
    {
        if(wHelper > wPayloadLen)
        {
            wHelper = wPayloadLen;
            wPayloadLen = 0;
        }
        else
        {
            /* wHelper is ok */
            wPayloadLen = (uint16_t) (wPayloadLen - wHelper);
        }
    }
    else
    {
        wHelper = 0;
    }

    memcpy(&aTmpBuf[wValidMacData], pBuffer, (uint16_t) wHelper);

    wValidMacData = wValidMacData + wHelper;

    if(wValidMacData == 16U && wPayloadLen != 0)
    {
        /* Switch to CMAC mode without padding*/
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pMACCryptoDataParams,
            (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_CONT),
            aTmpBuf,
            wValidMacData,
            pDataParams->bPendingRespMac,
            &bMacLen));

        /* Now add everything but the last block */
        /* now we calculate all blocks but the last one*/
        wValidMacData = wPayloadLen;

        /* calculate pending data of last block */
        wPayloadLen = (wValidMacData % 16U);
        if((wValidMacData >= 16) && (wPayloadLen == 0))
        {
            wValidMacData = (uint16_t) (wValidMacData - 16U);
            wPayloadLen = 16U;
        }
        else
        {
            wValidMacData = (uint16_t) (wValidMacData - wPayloadLen);
        }

        /* If we have data, we can now MAC it */
        if(wValidMacData)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pMACCryptoDataParams,
                (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_CONT),
                &pBuffer[wHelper],
                wValidMacData,
                pDataParams->bPendingRespMac,
                &bMacLen));
        }

        /* Recopy the last chunk into the tmp Buffer */
        memcpy(aTmpBuf, &pBuffer[wHelper + wValidMacData], (uint16_t) wPayloadLen);
        wValidMacData = wPayloadLen;
    }

    /* Last block - Verify MAC */
    if(bLast)
    {
        pDataParams->bPendingMacRespDataLength = 0;

        /* CMAC mode with padding*/
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pMACCryptoDataParams,
            (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_LAST),
            aTmpBuf,
            wValidMacData,
            pDataParams->bPendingRespMac,
            &bMacLen));

        /* we have to truncate the MAC*/
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_TruncateMacBuffer(pDataParams->bPendingRespMac, &bMacLen));

        if((bMacLen != 8U) || (wBuffLen < (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH + bMacLen)))
        {
            return PH_ADD_COMPCODE(PHHAL_HW_SAM_ERR_CRYPTO, PH_COMP_HAL);
        }

        /* compare the MACed in response with the calculated MAC*/
        if(memcmp(pDataParams->bPendingRespMac, &pBuffer[wBuffLen - (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH + bMacLen)], bMacLen))
        {
            return PH_ADD_COMPCODE(PHHAL_HW_SAM_ERR_CRYPTO, PH_COMP_HAL);
        }

        /* now, we can remove the MAC*/
        *pBuffLen = (uint16_t) (wBuffLen - bMacLen);

        /* Reorder SW1 SW2 */
        pBuffer[wBuffLen - (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH + bMacLen)] = pBuffer[wBuffLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH];
        pBuffer[wBuffLen - (1 + bMacLen)] = pBuffer[wBuffLen - 1U];
    }
    else
    {
        /* Setup pending data*/
        memcpy(pDataParams->bPendingMacRespData, aTmpBuf, (uint16_t) wValidMacData);
        pDataParams->bPendingMacRespDataLength = (uint8_t) wValidMacData;

        /* End UnMACing Process */
        *pBuffLen = wBuffLen;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_HSM_AES_GetFirstLastCommand(phhalHw_Sam_DataParams_t * pDataParams, uint8_t aCmd, uint8_t bP1, uint8_t bP2,
    uint8_t * bFirstCmd, uint8_t * bLastCmd)
{
    *bFirstCmd = PH_ON;
    *bLastCmd = PH_ON;

    /* Reset CommandChaining per default */
    if(pDataParams->bCommandChaining == PHHAL_HW_SAM_HSM_AES_CHAINING)
    {
        *bFirstCmd = PH_OFF;
    }

    /* In case of response chaining no CMD counter increment is allowed as well */
    if(pDataParams->bResponseChaining == PHHAL_HW_SAM_HSM_AES_CHAINING)
    {
        *bFirstCmd = PH_OFF;

        /* DESFire ReadX command needs special treatment */
        if(aCmd == PHHAL_HW_SAM_CMD_INS_DESFIRE_READ_X)
        {
            pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;
            *bLastCmd = PH_OFF;
        }
    }

    pDataParams->bCommandChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;

    /* Is the current command a chained one or not? */
    if(bP1 == PHHAL_HW_SAM_ISO7816_CHAINED_FRAME)
    {
        switch(aCmd)
        {
            /* Data Processing Instruction Codes. */
            case PHHAL_HW_SAM_CMD_INS_SAM_APPLY_SM:
            case PHHAL_HW_SAM_CMD_INS_SAM_REMOVE_SM:
            case PHHAL_HW_SAM_CMD_INS_SAM_VERIFY_MAC:
            case PHHAL_HW_SAM_CMD_INS_SAM_GENERATE_MAC:
            case PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_DATA:
            case PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_DATA:
            case PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_OFFLINE_DATA:
            case PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_OFFLINE_DATA:

            /* DESFire X-Mode Instruction Codes. */
            case PHHAL_HW_SAM_CMD_INS_DESFIRE_WRITE_X:

            /* Programmable Logic Instruction Codes. */
            case PHHAL_HW_CMD_INS_SAM_PL_EXEC:
            case PHHAL_HW_CMD_INS_SAM_PL_UPLOAD:

            /* ISO14443-4 Instruction Codes. */
            case PHHAL_HW_SAM_CMD_INS_ISO14443_4_EXCHANGE:
                pDataParams->bCommandChaining = PHHAL_HW_SAM_HSM_AES_CHAINING;
                *bLastCmd = PH_OFF;
                break;
            default:
                break;
        }
    }

    if(bP2 == PHHAL_HW_SAM_ISO7816_CHAINED_FRAME)
    {
        switch(aCmd)
        {
            /* PKI RSA Instruction Codes. */
            case PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_KEY_PAIR:
            case PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_KEY:
            case PHHAL_HW_SAM_CMD_INS_PKI_UPDATE_KEY_ENTRIES:
            case PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_HASH:
            case PHHAL_HW_SAM_CMD_INS_PKI_VERIFY_SIGNATURE:
            case PHHAL_HW_SAM_CMD_INS_PKI_DECIPHER_DATA:

            /* MIFARE Plus S-Mode Instruction Codes. */
            case PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_READ_MFP:
                pDataParams->bCommandChaining = PHHAL_HW_SAM_HSM_AES_CHAINING;
                *bLastCmd = PH_OFF;
                break;
            default:
                break;
        }
    }

    /* Set bFirstCmd is the status is 90AE. */
    switch(aCmd)
    {
        /* For Cmd.VCA_Select, for every command Command counter should be incremented. This is with
         * respect to the new error code added (0x90AE). Because for every command CmdCtr needs to be
         * incremented the below check is required.
         */
        case PHHAL_HW_SAM_CMD_INS_VCA_SELECT:
            *bFirstCmd = PH_ON;
            break;

        default:
            break;
    }

    /* Move out if the Host Protection Mode is PLAIN. */
    if(pDataParams->bAuthType == 0x00)
    {
        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
    }

    /* Apply the Host Protection. */
    else
    {
        if(*bFirstCmd)
        {
            /*
             * This flag is set by default for all commands. The command which require additional secure messaging options like
             * Ommit or NA is updated in the below switch statement.
             * Apply SM: Command.Enc, Command.Mac, Response.Enc, Response.MAC
             */

            /* Host Protection Mode is MAC. */
            if(pDataParams->bAuthType == 0x01U)
            {
                pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_MAC;
                pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_MAC;
            }
            /* Host Protection Mode is FULL. */
            else if(pDataParams->bAuthType == 0x02U)
            {
                pDataParams->bCmdSM = (PHHAL_HW_SAM_HSM_AES_MAC | PHHAL_HW_SAM_HSM_AES_ENC);
                pDataParams->bRespSM = (PHHAL_HW_SAM_HSM_AES_MAC | PHHAL_HW_SAM_HSM_AES_ENC);
            }

            switch(aCmd)
            {
                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * NA: Command.Enc, Command.Mac, Response.Enc, Response.MAC
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_LOCK_UNLOCK:
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_HOST:
                case PHHAL_HW_SAM_CMD_INS_ISO_INTERNAL_AUTHENTICATE:
                    pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * Apply SM: Command.Enc, Command.Mac
                 * Ommit SM: Response.Enc, Response.MAC
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_APPLY_SM:
                case PHHAL_HW_SAM_CMD_INS_SAM_SELECT_VC:
                case PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_PICC:
                case PHHAL_HW_SAM_CMD_INS_SAM_CREATE_TM_FILE_PICC:
                    pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * First Part
                 *      Apply SM: Command.Enc, Command.Mac
                 *      Ommit SM: Response.Enc, Response.MAC
                 *  Second Part
                 *      Apply SM: Response.Enc, Response.MAC
                 *      Ommit SM: Command.Enc, Command.Mac
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_PROXIMITY_CHECK:
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MFP:
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PDC:
                case PHHAL_HW_SAM_CMD_INS_SAM_PWD_AUTH_UL:
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_TAM:
                    if(pDataParams->bResponseChaining == PHHAL_HW_SAM_HSM_AES_CHAINING_NO_SM)
                    {
                        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    else
                    {
                        /* Note: In SamAV2, the commmand of this command is neither MACed nor Encrypted */
                        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * Apply SM     : Command.MAC, Response.MAC
                 * Ommit, N/A   : Command.Enc, Response.Enc
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_KILL_AUTHENTICATION:
                case PHHAL_HW_SAM_CMD_INS_SAM_SLEEP:
                case PHHAL_HW_SAM_CMD_INS_RC_INIT:
                case PHHAL_HW_SAM_CMD_INS_ISO14443_3_HALTA:
                case PHHAL_HW_SAM_CMD_INS_ISO14443_4_PRESENCE_CHECK:
                case PHHAL_HW_SAM_CMD_INS_ISO14443_4_DESELECT:
                    pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_MAC;
                    pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_MAC;
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * Apply SM: Command.Mac, Response.Enc, Response.MAC
                 * Ommit SM: Command.Enc
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_GET_VERSION:
                case PHHAL_HW_SAM_CMD_INS_SAM_GET_RANDOM:
                case PHHAL_HW_SAM_CMD_INS_SAM_GET_KEY_ENTRY:
                case PHHAL_HW_SAM_CMD_INS_SAM_GET_KUC_ENTRY:
                case PHHAL_HW_SAM_CMD_INS_SAM_REMOVE_SM:
                case PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_DATA:
                case PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_OFFLINE_DATA:
                case PHHAL_HW_SAM_CMD_INS_PKI_DECIPHER_DATA:
                case PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_ECC_PRIVATE_KEY:
                case PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_ECC_PUBLIC_KEY:
                case PHHAL_HW_CMD_INS_SAM_PL_UPLOAD:
                    pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_MAC;
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * Apply SM     : Command.Enc, Command.Mac, Response.MAC
                 * Ommit, N/A   : Response.Enc
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_CRYPTO:
                case PHHAL_HW_SAM_CMD_INS_SAM_ACTIVATE_OFFLINE_KEY:
                case PHHAL_HW_SAM_CMD_INS_SAM_LOAD_INIT_VECTOR:
                case PHHAL_HW_SAM_CMD_INS_SAM_SELECT_APPLICATION:
                case PHHAL_HW_SAM_CMD_INS_SAM_SET_CONFIGURATION:
                case PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_ENTRY:
                case PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KUC_ENTRY:
                case PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_KEY_ENTRY:
                case PHHAL_HW_SAM_CMD_INS_SAM_ENCHIPHER_KEY_ENTRY:
                case PHHAL_HW_SAM_CMD_INS_SAM_DERIVE_KEY:
                case PHHAL_HW_SAM_CMD_INS_SAM_VERIFY_MAC:
                case PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_DATA:
                case PHHAL_HW_SAM_CMD_INS_SAM_ENCIPHER_OFFLINE_DATA:
                case PHHAL_HW_SAM_CMD_INS_PKI_ENCIPHER_KEY_ENTRIES:
                case PHHAL_HW_SAM_CMD_INS_PKI_ENCIPHER_DATA:
                case PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_KEY:
                case PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_CURVE:
                case PHHAL_HW_SAM_CMD_INS_PKI_VERIFY_ECC_SIGNATURE:
                case PHHAL_HW_SAM_CMD_INS_RC_WRITE_REGISTER:
                case PHHAL_HW_SAM_CMD_INS_RC_RF_CONTROL:
                case PHHAL_HW_SAM_CMD_INS_RC_LOAD_REGISTER_VALUE_SET:
                case PHHAL_HW_SAM_CMD_INS_ISO14443_3_ACTIVATE_WAKEUP:
                case PHHAL_HW_SAM_CMD_INS_ISO14443_4_INIT:
                case PHHAL_HW_SAM_CMD_INS_ISO14443_4_FREE_CID:
                    pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_MAC;
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 *  First Part
                 *      Apply SM: Command.Enc, Command.Mac
                 *      Ommit SM: Response.Enc, Response.MAC
                 *  Second Part
                 *      Apply SM: Response.MAC
                 *      Ommit SM: Command.Enc, Command.Mac, Response.Enc
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PICC:
                case PHHAL_HW_SAM_CMD_INS_SAM_ISO_AUTHENTICATE_PICC:
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTH_SECTOR_SWITCH_MFP:
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MAM:
                    if(pDataParams->bResponseChaining == PHHAL_HW_SAM_HSM_AES_CHAINING_NO_SM)
                    {
                        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_MAC;
                    }
                    else
                    {
                        /* Note: In SamAV2, the command is neither MACed nor Encrypted */
                        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 *  First Part
                 *      Apply SM: Command.MAC
                 *      Ommit SM: Response.Enc, Response.Mac, Command.ENC
                 *  Second Part
                 *      Apply SM: Response.Enc, Response.MAC
                 *      Ommit SM: Command.Enc, Command.Mac
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_COMMIT_READER_ID:
                    if(pDataParams->bResponseChaining == PHHAL_HW_SAM_HSM_AES_CHAINING_NO_SM)
                    {
                        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    else
                    {
                        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_MAC;
                        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 *  Command
                 *      Apply SM: Command.Enc, Command.MAC
                 *      Ommit SM: Response.Enc, Response.Mac
                 *  Response
                 *      Apply SM: Response.Enc, Response.MAC
                 *      Ommit SM: Command.Enc, Command.Mac
                 *  Command + Response
                 *      Apply SM: Command.Enc, Command.Mac, Response.Enc, Response.MAC
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_READ_MFP:
                    if(bP1 == PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND)
                    {
                        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    else if(bP1 == PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE)
                    {
                        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    else
                    {
                        /* Else, nothing changed */
                    }
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 *  Command
                 *      Apply SM: Command.Enc, Command.MAC
                 *      Ommit SM: Response.Enc, Response.Mac
                 *  Response
                 *      Apply SM: Response.Enc, Response.MAC
                 *      Ommit SM: Command.Enc, Command.Mac
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_WRITE_MFP:
                case PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_MFP:
                    if(bP1 & PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE)
                    {
                        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    else
                    {
                        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
                    }
                    break;

                /*
                 * Commands that falls under below mentioned Secure-Messaging.
                 * Plain: No SM
                 * Mac  :
                 *          If  P1 = 0x00, Apply SM: Command.MAC Response.Enc Response.MAC and N/A: Command.Enc
                 *          If  P1 = 0x01, Apply SM: Command.MAC Response.Enc Response.MAC Command.Enc
                 * Enc  :
                 *          If  P1 = 0x00, Apply SM: Command.MAC Response.Enc Response.MAC Command.Enc
                 *          If  P1 = 0x01, Apply SM: Command.MAC Response.Enc Response.MAC Command.Enc
                 */
                case PHHAL_HW_SAM_CMD_INS_SAM_DUMP_SESSION_KEY:
                case PHHAL_HW_SAM_CMD_INS_SAM_DUMP_SECRET_KEY:
                    if((bP1 & 0x01U) && (pDataParams->bAuthType == 0x01U))
                    {
                        pDataParams->bRespSM |= PHHAL_HW_SAM_HSM_AES_ENC;
                    }
                    break;
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_HSM_AES_GetFirstLastResponse(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bSw1, uint8_t bSw2,
    uint8_t * bFirstResponse, uint8_t * bLastResponse)
{
    *bFirstResponse = PH_ON;
    *bLastResponse = PH_ON;

    /* Reset ResponseChaining per default */
    if(pDataParams->bResponseChaining == PHHAL_HW_SAM_HSM_AES_CHAINING)
    {
        *bFirstResponse = PH_OFF;
    }

    pDataParams->bResponseChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;

    /* Check for 0x90AF - in case of MACing of response is enabled, we should set the chaining option*/
    if((bSw1 != 0x90U) || (bSw2 != 0x00))
    {
        if((bSw1 == 0x90U) && (bSw2 == 0xAFU))
        {
            if(pDataParams->bRespSM != PHHAL_HW_SAM_HSM_AES_NO_SM)
            {
                pDataParams->bResponseChaining = PHHAL_HW_SAM_HSM_AES_CHAINING;
            }
            else
            {
                pDataParams->bResponseChaining = PHHAL_HW_SAM_HSM_AES_CHAINING_NO_SM;
            }

            *bLastResponse = PH_OFF;
        }
        else if((bSw1 == 0x90U) && (bSw2 == 0xAEU))
        {
            pDataParams->bResponseChaining = PHHAL_HW_SAM_HSM_AES_CHAINING_NO_SM;

            *bFirstResponse = PH_ON;
            *bLastResponse = PH_ON;
        }
        else
        {
            *bFirstResponse = PH_ON;
            pDataParams->bCommandChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_HSM_AES_InitAndLoadIV(phhalHw_Sam_DataParams_t * pDataParams, uint8_t* pIV, uint8_t bIsCommand)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;

    /* Initialization Vector
     *      Command : E(Ke; 0x01 || 0x01 || 0x01 || 0x01 || wCmd_Ctr || wCmd_Ctr || wCmd_Ctr)
     *      Response: E(Ke; 0x02 || 0x02 || 0x02 || 0x02 || wCmd_Ctr || wCmd_Ctr || wCmd_Ctr)
     *      wCmd_Ctr (MSB first)
     */

    /* Load null keys to encrypt IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pENCCryptoDataParams,
        phhalHw_Sam_HSM_AES_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    if(bIsCommand)
    {
        for(bIndex = 0; bIndex < 4U; bIndex++)
            pIV[bIndex] = 0x01U;

        for(bIndex = 1U; bIndex < 4U; bIndex++)
        {
            pIV[4U * bIndex] = (uint8_t) ((pDataParams->wCmd_Ctr & 0xFF000000U) >> 24U);
            pIV[4U * bIndex + 1U] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x00FF0000U) >> 16U);
            pIV[4U * bIndex + 2U] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x0000FF00U) >> 8U);
            pIV[4U * bIndex + 3U] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x000000FFU) >> 0);
        }
    }
    else
    {
        for(bIndex = 0; bIndex < 4U; bIndex++)
            pIV[bIndex] = 0x02U;

        for(bIndex = 1; bIndex < 4U; bIndex++)
        {
            pIV[4U * bIndex] = (uint8_t) ((pDataParams->wCmd_Ctr & 0xFF000000) >> 24U);
            pIV[4U * bIndex + 1U] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x00FF0000) >> 16U);
            pIV[4U * bIndex + 2U] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x0000FF00) >> 8U);
            pIV[4U * bIndex + 3U] = (uint8_t) ((pDataParams->wCmd_Ctr & 0x000000FF) >> 0);
        }
    }

    /* Encrypt IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pENCCryptoDataParams,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_FIRST),
        pIV,
        16U,
        pIV));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}
#endif /* NXPBUILD__PHHAL_HW_SAM */
