/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Specific implementation for RC523 Reader of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phhalHw.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "phhalHw_Sam_Rc523.h"
#include "phhalHw_Sam_Rc523_Reg.h"

phStatus_t phhalHw_Sam_Rc523_SetConfig(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bValue = 0;
    uint8_t     PH_MEMLOC_REM bModWidthReg = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    switch(wConfig)
    {
        case PHHAL_HW_CONFIG_PARITY:
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_MFRX, &bRegister));

            if(wValue == PH_OFF)
            {
                bValue = bRegister | PHHAL_HW_SAM_RC523_BIT_PARITYDISABLE;
            }
            else
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_BIT_PARITYDISABLE;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_MFRX, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXCRC:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TXMODE, &bRegister));

            if(wValue == PH_OFF)
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_BIT_CRCEN;
            }
            else
            {
                bValue = bRegister | PHHAL_HW_SAM_RC523_BIT_CRCEN;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TXMODE, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXCRC:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_RXMODE, &bRegister));

            if(wValue == PH_OFF)
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_BIT_CRCEN;
            }
            else
            {
                bValue = bRegister | PHHAL_HW_SAM_RC523_BIT_CRCEN;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_RXMODE, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXLASTBITS:

            /* Check parameter */
            if(wValue > PHHAL_HW_SAM_RC523_MASK_TXBITS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_BITFRAMING, &bRegister));
            bRegister &= (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_MASK_TXBITS;
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_BITFRAMING, bRegister | (uint8_t) wValue));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXALIGN:

            /* Check parameter */
            if(wValue > 7)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Adjust parameter */
            wValue = (uint16_t) (wValue << 4U);

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_BITFRAMING, &bRegister));
            bRegister &= (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_MASK_RXALIGN;
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_BITFRAMING, bRegister | (uint8_t) wValue));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXDEAFBITS:

            /* Check parameter */
            if(wValue > PHHAL_HW_SAM_RC523_MASK_RXWAIT)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Configure the Rx Deaf Time in bits */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_RXSEL, &bRegister));
            bRegister &= (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_MASK_RXWAIT;
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_RXSEL, bRegister | (uint8_t) wValue));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:

            /* Modify bit */
            if(wValue == PH_OFF)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COLL, PHHAL_HW_SAM_RC523_BIT_VALUESAFTERCOLL));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COLL, 0x00));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXDATARATE:

            /* Retrieve TxMode register */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TXMODE, &bRegister));
            bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_MASK_SPEED;

            /* Update current data rate */
            switch(wValue)
            {
                case PHHAL_HW_RF_DATARATE_106:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_106KBPS;
                    bModWidthReg = 0x26U;
                    break;

                case PHHAL_HW_RF_DATARATE_212:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_212KBPS;
                    bModWidthReg = 0x15U;
                    break;

                case PHHAL_HW_RF_DATARATE_424:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_424KBPS;
                    bModWidthReg = 0x0AU;
                    break;

                case PHHAL_HW_RF_DATARATE_848:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_848KBPS;
                    bModWidthReg = 0x05U;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Only apply the settings if the speed has changed */
            if(bValue != bRegister)
            {
                /* Apply data rate */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TXMODE, bValue));

                /* Configure the modulation width for ISO 14443-3A */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_MODWIDTH, bModWidthReg));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXDATARATE:

            /* Retrieve RxMode register */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_RXMODE, &bRegister));
            bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_MASK_SPEED;

            /* Update current data rate */
            switch(wValue)
            {
                case PHHAL_HW_RF_DATARATE_106:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_106KBPS;
                    break;

                case PHHAL_HW_RF_DATARATE_212:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_212KBPS;
                    break;

                case PHHAL_HW_RF_DATARATE_424:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_424KBPS;
                    break;

                case PHHAL_HW_RF_DATARATE_848:
                    bValue |= PHHAL_HW_SAM_RC523_BIT_848KBPS;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Apply data rate */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_RXMODE, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_MODINDEX:

            if(wValue & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_MASK_MODGSP)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Set ModGsP register */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_MODGSP, (uint8_t) wValue));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_ASK100:

            /* Switch off 100% ASK */
            if(wValue == PH_OFF)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TXASK, 0x00));
            }

            /* Switch on 100% ASK */
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TXASK,
                    PHHAL_HW_SAM_RC523_BIT_FORCE100ASK));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:

            /* Calculate values for Microsecond values */
            if(wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(pDataParams, PHHAL_HW_TIME_MICROSECONDS, wValue));
                pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
            }

            /* Calculate values for Millisecond values */
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(pDataParams, PHHAL_HW_TIME_MILLISECONDS, wValue));
                pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MILLISECONDS;
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc523_Wait(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wTimerShift = 0;
    uint16_t    PH_MEMLOC_REM wTimeoutNew = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;
    uint8_t     PH_MEMLOC_REM bIrqRq = 0;

    /* Parameter Check */
    if((bUnit != PHHAL_HW_TIME_MICROSECONDS) && (bUnit != PHHAL_HW_TIME_MILLISECONDS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Terminate a probably running command */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COMMAND, PHHAL_HW_SAM_RC523_CMD_IDLE));

    /* Retrieve RxDataRate */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(pDataParams, PHHAL_HW_CONFIG_RXDATARATE, &wTimerShift));
    ++wTimerShift;

    /* Calculate timershift */
    wTimerShift = (uint16_t) (PHHAL_HW_SAM_RC523_TIMER_SHIFT * (PHHAL_HW_SAM_RC523_ETU_106 / (float32_t) wTimerShift));

    /* Do as long as we have an overflow in the IC timer */
    do
    {
        /* Set temporary timeout */
        if(bUnit == PHHAL_HW_TIME_MICROSECONDS)
        {
            wTimeoutNew = (wTimeout > wTimerShift) ? (wTimeout - wTimerShift) : 0;
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(pDataParams, PHHAL_HW_TIME_MICROSECONDS, wTimeoutNew));
            wTimeout = 0;
        }
        else
        {
            /* Timer would overflow -> use maximum value and decrement overall value for next iteration */
            if(wTimeout > ((PHHAL_HW_SAM_RC523_TIMER_MAX_VALUE_US / 1000U) - 1U))
            {
                wTimeoutNew = (PHHAL_HW_SAM_RC523_TIMER_MAX_VALUE_US / 1000U) - 1U;
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(pDataParams, PHHAL_HW_TIME_MILLISECONDS, wTimeoutNew));
                wTimeout = wTimeout - wTimeoutNew;
            }
            /* No overflow -> set the complete time */
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(pDataParams, PHHAL_HW_TIME_MILLISECONDS, wTimeout));
                wTimeout = 0;
            }
        }

        /* Retrieve content of Control register */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_CONTROL, &bRegister));

        /* Enable timer interrupt */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COMMIEN,
            PHHAL_HW_SAM_RC523_BIT_TIMERI));

        /* Clear all irq flags */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COMMIRQ,
            (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_BIT_SET));

        /* Start timer */
        bRegister |= PHHAL_HW_SAM_RC523_BIT_TSTARTNOW;
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_CONTROL, bRegister));

        /* Wait for timer interrupt */
        do
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COMMIRQ, &bIrqRq));
        } while(!(bIrqRq & (PHHAL_HW_SAM_RC523_BIT_ERRI | PHHAL_HW_SAM_RC523_BIT_TIMERI)));

        /* Clear all irq flags */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_COMMIRQ,
            (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC523_BIT_SET));

    } while(wTimeout > 0);

    /* Restore previous timeout */
    if(pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(
            pDataParams,
            PHHAL_HW_TIME_MICROSECONDS,
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US]));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetFdt(
            pDataParams,
            PHHAL_HW_TIME_MILLISECONDS,
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS]));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc523_SetFdt(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    float32_t   PH_MEMLOC_REM fTimeoutUs = 0.0;
    float32_t   PH_MEMLOC_REM fTemp = 0.0;
    uint16_t    PH_MEMLOC_REM wDataRate = 0;
    uint16_t    PH_MEMLOC_REM wPrescaler = 0;
    uint16_t    PH_MEMLOC_REM wReload = 0;
    uint8_t     PH_MEMLOC_REM bTPrescaler = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    /* Retrieve RxDataRate */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(pDataParams, PHHAL_HW_CONFIG_RXDATARATE, &wDataRate));
    wDataRate++;

    /* Handle timeout in microseconds */
    if(bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        fTimeoutUs = (float32_t) wTimeout;
    }
    else
    {
        fTimeoutUs = (float32_t) wTimeout * 1000.0f;
    }

    /* Add timershift to timeout value */
    fTimeoutUs = fTimeoutUs + (PHHAL_HW_SAM_RC523_TIMER_SHIFT * (PHHAL_HW_SAM_RC523_ETU_106 / (float32_t) wDataRate));

    /* Max. timeout Check */
    if(fTimeoutUs > PHHAL_HW_SAM_RC523_TIMER_MAX_VALUE_US)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
    }

    /* Read Prescaler value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TPRESCALER, &bTPrescaler));
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TMODE, &bRegister));

    /* Get prescaler value */
    wPrescaler = ((uint16_t) (bRegister & PHHAL_HW_SAM_RC523_MASK_TPRESCALER_HI) << 8) | bTPrescaler;

    /* Calculate the reload value */
    fTemp = (fTimeoutUs * PHHAL_HW_SAM_RC523_TIMER_FREQ) / (1.0f + (2.0f * (float32_t) wPrescaler));

    /* Except for a special case, the float32_t value will have some commas
     * and therefore needs to be "ceiled"
     */
    if(fTemp > (float32_t) (uint16_t) fTemp)
    {
        wReload = (uint16_t) (fTemp + 1);
    }
    else
    {
        wReload = (uint16_t) fTemp;
    }

    /* Write reload value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TRELOADLO,
        (uint8_t) (wReload & 0xFFU)));

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TRELOADHI,
        (uint8_t) ((wReload >> 8U) & 0xFFU)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc523_GetFdt(phhalHw_Sam_DataParams_t * pDataParams, uint32_t * pTime)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    float32_t   PH_MEMLOC_REM fTimeout = 0.0;
    uint16_t    PH_MEMLOC_REM wDataRate = 0;
    uint16_t    PH_MEMLOC_REM wPsReload = 0;
    uint16_t    PH_MEMLOC_REM wTReload = 0;
    uint16_t    PH_MEMLOC_REM wTValue = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    /* Retrieve RxDataRate */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(pDataParams, PHHAL_HW_CONFIG_RXDATARATE, &wDataRate));
    wDataRate++;

    /* Get prescaler value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TMODE, &bRegister));
    wPsReload = (uint16_t) (bRegister & PHHAL_HW_SAM_RC523_MASK_TPRESCALER_HI) << 8;
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TPRESCALER, &bRegister));
    wPsReload |= (uint16_t) bRegister;

    /* Get reload value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TRELOADHI, &bRegister));
    wTReload = (uint16_t) bRegister << 8;
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TRELOADLO, &bRegister));
    wTReload |= (uint16_t) bRegister;

    /* Get counter value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TCOUNTERVALHI, &bRegister));
    wTValue = (uint16_t) bRegister << 8;
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC523_REG_TCOUNTERVALLO, &bRegister));
    wTValue |= (uint16_t) bRegister;

    /* Subtract reload- and counter values */
    wTValue = wTReload - wTValue;

    /* Calculate the timeout value */
    fTimeout = ((float32_t) wTValue * (1.0f + (2.0f * (float32_t) wPsReload))) / PHHAL_HW_SAM_RC523_TIMER_FREQ;

    /* Subtract timershift from timeout value */
    fTimeout = fTimeout - (PHHAL_HW_SAM_RC523_TIMER_SHIFT * (PHHAL_HW_SAM_RC523_ETU_106 / (float32_t) wDataRate));

    /* Return the value */
    *pTime = (uint32_t) fTimeout;

    /* Round the value if neccessary */
    if((fTimeout - (float32_t) *pTime) >= 0.5f)
    {
        ++(*pTime);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
