/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Specific implementation for RC663 Reader of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phhalHw.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "phhalHw_Sam_Rc663.h"
#include "phhalHw_Sam_Rc663_Config.h"
#include "phhalHw_Sam_Rc663_Reg.h"

/** Register Settings to return"*/
static const uint8_t PH_MEMLOC_CONST_ROM gkphhalHw_Sam_Rc663_Registers[] =
{
    PHHAL_HW_RC663_REG_DRVMODE,
    PHHAL_HW_RC663_REG_TXAMP,
    PHHAL_HW_RC663_REG_DRVCON,
    PHHAL_HW_RC663_REG_TXI,
    PHHAL_HW_RC663_REG_TXCRCCON,
    PHHAL_HW_RC663_REG_RXCRCCON,
    PHHAL_HW_RC663_REG_TXDATANUM,
    PHHAL_HW_RC663_REG_TXMODWIDTH,
    PHHAL_HW_RC663_REG_TXSYM10BURSTLEN,
    PHHAL_HW_RC663_REG_TXWAITCTRL,
    PHHAL_HW_RC663_REG_FRAMECON,
    PHHAL_HW_RC663_REG_RXCTRL,
    PHHAL_HW_RC663_REG_RXTHRESHOLD,
    PHHAL_HW_RC663_REG_RCV,
    PHHAL_HW_RC663_REG_RXANA
};

/** Register Settings for ISO14443A @ 106kbit/s */
static const uint8_t PH_MEMLOC_CONST_ROM gkphhalHw_Sam_Rc663_Reg_106[] =
{
    0x8EU,
    0x15U,
    0x11U,
    0x0AU,
    0x18U,
    0x18U,
    0x0FU,
    0x27U,
    0x00U,
    0xC0U,
    0xCFU,
    0x04U,
    PHHAL_HW_SAM_RC663_RXTHRESHOLD_I14443A_106,
    0x12U,
    0x0AU
};

/** Register Settings for ISO14443A @ 212kbit/s */
static const uint8_t PH_MEMLOC_CONST_ROM gkphhalHw_Sam_Rc663_Reg_212[] =
{
    0x8EU,
    0x12U,
    0x11U,
    0x0AU,
    0x18U,
    0x18U,
    0x0FU,
    0x10U,
    0x00U,
    0xC0U,
    0xCFU,
    0x05U,
    PHHAL_HW_SAM_RC663_RXTHRESHOLD_I14443A_212,
    0x12U,
    0x02U
};

/** Register Settings for ISO14443A @ 424kbit/s */
static const uint8_t PH_MEMLOC_CONST_ROM gkphhalHw_Sam_Rc663_Reg_424[] =
{
    0x8EU,
    0x12U,
    0x11U,
    0x0AU,
    0x18U,
    0x18U,
    0x0FU,
    0x08U,
    0x00U,
    0xC0U,
    0xCFU,
    0x06U,
    PHHAL_HW_SAM_RC663_RXTHRESHOLD_I14443A_424,
    0x12U,
    0x02U
};

/** Register Settings for ISO14443A @ 848kbit/s */
static const uint8_t PH_MEMLOC_CONST_ROM gkphhalHw_Sam_Rc663_Reg_848[] =
{
    0x8FU,
    0xDBU,
    0x11U,
    0x0AU,
    0x18U,
    0x18U,
    0x0FU,
    0x02U,
    0x00U,
    0xC0U,
    0xCFU,
    0x07U,
    PHHAL_HW_SAM_RC663_RXTHRESHOLD_I14443A_848,
    0x12U,
    0x02U
};

/* Digital delay for all protocols and speeds in microseconds */
static const uint16_t PH_MEMLOC_CONST_ROM gkphhalHw_Sam_Rc663_DigiDelays_Us[] =
{
    17U /* 106KBaud */,
    49U /* 212KBaud */,
    45U /* 424KBaud */,
    42U /* 848KBaud */
};

phStatus_t phhalHw_Sam_Rc663_SetCardMode(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wTxDataRate, uint16_t wRxDataRate)
{
    phStatus_t      PH_MEMLOC_REM wStatus = 0;
    uint16_t        PH_MEMLOC_COUNT wIndex = 0;
    uint8_t         PH_MEMLOC_REM bTxConfig = 0;
    uint8_t         PH_MEMLOC_REM bRxConfig = 0;
    uint8_t         PH_MEMLOC_REM bRegister = 0;

    const uint8_t * PH_MEMLOC_REM pTxRegisterSet = NULL;
    const uint8_t * PH_MEMLOC_REM pRxRegisterSet = NULL;

    /* No RegisterSet by default */
    pTxRegisterSet = NULL;

    switch(pDataParams->bCardType)
    {
        case PHHAL_HW_CARDTYPE_ISO14443A:

            switch(wTxDataRate)
            {
                case PHHAL_HW_RF_DATARATE_106:
                    bTxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_106;
                    pTxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_106;
                    break;

                case PHHAL_HW_RF_DATARATE_212:
                    bTxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_212;
                    pTxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_212;
                    break;

                case PHHAL_HW_RF_DATARATE_424:
                    bTxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_424;
                    pTxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_424;
                    break;

                case PHHAL_HW_RF_DATARATE_848:
                    bTxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_848;
                    pTxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_848;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            switch(wRxDataRate)
            {
                case PHHAL_HW_RF_DATARATE_106:
                    bRxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_106;
                    pRxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_106;
                    break;

                case PHHAL_HW_RF_DATARATE_212:
                    bRxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_212;
                    pRxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_212;
                    break;

                case PHHAL_HW_RF_DATARATE_424:
                    bRxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_424;
                    pRxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_424;
                    break;

                case PHHAL_HW_RF_DATARATE_848:
                    bRxConfig = PHHAL_HW_SAM_RC663_RXTX_I14443A_848;
                    pRxRegisterSet = (const uint8_t*) gkphhalHw_Sam_Rc663_Reg_848;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* Perform LoadProtocol */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_FIFODATA, bRxConfig));
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_FIFODATA, bTxConfig));
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_COMMAND, PHHAL_HW_SAM_RC663_CMD_LOADPROTOCOL));
    do
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_RC663_REG_IRQ0, &bRegister));
    } while(!(bRegister & PHHAL_HW_RC663_BIT_IDLEIRQ));

    /* Apply RegisterSet using library register set */
    for(wIndex = 0; wIndex < sizeof(gkphhalHw_Sam_Rc663_Registers); ++wIndex)
    {
        /* TX register set is valid */
        if(pTxRegisterSet)
        {
            /* Initially take everything from TX register set */
            bRegister = pTxRegisterSet[wIndex];

            /* RX register set is also valid */
            if(pRxRegisterSet)
            {
                switch(gkphhalHw_Sam_Rc663_Registers[wIndex])
                {
                    /* Take the following registers from the RX register set */
                    case PHHAL_HW_RC663_REG_RXCRCCON:
                    case PHHAL_HW_RC663_REG_RXSOFD:
                    case PHHAL_HW_RC663_REG_RXCTRL:
                    case PHHAL_HW_RC663_REG_RXWAIT:
                    case PHHAL_HW_RC663_REG_RXTHRESHOLD:
                    case PHHAL_HW_RC663_REG_RCV:
                    case PHHAL_HW_RC663_REG_RXANA:
                        bRegister = pRxRegisterSet[wIndex];
                        break;

                    default:
                        break;
                }
            }
        }
        /* ONLY RX register set is valid */
        else if(pRxRegisterSet)
        {
            /* Take everything from RX register set */
            bRegister = pRxRegisterSet[wIndex];
        }
        /* At least one register set is mandatory */
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        /* Preserve TxEn bit in DrvMode register */
        if(gkphhalHw_Sam_Rc663_Registers[wIndex] == PHHAL_HW_RC663_REG_DRVMODE)
        {
            /* Read out DrvMode register */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_RC663_REG_DRVMODE, &bTxConfig));

            /* Preserve TxEn bit, take other bits from register set */
            bRegister = (bTxConfig & PHHAL_HW_RC663_BIT_TXEN) | (bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_RC663_BIT_TXEN);
        }

        /* Actual write to register */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(
            pDataParams,
            gkphhalHw_Sam_Rc663_Registers[wIndex],
            bRegister));
    }

    /* Restore ASK 100 Setting */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams, PHHAL_HW_CONFIG_ASK100,
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_ASK100]));

    /* A change in the bitrate also affects TxWait */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetTxWait(pDataParams,
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXWAIT_US]));

    /* Update Datarate in shadow for parity setting */
    pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] = wTxDataRate;
    pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] = wRxDataRate;

    /* Apply Parity settings from Shadow */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams, PHHAL_HW_CONFIG_PARITY,
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY]));

    /* Apply Tx-CRC settings from Shadow */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXCRC,
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXCRC]));

    /* Apply Rx-CRC settings from Shadow */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams, PHHAL_HW_CONFIG_RXCRC,
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXCRC]));

    /* Apply TxLastBits settings from Shadow */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXLASTBITS,
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS]));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_SetConfig(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bValue = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    switch(wConfig)
    {
        case PHHAL_HW_CONFIG_PARITY:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_FRAMECON, &bRegister));

            /* Parity is OFF */
            if(wValue == PH_OFF)
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) (PHHAL_HW_SAM_RC663_BIT_TXPARITYEN | PHHAL_HW_SAM_RC663_BIT_RXPARITYEN);
            }
            /* Parity is ON */
            else
            {
                bValue = bRegister | PHHAL_HW_SAM_RC663_BIT_TXPARITYEN | PHHAL_HW_SAM_RC663_BIT_RXPARITYEN;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_FRAMECON, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXCRC:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXCRCCON, &bRegister));

            if(wValue == PH_OFF)
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_BIT_CRCEN;
            }
            else
            {
                bValue = bRegister | PHHAL_HW_SAM_RC663_BIT_CRCEN;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXCRCCON, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXCRC:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXCRCCON, &bRegister));

            if(wValue == PH_OFF)
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_BIT_CRCEN;
            }
            else
            {
                bValue = bRegister | PHHAL_HW_SAM_RC663_BIT_CRCEN;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXCRCCON, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXLASTBITS:

            /* Check parameter */
            if(wValue > PHHAL_HW_SAM_RC663_MASK_TXLASTBITS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXDATANUM, &bRegister));
            bRegister &= (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_MASK_TXLASTBITS;
            bRegister |= (uint8_t) wValue;
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXDATANUM, bRegister));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXALIGN:

            /* Check parameter */
            if(wValue > (PHHAL_HW_SAM_RC663_MASK_RXALIGN >> 4))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXBITCTRL, &bRegister));
            bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_MASK_RXALIGN;
            bValue |= (uint8_t) (wValue << 4);

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXBITCTRL, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_RXDEAFBITS:

            /* Set RxWait */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetRxWait(pDataParams, wValue));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXWAIT_US:

            /* Set TxWait */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetTxWait(pDataParams, wValue));

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXBITCTRL, &bRegister));

            /* Disable */
            if(wValue == PH_OFF)
            {
                bValue = bRegister | PHHAL_HW_SAM_RC663_BIT_VALUESAFTERCOLL;
            }
            /* Enable */
            else
            {
                bValue = bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_BIT_VALUESAFTERCOLL;
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXBITCTRL, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TXDATARATE:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetCardMode(pDataParams, wConfig,
                pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE]));
            break;

        case PHHAL_HW_CONFIG_RXDATARATE:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetCardMode(pDataParams,
                pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE], wConfig));
            break;

        case PHHAL_HW_CONFIG_MODINDEX:

            /* Parameter Check */
            if(wValue > PHHAL_HW_SAM_RC663_MASK_RESIDUAL_CARRIER)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Read out TxAmp register */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXAMP, &bRegister));

            /* Set modulation index value */
            bValue = (bRegister & (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_MASK_RESIDUAL_CARRIER) | (uint8_t) wValue;

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXAMP, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_ASK100:

            /* Read out DrvMod register */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_DRVMODE, &bRegister));

            /* Clear ClkMode bits */
            bValue = bRegister & (uint8_t) ~(uint8_t) (PHHAL_HW_SAM_RC663_BIT_RFON | PHHAL_HW_SAM_RC663_BIT_TPUSHON |
                PHHAL_HW_SAM_RC663_BIT_TPULLON);

            /* Update DrvMod register */
            if(wValue == PH_OFF)
            {
                bValue |= PHHAL_HW_SAM_RC663_BIT_RFON | PHHAL_HW_SAM_RC663_BIT_TPUSHON | PHHAL_HW_SAM_RC663_BIT_TPULLON;
            }
            else
            {
                bValue |= 0x01;  /* 100%ASK: ClkMode is the pause. */
            }

            /* Only perform the operation, if the new value is different */
            if(bValue != bRegister)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_DRVMODE, bValue));
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:

            /* Calculate values for Microsecond values */
            if(wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetFdt(pDataParams, PHHAL_HW_TIME_MICROSECONDS, wValue));
                pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
            }
            /* Calculate values for Millisecond values */
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetFdt(pDataParams, PHHAL_HW_TIME_MILLISECONDS, wValue));
                pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MILLISECONDS;
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_Wait(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bControlReg = 0;
    uint8_t     PH_MEMLOC_REM bIrqRq = 0;

    /* Parameter Check */
    if((bUnit != PHHAL_HW_TIME_MICROSECONDS) && (bUnit != PHHAL_HW_TIME_MILLISECONDS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Set temporary timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetFdt(pDataParams, bUnit, wTimeout));

    /* retrieve content of Timer Control register */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TCONTROL, &bControlReg));

    /* Clear all irq flags */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_IRQ1,
        (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_BIT_SET));

    /* Manually start T0 */
    bControlReg |= PHHAL_HW_SAM_RC663_BIT_T0STARTSTOPNOW | PHHAL_HW_SAM_RC663_BIT_T0RUNNING;

    /* Manually start T1 */
    bControlReg |= PHHAL_HW_SAM_RC663_BIT_T1STARTSTOPNOW | PHHAL_HW_SAM_RC663_BIT_T1RUNNING;
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TCONTROL, bControlReg));

    /* Wait for timer interrupt */
    do
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_IRQ1, &bIrqRq));
    } while(!(bIrqRq & PHHAL_HW_SAM_RC663_BIT_TIMER1IRQ));

    /* Manually stop T0 */
    bControlReg &= (uint8_t) ~(uint8_t) (PHHAL_HW_SAM_RC663_BIT_T0RUNNING | PHHAL_HW_SAM_RC663_BIT_T1RUNNING);
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TCONTROL, bControlReg));

    /* Restore previous timeout */
    if(pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetFdt(
            pDataParams,
            PHHAL_HW_TIME_MICROSECONDS,
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US]));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetFdt(
            pDataParams,
            PHHAL_HW_TIME_MILLISECONDS,
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS]));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_SetRxWait(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wTimeEtu)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    /* Since we use DBFREQ*2, we need to adjust the value for ETU granularity */

    /* DBFREQ*2 = 53KHz*2 = 106KHz = 1/2 ETU */
    /* Parameter Check */
    if(wTimeEtu > (PHHAL_HW_SAM_RC663_MASK_RXWAIT >> 1U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    wTimeEtu <<= 1;

    /* Set the RxWait register  */
    bRegister = PHHAL_HW_SAM_RC663_BIT_RXWAITDBFREQ | (uint8_t) wTimeEtu;
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_RXWAIT, bRegister));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_SetTxWait(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wTimeUs)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    float32_t   PH_MEMLOC_REM fTime = 0.0;
    uint16_t    PH_MEMLOC_REM wDataRate = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    /* Retrieve Tx-Datarate */
    wDataRate = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE];

    /* Since we use DBFREQ*2, we need to adjust the value for ETU granularity */
    fTime = ((float32_t) wDataRate + 1.0f) * (PHHAL_HW_SAM_RC663_ETU_106 / 2.0f);

    /* Perform calculation [value] / [timer granularity] */
    fTime = (float32_t) wTimeUs / fTime;
    wTimeUs = (uint16_t) fTime;
    if((float32_t) wTimeUs < fTime)
    {
        ++wTimeUs;
    }

    /* Remove digital circutry delay */
    if(wTimeUs > PHHAL_HW_SAM_RC663_DIGI_DELAY_TXWAIT_DBFREQ)
    {
        wTimeUs = wTimeUs - PHHAL_HW_SAM_RC663_DIGI_DELAY_TXWAIT_DBFREQ;
    }
    else
    {
        wTimeUs = 0;
    }

    /* Check parameter */
    if(wTimeUs > 0x07FFU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Read TxWaitControl register*/
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXWAITCTRL, &bRegister));

    /* Set TxWait configuration */
    bRegister |= PHHAL_HW_SAM_RC663_BIT_TXWAIT_START_RX | PHHAL_HW_SAM_RC663_BIT_TXWAIT_DBFREQ;

    /* Set the 3 MSBits */
    bRegister &= (uint8_t) ~(uint8_t) PHHAL_HW_SAM_RC663_MASK_TXWAITHI;
    bRegister |= (uint8_t) ((wTimeUs >> 5) & PHHAL_HW_SAM_RC663_MASK_TXWAITHI);

    /* Write changed register */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXWAITCTRL, bRegister));

    /* Set the 8 LSBs */
    bRegister = (uint8_t) wTimeUs;
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_TXWAITLO, bRegister));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_SetFdt(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    float32_t   PH_MEMLOC_REM fTime = 0.0;
    uint16_t    PH_MEMLOC_REM wT0Reload = 0;
    uint16_t    PH_MEMLOC_REM wT1Reload = 0;
    uint16_t    PH_MEMLOC_REM wDelay = 0;

    /* Calculate values for Microsecond values */
    if(bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        /* Retrieve and add digital delay */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_GetDigiDelay(pDataParams, PH_ON, &wDelay));
        fTime = (float32_t) wTimeout + (float32_t) wDelay;

        /* Timeout can be realized with just one timer */
        if(!((fTime * PHHAL_HW_SAM_RC663_TIMER_FREQ) > (float32_t) PHHAL_HW_SAM_RC663_TIMER_RELOAD_MAX))
        {
            fTime = fTime * PHHAL_HW_SAM_RC663_TIMER_FREQ;
            wT0Reload = (uint16_t) fTime;

            /* Since timer IRQ is raised at underflow, wReload has to decremented by one.
             * However, if the resulting reload value has commas, it must be 'ceiled'
             */
            if(!(fTime > (float32_t) wT0Reload))
            {
                --wT0Reload;
            }

            /* No timer 1 */
            wT1Reload = 0x0000;
        }
        /* If not, set prescaler to 1 ETU */
        else
        {
            /* T0 Overflow occurs at exactly 1 ETU (from ISO14443 protocol @ 106) */
            wT0Reload = (uint16_t) (PHHAL_HW_SAM_RC663_TIMER_FREQ * PHHAL_HW_SAM_RC663_ETU_106);

            /* T1 reload is the given value divided by the prescaler T0 frequency */
            fTime = fTime / PHHAL_HW_SAM_RC663_ETU_106;
            wT1Reload = (uint16_t) fTime;

            /* Since timer IRQ is raised at underflow, wReload has to decremented by one.
             * However, if the resulting reload value has commas, it must be 'ceiled'
             */
            if(!(fTime > (float32_t) wT1Reload))
            {
                --wT1Reload;
            }
        }
    }
    /* Calculate values for Millisecond values */
    else
    {
        /* T0 overflow occurs every millisecond */
        wT0Reload = (uint16_t) (1000.0f * PHHAL_HW_SAM_RC663_TIMER_FREQ);

        /* T1 reload matches exactly the given value */
        wT1Reload = ((wTimeout > 0) ? --wTimeout : wTimeout);
    }

    /* Write T0 reload value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T0RELOADHI, (uint8_t) (wT0Reload >> 8U)));
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T0RELOADLO, (uint8_t) (wT0Reload)));

    /* Write T1 reload value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T1RELOADHI, (uint8_t) (wT1Reload >> 8U)));
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_WriteRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T1RELOADLO, (uint8_t) (wT1Reload)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_GetFdt(phhalHw_Sam_DataParams_t * pDataParams, phStatus_t wExchangeStatus, uint32_t * pTime)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    float32_t   PH_MEMLOC_REM fTime = 0.0;
    uint16_t    PH_MEMLOC_REM wT0Reload = 0;
    uint16_t    PH_MEMLOC_REM wT1Reload = 0;
    uint16_t    PH_MEMLOC_REM wT0Value = 0;
    uint16_t    PH_MEMLOC_REM wT1Value = 0;
    uint16_t    PH_MEMLOC_REM wDelay = 0;
    uint8_t     PH_MEMLOC_REM bRegister = 0;

    /* Get T0 reload value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T0RELOADHI, &bRegister));
    wT0Reload = (uint16_t) bRegister << 8U;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T0RELOADLO, &bRegister));
    wT0Reload |= (uint16_t) bRegister;

    /* Get T1 reload value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T1RELOADHI, &bRegister));
    wT1Reload = (uint16_t) bRegister << 8U;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T1RELOADLO, &bRegister));
    wT1Reload |= (uint16_t) bRegister;

    /* Get T0 counter value */
    if((wExchangeStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T0COUNTERVALHI, &bRegister));
        wT0Value = (uint16_t) bRegister << 8;
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T0COUNTERVALLO, &bRegister));
        wT0Value |= (uint16_t) bRegister;
    }
    else
    {
        wT0Value = 0;
    }

    /* Get T1 counter value */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T1COUNTERVALHI, &bRegister));
    wT1Value = (uint16_t) bRegister << 8U;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_ReadRegister(pDataParams, PHHAL_HW_SAM_RC663_REG_T1COUNTERVALLO, &bRegister));
    wT1Value |= (uint16_t) bRegister;

    /* Subtract reload- and counter values */
    wT0Value = wT0Reload - wT0Value;
    wT1Value = wT1Reload - wT1Value;

    /*
     * Now we can calculate the time in microseconds according to the following formular:
     *  time[us] = (wT1Value * (1 / (PHHAL_HW_SAM_TIMER_FREQ / wT0Reload))) + (wT0Value * (1 / PHHAL_HW_SAM_TIMER_FREQ))
     */
    fTime = (float32_t) wT1Value / (PHHAL_HW_SAM_RC663_TIMER_FREQ / (float32_t) wT0Reload);
    fTime = fTime + ((float32_t) wT0Value / PHHAL_HW_SAM_RC663_TIMER_FREQ);

    /* Return the value */
    *pTime = (uint32_t) fTime;
    if((float32_t) *pTime < fTime)
    {
        ++(*pTime);
    }

    /* Retrieve and subtract digital delay */
    if((wExchangeStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_GetDigiDelay(pDataParams, PH_OFF, &wDelay));
        *pTime = *pTime - wDelay;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Rc663_GetDigiDelay(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bIsTimeout, uint16_t * pDelayUs)
{
    uint16_t PH_MEMLOC_REM wDataRate = 0;

    wDataRate = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE];

    *pDelayUs = gkphhalHw_Sam_Rc663_DigiDelays_Us[wDataRate];

    /* In a timeout case we have to add an additional delay */
    if(bIsTimeout != PH_OFF)
    {
        *pDelayUs = *pDelayUs + PHHAL_HW_SAM_RC663_I14443_ADD_DELAY_US;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
