/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Specific SAM (AV4 and future SAM's) HAL Component of Reader Library Framework.
 * Implementation of generic HAL interfaces
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <phCryptoSym.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "phhalHw_Sam.h"
#include "Commands/phhalHw_Sam_Cmd.h"
#include "Commands/02_SecurityConfiguration/phhalHw_Sam_Cmd_SC.h"
#include "Commands/03_KeyManagement/phhalHw_Sam_Cmd_KM.h"
#include "Commands/05_DataProcessing/phhalHw_Sam_Cmd_DP.h"
#include "Commands/15_ReaderIC/phhalHw_Sam_Cmd_RC.h"
#include "Commands/16_ISO14443_3/phhalHw_Sam_Cmd_ISO14443_3.h"

#include "HSM_AES/phhalHw_Sam_HSM_AES.h"
#include "Utils/phhalHw_Sam_HcUtils.h"
#include "Utils/Rc523/phhalHw_Sam_Rc523.h"
#include "Utils/Rc663/phhalHw_Sam_Rc663.h"

/* Default shadow for ISO14443-3A Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wSam_DefaultShadow_I14443a[][2U] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_ON},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0008U},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_SAM_DEFAULT_TIMEOUT},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0x0000},
    {PHHAL_HW_CONFIG_ASK100,                PH_ON}
};

phStatus_t phhalHw_Sam_Init(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pBalDataParams,
    void * pReaderHalDataParams, void * pKeyStoreDataParams, void * pCryptoENCDataParams, void * pCryptoMACDataParams,
    void * pCryptoRngDataParams, void * pPLUpload_CryptoENCDataParams, void * pPLUpload_CryptoMACDataParams,
    uint8_t bOpMode, uint8_t bLogicalChannel, uint8_t* pTxBuffer, uint16_t wTxBufSize, uint8_t* pRxBuffer,
    uint16_t wRxBufSize, uint8_t* pPLUploadBuf, uint8_t * pCmdBuff, uint16_t wCmdBuffSize)
{
    if(sizeof(phhalHw_Sam_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pBalDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCryptoENCDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCryptoMACDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCryptoRngDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPLUpload_CryptoENCDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPLUpload_CryptoMACDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pTxBuffer, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRxBuffer, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPLUploadBuf, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pCmdBuff, PH_COMP_HAL);

    /* Verify exchange buffers */
    if((wTxBufSize <= PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN) || (wRxBufSize <= PHHAL_HW_SAM_RESERVED_RX_BUFFER_LEN))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Verify operation mode */
    if(bOpMode > PHHAL_HW_SAM_OPMODE_X_RC663)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Verify NonX reader HAL pointer */
    if((bOpMode == PHHAL_HW_SAM_OPMODE_NON_X) && (pReaderHalDataParams == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    pDataParams->wId = PH_COMP_HAL | PHHAL_HW_SAM_ID;
    pDataParams->pBalDataParams = pBalDataParams;
    pDataParams->pReaderHalDataParams = pReaderHalDataParams;
    pDataParams->pKeyStoreDataParams = pKeyStoreDataParams;
    pDataParams->pENCCryptoDataParams = pCryptoENCDataParams;
    pDataParams->pMACCryptoDataParams = pCryptoMACDataParams;
    pDataParams->pCryptoRngDataParams = pCryptoRngDataParams;
    pDataParams->pPLUpload_ENCCryptoDataParams = pPLUpload_CryptoENCDataParams;
    pDataParams->pPLUpload_MACCryptoDataParams = pPLUpload_CryptoMACDataParams;
    pDataParams->wCmd_Ctr = 0;
    pDataParams->bHostMode = PHHAL_HW_SAM_HC_AV3_MODE;
    pDataParams->bAuthType = 0x00;
    pDataParams->bPendingEncCmdDataLength = 0;
    pDataParams->bPendingMacCmdDataLength = 0;
    pDataParams->bPendingMacRespDataLength = 0;
    pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
    pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
    pDataParams->bCommandChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;
    pDataParams->bResponseChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;
    pDataParams->bOpMode = bOpMode;
    pDataParams->bLogicalChannel = bLogicalChannel;
    pDataParams->pTxBuffer = pTxBuffer;
    pDataParams->wTxBufSize = wTxBufSize;
    pDataParams->wTxBufLen = 0;
    pDataParams->pRxBuffer = pRxBuffer;
    pDataParams->wRxBufSize = wRxBufSize;
    pDataParams->wRxBufLen = 0;
    pDataParams->wRxBufStartPos = 0;
    pDataParams->wTxBufStartPos = 0;
    pDataParams->bCardType = PHHAL_HW_CARDTYPE_ISO14443A;
    pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
    pDataParams->wFieldOffTime = PHHAL_HW_FIELD_OFF_DEFAULT;
    pDataParams->wFieldRecoveryTime = PHHAL_HW_FIELD_RECOVERY_DEFAULT;
    pDataParams->wAdditionalInfo = 0;
    pDataParams->wErrorCode = 0;
    pDataParams->wTimingMode = PHHAL_HW_TIMING_MODE_OFF;
    pDataParams->dwTimingUs = 0;
    pDataParams->bRfResetAfterTo = PH_OFF;
    pDataParams->bDisableNonXCfgMapping = PH_OFF;
    pDataParams->pPLUploadBuf = pPLUploadBuf;
    pDataParams->wPLUploadBufLen = 0;
    pDataParams->pCmdBuff = pCmdBuff;
    pDataParams->wCmdBuffSize = wCmdBuffSize;

    /* Configure Default I2C Parameters for SAM. */
    phbalReg_SetConfig(pBalDataParams, PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE,
        PHHAL_HW_SAM_COMMUNICATION_TYPE_TDA);

    phbalReg_SetConfig(pBalDataParams, PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS,
        0x2BU);

    phbalReg_SetConfig(pBalDataParams, PHHAL_HW_CONFIG_I2C_TIMEOUT_MS,
        200U);

    phbalReg_SetConfig(pBalDataParams, PHHAL_HW_CONFIG_I2C_BITRATE,
        PHHAL_HW_CONFIG_I2C_BITRATE_FAST_MODE);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_DetectMode(phhalHw_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aVersion[PHHAL_HW_SAM_CMD_GET_VERSION_RESPONSE_LENGTH];
    uint8_t     PH_MEMLOC_REM bVerLen = 0;

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Issue GetVersion command */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_GetVersion(
        pDataParams,
        aVersion,
        &bVerLen));

    /* Check length of received response */
    if(bVerLen == PHHAL_HW_SAM_CMD_GET_VERSION_RESPONSE_LENGTH)
    {
        /* Retrieve Host-Mode */
        switch(aVersion[PHHAL_HW_SAMAV2_CMD_GET_VERSION_RESPONSE_HOSTMODE_POS])
        {
            case 0x03U: /* Unactivated State. */
            case 0xA3U: /* Activate State. */
                pDataParams->bHostMode = PHHAL_HW_SAM_HC_AV3_MODE;
                break;

            case 0x04U: /* Unactivated State. */
            case 0xA4U: /* Activate State. */
                pDataParams->bHostMode = PHHAL_HW_SAM_HC_AV4_MODE;
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_HAL);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Store the UID globally */
    memcpy(pDataParams->bUid, &aVersion[PHHAL_HW_SAM_CMD_GET_VERSION_RESPONSE_UID_OFFSET],
        PHHAL_HW_SAMAV2_HC_SAM_UID_SIZE);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Exchange(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pData, uint16_t wDataLen,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;

    uint32_t    PH_MEMLOC_REM dwTimingSingle = 0;
    uint16_t    PH_MEMLOC_REM wValidBits = 0;
    uint16_t    PH_MEMLOC_REM wParity = PH_OFF;
    uint16_t    PH_MEMLOC_REM wCrc = PH_OFF;
    uint8_t     PH_MEMLOC_REM bLeByte = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    uint8_t     PH_MEMLOC_REM aCmd[PHHAL_HW_SAM_ISO7816_HEADER_LENGTH];

    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* TODO: Check if this is required */
    /* uint16_t    PH_MEMLOC_REM wTxStartPosTmp = 0; */

    /* Check options */
    if(wOption & (uint16_t) ~(uint16_t) (PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Parameter check */
    if((wOption & PH_EXCHANGE_BUFFERED_BIT) && (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] > 0))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check if caller has provided valid RxBuffer */
    if(ppResponse == NULL)
    {
        ppResponse = &pResponse;
    }

    if(pRespLen == NULL)
    {
        pRespLen = &wRespLen;
    }

    *pRespLen = 0;
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos;
    pDataParams->wAdditionalInfo = 0;

    /* Non-X Mode : Exchange via Reader HAL */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        /* Perform Exchange */
        wStatus = phhalHw_Exchange(
            pDataParams->pReaderHalDataParams,
            wOption & (uint16_t) ~(uint16_t) PH_EXCHANGE_CUSTOM_BITS_MASK,
            pData,
            wDataLen,
            ppResponse,
            pRespLen);

        /* do not perform real exchange, just fill the global TxBuffer */
        if(wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return wStatus;
        }

        /* Restore Parity-setting again since many PAL layers expect it */
        if(wParity != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(pDataParams->pReaderHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_ON));
        }

        /* Restore RxCRC-setting again */
        if(wCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(pDataParams->pReaderHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_ON));
        }

        /* Retrieve RxLastBits */
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_GetConfig(pDataParams->pReaderHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &wValidBits));
        pDataParams->wAdditionalInfo = (uint8_t) wValidBits;

        /* Clear TxLastBits */
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] = 0;

        /* status check */
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
        {
            PH_CHECK_SUCCESS(wStatus);
        }
    }
    /* X Mode : Exchange ourself */
    else
    {
        /* Perpend Header */
        if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
        {
            /* Filling in ISO7816 header */
            aCmd[PHHAL_HW_SAM_ISO7816_CLA_POS] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
            aCmd[PHHAL_HW_SAM_ISO7816_INS_POS] = PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE;
            aCmd[PHHAL_HW_SAM_ISO7816_P1_POS] = 0x00;
            aCmd[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
            aCmd[PHHAL_HW_SAM_ISO7816_LC_POS] = 0x00;

            /* Buffer header */
            PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                aCmd,
                PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
                NULL,
                NULL));
        }

        /* Check for buffer overflow (include LE) */
        if((pDataParams->wTxBufStartPos + pDataParams->wTxBufLen + wDataLen + 1U) > pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Append data */
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pData,
            wDataLen,
            NULL,
            NULL));

        /* if we do not perform real exchange we're finished */
        if(wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

        /* Modify P1 byte (TxLastBits) */
        pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + PHHAL_HW_SAM_ISO7816_P1_POS] =
            (uint8_t) pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS];

        /* Modify LC byte */
        pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + PHHAL_HW_SAM_ISO7816_LC_POS] =
            (uint8_t) (pDataParams->wTxBufLen - PHHAL_HW_SAM_ISO7816_HEADER_LENGTH);

        /* Append LE byte and exchange */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            &bLeByte,
            1U,
            ppResponse,
            pRespLen);

        /* Clear TxLastBits */
        if(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] > 0)
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXLASTBITS, 0));
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_X_RC663)
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_Rc663_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXLASTBITS, 0));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_Rc523_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXLASTBITS, 0));
            }
        }

        /* Reset buffered bytes */
        pDataParams->wTxBufLen = 0;

        /* Timing */
        if((pDataParams->wTimingMode & (uint16_t) ~(uint16_t) PHHAL_HW_TIMING_MODE_OPTION_MASK) == PHHAL_HW_TIMING_MODE_FDT)
        {
            /* Retrieve FDT Value */
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_X_RC663)
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_Rc663_GetFdt(pDataParams, wStatus, &dwTimingSingle));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_Rc523_GetFdt(pDataParams, &dwTimingSingle));
            }

            /* FDT single -> replace the previous value */
            if(pDataParams->wTimingMode & PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR)
            {
                pDataParams->dwTimingUs = dwTimingSingle;
            }
            /* FDT -> add current value to current count */
            else
            {
                /* Check for overflow */
                if((0xFFFFFFFFU - pDataParams->dwTimingUs) < dwTimingSingle)
                {
                    pDataParams->dwTimingUs = 0xFFFFFFFFU;
                }
                /* Update global timing value */
                else
                {
                    pDataParams->dwTimingUs = pDataParams->dwTimingUs + dwTimingSingle;
                }
            }
        }

        /* Allow incomplete byte error */
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
        {
            /* Reset after timeout functionality */
            if((pDataParams->bRfResetAfterTo != PH_OFF) &&
                ((wStatus & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT))
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_FieldReset(pDataParams));
            }

            /* Collision error together with no data means protocol error for the library */
            if(((wStatus & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR) && (*pRespLen == 0))
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
            }

            /* check return code */
            PH_CHECK_SUCCESS(wStatus);
        }
    }
    return wStatus;
}

phStatus_t phhalHw_Sam_GetConfig(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* In case of Non-X mode, the GetConfig is directly redirected to the Reader IC if not disabled. */
    /* Exceptions are: RxLastbits and custom configs. */
    if(pDataParams->bDisableNonXCfgMapping == PH_OFF)
    {
        if((pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X) &&
            (wConfig != PHHAL_HW_CONFIG_RXLASTBITS) &&
            (wConfig != PHHAL_HW_SAM_CONFIG_HOSTMODE) &&
            (wConfig != PHHAL_HW_SAM_CONFIG_DISABLE_NONX_CFG_MAPPING) &&
            (wConfig != PHHAL_HW_SAM_CONFIG_GET_SAM_ERROR_CODE) &&
            (wConfig != PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE) &&
            (wConfig != PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS) &&
            (wConfig != PHHAL_HW_CONFIG_I2C_TIMEOUT_MS) &&
            (wConfig != PHHAL_HW_CONFIG_I2C_BITRATE))
        {
            return phhalHw_GetConfig(pDataParams->pReaderHalDataParams, wConfig, pValue);
        }
    }

    switch(wConfig)
    {
        case PHHAL_HW_CONFIG_PARITY:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_TXCRC:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_RXCRC:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_TXLASTBITS:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
        case PHHAL_HW_CONFIG_RXLASTBITS:

            *pValue = pDataParams->wAdditionalInfo;
            break;

        case PHHAL_HW_CONFIG_RXDEAFBITS:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_TXDATARATE:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_RXDATARATE:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_MODINDEX:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_ASK100:

            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:

            if(pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
            {
                *pValue = pDataParams->wCfgShadow[wConfig];
            }
            else
            {
                if(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] > (0xFFFFU / 1000U))
                {
                    return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
                }
                *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] * 1000U;
            }
            break;

        case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:

            if(pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
            {
                *pValue = pDataParams->wCfgShadow[wConfig];
            }
            else
            {
                *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US] / 1000U;
            }
            break;

        case PHHAL_HW_CONFIG_TIMING_MODE:

            *pValue = pDataParams->wTimingMode;
            break;

        case PHHAL_HW_CONFIG_TIMING_US:

            if(pDataParams->dwTimingUs > 0xFFFFU)
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }

            *pValue = (uint16_t) pDataParams->dwTimingUs;
            pDataParams->dwTimingUs = 0;
            break;

        case PHHAL_HW_CONFIG_TIMING_MS:

            if(pDataParams->dwTimingUs > (0xFFFFU * 1000U))
            {
                pDataParams->dwTimingUs = 0;
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }

            *pValue = (uint16_t) (pDataParams->dwTimingUs / 1000U);
            pDataParams->dwTimingUs = 0;
            break;

        case PHHAL_HW_CONFIG_FIELD_OFF_TIME:

            *pValue = pDataParams->wFieldOffTime;
            break;

        case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:

            *pValue = pDataParams->wFieldRecoveryTime;
            break;

        case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:

            /* Return parameter */
            *pValue = pDataParams->wRxBufStartPos;
            break;

        case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:

            /* Return parameter */
            *pValue = pDataParams->wRxBufSize - PHHAL_HW_SAMAV2_RESERVED_RX_BUFFER_LEN;
            break;

        case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:

            /* Return parameter */
            *pValue = pDataParams->wTxBufSize - (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN + pDataParams->wTxBufStartPos);
            break;

        case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:

            /* Needed for MIFARE Encrypted buffered data */
            if(pDataParams->wTxBufLen_Cmd >= (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1U))
            {
                *pValue = pDataParams->wTxBufLen_Cmd - (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1U);
            }
            else
            {
                /* Normal Exchange */
                if(pDataParams->wTxBufLen >= PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN)
                {
                    *pValue = pDataParams->wTxBufLen - PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN;
                }
                else
                {
                    *pValue = 0;
                }
            }
            break;

        case PHHAL_HW_CONFIG_TXBUFFER:

            /* Needed for MIFARE Encrypted buffered data */
            if(pDataParams->wTxBufLen_Cmd >= (PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U))
            {
                /* Check additional info parameter */
                if(((PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1) + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
                }

                /* Return TxBuffer byte */
                *pValue = (uint16_t) pDataParams->pTxBuffer[(PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1U) +
                    pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
            }
            /* Normal Exchange */
            else
            {
                /* Check additional info parameter */
                if((PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
                }

                /* Return TxBuffer byte */
                *pValue = (uint16_t) pDataParams->pTxBuffer[PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN +
                    pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
            }
            break;

        case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:

            *pValue = (uint16_t) pDataParams->bRfResetAfterTo;
            break;

        case PHHAL_HW_CONFIG_CARD_TYPE:
            /* Return parameter */
            *pValue = (uint16_t) pDataParams->bCardType;
            break;

        case PHHAL_HW_SAM_CONFIG_HOSTMODE:

            *pValue = pDataParams->bHostMode;
            break;

        case PHHAL_HW_SAM_CONFIG_DISABLE_NONX_CFG_MAPPING:
            *pValue = pDataParams->bDisableNonXCfgMapping;
            break;

        case PHHAL_HW_CONFIG_SET_READER_IC:
            *pValue = pDataParams->bOpMode;
            break;

        case PHHAL_HW_SAM_CONFIG_GET_SAM_ERROR_CODE:
            *pValue = pDataParams->wErrorCode;
            break;

        case PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE:
            wStatus = phbalReg_GetConfig(
                pDataParams->pBalDataParams,
                PHBAL_REG_SAM_CONFIG_COMMUNICATION_TYPE,
                pValue);
            break;

        case PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS:
            wStatus = phbalReg_GetConfig(
                pDataParams->pBalDataParams,
                PHBAL_REG_SAM_CONFIG_I2C_SLAVE_ADDRESS,
                pValue);
            break;

        case PHHAL_HW_CONFIG_I2C_TIMEOUT_MS:
            wStatus = phbalReg_GetConfig(
                pDataParams->pBalDataParams,
                PHBAL_REG_SAM_CONFIG_I2C_TIMEOUT_MS,
                pValue);
            break;

        case PHHAL_HW_CONFIG_I2C_BITRATE:
            wStatus = phbalReg_GetConfig(
                pDataParams->pBalDataParams,
                PHBAL_REG_SAM_CONFIG_I2C_BITRATE,
                pValue);
            break;

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            break;
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_SetConfig(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* In case of Non-X mode, the SetConfig is directly redirected to the Reader IC if not disabled. */
    if(pDataParams->bDisableNonXCfgMapping == PH_OFF)
    {
        if((pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X) &&
            (wConfig != PHHAL_HW_SAM_CONFIG_HOSTMODE) &&
            (wConfig != PHHAL_HW_SAM_CONFIG_DISABLE_NONX_CFG_MAPPING) &&
            (wConfig != PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE) &&
            (wConfig != PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS) &&
            (wConfig != PHHAL_HW_CONFIG_I2C_TIMEOUT_MS) &&
            (wConfig != PHHAL_HW_CONFIG_I2C_BITRATE))
        {
            return phhalHw_SetConfig(pDataParams->pReaderHalDataParams, wConfig, wValue);
        }
    }

    switch(wConfig)
    {
        case PHHAL_HW_CONFIG_TXLASTBITS:

            /* Check parameter */
            if(wValue > 7U)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = wValue;
            break;

        case PHHAL_HW_CONFIG_TIMING_MODE:

            /* Check supported option bits */
            switch(wValue & PHHAL_HW_TIMING_MODE_OPTION_MASK)
            {
                case PHHAL_HW_TIMING_MODE_OPTION_DEFAULT:
                case PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR:
                    break;
                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Check supported timing modes */
            switch(wValue & (uint16_t) ~(uint16_t) PHHAL_HW_TIMING_MODE_OPTION_MASK)
            {
                case PHHAL_HW_TIMING_MODE_OFF:
                case PHHAL_HW_TIMING_MODE_FDT:
                    pDataParams->dwTimingUs = 0;
                    pDataParams->wTimingMode = wValue;
                    break;
                case PHHAL_HW_TIMING_MODE_COMM:
                    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONFIG_FIELD_OFF_TIME:

            /* Parameter Check */
            if((wValue == 0) || (wValue > 0xFFU))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Store config data */
            pDataParams->wFieldOffTime = wValue;
            break;

        case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:

            /* Store config data */
            pDataParams->wFieldRecoveryTime = wValue;
            break;

        case PHHAL_HW_CONFIG_ADDITIONAL_INFO:

            /* Modify additional info parameter */
            pDataParams->wAdditionalInfo = wValue;
            break;

        case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:

            /* Boundary check */
            if((PHHAL_HW_SAM_RESERVED_RX_BUFFER_LEN + wValue) >= pDataParams->wRxBufSize)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Set start position */
            pDataParams->wRxBufStartPos = wValue;
            pDataParams->wRxBufLen = wValue;

            /* Preserve RxBuffer contents if needed */
            if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
            {
                pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
            }
            else
            {
                pDataParams->wTxBufStartPos = 0;
            }
            break;

        case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:

            /* Needed for MIFARE Encrypted buffered data */
            if(pDataParams->wTxBufLen_Cmd >= (PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U))
            {
                /* Check parameter */
                if(((PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U) + pDataParams->wTxBufStartPos + wValue) > pDataParams->wTxBufSize)
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
                }

                /* set buffer length */
                pDataParams->wTxBufLen_Cmd = (PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U) + wValue;
            }
            /* Normal Exchange */
            else
            {
                /* Check parameter */
                if((PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN + pDataParams->wTxBufStartPos + wValue) > pDataParams->wTxBufSize)
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
                }

                /* set buffer length */
                pDataParams->wTxBufLen = PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN + wValue;
            }
            break;

        case PHHAL_HW_CONFIG_TXBUFFER:

            /* Needed for MIFARE Encrypted buffered data */
            if(pDataParams->wTxBufLen_Cmd >= (PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U))
            {
                /* Check additional info parameter */
                if(((PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U) + pDataParams->wTxBufStartPos +
                    pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
                }

                /* Modify TxBuffer byte */
                pDataParams->pTxBuffer[(PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN - 1U) + pDataParams->wTxBufStartPos +
                    pDataParams->wAdditionalInfo] = (uint8_t) wValue;
            }
            /* Normal Exchange */
            else
            {
                /* Check additional info parameter */
                if((PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN + pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo)
                    >= pDataParams->wTxBufSize)
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
                }

                /* Modify TxBuffer byte */
                pDataParams->pTxBuffer[PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN + pDataParams->wTxBufStartPos +
                    pDataParams->wAdditionalInfo] = (uint8_t) wValue;
            }
            break;

        case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:

            if(wValue == PH_OFF)
            {
                pDataParams->bRfResetAfterTo = PH_OFF;
            }
            else
            {
                pDataParams->bRfResetAfterTo = PH_ON;
            }
            break;

        case PHHAL_HW_SAM_CONFIG_HOSTMODE:

            pDataParams->bHostMode = (uint8_t) wValue;
            break;

        case PHHAL_HW_SAM_CONFIG_DISABLE_NONX_CFG_MAPPING:

            if(wValue != PH_OFF)
            {
                pDataParams->bDisableNonXCfgMapping = PH_ON;
            }
            else
            {
                pDataParams->bDisableNonXCfgMapping = PH_OFF;
            }
            break;

        case PHHAL_HW_CONFIG_SETMINFDT:

            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_SetMinFDT(pDataParams, wValue));
            break;

        case PHHAL_HW_CONFIG_SET_READER_IC:
            pDataParams->bOpMode = (uint8_t) wValue;
            break;

        case PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE:
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
            {
                wStatus = phbalReg_SetConfig(
                    pDataParams->pBalDataParams,
                    PHBAL_REG_SAM_CONFIG_COMMUNICATION_TYPE,
                    wValue);
            }
            else
            {
                wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS:
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
            {
                wStatus = phbalReg_SetConfig(
                    pDataParams->pBalDataParams,
                    PHBAL_REG_SAM_CONFIG_I2C_SLAVE_ADDRESS,
                    wValue);
            }
            else
            {
                wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONFIG_I2C_TIMEOUT_MS:
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
            {
                wStatus = phbalReg_SetConfig(
                    pDataParams->pBalDataParams,
                    PHBAL_REG_SAM_CONFIG_I2C_TIMEOUT_MS,
                    wValue);
            }
            else
            {
                wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONFIG_I2C_BITRATE:
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
            {
                wStatus = phbalReg_SetConfig(
                    pDataParams->pBalDataParams,
                    PHBAL_REG_SAM_CONFIG_I2C_BITRATE,
                    wValue);
            }
            else
            {
                wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            break;

        default:

            /* Perform ReaderIC specific configuration */
            if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_X_RC663)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams, wConfig, wValue));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetConfig(pDataParams, wConfig, wValue));
            }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_SetMinFDT(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wValue)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint16_t   PH_MEMLOC_REM wTimer = 0;
    uint16_t   PH_MEMLOC_REM wTxRate = 0;

    if(wValue == PH_ON)
    {
        /*Backup the old Timer values and set min FDT*/
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_GetConfig(pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, &wTimer));
        pDataParams->dwFdtPc = wTimer;

        /* Get the data rate */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_GetConfig(pDataParams,
            PHHAL_HW_CONFIG_TXDATARATE, &wTxRate));

        switch(wTxRate)
        {
            case PHHAL_HW_RF_DATARATE_106:
                wTimer = PHHAL_HW_MINFDT_106_US;
                break;

            case PHHAL_HW_RF_DATARATE_212:
                wTimer = PHHAL_HW_MINFDT_212_US;
                break;

            case PHHAL_HW_RF_DATARATE_424:
                wTimer = PHHAL_HW_MINFDT_424_US;
                break;

            case PHHAL_HW_RF_DATARATE_848:
                wTimer = PHHAL_HW_MINFDT_848_US;
                break;

            default:
                break;
        }

        /* Perform ReaderIC specific configuration */
        if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_X_RC663)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, wTimer));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetConfig(pDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, wTimer));
        }
    }
    else if(wValue == PH_OFF)
    {
        /* Perform ReaderIC specific configuration */
        if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_X_RC663)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc663_SetConfig(pDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, pDataParams->dwFdtPc));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Rc523_SetConfig(pDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, pDataParams->dwFdtPc));
        }
    }
    else
    {
        /* Do nothing*/
    }
    return wStatus;
}

phStatus_t phhalHw_Sam_ApplyProtocolSettings(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCardType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_COUNT wIndex = 0;
    uint8_t     PH_MEMLOC_REM bUseDefaultShadow = 0;
    uint16_t    PH_MEMLOC_REM wConfig = 0;

    uint16_t *  PH_MEMLOC_REM pShadowDefault = NULL;
    uint16_t    PH_MEMLOC_REM wShadowCount = 0;

    /* In case of Non-X mode, the ApplyProtocolSettings is directly redirected to the Reader IC */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        return phhalHw_ApplyProtocolSettings(pDataParams->pReaderHalDataParams, bCardType);
    }

    /* We only support ISO 14443A */
    if(pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443A)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Store new card type */
    if(bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        pDataParams->bCardType = bCardType;
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
        bUseDefaultShadow = 1;

        /* Initialize config shadow */
        memset(pDataParams->wCfgShadow, 0x00, PHHAL_HW_RC523_SHADOW_COUNT);
    }
    else
    {
        bUseDefaultShadow = 0;
    }

    /* Use 14443a default shadow */
    pShadowDefault = (uint16_t*) wSam_DefaultShadow_I14443a;
    wShadowCount = sizeof(wSam_DefaultShadow_I14443a) / (sizeof(uint16_t) * 2U);

    /* Apply shadowed registers */
    for(wIndex = 0; wIndex < wShadowCount; ++wIndex)
    {
        /* Get wConfig */
        wConfig = pShadowDefault[wIndex << 1U];

        /* Apply only one the correct timeout unit */
        if(!(((wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US) &&
            (pDataParams->bTimeoutUnit != PHHAL_HW_TIME_MICROSECONDS)) ||
            ((wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS) &&
            (pDataParams->bTimeoutUnit != PHHAL_HW_TIME_MILLISECONDS))))
        {
            /* Default shadow: */
            if(bUseDefaultShadow)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_SetConfig(pDataParams, wConfig, pShadowDefault[(wIndex << 1U) + 1U]));
            }
            /* Current shadow: */
            else
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_ReadRegister(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bAddress, uint8_t * pValue)
{
    /* In case of Non-X mode, the ReadRegister is directly redirected to the Reader IC */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        return phhalHw_ReadRegister(pDataParams->pReaderHalDataParams, bAddress, pValue);
    }

    /* perform command */
    return phhalHw_Sam_Cmd_RC_ReadRegister(pDataParams, &bAddress, 1, pValue);
}

phStatus_t phhalHw_Sam_WriteRegister(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bAddress, uint8_t bValue)
{
    uint8_t PH_MEMLOC_REM aData[2U];

    /* In case of Non-X mode, the WriteRegister is directly redirected to the Reader IC */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        return phhalHw_WriteRegister(pDataParams->pReaderHalDataParams, bAddress, bValue);
    }

    aData[0] = bAddress;
    aData[1U] = bValue;

    /* perform command */
    return phhalHw_Sam_Cmd_RC_WriteRegister(pDataParams, aData, 2U);
}

phStatus_t phhalHw_Sam_FieldReset(phhalHw_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* In case of Non-X mode, the FieldReset is directly redirected to the Reader */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        return phhalHw_FieldReset(pDataParams->pReaderHalDataParams);
    }

    /* Perform field reset */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_RC_RFControl(pDataParams, (uint8_t) pDataParams->wFieldOffTime));

    /* Wait recovery time */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Wait(
        pDataParams,
        PHHAL_HW_TIME_MILLISECONDS,
        pDataParams->wFieldRecoveryTime));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_FieldOn(phhalHw_Sam_DataParams_t * pDataParams)
{
    /* In case of Non-X mode, the FieldOn is directly redirected to the Reader IC */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        return phhalHw_FieldOn(pDataParams->pReaderHalDataParams);
    }

    return phhalHw_Sam_Cmd_RC_RFControl(pDataParams, 1U);
}

phStatus_t phhalHw_Sam_FieldOff(phhalHw_Sam_DataParams_t * pDataParams)
{
    /* In case of Non-X mode, the FieldOn is directly redirected to the Reader IC */
    if(pDataParams->bOpMode == PHHAL_HW_SAM_OPMODE_NON_X)
    {
        return phhalHw_FieldOff(pDataParams->pReaderHalDataParams);
    }

    return phhalHw_Sam_Cmd_RC_RFControl(pDataParams, 1U);
}

phStatus_t phhalHw_Sam_Wait(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    switch(pDataParams->bOpMode)
    {
        /* In case of Non-X mode, the Wait is directly redirected to the Reader IC */
        case PHHAL_HW_SAM_OPMODE_NON_X:
            wStatus = phhalHw_Wait(pDataParams->pReaderHalDataParams, bUnit, wTimeout);
            break;

        /* Rc663 in X-Mode */
        case PHHAL_HW_SAMAV2_OPMODE_X_RC663:
            wStatus = phhalHw_Sam_Rc663_Wait(pDataParams, bUnit, wTimeout);
            break;

        /* Rc523 in X-Mode */
        default:
            wStatus = phhalHw_Sam_Rc523_Wait(pDataParams, bUnit, wTimeout);
            break;
    }

    return wStatus;
}
#endif /* NXPBUILD__PHHAL_HW_SAM */
