/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic HAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHwContact.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS

#include <phToolsAtrParser.h>
#include <phToolsMicropross.h>
#include "phhalHwContact_Micropross_Int.h"
#include "phhalHwContact_Micropross.h"
#include <stdio.h>              /* PRQA S 5124 */
#include <time.h>

phStatus_t phhalHwContact_Micropross_Init(
                                          phhalHwContact_Micropross_DataParams_t * pDataParams,
                                          uint16_t wSizeOfDataParams,
                                          void * pBalDataParams,
                                          uint8_t * pTxBuffer,
                                          uint16_t wTxBufSize,
                                          uint8_t * pRxBuffer,
                                          uint16_t wRxBufSize,
                                          uint8_t * pIntBuffer,
                                          uint16_t wIntBufferSize
                                          )
{
    if (sizeof(phhalHwContact_Micropross_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    if (wRxBufSize == 0) /* wTxBufSize == 0 is allowed */
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Same buffer for rx and tx is forbidden */
    if (pTxBuffer == pRxBuffer)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);
    PH_ASSERT_NULL (pIntBuffer);

    /* Init. private data */
    pDataParams->wId                                        = PH_COMP_HAL | PHHAL_HW_CONTACT_MICROPROSS_ID;
    pDataParams->pBalDataParams                             = pBalDataParams;
    pDataParams->pTxBuffer                                  = pTxBuffer;
    pDataParams->wTxBufSize                                 = wTxBufSize;
    pDataParams->wTxBufLen                                  = 0;
    pDataParams->pRxBuffer                                  = pRxBuffer;
    pDataParams->wRxBufSize                                 = wRxBufSize;
    pDataParams->wRxBufLen                                  = 0;
    pDataParams->wRxBufStartPos                             = 0;
    pDataParams->wTxBufStartPos                             = 0;
    pDataParams->dwNumExpectedBytes                         = 0;
    pDataParams->bProtocolType                              = PHHAL_HW_CONTACT_PROTOCOLTYPE_T0;
    pDataParams->dwI2CRxPollingTimeoutMs                    = 1000;
    pDataParams->wTimingMode                                = PHHAL_HW_TIMING_MODE_OFF;
    pDataParams->bTimeoutUnit                               = PHHAL_HW_CONTACT_TIME_MICROSECONDS;
    pDataParams->qwTiming_ns                                = 0;
    pDataParams->bRfResetAfterTo                            = PH_OFF;
    pDataParams->pIntBuffer                                 = pIntBuffer;
    pDataParams->wIntBufferSize                             = wIntBufferSize;
    pDataParams->wIntBufferLen                              = 0x00;
    pDataParams->wIntBufferReadPos                          = 0x00;
    pDataParams->bCouplerNumber                             = PHHAL_HW_CONTACT_MICROPROSS_COUPER_NUMBER_DEFAULT;
    pDataParams->dwGuardTimeEtu                             = 2;
    pDataParams->dwGuardTimeClk                             = 0;
    pDataParams->wCouplerResourceId                         = NULL;
    pDataParams->bSerialNumber[0]                           = 0;
    pDataParams->bDeviceFamily[0]                           = 0;
    pDataParams->bDeviceReferenceWithinFamily[0]            = 0;
    pDataParams->bMotherboardSystemVersion[0]               = 0;
    pDataParams->bMotherboardBootVersion[0]                 = 0;
    pDataParams->bCouplerName[0]                            = 0;
    pDataParams->bCouplerSerialNumber[0]                    = 0;
    pDataParams->bCouplerDriverVersion[0]                   = 0;
    pDataParams->wCouplerFPGAVersion                        = 0;
    pDataParams->dwScenarioID                               = 0;
    pDataParams->SpyThreadHandle                            = 0;
    pDataParams->GEvtReady                                  = 0;
    pDataParams->GEvtFinished                               = 0;
    pDataParams->bCommunicationChannel                      = PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD;
    pDataParams->wSlaveAddress                              = 0;
    pDataParams->dwTHighScl                                 = 4000;
    pDataParams->dwTLowScl                                  = 4700;
    pDataParams->dwTSuDat                                   = 240;
    pDataParams->dwTHdDat                                   = 500;

    memset(pDataParams->bSpyFilename, 0x00, sizeof(pDataParams->bSpyFilename)); /* PRQA S 3200 */
    memset(pDataParams->pIntBuffer, 0x00, wIntBufferSize);                      /* PRQA S 3200 */

    pDataParams->pBalTimeoutHandling = NULL;
    pDataParams->pBalTimeoutHandlingContext = NULL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Exchange(
                                              phhalHwContact_Micropross_DataParams_t * pDataParams,
                                              uint16_t wOption,
                                              uint8_t * pTxBuffer,
                                              uint16_t wTxLength,
                                              uint8_t ** ppRxBuffer,
                                              uint16_t * pRxLength
                                              )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    phStatus_t  PH_MEMLOC_REM statusExchange = PH_ERR_SUCCESS;

    uint8_t *   PH_MEMLOC_REM pTmpBuffer;
    uint16_t    PH_MEMLOC_REM wTmpBufferLen = 0;
    uint16_t    PH_MEMLOC_REM wTmpBufferSize;

    uint8_t     PH_MEMLOC_REM pCrc[2];
    uint8_t     PH_MEMLOC_REM pLrc[1];

    uint8_t    PH_MEMLOC_REM bLrc;
    uint16_t   PH_MEMLOC_REM wCrc;

    uint16_t    PH_MEMLOC_REM wNumReceivedBytes = 0;
    int32_t     PH_MEMLOC_REM nbytes;
    //uint32_t    PH_MEMLOC_REM dwIccResponseTime = 0;

    uint16_t    PH_MEMLOC_REM wOldBalTimoutMs = 0;
    uint32_t    PH_MEMLOC_REM dwNewBalTimeoutMs = 0;
    uint32_t    PH_MEMLOC_REM dwCurrentProtocol = 0;
    uint32_t    PH_MEMLOC_REM dwCurrentClk = 0;
    clock_t     PH_MEMLOC_REM start;
    clock_t     PH_MEMLOC_REM end;

    uint16_t   PH_MEMLOC_REM wCrcLength;
    uint8_t *  PH_MEMLOC_REM pCrcData;

    uint32_t PH_MEMLOC_REM dwI2CReceiveTimeout;

    /* Check options */
    if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* set the receive length */
    if (pRxLength != NULL)
    {
        *pRxLength = 0;
    }

    /* Copy data to internal buffer */
    if (wTxLength != 0)
    {
        /* TxBuffer overflow check */
        if (wTxLength > (pDataParams->wTxBufSize - pDataParams->wTxBufLen))
        {
            pDataParams->wTxBufLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* copy data */
        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;
    }

    /* Fill the global TxBuffer */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        /* Buffer operation -> finished */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Get WT or BWT which is needed later to set correct timeout (if we get this timing later on we will probably overwrite the exchange buffer!) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, &dwCurrentProtocol));
    switch(dwCurrentProtocol)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, &dwNewBalTimeoutMs));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_BWT_CLK, &dwNewBalTimeoutMs));
        break;
    }

    /* convert clk cycles to ms plus 5 seconds for pcd communication and 5 seconds for picc communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ, &dwCurrentClk));
    dwNewBalTimeoutMs = (uint32_t)((float64_t)dwNewBalTimeoutMs / dwCurrentClk * 1000.0 + 10000);

    if(dwNewBalTimeoutMs > 65535)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    // Timing Mode not implemented for now
    //TODOGL: use for FDT because GetIccResponseTimeNs does not work if command gets timeout
    //if(pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_COMM || pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_FDT)
    //{
    //    //TODOGL: implementation for now evaluates FDT not COMM!
    //    /* Enable Spy */
    //    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_EnableSpy(pDataParams, CONTACT_MICROPROSS_EVENT_MASK_FDT, PHHAL_HW_CONTACT_MICROPROSS_DATEFORMAT_INTERNALCLK));
    //}

    /* Add CRC */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_CRC)
    {
        /* TxBuffer overflow check */
        if (2 > (pDataParams->wTxBufSize - pDataParams->wTxBufLen) ||
            2 > (0xFFFF - pDataParams->wTxBufLen))
        {
            pDataParams->wTxBufLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_CONTACTBASED,
            PH_TOOLS_CRC16_POLY_CONTACTBASED,
            pTxBuffer,
            wTxLength,
            &wCrc));

        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED)
        {
            pCrc[0] = (uint8_t)(wCrc >> 8);
            pCrc[1] = (uint8_t)(wCrc);
        }
        else
        {
            pCrc[0] = (uint8_t)(wCrc);
            pCrc[1] = (uint8_t)(wCrc >> 8);
        }

        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pCrc, 2); /* PRQA S 3200 */
        pDataParams->wTxBufLen += 2;
    }
    else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_LRC)
    {
        /* TxBuffer overflow check */
        if (1 > (pDataParams->wTxBufSize - pDataParams->wTxBufLen) ||
            1 > (0xFFFF - pDataParams->wTxBufLen))
        {
            pDataParams->wTxBufLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateLrc(pTxBuffer, wTxLength, pLrc));

        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pLrc, 1); /* PRQA S 3200 */

        pDataParams->wTxBufLen += 1;
    }

    /* check max transmit len */
    if (pDataParams->wTxBufLen > 4096)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Build Exchange command */
    /* Reset the scratch buffer len and add string to buffer */
    pDataParams->wIntBufferLen = 0;

    /* Buffer overflow check +1 for the '\0' */
    switch(pDataParams->bCommunicationChannel)
    {
    case PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD:
        nbytes = _snprintf(NULL, 0, "%s %u %u", CONTACT_MICROPROSS_CMD_EXCHANGE, pDataParams->bCouplerNumber, pDataParams->wTxBufLen) + 1;
        if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        pDataParams->wIntBufferLen += (uint16_t)_snprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
                pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, "%s %u %u",
                CONTACT_MICROPROSS_CMD_EXCHANGE, pDataParams->bCouplerNumber, pDataParams->wTxBufLen);

        /* Copy dummy data if not data should be transmitted */
        if (pDataParams->wTxBufLen == 0)
        {
            /* this was is needed for dummy sends where we just want to receive data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_AddToIntBufferHexString(pDataParams, pDataParams->pTxBuffer, 1));
        }
        else
        {
            /* Add data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_AddToIntBufferHexString(pDataParams, pDataParams->pTxBuffer, pDataParams->wTxBufLen));
        }

        /* Expected data */
        /* Buffer overflow check +1 for the '\0' */
        nbytes = _snprintf(NULL, 0, " %u", pDataParams->dwNumExpectedBytes) + 1;
        if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        pDataParams->wIntBufferLen += (uint16_t)_snprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
            pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, " %u", pDataParams->dwNumExpectedBytes);

        /* extend BAL USB timeouts */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

        statusExchange = phToolsMicropross_SendCommand(pDataParams);

        /* Reset buffered bytes */
        pDataParams->wTxBufLen = 0;

        /* Restore old timeout */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

        PH_CHECK_SUCCESS(statusExchange);

        /* get the return status */
        statusExchange = phToolsMicropross_GetStatus(pDataParams, CONTACT_MICROPROSS_CMD_EXCHANGE);

        /* get number of received bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_Uint16(pDataParams, &wNumReceivedBytes));

        break;

    case PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C:
        /* for t=1 we want first 3 bytes and receive LEN bytes afterwards + (LRC || CRC) length */
        /* for t=1 remove NAD calc redundancy code and add the correct I2C NAD afterwards*/
        /* adjust dwNumExpectedBytes for later use */

        /* for t=0 we already have the right number of expected bytes */

        /* I2C Write */
        if (pDataParams->wTxBufLen != 0)
        {
            nbytes = _snprintf(NULL, 0, "%s %u %X %X", CONTACT_MICROPROSS_CMD_I2C_WRITE, pDataParams->bCouplerNumber, pDataParams->wSlaveAddress, pDataParams->wTxBufLen) + 1;
            if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            pDataParams->wIntBufferLen += (uint16_t)_snprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
                    pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, "%s %u %X %X",
                    CONTACT_MICROPROSS_CMD_I2C_WRITE, pDataParams->bCouplerNumber, pDataParams->wSlaveAddress, pDataParams->wTxBufLen);

            /* Add data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_AddToIntBufferHexString(pDataParams, pDataParams->pTxBuffer, pDataParams->wTxBufLen));

            statusExchange = phToolsMicropross_SendCommand(pDataParams);

            /* Reset buffered bytes */
            pDataParams->wTxBufLen = 0;

            PH_CHECK_SUCCESS(statusExchange);

            /* get the return status */
            PH_CHECK_SUCCESS_FCT(statusExchange, phToolsMicropross_GetStatus(pDataParams, CONTACT_MICROPROSS_CMD_I2C_WRITE));

            if(pDataParams->wCouplerFPGAVersion < 40)
            {
                /* Calc tx delay (at least duration of one byte */
                phTools_Sleep(1 + (((pDataParams->dwTLowScl + pDataParams->dwTHighScl) * 9))/1000000);
            }
        }

        /* we do not expect any bytes so we can already return */
        if(pDataParams->dwNumExpectedBytes == 0)
        {
            /* Store received data length in dataparams */
            pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + 0;

            /* Return RxBuffer pointer */
            if (ppRxBuffer != NULL)
            {
                *ppRxBuffer = pDataParams->pRxBuffer;
            }

            /* Return RxBuffer length */
            if (pRxLength != NULL)
            {
                *pRxLength = pDataParams->wRxBufLen;
            }

            return PH_ADD_COMPCODE(statusExchange, PH_COMP_HAL);
        }

        /* Check length of read command */
        pDataParams->wIntBufferLen = 0;
        nbytes = _snprintf(NULL, 0, "%s %u %X %X", CONTACT_MICROPROSS_CMD_I2C_READ, pDataParams->bCouplerNumber, pDataParams->wSlaveAddress, pDataParams->dwNumExpectedBytes) + 1;
        if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Adjust timeout for I2C read to 5ms + length */
        /* FIXXME: 5ms is only a workaround for a MP issue with long Start Hold time */
        dwI2CReceiveTimeout = 5 + (((pDataParams->dwTLowScl + pDataParams->dwTHighScl) * (pDataParams->dwNumExpectedBytes + 1) * 9) + 999999)/1000000;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_RX_TIMEOUT, dwI2CReceiveTimeout, 0));

        start = clock();
        do
        {
            end = clock();

             /* Inital delay */
            phTools_Sleep(1);

            pDataParams->wIntBufferLen = 0;
            pDataParams->wIntBufferLen += (uint16_t)_snprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
                pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, "%s %u %X %X",
                CONTACT_MICROPROSS_CMD_I2C_READ, pDataParams->bCouplerNumber, pDataParams->wSlaveAddress, pDataParams->dwNumExpectedBytes);

            PH_CHECK_SUCCESS_FCT(statusExchange, phToolsMicropross_SendCommand(pDataParams));
            /* get the return status */
            statusExchange = phToolsMicropross_GetStatus(pDataParams, CONTACT_MICROPROSS_CMD_I2C_READ);
        } while((1000 * ((double)end - start)) / CLOCKS_PER_SEC <= pDataParams->dwI2CRxPollingTimeoutMs && (statusExchange & PH_ERR_MASK) != PH_ERR_SUCCESS);

        /* get number of received bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_Uint16Hex(pDataParams, &wNumReceivedBytes));

        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    /* Extract Data before any other action is done because buffer is reused (only everything is ok or only a collision occurs)*/
    if((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS || (statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        if(wNumReceivedBytes > 0)
        {
            /* Handle as it was success */
            statusExchange = (statusExchange & PH_COMP_MASK);

            /* Reset receive buffer length */
            pDataParams->wRxBufLen = 0;

            /* Retrieve receive buffer properties */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetRxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

            /* Extract the received data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_HexString(pDataParams, pTmpBuffer, wTmpBufferSize, &wTmpBufferLen));
        }
    }

    // Timing Mode not implemented for now
    ////TODOGL: use for FDT because GetIccResponseTimeNs does not work if command gets timeout
    //if(pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_COMM || pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_FDT)
    //{
    //    /* Disable Spy */
    //    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_DisableSpy(pDataParams));
    //}
    //else if(pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_FDT)
    //{
    //    /* Get timing via the IccResponseTime Command */
    //    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetIccResponseTimeNs(pDataParams, &dwIccResponseTime));

    //    pDataParams->qwTiming_ns = dwIccResponseTime;
    //}

    /* Extract Data only everything is ok or only a collision occurs*/
    if ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Check received bit count match with extraced data */
        if (wNumReceivedBytes != wTmpBufferLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        /* Store received data length in dataparams */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpBufferLen;

        /* get data for CRC check */
        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER) == PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER)
        {
            wCrcLength = pDataParams->wRxBufLen;
            pCrcData = pDataParams->pRxBuffer;
        }
        else
        {
            wCrcLength = pDataParams->wRxBufLen - pDataParams->wRxBufStartPos;
            pCrcData = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
        }

        /* Check and remove CRC in software if required */
        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
        {
            if (wCrcLength < 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
            }

            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
                PH_TOOLS_CRC16_PRESET_CONTACTBASED,
                PH_TOOLS_CRC16_POLY_CONTACTBASED,
                pCrcData,
                wCrcLength - 2,
                &wCrc));

            if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED)
            {
                pCrc[0] = (uint8_t)(wCrc >> 8);
                pCrc[1] = (uint8_t)(wCrc);
            }
            else
            {
                pCrc[0] = (uint8_t)(wCrc);
                pCrc[1] = (uint8_t)(wCrc >> 8);
            }

            /* CRC must match */
            if (pCrc[0] != pCrcData[wCrcLength-2] || pCrc[1] != pCrcData[wCrcLength-1])
            {
                return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
            }

            pDataParams->wRxBufLen -= 2;
        }
        else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_LRC)
        {
            if (wCrcLength < 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
            }

            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateLrc(
                pCrcData,
                wCrcLength,
                &bLrc));

            /* LRC over all data must be zero */
            if (bLrc != 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
            }

            pDataParams->wRxBufLen -= 1;
        }
        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    else if ((statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        /* Store received data length in dataparams */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + 0;

        /* If timeout then the response time is Zero */
        statusTmp = PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }
    else
    {
        /* Just delegate error */
        statusTmp = statusExchange;
    }

    /* Return RxBuffer pointer */
    if (ppRxBuffer != NULL)
    {
        *ppRxBuffer = pDataParams->pRxBuffer;
    }

    /* Return RxBuffer length */
    if (pRxLength != NULL)
    {
        *pRxLength = pDataParams->wRxBufLen;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_ApplyProtocolSettings(
                                                           phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                           uint8_t * pAtr,
                                                           uint16_t dwAtrLength,
                                                           uint8_t bProtocolType
                                                           )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t    PH_MEMLOC_REM wConfig;
    uint8_t     PH_MEMLOC_REM bConfig = 0;
    uint8_t     PH_MEMLOC_REM bSupported;
    uint32_t    PH_MEMLOC_REM wFDIndexValue;

    /* if ATR available check if card supports specified ProtocolType */
    if(dwAtrLength != 0 && pAtr != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(pAtr, dwAtrLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(bProtocolType, &bSupported));

        if(!bSupported)
        {
            /* not supported so reset to default values */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }

    /* Store new protocol type */
    if (bProtocolType != PHHAL_HW_CONTACT_PROTOCOLTYPE_CURRENT)
    {
        pDataParams->bProtocolType = bProtocolType;
    }

    /* Apply global Parameters */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_Convention(&bConfig));
    if(bConfig != 0x00)
    {
        /* ATR was recieved so we can set the convention */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CONVENTION, (uint32_t)bConfig));
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &wFDIndexValue));

    /* Apply protocol dependent Parameters */
    switch(bProtocolType)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_WTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_GTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_GT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, PHHAL_HW_CONTACT_MODE_DISABLED));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CGTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CGT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CWT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_BWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_BWT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_RedundancyMode(&bConfig));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, (uint32_t)bConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_IFSC(&bConfig));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_IFSC, (uint32_t)bConfig));

        pDataParams->dwNumExpectedBytes = PHHAL_HW_CONTACT_MAX_T1_BLOCK_SIZE;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, bProtocolType));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_ShadowDefaultPinValues(pDataParams));

    return statusTmp;
}

phStatus_t phhalHwContact_Micropross_SetConfig32(
                                                 phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                 uint16_t wConfig,
                                                 uint32_t dwValue
                                                 )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t PH_MEMLOC_REM dwTmp;

    switch (wConfig)
    {
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_COUPLER_NUMBER:
            if ( dwValue > CONTACT_MICROPROSS_MP_COUPLER_3 )
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Check if this coupler exists */
            statusTmp = phToolsMicropross_GetResourceID(pDataParams, (uint8_t)dwValue, &dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->bCouplerNumber = (uint8_t)dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_VCC:
            statusTmp = phhalHwContact_Micropross_Int_SetVoltageClass(pDataParams, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
            statusTmp = phhalHwContact_Micropross_Int_SetClkFrequency(pDataParams, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_I2C_RX_POLLING_TIMEOUT_MS:
            pDataParams->dwI2CRxPollingTimeoutMs = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
            if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
                (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_LRC &&
                (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_CRC)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
                (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_LRC &&
                (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_CRC)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_POWER_OFF_ON_ATR_ERROR:
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_POWER_OFF_ON_ATR_ERROR, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_FD:
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_CURRENT_F_D, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PARITY_ERROR_SIGNAL:
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ERROR_SIGNAL, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
        case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));

            /* Micropross does not count the first 10 ETUs */
            if(dwValue >= (dwTmp * 10))
            {
                dwTmp = dwValue - dwTmp * 10;
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            statusTmp = phhalHwContact_Micropross_Int_SetGuardTimeClk(pDataParams, 0, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->dwGuardTimeClk = dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            dwTmp = (uint32_t)(dwValue / dwTmp + 0.5);
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_BWT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_BGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            dwTmp = (uint32_t)(dwValue / dwTmp + 0.5);
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_BGT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            dwTmp = (uint32_t)(dwValue / dwTmp + 0.5);
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_CWT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_EGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            dwTmp = (uint32_t)(dwValue / dwTmp + 0.5);
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_EGT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_RGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            dwTmp = (uint32_t)(dwValue / dwTmp + 0.5);
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_RGT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            dwTmp = (uint32_t)(dwValue / dwTmp + 0.5);

            /* If WT is set we also set CWT and BWT as those timings are used in the CPSR (send recieve) command */
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_WWT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_BWT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_CWT, dwTmp);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_IFSC:
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_IFSC, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_CONVENTION:
            if(dwValue != PHHAL_HW_CONTACT_DIRECT_CONVENTION && dwValue != PHHAL_HW_CONTACT_INVERSE_CONVENTION)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            phhalHwContact_Micropross_Int_SetConvention(pDataParams, dwValue);
            /* Write config data into shadow */
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_TIMING_MODE:
            /* No timing mode implemented for now */
            if(dwValue != PHHAL_HW_TIMING_MODE_OFF)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wTimingMode = (uint16_t)dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_VCC_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_VCC_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_RISE_TIME:
            statusTmp = phhalHwContact_Micropross_Int_SetFallRiseTime(pDataParams, wConfig, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_HIGH:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_HIGH:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_HIGH:
            statusTmp = phhalHwContact_Micropross_Int_SetOutVoltLowHigh(pDataParams, wConfig, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_DUTY_CYCLE:
            statusTmp = phhalHwContact_Micropross_Int_SetDutyCycle(pDataParams, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_HIGH:
            statusTmp = phhalHwContact_Micropross_Int_SetInVoltLowHigh(pDataParams, wConfig, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_PULL_UP:
            statusTmp = phhalHwContact_Micropross_Int_SetPullUpResistor(pDataParams, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wCfgShadow[wConfig] = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
            statusTmp = phhalHwContact_Micropross_Int_SetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_MAX_CLK_ATR, dwValue);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
            pDataParams->dwNumExpectedBytes = dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL:
            if(dwValue == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD && (uint8_t)dwValue != pDataParams->bCommunicationChannel)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_DeactivateI2C(pDataParams));
            }
            else if(dwValue == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C && (uint8_t)dwValue != pDataParams->bCommunicationChannel)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_ActivateI2C(pDataParams));
            }
            else if(dwValue != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD && dwValue != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            pDataParams->bCommunicationChannel = (uint8_t)dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_I2C_SLAVE_ADDRESS:
            if(dwValue < 0 || dwValue > 127)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->wSlaveAddress = (uint16_t)dwValue;
            break;

        case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
            /* Boundary check */
            if (dwValue > 0xFFFF || dwValue >= pDataParams->wRxBufSize)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }
            /* Set start position */
            pDataParams->wRxBufStartPos = (uint16_t)dwValue;
            pDataParams->wRxBufLen = (uint16_t)dwValue;
            break;

        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_ADDRESS_WIDTH:
            if (dwValue != CONTACT_MICROPROSS_I2C_CP_ADDRESS_7BIT_WIDTH && dwValue != CONTACT_MICROPROSS_I2C_CP_ADDRESS_10BIT_WIDTH)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_ADDRESS_WIDTH, dwValue, 0);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_RX_TIMEOUT:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TX_TIMEOUT:
            if (dwValue > CONTACT_MICROPROSS_I2C_CP_MAX_TIMEOUT_MS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_RX_TIMEOUT)
            {
                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_RX_TIMEOUT, dwValue, 0);
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TX_TIMEOUT)
            {
                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_TX_TIMEOUT, dwValue, 0);
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_DAT:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THD_DAT:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_STA:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THD_STA:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_STO:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THIGH_SCL:
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TLOW_SCL:
            if (dwValue > CONTACT_MICROPROSS_I2C_CP_MAX_TIMEING_NS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            if (dwValue % CONTACT_MICROPROSS_I2C_CP_TIMEING_RESOLUTION_NS != 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_DAT)
            {
                /* check if clk low time in correct range */
                if((pDataParams->dwTLowScl - pDataParams->dwTHdDat) < dwValue)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
                }

                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_TSU_DAT, dwValue, 4 /*TOU_NANO*/);

                if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
                {
                    pDataParams->dwTSuDat = dwValue;
                }
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THD_DAT)
            {
                /* check if clk low time in correct range */
                if((pDataParams->dwTLowScl - pDataParams->dwTSuDat) < dwValue)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
                }

                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_THD_DAT, dwValue, 4 /*TOU_NANO*/);

                if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
                {
                    pDataParams->dwTHdDat = dwValue;
                }
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_STA)
            {
                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_TSU_STA, dwValue, 4 /*TOU_NANO*/);
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THD_STA)
            {
                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_THD_STA, dwValue, 4 /*TOU_NANO*/);
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_STO)
            {
                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_TSU_STO, dwValue, 4 /*TOU_NANO*/);
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THIGH_SCL)
            {
                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_THIGH_SCL, dwValue, 4 /*TOU_NANO*/);
                if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
                {
                    pDataParams->dwTHighScl = dwValue;
                }
            }
            else if (wConfig == PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TLOW_SCL)
            {
                /* check if sda setup and hold time are in correct range */
                if((pDataParams->dwTSuDat + pDataParams->dwTHdDat) > dwValue)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
                }

                statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_TLOW_SCL, dwValue, 4 /*TOU_NANO*/);
                if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
                {
                    pDataParams->dwTLowScl = dwValue;
                }
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_AUTO_NACK:
            if (dwValue != PH_OFF && dwValue != PH_ON)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            statusTmp = phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_AUTO_NACK, dwValue, 0);
            if ((statusTmp) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_CONTACT_CONFIG_I2C_BITRATE_KBITSEC:
            /* always in nanoseconds and with 20ns resolution */
            dwTmp = (uint32_t)(1.0 / dwValue * 1000000 / 2.0);
            dwTmp = ((dwTmp % 20) <= 10) ? dwTmp - (dwTmp % 20) : dwTmp + (20 - (dwTmp % 20));

            /* first set clk low in order to check if sda setup and hold time for data are in a correct range */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_SetConfig32(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TLOW_SCL, dwTmp));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_SetConfig32(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THIGH_SCL, dwTmp));
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_GetConfig32(
                                                 phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                 uint16_t wConfig,
                                                 uint32_t * pValue
                                                 )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint32_t   PH_MEMLOC_REM dwTmp;

    switch (wConfig)
    {
        case PHHAL_HW_CONTACT_MICROPROSS_CONFIG_COUPLER_NUMBER:
            /* Return parameter */
            *pValue = (uint16_t)pDataParams->bCouplerNumber;
            break;

        case PHHAL_HW_CONTACT_CONFIG_I2C_RX_POLLING_TIMEOUT_MS:
            /* Return parameter */
            *pValue = pDataParams->dwI2CRxPollingTimeoutMs;
            break;

        case PHHAL_HW_CONTACT_CONFIG_VCC:
        case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
        case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
        case PHHAL_HW_CONTACT_CONFIG_PARITY_ERROR_SIGNAL:
        case PHHAL_HW_CONTACT_CONFIG_FD:
        case PHHAL_HW_CONTACT_CONFIG_PIN_VCC_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_VCC_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_FALL_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_RISE_TIME:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_HIGH:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_HIGH:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_HIGH:
        case PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_DUTY_CYCLE:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_LOW:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_HIGH:
        case PHHAL_HW_CONTACT_CONFIG_PIN_IO_PULL_UP:
        case PHHAL_HW_CONTACT_CONFIG_CONVENTION:
        case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
            /* Read config from shadow */
            *pValue = pDataParams->wCfgShadow[wConfig];
            break;

        case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_BWT, pValue));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            *pValue = *pValue * dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_BGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_BGT, pValue));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            *pValue = *pValue * dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_CWT, pValue));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            *pValue = *pValue * dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_EGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_EGT, pValue));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            *pValue = *pValue * dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_RGT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_RGT, pValue));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            *pValue = *pValue * dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_WWT, pValue));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, &dwTmp));
            *pValue = *pValue * dwTmp;
            break;

        case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_MAX_CLK_ATR, pValue));
            break;

        case PHHAL_HW_CONTACT_CONFIG_IFSC:
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_IFSC, pValue));
            break;

        case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
        case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
            *pValue = pDataParams->dwGuardTimeClk;
            break;

        case PHHAL_HW_CONTACT_CONFIG_TIMING_MODE:
            *pValue = pDataParams->wTimingMode;
            break;

        case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
            *pValue = pDataParams->dwNumExpectedBytes;
            break;

        case PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL:
            *pValue = (uint32_t)pDataParams->bCommunicationChannel;
            break;

        case PHHAL_HW_CONTACT_CONFIG_I2C_SLAVE_ADDRESS:
            *pValue = (uint32_t)pDataParams->wSlaveAddress;
            break;

        case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
            *pValue = (uint32_t)pDataParams->wRxBufStartPos;
            break;

        case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_BUFSIZE:
            *pValue = pDataParams->wRxBufSize - PHHAL_HW_CONTACT_MAX_CRC_BUFFER_LEN;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Wait(
                        phhalHwContact_Micropross_DataParams_t * pDataParams,
                        uint8_t bUnit,
                        uint16_t wTimeout
                        )
{
    if (bUnit == PHHAL_HW_CONTACT_TIME_MICROSECONDS)
    {
        return phhalHwContact_Micropross_Int_Sleep(pDataParams, wTimeout);
    }

    if (bUnit == PHHAL_HW_CONTACT_TIME_MILLISECONDS)
    {
        return phhalHwContact_Micropross_Int_Sleep(pDataParams, wTimeout*1000);
    }

    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Activate(
                        phhalHwContact_Micropross_DataParams_t * pDataParams,
                        uint8_t * pATRBuffer,
                        uint16_t * pwATRLength
                        )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint16_t   PH_MEMLOC_REM wDataSize;
    uint8_t    PH_MEMLOC_REM bColdReset;
    uint32_t   PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C communication */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    if(pDataParams->dwAtrLength == 0)
    {
        bColdReset = 1;
    }
    else
    {
        bColdReset = 0;
    }

    /* Reset ATR before doing anything else */
    memset(pDataParams->pAtrBuffer, 0x00, PHHAL_HW_CONTACT_MAX_ATR_SIZE);
    pDataParams->dwAtrLength = 0;

    /* Before activation bring reader to default state */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    if(bColdReset)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_ISOF, pDataParams->bCouplerNumber));
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 0", CONTACT_MICROPROSS_CMD_ISON, pDataParams->bCouplerNumber));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 0", CONTACT_MICROPROSS_CMD_ISON, pDataParams->bCouplerNumber));
    }

    /* get number of received bits */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_Uint16(pDataParams, &wDataSize));

    if(wDataSize > PHHAL_HW_CONTACT_MAX_ATR_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    if(wDataSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    /* Extract the received data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_HexString(pDataParams, pATRBuffer, wDataSize, pwATRLength));

    /* Copy ATR to internal atr buffer */
    memcpy(&pDataParams->pAtrBuffer[0], pATRBuffer, wDataSize);
    pDataParams->dwAtrLength = wDataSize;

    /* Set ATR in phToolsAtrParser to parse certain values later */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(&pDataParams->pAtrBuffer[0], pDataParams->dwAtrLength));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    if(wDataSize != *pwATRLength)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Deactivate(
                        phhalHwContact_Micropross_DataParams_t * pDataParams
                        )
{
    phStatus_t statusTmp;
    uint32_t   PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C communication */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    pDataParams->dwAtrLength = 0;
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_ISOF, pDataParams->bCouplerNumber);
}

phStatus_t phhalHwContact_Micropross_ClockStop(
                        phhalHwContact_Micropross_DataParams_t * pDataParams,
                        uint16_t wTgClockCount,
                        uint16_t wThClockCount,
                        uint32_t dwPinState
                        )
{
    phStatus_t statusTmp;
    uint32_t   PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C communication */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u %u",
        CONTACT_MICROPROSS_CMD_CLK_STOP, pDataParams->bCouplerNumber, wTgClockCount, wThClockCount, dwPinState);
}

phStatus_t phhalHwContact_Micropross_InitContactReader(
                                                       phhalHwContact_Micropross_DataParams_t * pDataParams
                                                       )
{
    phStatus_t statusTmp;
    phStatus_t status;
    uint32_t coupler;

    if(pDataParams->bCouplerNumber == PHHAL_HW_CONTACT_MICROPROSS_COUPER_NUMBER_DEFAULT)
    {
        /* Find the correct coupler */
        for (coupler = 0; coupler < 4; coupler++)
        {
            status = phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CONFIG_COUPLER_NUMBER, coupler);
            if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                break;
            }
            if(coupler == CONTACT_MICROPROSS_MP_COUPLER_3)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
            }
        }
    }

    /* If resource not is open open it */
    if(pDataParams->wCouplerResourceId == NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_OpenResource(pDataParams));
    }

    /* Reset Reader */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_ResetHard(pDataParams));

    /* Reader Reset resets also I2C parameter */
    pDataParams->bCommunicationChannel  = PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD;
    pDataParams->dwTHighScl             = 4000;
    pDataParams->dwTLowScl              = 4700;

    /* Set internal default values for communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());

    /* Set default Datarate */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));

    /* Set default Max Clk Atr timing */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR, PHHAL_HW_CONTACT_MAX_CLK_ATR_DEFAULT));

    /* ApplyProtocolSettings to have default values applied */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    /* Deactivate set power and ATR length to zero */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Deactivate(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS */
