/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Micropross specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHwContact.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS

#include "phhalHwContact_Micropross_Int.h"
#include <phhalHwContact_Micropross_Cmd.h>
#include <phToolsMicropross.h>
#include <stdio.h>              /* PRQA S 5124 */
#include <time.h>

phStatus_t phhalHwContact_Micropross_Cmd_EnableSpy(
                                                    phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                    uint32_t dwEventMask,
                                                    uint8_t * pFilename,
                                                    uint16_t wDateFormat
                                                    )
{
    FILE * fh;

    /* Copy filename to pDataParams */
    if(pFilename != NULL)
    {
        memcpy(pDataParams->bSpyFilename, pFilename, strlen((const char *)pFilename) + 1);

        /* try to open file */
        fh = fopen((const char*)pDataParams->bSpyFilename, "ab");

        if(fh == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        fclose(fh);
    }

    return phhalHwContact_Micropross_Int_EnableSpy(pDataParams, dwEventMask, wDateFormat);
}

phStatus_t phhalHwContact_Micropross_Cmd_DisableSpy(
                                                    phhalHwContact_Micropross_DataParams_t * pDataParams
                                                    )
{
    phStatus_t  PH_MEMLOC_REM status;
    PH_CHECK_SUCCESS_FCT(status, phhalHwContact_Micropross_Int_DisableSpy(pDataParams));

    /* Reset filename */
    memset(pDataParams->bSpyFilename, 0x00, sizeof(pDataParams->bSpyFilename));
    return status;
}

phStatus_t phhalHwContact_Micropross_Cmd_SimAtrString(
                                                    phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                    uint8_t * pAtrString
                                                    )
{
    return phhalHwContact_Micropross_Int_SimAtrString(pDataParams, pAtrString);
}

phStatus_t phhalHwContact_Micropross_Cmd_OpenScenario(
    phhalHwContact_Micropross_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    /* If there is an open sequence close it */
    if (pDataParams->dwScenarioID != 0)
    {
        phhalHwContact_Micropross_Cmd_CloseScenario(pDataParams);
    }

    /* Open Scenario */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_SEQUENCER_OPEN, pDataParams->bCouplerNumber));

    /* Extract the received data */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32Hex(pDataParams, &pDataParams->dwScenarioID));
    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    return status;
}

phStatus_t phhalHwContact_Micropross_Cmd_ExecuteScenario(
    phhalHwContact_Micropross_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwTimeout                                      /**< [In] timeout of command in seconds. 0 direct return, 0xFFFFFFFF infinite timeout */
    )
{
    phStatus_t PH_MEMLOC_REM status;
    phStatus_t PH_MEMLOC_REM statusTimeout;
    uint16_t   PH_MEMLOC_REM wOldBalTimoutMs; /* Store the old Timeout Value */
    uint32_t   PH_MEMLOC_REM dwNewBalTimeoutMs;

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Get the old BAL timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));

    /* calculate the new BAL timeout value */
    dwNewBalTimeoutMs = (dwTimeout * 1000) + (uint32_t)PHHAL_HW_CONTACT_MICROPROSS_COM_TIMEOUT_OFFSET_MS;

    /* Check if there would be an overflow */
    if (dwNewBalTimeoutMs > 0xFFFF)
    {
        dwNewBalTimeoutMs = 0xFFFF;
    }

    /* Set the new (temporary) timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

    /* Execute Scenario */
    status = phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %x %u",
        CONTACT_MICROPROSS_CMD_SEQUENCER_EXECUTE, pDataParams->bCouplerNumber, pDataParams->dwScenarioID, dwTimeout);

    /* Restore old timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

    return status;
}

phStatus_t phhalHwContact_Micropross_Cmd_CloseScenario(
    phhalHwContact_Micropross_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Close Scenario */
    status = phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %x",
        CONTACT_MICROPROSS_CMD_SEQUENCER_CLOSE, pDataParams->bCouplerNumber, pDataParams->dwScenarioID);

    pDataParams->dwScenarioID = 0;

    return status;
}

phStatus_t phhalHwContact_Micropross_Cmd_AddToScenario(
    phhalHwContact_Micropross_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t * pAction,                                       /**< [In] string with the action. */
    uint16_t wActionLength,                                  /**< [In] Length of the action string. */
    uint8_t * pParameter,                                    /**< [In] string with parameter for the action. */
    uint16_t wParameterLength                                /**< [In] Length of the parameter string. */
    )
{
    if (pAction == NULL && wActionLength != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if (pParameter == NULL && wParameterLength != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if (pDataParams->dwScenarioID == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Add */
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %x %s %s",
        CONTACT_MICROPROSS_CMD_SEQUENCER_ADD, pDataParams->bCouplerNumber, pDataParams->dwScenarioID, pAction, pParameter);
}

phStatus_t phhalHwContact_Micropross_Cmd_GetRxBuffer(
    phhalHwContact_Micropross_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t ** ppRxBuffer,                                   /**< [Out] Pointer to received data. */
    uint16_t * pRxLength                                     /**< [Out] Number of received data bytes. */
    )
{
    phStatus_t statusTmp;
    phStatus_t statusExchange;
    uint16_t   wNumReceivedBytes = 0;
    uint8_t *  pTmpBuffer;
    uint16_t   wTmpBufferLen;
    uint16_t   wTmpBufferSize;
    int32_t    nbytes;
    uint32_t   dwI2CReceiveTimeout;
    clock_t    start;
    clock_t    end;


    switch(pDataParams->bCommunicationChannel)
    {
    case PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD:
        statusExchange = phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 0 0 %u",
            CONTACT_MICROPROSS_CMD_EXCHANGE, pDataParams->bCouplerNumber, pDataParams->dwNumExpectedBytes);
        /* get number of received bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_Uint16(pDataParams, &wNumReceivedBytes));
        break;

    case PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C:
        /* Check length of read command */
        pDataParams->wIntBufferLen = 0;
        nbytes = _snprintf(NULL, 0, "%s %u %X %X", CONTACT_MICROPROSS_CMD_I2C_READ, pDataParams->bCouplerNumber, pDataParams->wSlaveAddress, pDataParams->dwNumExpectedBytes) + 1;
        if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Adjust timeout for I2C read to 5ms + length */
        /* FIXXME: 5ms is only a workaround for a MP issue with long Start Hold time */
        dwI2CReceiveTimeout = 5 + (((pDataParams->dwTLowScl + pDataParams->dwTHighScl) * (pDataParams->dwNumExpectedBytes + 1) * 9) + 999999)/1000000;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_I2CChangeParameter(pDataParams, CONTACT_MICROPROSS_CMD_I2C_PARAMETER_I2C_CP_RX_TIMEOUT, dwI2CReceiveTimeout, 0));

        start = clock();
        do
        {
            end = clock();

             /* Inital delay */
            phTools_Sleep(1);

            pDataParams->wIntBufferLen = 0;
            pDataParams->wIntBufferLen += (uint16_t)_snprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
                pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, "%s %u %X %X",
                CONTACT_MICROPROSS_CMD_I2C_READ, pDataParams->bCouplerNumber, pDataParams->wSlaveAddress, pDataParams->dwNumExpectedBytes);

            PH_CHECK_SUCCESS_FCT(statusExchange, phToolsMicropross_SendCommand(pDataParams));
            /* get the return status */
            statusExchange = phToolsMicropross_GetStatus(pDataParams, CONTACT_MICROPROSS_CMD_I2C_READ);
        } while((1000 * ((double)end - start)) / CLOCKS_PER_SEC <= pDataParams->dwI2CRxPollingTimeoutMs && (statusExchange & PH_ERR_MASK) != PH_ERR_SUCCESS);

        /* get number of received bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_Uint16Hex(pDataParams, &wNumReceivedBytes));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    if ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS || (statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        if(wNumReceivedBytes > 0)
        {
            /* Handle as it was success */
            statusExchange = (PH_ERR_SUCCESS & PH_COMP_MASK);

            /* Reset receive buffer length */
            pDataParams->wRxBufLen = 0;

            /* Retrieve receive buffer properties */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetRxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

            /* Extract the received data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_GetParam_HexString(pDataParams, pTmpBuffer, wTmpBufferSize, &wTmpBufferLen));

            /* Return RxBuffer pointer */
            if (ppRxBuffer != NULL)
            {
                *ppRxBuffer = pTmpBuffer;
            }

            /* Return RxBuffer length */
            if (pRxLength != NULL)
            {
                *pRxLength = wTmpBufferLen;
            }
        }
    }

    return PH_ADD_COMPCODE(statusExchange, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Cmd_SetBalTimeoutHandler(
    phhalHwContact_Micropross_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    cbBalTimeoutHandling pBalTimeoutHandling,                /**< [In] Pointer to the handler function. */
    void * pContext                                          /**< [In] Pointer to the context of the callback function */
)
{
    pDataParams->pBalTimeoutHandling = pBalTimeoutHandling;
    pDataParams->pBalTimeoutHandlingContext = pContext;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Cmd_CouplerCheckLicense(
    phhalHwContact_Micropross_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
    uint32_t dwLicenseId,                                    /**< [In] Specifies the license. */
    int32_t *pValidLicense                                   /**< [Out] license status. */
)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_ASSERT_NULL(pDataParams);

    if (pValidLicense == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* MPS_CouplerCheckLicense */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_COUPLER_CHECK_LICENSE, pDataParams->bCouplerNumber, dwLicenseId));

    /* Extract the received data */
    return phToolsMicropross_GetParam_Int32(pDataParams, pValidLicense);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS */
