/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MP300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS

#include "phhalHwContact_Micropross.h"
#include "phhalHwContact_Micropross_Int.h"
#include <phToolsAtrParser.h>
#include <phTools.h>
#include <phToolsMicropross.h>
#include <stdio.h>              /* PRQA S 5124 */
#include <WinSock2.h>

#include "../../../phbalReg/src/Mp300Usb/external/lusb0_usb.h"

phStatus_t phhalHwContact_Micropross_Int_Sleep(
                                            phhalHwContact_Micropross_DataParams_t * pDataParams,
                                            uint32_t dwTime
                                            )
{
    phStatus_t PH_MEMLOC_REM status;
    phStatus_t PH_MEMLOC_REM statusTimeout;
    uint16_t   PH_MEMLOC_REM wOldBalTimoutMs; /* Store the old Timeout Value */
    uint32_t   PH_MEMLOC_REM dwNewBalTimeoutMs;

    /* Get the old BAL timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldBalTimoutMs));

    /* calculate the new BAL timeout value */
    dwNewBalTimeoutMs = (dwTime / 1000) + (uint32_t)PHHAL_HW_CONTACT_MICROPROSS_COM_TIMEOUT_OFFSET_MS;

    /* Check if there would be an overflow */
    if (dwNewBalTimeoutMs > 0xFFFF)
    {
        dwNewBalTimeoutMs = 0xFFFF;
    }
    /* Set the new (temporary) timeout value */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, (uint16_t)dwNewBalTimeoutMs));

    status = phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %lu\n", CONTACT_MICROPROSS_CMD_DO_TEMPO, dwTime);

    /* Restore old timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldBalTimoutMs));

    return status;
}

phStatus_t phhalHwContact_Micropross_Int_SetClkFrequency(
                                                      phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                      uint32_t dwClkFrequency
                                                      )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_CLK_FREQUENCY, pDataParams->bCouplerNumber, dwClkFrequency);
}

phStatus_t phhalHwContact_Micropross_Int_SetVoltageClass(
                                                      phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                      uint32_t dwVoltage
                                                      )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_VOLTAGE_CLASS, pDataParams->bCouplerNumber, dwVoltage);
}

phStatus_t phhalHwContact_Micropross_Int_I2CChangeParameter(
                                                          phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                          uint32_t dwConfig,
                                                          uint32_t dwParam1,
                                                          uint32_t dwParam2
                                                          )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %d %d %d",
        CONTACT_MICROPROSS_CMD_I2C_CHANGE_PARAMETERS, pDataParams->bCouplerNumber, dwConfig, dwParam1, dwParam2);
}

phStatus_t phhalHwContact_Micropross_Int_SetProtocolParameter(
                                                          phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                          uint32_t dwConfig,
                                                          uint32_t dwValue
                                                          )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u 0 %u",
        CONTACT_MICROPROSS_CMD_PROTOCOL_PARAM, pDataParams->bCouplerNumber, dwConfig, dwValue);
}

phStatus_t phhalHwContact_Micropross_Int_GetProtocolParameter(
                                                          phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                          uint32_t dwConfig,
                                                          uint32_t * pValue
                                                          )
{
    phStatus_t PH_MEMLOC_REM status;

    /* We need to get inital etu width if we are somewhere in the activation process */
    if(dwConfig == PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH && pDataParams->dwAtrLength == 0)
    {
        dwConfig = PHHAL_HW_CONTACT_MICROPROSS_CPP_INITIAL_ETU_WIDTH;
    }

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_PROTOCOL_PARAM_GET, pDataParams->bCouplerNumber, dwConfig));

    /* Extract the received data */
    return phToolsMicropross_GetParam_Uint32(pDataParams, pValue);
}

phStatus_t phhalHwContact_Micropross_Int_SetPullUpResistor(phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                       uint32_t dwValue
                                                       )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u",
        CONTACT_MICROPROSS_CMD_PULLUPRES, pDataParams->bCouplerNumber, PHHAL_HW_PIN_C7_IO, dwValue);
}

phStatus_t phhalHwContact_Micropross_Int_GetIccResponseTimeNs(phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                            uint32_t * pResponseTime
                                                            )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t pNumberOfTimings = 0;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_GET_RESPONSE_TIME, pDataParams->bCouplerNumber));

    /* First we get the Number of Timings */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32(pDataParams, &pNumberOfTimings));

    if(pNumberOfTimings == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* Extract the first timing as this should be the timing of the last command sent */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32(pDataParams, pResponseTime));
    *pResponseTime = (uint32_t)(*pResponseTime * (1.0 / pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ] * 1000000000));

    return status;
}

phStatus_t phhalHwContact_Micropross_Int_SetFallRiseTime(phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                     uint16_t wConfig,
                                                     uint32_t dwValue
                                                     )
{
    uint8_t bPinNum = 0;
    uint32_t dwRiseTime = 0;
    uint32_t dwFallTime = 0;

    /* Get Pin Number encoded in wConfig */
    bPinNum = (((wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_MASK) >> 4) - PHHAL_HW_CONTACT_CONFIG_PIN_NUMBER_OFFSET);

    /* Check if dwValue is Fall or Rise Time */
    switch(wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_CONFIG_MASK)
    {
        case PHHAL_HW_CONTACT_CONFIG_PIN_FALL_TIME:
            dwFallTime = dwValue;
            dwRiseTime = pDataParams->wCfgShadow[((wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_MASK) | PHHAL_HW_CONTACT_CONFIG_PIN_RISE_TIME)];
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_RISE_TIME:
            dwRiseTime = dwValue;
            dwFallTime = pDataParams->wCfgShadow[((wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_MASK) | PHHAL_HW_CONTACT_CONFIG_PIN_FALL_TIME)];
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Set Fall or Rise Time and get other Value from shadow */
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u %u",
        CONTACT_MICROPROSS_CMD_SETFALLRISETIME, pDataParams->bCouplerNumber, bPinNum, dwRiseTime, dwFallTime);
}

phStatus_t phhalHwContact_Micropross_Int_SetOutVoltLowHigh(phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                       uint16_t wConfig,
                                                       uint32_t dwValue
                                                       )
{
    uint8_t bPinNum = 0;
    uint32_t dwLow = 0;
    uint32_t dwHigh = 0;

    /* Get Pin Number encoded in wConfig */
    bPinNum = (((wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_MASK) >> 4) - PHHAL_HW_CONTACT_CONFIG_PIN_NUMBER_OFFSET);

    /* Check if dwValue is Fall or Rise Time */
    switch(wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_CONFIG_MASK)
    {
        case PHHAL_HW_CONTACT_CONFIG_PIN_OUTPUT_LOW:
            dwLow = dwValue;
            dwHigh = pDataParams->wCfgShadow[((wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_MASK) | PHHAL_HW_CONTACT_CONFIG_PIN_OUTPUT_HIGH)];
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_OUTPUT_HIGH:
            dwHigh = dwValue;
            dwLow = pDataParams->wCfgShadow[((wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_MASK) | PHHAL_HW_CONTACT_CONFIG_PIN_OUTPUT_LOW)];
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Set Fall or Rise Time and get other Value from shadow */
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u %u",
        CONTACT_MICROPROSS_CMD_SETVOLVOH, pDataParams->bCouplerNumber, bPinNum, dwLow, dwHigh);
}

phStatus_t phhalHwContact_Micropross_Int_SetDutyCycle(phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                   uint32_t dwValue
                                                   )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_SETDUTYCYCLE, pDataParams->bCouplerNumber, dwValue);
}

phStatus_t phhalHwContact_Micropross_Int_SetInVoltLowHigh(phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                   uint16_t wConfig,
                                                   uint32_t dwValue
                                                   )
{
    uint32_t dwLow = 0;
    uint32_t dwHigh = 0;

    /* Check if dwValue is Fall or Rise Time */
    switch(wConfig & PHHAL_HW_CONTACT_CONFIG_PIN_CONFIG_MASK)
    {
        case PHHAL_HW_CONTACT_CONFIG_PIN_INPUT_LOW:
            dwLow = dwValue;
            dwHigh = pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_HIGH];
            break;

        case PHHAL_HW_CONTACT_CONFIG_PIN_INPUT_HIGH:
            dwHigh = dwValue;
            dwLow = pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_LOW];
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u %u",
        CONTACT_MICROPROSS_CMD_SETVILVIH, pDataParams->bCouplerNumber, PHHAL_HW_PIN_C7_IO, dwLow, dwHigh);
}

phStatus_t phhalHwContact_Micropross_Int_ShadowDefaultPinValues(phhalHwContact_Micropross_DataParams_t * pDataParams)
{
    /* C1 .. Vcc */
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_VCC_FALL_TIME]      = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_VCC_FALL_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_VCC_RISE_TIME]      = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_VCC_RISE_TIME;
    /* C2 .. Reset */
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_RESET_FALL_TIME]    = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_FALL_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_RESET_RISE_TIME]    = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_RISE_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_LOW]   = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_OUTPUT_LOW;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_HIGH]  = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_OUTPUT_HIGH;
    /* C3 .. Clock */
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_FALL_TIME]    = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_FALL_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_RISE_TIME]    = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_RISE_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_LOW]   = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_OUTPUT_LOW;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_HIGH]  = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_OUTPUT_HIGH;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_DUTY_CYCLE]   = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_DUTY_CYCLE;
    /* C7 .. I/O */
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_FALL_TIME]       = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_FALL_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_RISE_TIME]       = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_RISE_TIME;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_LOW]      = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_OUTPUT_LOW;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_HIGH]     = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_OUTPUT_HIGH;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_LOW]       = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_INPUT_LOW;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_HIGH]      = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_INPUT_HIGH;
    pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_PIN_IO_PULL_UP]         = PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_PULL_UP;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Int_EnableSpy(phhalHwContact_Micropross_DataParams_t * pDataParams, uint32_t dwEventMask, uint16_t wDateFormat)
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    phStatus_t PH_MEMLOC_REM status;
    DWORD PH_MEMLOC_REM GThreadId;
    uint8_t EventMask[4];
    uint8_t localIp[30];
    uint32_t networkaddress;
    uint32_t subnetmask;

    if(pDataParams->SpyThreadHandle != NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Get EventMask from DataParams */
    EventMask[0] = (dwEventMask >> 24) & 0xFF;
    EventMask[1] = (dwEventMask >> 16) & 0xFF;
    EventMask[2] = (dwEventMask >> 8) & 0xFF;
    EventMask[3] = dwEventMask & 0xFF;

    /* Open and Log with given Event Mask and start downloading the Trace */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_SPY_OPEN, pDataParams->bCouplerNumber));

    /* Set Dating of events */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_SPY_DATE_FORMAT, pDataParams->bCouplerNumber, wDateFormat)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %02X%02X%02X%02X 00000000",
        CONTACT_MICROPROSS_CMD_OPEN_LOG, pDataParams->bCouplerNumber, EventMask[0], EventMask[1], EventMask[2], EventMask[3])));

    /* Initialize events */
    pDataParams->GEvtReady = CreateEvent (NULL, 0, 0, "Ready");
    pDataParams->GEvtFinished = CreateEvent (NULL, 0, 0, "Finished");

    /* Create Thread to read data from socket */
    pDataParams->SpyThreadHandle = CreateThread (NULL, 0, phhalHwContact_Micropross_Int_ReadSpyDataThread, pDataParams, NULL, &GThreadId);

    /* Set highest priority for thread */
    SetThreadPriority(pDataParams->SpyThreadHandle, THREAD_PRIORITY_HIGHEST);

    /* Now waiting for thread to be ready */
    WaitForSingleObject (pDataParams->GEvtReady, 0xFFFFFFFF);

    if (((phbalReg_Mp300Usb_DataParams_t *)pDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_MP300USB_ID))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 00000001 %s",
            CONTACT_MICROPROSS_CMD_LAUNCH_DOWNLOAD, pDataParams->bCouplerNumber, CONTACT_MICROPROSS_USB_PORT)));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams,
            phhalHwContact_Micropross_Int_GetMpNetworkAddress(pDataParams, &networkaddress, &subnetmask)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_GetLocalIp(networkaddress, subnetmask, localIp)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 00000001 %s:%s",
            CONTACT_MICROPROSS_CMD_LAUNCH_DOWNLOAD, pDataParams->bCouplerNumber, CONTACT_MICROPROSS_TCPIP_PORT, localIp)));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Int_DisableSpy(phhalHwContact_Micropross_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;

    /* Close and Flush Log, End Download and Close Spy */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_CLOSE_LOG, pDataParams->bCouplerNumber)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_FLUSH_LOG, pDataParams->bCouplerNumber)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_END_DOWNLOAD, pDataParams->bCouplerNumber)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_SPY_CLOSE, pDataParams->bCouplerNumber)));

    /* Now waiting for EOF in SpyThread */
    WaitForSingleObject(pDataParams->GEvtFinished, CONTACT_MICROPROSS_THREAD_TIMEOUT);

    if (WaitForSingleObject(pDataParams->SpyThreadHandle, CONTACT_MICROPROSS_THREAD_TIMEOUT) == WAIT_OBJECT_0)
    {
        /* the thread handle is signaled - the thread has terminated */
        pDataParams->SpyThreadHandle = NULL;
    }
    else
    {
        /* the thread handle is not signaled - the thread is still alive so kill it */
        TerminateThread(pDataParams->SpyThreadHandle, 0);
        pDataParams->SpyThreadHandle = NULL;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

unsigned long __stdcall phhalHwContact_Micropross_Int_ReadSpyDataThread (void *voidParams)
{
    phStatus_t statusTmp;
    int32_t dwError;
    int32_t wActRxBufSize = 512;
    uint8_t Buffer[512];
    uint8_t pData[CONTACT_MICROPROSS_SPY_BUFFER_SIZE];
    uint32_t pSpyEvents[CONTACT_MICROPROSS_SPY_BUFFER_SIZE];
    uint32_t bDataOffset = 0, remainingSize = 0;
    uint8_t *pRxBuffer = Buffer;
    uint8_t setRdyFlag = 1;
    SpyHeaderContact gHeader;
    phStatus_t finished = 0;
    phbalReg_Mp300Usb_DataParams_t *pBalUsbDataParams = NULL;
    phbalReg_SocketWin_DataParams_t * pBalWinSocketDataParams;
    uint32_t DataType = 0, Size = 0;
    uint32_t dwETUns = 0;
    uint8_t bDataTypeReceived = 0;
    uint8_t bSizeReceived = 0;
    SOCKET ClientSocket = NULL, ServerSocket = NULL;

    phhalHwContact_Micropross_DataParams_t *pHalDataParams = (phhalHwContact_Micropross_DataParams_t *)voidParams;

    /* Only communication with reader in this thread - main thread is currently waiting so get ETU width if we want to measure FDT */
    if(!strstr((char *)pHalDataParams->bSpyFilename, (const char *)CONTACT_MICROPROSS_TRACEFILE_EXTENSION))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_CalcOneETUns(pHalDataParams, &dwETUns));
    }

    if (((phbalReg_Mp300Usb_DataParams_t *)pHalDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_MP300USB_ID))
    {
        pBalUsbDataParams = (phbalReg_Mp300Usb_DataParams_t*)pHalDataParams->pBalDataParams;
    }
    else
    {
        pBalWinSocketDataParams = (phbalReg_SocketWin_DataParams_t*)pHalDataParams->pBalDataParams;
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsMicropross_CreateServerSocket(pHalDataParams->GEvtReady, ((phbalReg_SocketWin_DataParams_t*)(pHalDataParams->pBalDataParams))->pSocketHostName, &ClientSocket, &ServerSocket));
    }

    memset(&gHeader, 0x00, sizeof(gHeader));
    memset(pSpyEvents, 0x0000, CONTACT_MICROPROSS_SPY_BUFFER_SIZE);

    /* Erase old File */
    if(strstr((char *)pHalDataParams->bSpyFilename, (const char *)CONTACT_MICROPROSS_TRACEFILE_EXTENSION))
    {
        remove((const char *)pHalDataParams->bSpyFilename);
    }

    while(!finished)
    {
        /* Always read 512 Byte */
        if (((phbalReg_Mp300Usb_DataParams_t *)pHalDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_MP300USB_ID))
        {
            dwError = pBalUsbDataParams->pfBulkRead((usb_dev_handle *)pBalUsbDataParams->pUsbDevHandle,
                0x82,
                (char*)pRxBuffer,
                wActRxBufSize,
                5000);

            /* Read once before LGBD to say we are ready */
            if(setRdyFlag == 1)
            {
                /* Thread ready */
                SetEvent (pHalDataParams->GEvtReady);
                setRdyFlag = 0;
            }

            /* We always get Usb Packets of specific size (8 Byte Header or up to 512 Byte Data */
            if(dwError > 0)
            {
                finished = phhalHwContact_Micropross_Int_EvaluateData(pHalDataParams, dwError, (uint32_t *)&gHeader, pSpyEvents, pRxBuffer, pData, &bDataOffset, &remainingSize, &DataType, &Size, &dwETUns, &bDataTypeReceived, &bSizeReceived);
            }
        }
        else
        {
            /* We always get 512 Byte so we have to parse Header and Data */
            dwError = recv(ClientSocket, (char *)pRxBuffer, wActRxBufSize, 0);

            finished = phhalHwContact_Micropross_Int_EvaluateData(pHalDataParams, dwError, (uint32_t *)&gHeader, pSpyEvents, pRxBuffer, pData, &bDataOffset, &remainingSize, &DataType, &Size, &dwETUns, &bDataTypeReceived, &bSizeReceived);
        }
    }

    if (((phbalReg_SocketWin_DataParams_t *)pHalDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_SOCKETWIN_ID))
    {
        closesocket(ServerSocket);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Int_ProcessData(
                                                     phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                     uint32_t DataType,
                                                     uint32_t Size,
                                                     uint8_t *pData,
                                                     uint32_t *pSpyEvents,
                                                     uint32_t *gHeader,
                                                     uint8_t *pbFinished
                                                     )
{
    uint32_t counter, i;
    uint32_t eventsNumber;
    uint32_t *pEvt;
    SpyHeaderContact *spyHeader = (SpyHeaderContact *)gHeader;

    *pbFinished = 0;

    switch (DataType)
    {
    case 1: /* EOF */
        *pbFinished = 1;
        break;

    case 2: /* Header */
        memcpy(gHeader, pData, Size);

        if (spyHeader->DataFormat == 0x4D4D)
        {
            phToolsMicropross_SwapHeaderContact(spyHeader);
            memcpy(pData, gHeader, Size);
        }

        if(strstr((char *)pDataParams->bSpyFilename, (const char *)CONTACT_MICROPROSS_TRACEFILE_EXTENSION))
        {
            FILE * fh = fopen((const char*)pDataParams->bSpyFilename, "ab");

            if(fh == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
            }

            for(counter = 0; counter < Size; counter++)
            {
                fprintf(fh, "%c", pData[counter]);
            }

            fclose(fh);
        }
        break;

    case 3: /* Events */
        {
            eventsNumber = Size / sizeof(uint32_t);
            pEvt = (uint32_t *)pData;

            if (spyHeader->DataFormat == 0x4D4D)
            {
                phToolsMicropross_SwapEvents32((uint32_t *)pData, eventsNumber);
            }

            if(strstr((char *)pDataParams->bSpyFilename, (const char *)CONTACT_MICROPROSS_TRACEFILE_EXTENSION))
            {
                FILE * fh = fopen((const char*)pDataParams->bSpyFilename,"ab");

                if(fh == NULL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
                }

                for(counter = 0; counter < Size; counter++)
                {
                    fprintf(fh, "%c", pData[counter]);
                }

                fclose(fh);
            }
            else
            {
                for (i = 0; i < eventsNumber; i++)
                {
                    /* Check overflow */
                    if((spyHeader->NumberOfEvents + eventsNumber) <= CONTACT_MICROPROSS_SPY_BUFFER_SIZE)
                    {
                        /* Save everything into Buffer to calc FDT afterwards */
                        memcpy(&pSpyEvents[spyHeader->NumberOfEvents + i], &pEvt[i], sizeof(uint32_t));
                    }
                }
            }

            /* Add number of events to header */
            spyHeader->NumberOfEvents += eventsNumber;
        }
        break;

    case 7: /* Burst */
        {
            uint32_t burstNumber = Size / sizeof(uint16_t);
            uint16_t *pBurst = (uint16_t *)pData;

            if (spyHeader->DataFormat == 0x4D4D)
            {
                phToolsMicropross_SwapEvents16(pBurst, burstNumber);
            }

            for (i = 0; i < burstNumber; i++)
            {
                /* printf("\t\tBurst 0x%04X\n", pBurst[i]); */
            }
        }
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Micropross_Int_CalcOneETUns(phhalHwContact_Micropross_DataParams_t * pDataParams, uint32_t * pValue)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_CHECK_SUCCESS_FCT(status, phhalHwContact_Micropross_Int_GetProtocolParameter(pDataParams, PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH, pValue));
    *pValue = (uint32_t)(*pValue * (1000000000.0 / pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ]));
    return status;
}

uint16_t phhalHwContact_Micropross_Int_EvaluateData(
                                                    phhalHwContact_Micropross_DataParams_t *pDataParams,
                                                    int32_t dwReceivedBytes,
                                                    uint32_t *gHeader,
                                                    uint32_t *pSpyEvents,
                                                    uint8_t *pRxBuffer,
                                                    uint8_t *pData,
                                                    uint32_t *dwDataOffset,
                                                    uint32_t *remainingSize,
                                                    uint32_t *DataType,
                                                    uint32_t *Size,
                                                    uint32_t *dwETUns,
                                                    uint8_t *bDataTypeReceived,
                                                    uint8_t *bSizeReceived
                                                    )
{
    phStatus_t statusTmp;
    uint8_t finished = 0;

    if((uint32_t)dwReceivedBytes == 0)
    {
        return finished;
    }

    /* Parse Data */
    if(*bDataTypeReceived && *bSizeReceived)
    {
        /* We got all the data for current DataType */
        if(*remainingSize <= (uint32_t)dwReceivedBytes)
        {
            memcpy(pData + *dwDataOffset, pRxBuffer, *remainingSize);

            /* 4 Bytes DataType - 4 Bytes Size of Data - Actual Data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Micropross_Int_ProcessData(pDataParams, *DataType, *Size, pData, pSpyEvents, gHeader, &finished));

            /* All bytes received */
            *bDataTypeReceived = 0;
            *bSizeReceived = 0;
            *dwDataOffset = 0;

            dwReceivedBytes -= *remainingSize;

            /* Still bytes in buffer */
            if(dwReceivedBytes)
            {
                phhalHwContact_Micropross_Int_EvaluateData(pDataParams, dwReceivedBytes, gHeader, pSpyEvents, pRxBuffer + *remainingSize,
                    pData, dwDataOffset, remainingSize, DataType, Size, dwETUns, bDataTypeReceived, bSizeReceived);
            }
        }
        else
        {
            memcpy(pData + *dwDataOffset, pRxBuffer, dwReceivedBytes);
            *remainingSize -= dwReceivedBytes;
            *dwDataOffset += dwReceivedBytes;
        }
    }
    else if(*bDataTypeReceived)
    {
        if(dwReceivedBytes >= 4)
        {
            *Size = (pRxBuffer[0] << 24) | (pRxBuffer[1] << 16) | (pRxBuffer[2] << 8) | pRxBuffer[3];
            *remainingSize = *Size;
            *bSizeReceived = 1;
            dwReceivedBytes -= 4;
        }

        /* Receive remaining bytes */
        phhalHwContact_Micropross_Int_EvaluateData(pDataParams, dwReceivedBytes, gHeader, pSpyEvents, pRxBuffer + *bSizeReceived * 4,
            pData, dwDataOffset, remainingSize, DataType, Size, dwETUns, bDataTypeReceived, bSizeReceived);
    }
    else
    {
        /* Parse Data Type */
        if(dwReceivedBytes >= 4)
        {
            *DataType = (pRxBuffer[0] << 24) | (pRxBuffer[1] << 16) | (pRxBuffer[2] << 8) | pRxBuffer[3];
            *bDataTypeReceived = 1;
            dwReceivedBytes -= 4;
        }
        if(dwReceivedBytes >= 4)
        {
            *Size = (pRxBuffer[4] << 24) | (pRxBuffer[5] << 16) | (pRxBuffer[6] << 8) | pRxBuffer[7];
            *remainingSize = *Size;
            *bSizeReceived = 1;
            dwReceivedBytes -= 4;
        }

        if(*DataType < 1 || *DataType > 5)
        {
            return 1;
        }

        /* EOF */
        if(*DataType == 1 && *Size == 0)
        {
            finished = 1;
            phhalHwContact_Micropross_Int_FinishSpying(pDataParams, pSpyEvents, gHeader, dwETUns);
            return finished;
        }

        /* Receive remaining bytes */
        phhalHwContact_Micropross_Int_EvaluateData(pDataParams, dwReceivedBytes, gHeader, pSpyEvents, pRxBuffer + (*bDataTypeReceived * 4 + *bSizeReceived * 4),
            pData, dwDataOffset, remainingSize, DataType, Size, dwETUns, bDataTypeReceived, bSizeReceived);
    }

    return finished;
}

phStatus_t phhalHwContact_Micropross_Int_FinishSpying(
                                                      phhalHwContact_Micropross_DataParams_t *pDataParams,
                                                      uint32_t *pSpyEvents,
                                                      uint32_t *gHeader,
                                                      uint32_t *dwETUns
                                                      )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint32_t counter = 0;
    uint32_t start = 0, end = 0;
    FILE * fh;
    uint8_t littleEndian = 0x49;
    uint32_t dateCountZero = 0;
    SpyHeaderContact *spyHeader = (SpyHeaderContact *)gHeader;

    if(strstr((char *)pDataParams->bSpyFilename, (const char *)CONTACT_MICROPROSS_TRACEFILE_EXTENSION))
    {
        /* write new number of events into file header */
        fh = fopen((const char*)(pDataParams->bSpyFilename),"r+b");

        fseek(fh, 80, SEEK_SET);
        for(counter = 0; counter < 4; counter ++)
        {
            fprintf(fh, "%c", ((spyHeader->NumberOfEvents >> (8 * counter)) & 0xFF));
        }

        fseek(fh, 4, SEEK_SET);
        fwrite(&littleEndian, 1, 1, fh);
        fseek(fh, 5, SEEK_SET);
        fwrite(&littleEndian, 1, 1, fh);
        fclose(fh);
    }
    else
    {
        /* Calculate FDT */
        for(counter = 0; counter < spyHeader->NumberOfEvents; counter++)
        {
            if(((pSpyEvents[counter] & 0x000F) == 4) && (((pSpyEvents[counter] >> 4) & 0x0FFF) == 0x001))
            {
                end = pSpyEvents[counter] >> 16;
                break;
            }
            else if(counter == (CONTACT_MICROPROSS_SPY_BUFFER_SIZE - 1))
            {
                break;
            }
        }

        if(end)
        {
            for(; counter > 0; counter--)
            {
                if((pSpyEvents[counter] & 0x000F) == 5)
                {
                    start = pSpyEvents[counter] >> 16;
                    break;
                }
                else if(((pSpyEvents[counter] & 0x000F) == 0) && (((pSpyEvents[counter] >> 4) & 0x0FFF) == 0x00F))
                {
                    dateCountZero++;
                }
            }
        }

        if(start && end)
        {
            pDataParams->qwTiming_ns = (uint64_t)(end - start + dateCountZero * MAXWORD) * 20 - *dwETUns;
        }
        else
        {
            //TODOGL: return error in fail case? or just set FDT to zero?
            /* Something went wrong */
            pDataParams->qwTiming_ns = 0;
        }
    }

    /* Thread finished */
    SetEvent (pDataParams->GEvtFinished);

    return status;
}

phStatus_t phhalHwContact_Micropross_Int_SimAtrString(
                                                    phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                    uint8_t * pAtrString
                                                    )
{
    phStatus_t status;
    uint32_t i = 0;
    uint8_t tmpAtr[PHHAL_HW_CONTACT_MAX_ATR_SIZE];
    uint32_t dwAtrLength = (uint32_t)strlen((char *)pAtrString)/2;

    if(dwAtrLength > PHHAL_HW_CONTACT_MAX_ATR_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u %s",
        CONTACT_MICROPROSS_CMD_PROTOCOL_PARAM, pDataParams->bCouplerNumber, PHHAL_HW_CONTACT_MICROPROSS_CPP_SIM_ATR, dwAtrLength, pAtrString));

    /* Convert String to actual hex values */
    for(i = 0; i < dwAtrLength; i++)
    {
        uint8_t hex[2];
        hex[0] = pAtrString[i * 2];
        hex[1] = pAtrString[i * 2 + 1];
        tmpAtr[i] = (uint8_t)strtoul((const char*)hex, NULL, 16);
    }

    PH_CHECK_SUCCESS_FCT(status, phToolsAtrParser_SetInternalAtr(tmpAtr, dwAtrLength));

    return status;
}

phStatus_t phhalHwContact_Micropross_Int_SetGuardTimeClk(
                                                    phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                    uint32_t dwEtuCount,
                                                    uint32_t dwClockCount
                                                    )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u",
        CONTACT_MICROPROSS_CMD_SET_GUARD_TIME_CLK, pDataParams->bCouplerNumber, dwEtuCount, dwClockCount);
}

phStatus_t phhalHwContact_Micropross_Int_SetConvention(
                                                       phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                       uint32_t dwConvention
                                                       )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_SET_CONVENTION, pDataParams->bCouplerNumber, dwConvention);
}

phStatus_t phhalHwContact_Micropross_Int_ResetHard(
                                                   phhalHwContact_Micropross_DataParams_t * pDataParams
                                                   )
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u",
        CONTACT_MICROPROSS_CMD_RESET_HARD, pDataParams->bCouplerNumber);
}

phStatus_t phhalHwContact_Micropross_Int_ActivateI2C(
                                                     phhalHwContact_Micropross_DataParams_t * pDataParams
                                                     )
{
    phStatus_t status;
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 0 0", CONTACT_MICROPROSS_CMD_I2C_PIN_WRITE, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phTools_Sleep(2000));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_I2C_OPEN, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 3", CONTACT_MICROPROSS_CMD_I2C_FLUSH, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_I2C_ON, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 0 1", CONTACT_MICROPROSS_CMD_I2C_PIN_WRITE, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phTools_Sleep(2000));
    return status;
}

phStatus_t phhalHwContact_Micropross_Int_DeactivateI2C(
                                                       phhalHwContact_Micropross_DataParams_t * pDataParams
                                                       )
{
    phStatus_t status;
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_I2C_OFF, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 3", CONTACT_MICROPROSS_CMD_I2C_FLUSH, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_I2C_CLOSE, pDataParams->bCouplerNumber));
    return status;
}

phStatus_t phhalHwContact_Micropross_Int_GetMpNetworkAddress(
                                                             phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                             uint32_t * NetworkAddress,
                                                             uint32_t * SubnetMask
                                                             )
{
    phStatus_t status;
    uint32_t subnetmask[4];
    uint32_t ipaddress[4];
    uint32_t dwip;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_CheckSuccessCloseSpy(pDataParams, phToolsMicropross_SendAndCheckCmd(
        pDataParams, "%s", CONTACT_MICROPROSS_CMD_GET_NETWORK_INFO)));

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }

    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%d.%d.%d.%d %d.%d.%d.%d",
        &ipaddress[0], &ipaddress[1], &ipaddress[2], &ipaddress[3], &subnetmask[0], &subnetmask[1], &subnetmask[2], &subnetmask[3]) != 8)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    dwip = (ipaddress[0] << 24) + (ipaddress[1] << 16) + (ipaddress[2] << 8) + ipaddress[3];
    *SubnetMask = (subnetmask[0] << 24) + (subnetmask[1] << 16) + (subnetmask[2] << 8) + subnetmask[3];

    *NetworkAddress = dwip & *SubnetMask;

    return status;
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS */
