/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic HAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phbalReg.h>
#include <phhalHwContact.h>
#include <phTools.h>
#include <phToolsAtrParser.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_RDSCR01

#include "phhalHwContact_RdScr01.h"
#include "phhalHwContact_RdScr01_Int.h"
#include <time.h>

phStatus_t phhalHwContact_RdScr01_Init(
    phhalHwContact_RdScr01_DataParams_t * pDataParams,
    uint16_t wSizeOfDataParams,
    void * pBalDataParams,
    uint8_t * pTxBuffer,
    uint16_t  wTxBufSize,
    uint8_t * pRxBuffer,
    uint16_t  wRxBufSize,
    uint8_t * pIntBuffer,
    uint32_t wIntBufferSize
   )
{
    if (sizeof(phhalHwContact_RdScr01_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    /* Init. private data */
    pDataParams->wId                    = PH_COMP_HAL | PHHAL_HW_CONTACT_RDSCR01_ID;
    pDataParams->pBalDataParams         = (phbalReg_RdScr01_DataParams_t *)pBalDataParams;
    pDataParams->pTxBuffer              = pTxBuffer;
    pDataParams->wTxBufSize             = wTxBufSize;
    pDataParams->wTxBufLen              = 0;
    pDataParams->pRxBuffer              = pRxBuffer;
    pDataParams->wRxBufSize             = wRxBufSize;
    pDataParams->wRxBufLen              = 0;
    pDataParams->wRxBufStartPos         = 0;
    pDataParams->pIntBuffer             = pIntBuffer;
    pDataParams->dwIntBufferLen         = 0x00;
    pDataParams->dwIntBufferSize        = wIntBufferSize;
    pDataParams->dwNumExpectedBytes     = 0;
    pDataParams->dwI2CRxPollingTimeoutMs= 1000;
    pDataParams->bCommunicationChannel  = PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD;
    pDataParams->wSlaveAddress          = 0;
    pDataParams->dwBitrate              = 0;
    pDataParams->bClkStretch            = PH_ON;
    pDataParams->bHsMode                = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_SetConfig32(
    phhalHwContact_RdScr01_DataParams_t * pDataParams,
    uint16_t  wConfig,
    uint32_t  dwValue
    )
{
    phStatus_t  statusTmp = PH_ERR_SUCCESS;
    float64_t configValue = 0;
    uint32_t value;

    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
        if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_LRC &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_CRC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_LRC &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_CRC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_FD:
        statusTmp = phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfRawSetBaudrate((dwValue >> 4) & 0x0F, dwValue & 0x0F));
        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_VCC:
        configValue = (float64_t)(dwValue / 1000.0);
        statusTmp = phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfSetVcc(configValue));
        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
        if (pDataParams->bCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        configValue = (float64_t)(dwValue / 1000000.0);
        statusTmp = phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfSetUartFreq(configValue));
        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_RX_POLLING_TIMEOUT_MS:
        pDataParams->dwI2CRxPollingTimeoutMs = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_TIMING_MODE:
        pDataParams->wTimingMode = (uint16_t)dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        /* Timing in us */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ, &value));
        value = (uint32_t)(((float64_t)dwValue / value) * 1000000);
        statusTmp = phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pf7816ModifyTimeout(value));
        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
        pDataParams->dwNumExpectedBytes = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL:
        if(dwValue == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD && (uint8_t)dwValue != pDataParams->bCommunicationChannel)
        {
            pDataParams->dwBitrate = 0;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfSetIoConfig(SECIF_NONE, RST_MODE_ISO7816, CLK_MODE_ISO7816)));
        }
        else if(dwValue == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C && (uint8_t)dwValue != pDataParams->bCommunicationChannel)
        {
            /* 400 kBit/s Bitrate */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfI2CSetBitrate(400)));
            pDataParams->dwBitrate = 400;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfSetIoConfig(SECIF_SCI2C_IO2IO3, RST_MODE_HIGH, CLK_MODE_LOW)));
            pDataParams->pBalDataParams->pfRawGetColdAtr(&pDataParams->pIntBuffer[0], &pDataParams->dwIntBufferLen);
        }
        else if(dwValue != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD && dwValue != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        pDataParams->bCommunicationChannel = (uint8_t)dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_SLAVE_ADDRESS:
        if(dwValue < 0 || dwValue > 127)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfI2CSetControlByte((uint8_t)dwValue)));
        pDataParams->wSlaveAddress = (uint16_t)dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (dwValue > 0xFFFF || dwValue >= pDataParams->wRxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Set start position */
        pDataParams->wRxBufStartPos = (uint16_t)dwValue;
        pDataParams->wRxBufLen = (uint16_t)dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_BITRATE_KBITSEC:
        if (pDataParams->bCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }

        /* Boundary check */
        if (dwValue < 20 || dwValue > 3500)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfI2CSetBitrate((float64_t)dwValue)));
        pDataParams->dwBitrate = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_CLK_STRETCH:
        if(dwValue != PH_ON && dwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Setting is called disable clock strechting so if dwValue == PH_ON we set the config so zero */
        value = dwValue == PH_ON ? 0: 1;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfI2CSetParameter(SCR01_RAWI2C_PROTOCOL_DIS_ALL_STRETCH, value)));
        pDataParams->bClkStretch = (uint8_t)dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_HS_MODE:
        if(dwValue != PH_ON && dwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Setting is called disable hs mode so if dwValue == PH_ON we set the config so zero */
        value = dwValue == PH_ON ? 0: 1;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfI2CSetParameter(SCR01_RAWI2C_PROTOCOL_DIS_HS_MODE, value)));
        pDataParams->bHsMode = (uint8_t)dwValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_GetConfig32(
    phhalHwContact_RdScr01_DataParams_t * pDataParams,
    uint16_t   wConfig,
    uint32_t * pValue
    )
{
    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
    case PHHAL_HW_CONTACT_CONFIG_FD:
    case PHHAL_HW_CONTACT_CONFIG_VCC:
    case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_RX_POLLING_TIMEOUT_MS:
        /* Return parameter */
        *pValue = pDataParams->dwI2CRxPollingTimeoutMs;
        break;

    case PHHAL_HW_CONTACT_CONFIG_TIMING_MODE:
        *pValue = pDataParams->wTimingMode;
        break;

    case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
        *pValue = pDataParams->dwNumExpectedBytes;
        break;

    case PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL:
        *pValue = (uint32_t)pDataParams->bCommunicationChannel;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_SLAVE_ADDRESS:
        *pValue = (uint32_t)pDataParams->wSlaveAddress;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
        *pValue = (uint32_t)pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_BUFSIZE:
        *pValue = pDataParams->wRxBufSize - PHHAL_HW_CONTACT_MAX_CRC_BUFFER_LEN;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_BITRATE_KBITSEC:
        *pValue = pDataParams->dwBitrate;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_CLK_STRETCH:
        *pValue = (uint32_t)pDataParams->bClkStretch;
        break;

    case PHHAL_HW_CONTACT_CONFIG_I2C_HS_MODE:
        *pValue = (uint32_t)pDataParams->bHsMode;
        break;

    default:
        *pValue = 0;
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_Wait(
    phhalHwContact_RdScr01_DataParams_t * pDataParams,
    uint8_t   bUnit,
    uint16_t  wTimeout
    )
{
    if (bUnit == PHHAL_HW_CONTACT_TIME_MICROSECONDS)
    {
        pDataParams->pBalDataParams->pfSleep(wTimeout / 1000);
    }
    else if (bUnit == PHHAL_HW_CONTACT_TIME_MILLISECONDS)
    {
        pDataParams->pBalDataParams->pfSleep(wTimeout);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_Exchange(
    phhalHwContact_RdScr01_DataParams_t * pDataParams,
    uint16_t    wOption,
    uint8_t   * pTxBuffer,
    uint16_t    wTxLength,
    uint8_t  ** ppRxBuffer,
    uint16_t  * pRxLength
    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    phStatus_t PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    uint8_t    PH_MEMLOC_REM pCrc[2];
    uint8_t    PH_MEMLOC_REM pLrc[1];
    uint8_t    PH_MEMLOC_REM bLrc;
    uint16_t   PH_MEMLOC_REM wCrc;
    uint32_t   PH_MEMLOC_REM pFdt;
    uint32_t   PH_MEMLOC_REM protocolType;
    clock_t    PH_MEMLOC_REM start;
    clock_t    PH_MEMLOC_REM end;
    uint32_t   PH_MEMLOC_REM dwReceivedBytes;
    uint16_t   PH_MEMLOC_REM wCrcLength;
    uint8_t *  PH_MEMLOC_REM pCrcData;

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pDataParams->pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &pDataParams->wRxBufLen;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* copy data */
    memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength); /* PRQA S 3200 */
    pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;

    /* do not perform real exchange, just fill the global TxBuffer */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* If data for Tx available */
    if(pDataParams->wTxBufLen != 0)
    {
        /* Calc LRC/CRC */
        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_CRC)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
                PH_TOOLS_CRC16_PRESET_CONTACTBASED,
                PH_TOOLS_CRC16_POLY_CONTACTBASED,
                pTxBuffer,
                wTxLength,
                &wCrc));

            if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED)
            {
                pCrc[0] = (uint8_t)(wCrc >> 8);
                pCrc[1] = (uint8_t)(wCrc);
            }
            else
            {
                pCrc[0] = (uint8_t)(wCrc);
                pCrc[1] = (uint8_t)(wCrc >> 8);
            }

            memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pCrc, 2); /* PRQA S 3200 */
            pDataParams->wTxBufLen += 2;
        }
        else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_LRC)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateLrc(pTxBuffer, wTxLength, pLrc));

            memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pLrc, 1); /* PRQA S 3200 */
            pDataParams->wTxBufLen += 1;
        }

        /* Copy data to int buffer and update wTxLength */
        /* copy data to pIntTmpBuffer as this buffer is used as RxBuffer afterwards */
        memcpy(pDataParams->pIntBuffer, &pDataParams->pTxBuffer[0], pDataParams->wTxBufLen); /* PRQA S 3200 */
        wTxLength = (uint32_t)pDataParams->wTxBufLen;
    }
    else
    {
        wTxLength = 0;
    }

    /* Clear TxBuffer Data */
    pDataParams->wTxBufLen = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, &protocolType));

    switch(pDataParams->bCommunicationChannel)
    {
    case PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD:
        switch(protocolType)
        {
        case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
            if(wTxLength != 0)
            {
                /* Set number bytes to transmit (dwIntBufferLen is used because it is also used for return value) */
                pDataParams->dwIntBufferLen = wTxLength;
                /* exchange data */
                status = phhalHwContact_RdScr01_Int_ConvertError(
                    pDataParams->pBalDataParams->pfRawExchange(pDataParams->pIntBuffer, &pDataParams->dwIntBufferLen, pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_WT_CLK]));
            }
            else
            {
                /* we just want to receive bytes so check if we have something buffered */
                status = (pDataParams->dwIntBufferLen > 0) ? PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL) : PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
            }

            dwReceivedBytes = (pDataParams->dwIntBufferLen > pDataParams->dwNumExpectedBytes) ? pDataParams->dwNumExpectedBytes : pDataParams->dwIntBufferLen;
            break;

        case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
            /* Set number bytes to transmit (dwIntBufferLen is used because it is also used for return value)  */
            pDataParams->dwIntBufferLen = wTxLength;
            /* exchange data */
            status = phhalHwContact_RdScr01_Int_ConvertError(
                pDataParams->pBalDataParams->pfRawExchange(pDataParams->pIntBuffer, &pDataParams->dwIntBufferLen, pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_BWT_CLK]));
            dwReceivedBytes = pDataParams->dwIntBufferLen;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        break;

    case PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C:
        /* exchange data */
        /* Perform send command only if data are available for send */
        if (wTxLength != 0)
        {
            status = phhalHwContact_RdScr01_Int_ConvertError(
                pDataParams->pBalDataParams->pfI2CSendData(pDataParams->pIntBuffer, wTxLength));

            if((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                return status;
            }
        }

        /* we do not expect any bytes so we can already return */
        if(pDataParams->dwNumExpectedBytes == 0)
        {
            /* Store received data length in dataparams */
            pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + 0;

            /* Return RxBuffer pointer */
            if (ppRxBuffer != NULL)
            {
                *ppRxBuffer = pDataParams->pRxBuffer;
            }

            /* Return RxBuffer length */
            if (pRxLength != NULL)
            {
                *pRxLength = pDataParams->wRxBufLen;
            }

            return PH_ADD_COMPCODE(status, PH_COMP_HAL);
        }

        /* Polling for one second */
        start = clock();
        do
        {
            end = clock();

            /* Inital delay */
            phTools_Sleep(1);

            status = phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pfI2CReadData(pDataParams->pIntBuffer, pDataParams->dwNumExpectedBytes));
        } while((1000 * ((double)end - start)) / CLOCKS_PER_SEC <= pDataParams->dwI2CRxPollingTimeoutMs && (status & PH_ERR_MASK) != PH_ERR_SUCCESS);
        dwReceivedBytes = pDataParams->dwNumExpectedBytes;
        pDataParams->dwIntBufferLen = pDataParams->dwNumExpectedBytes;
        break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* if success copy data to RxBuffer */
    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* check if buffer fits (also if offset is set) */
        if(dwReceivedBytes <= (uint32_t)(pDataParams->wRxBufSize - pDataParams->wRxBufStartPos))
        {
            /* Add new data to rxBuffer*/
            memcpy(&pDataParams->pRxBuffer[pDataParams->wRxBufStartPos], pDataParams->pIntBuffer, dwReceivedBytes); /* PRQA S 3200 */
            pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + (uint16_t)dwReceivedBytes;

            /* Save remaining bytes for later use */
            if(pDataParams->dwNumExpectedBytes < pDataParams->dwIntBufferLen)
            {
                memmove(&pDataParams->pIntBuffer[0], &pDataParams->pIntBuffer[dwReceivedBytes], pDataParams->dwIntBufferLen - dwReceivedBytes);
                pDataParams->dwIntBufferLen -= dwReceivedBytes;
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(status, PH_COMP_HAL);
    }

    if(pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_COMM || pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_FDT)
    {
        /* get execution time of last exchange */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(
            pDataParams->pBalDataParams->pf7816GetFdt(&pFdt)));

        if(pFdt != 0)
        {
            pDataParams->qwTiming_ns = pFdt * 1000;
        }
    }

    /* get data for CRC check */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER) == PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER)
    {
        wCrcLength = pDataParams->wRxBufLen;
        pCrcData = pDataParams->pRxBuffer;
    }
    else
    {
        wCrcLength = pDataParams->wRxBufLen - pDataParams->wRxBufStartPos;
        pCrcData = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    }

    /* Check and remove CRC in software if required */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
    {
        if (wCrcLength < 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_CONTACTBASED,
            PH_TOOLS_CRC16_POLY_CONTACTBASED,
            pCrcData,
            wCrcLength - 2,
            &wCrc));

        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED)
        {
            pCrc[0] = (uint8_t)(wCrc >> 8);
            pCrc[1] = (uint8_t)(wCrc);
        }
        else
        {
            pCrc[0] = (uint8_t)(wCrc);
            pCrc[1] = (uint8_t)(wCrc >> 8);
        }

        /* CRC over all data must be zero */
        if (pCrc[0] != pCrcData[wCrcLength-2] || pCrc[1] != pCrcData[wCrcLength-1])
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        pDataParams->wRxBufLen -= 2;
    }
    else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_LRC)
    {
        if (wCrcLength < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateLrc(
            pCrcData,
            wCrcLength,
            &bLrc));

        /* LRC over all data must be zero */
        if (bLrc != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        pDataParams->wRxBufLen -= 1;
    }

    *ppRxBuffer = pDataParams->pRxBuffer;
    *pRxLength  = pDataParams->wRxBufLen;

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_ApplyProtocolSettings(
    phhalHwContact_RdScr01_DataParams_t * pDataParams,
    uint8_t * pAtr,
    uint16_t dwAtrLength,
    uint8_t bProtocolType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t    PH_MEMLOC_REM wConfig;
    uint8_t     PH_MEMLOC_REM bConfig = 0;
    uint8_t     PH_MEMLOC_REM bSupported;
    uint32_t    PH_MEMLOC_REM wFDIndexValue;

    /* if ATR available check if card supports specified ProtocolType */
    if(dwAtrLength != 0 && pAtr != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(pAtr, dwAtrLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(bProtocolType, &bSupported));

        if(!bSupported)
        {
            /* not supported so reset to default values */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }

    /* Store new protocol type */
    if (bProtocolType != PHHAL_HW_CONTACT_PROTOCOLTYPE_CURRENT)
    {
        pDataParams->bProtocolType = bProtocolType;
    }

    /* Apply global Parameters */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_Convention(&bConfig));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CONVENTION, (uint32_t)bConfig));*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &wFDIndexValue));

    /* Apply protocol dependent Parameters */
    switch(bProtocolType)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_WTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, wConfig));

        /*PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_GTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_GT_CLK, wConfig));*/

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, PHHAL_HW_CONTACT_MODE_DISABLED));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        /*PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CGTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CGT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CWT_CLK, wConfig));*/

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_BWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_BWT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_RedundancyMode(&bConfig));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, (uint32_t)bConfig));

        pDataParams->dwNumExpectedBytes = PHHAL_HW_CONTACT_MAX_T1_BLOCK_SIZE;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, bProtocolType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_Activate(
                        phhalHwContact_RdScr01_DataParams_t * pDataParams,
                        uint8_t * pATRBuffer,
                        uint16_t * pwATRLength
                        )
{
    phStatus_t statusTmp;
    phStatus_t status = PH_ERR_SUCCESS;
    uint8_t bColdReset;
    uint32_t dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C communication */
    if (dwCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    if(pDataParams->dwAtrLength == 0)
    {
        bColdReset = 1;
    }
    else
    {
        bColdReset = 0;
    }

    /* Reset ATR before doing anything else */
    memset(pDataParams->pAtr, 0x00, PHHAL_HW_CONTACT_MAX_ATR_SIZE);
    pDataParams->dwAtrLength = 0;

    /* Before activation bring reader to default state */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    if(bColdReset)
    {
        status = phhalHwContact_RdScr01_Int_ConvertError(
            pDataParams->pBalDataParams->pfRawGetColdAtr(&pDataParams->pAtr[0], &pDataParams->dwAtrLength));
    }
    else
    {
        status = phhalHwContact_RdScr01_Int_ConvertError(
            pDataParams->pBalDataParams->pfRawGetWarmAtr(&pDataParams->pAtr[0], &pDataParams->dwAtrLength));
    }

    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        if(pDataParams->dwAtrLength > PHHAL_HW_CONTACT_MAX_ATR_SIZE)
        {
            *pwATRLength = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        memcpy(pATRBuffer, &pDataParams->pAtr[0], pDataParams->dwAtrLength);
        *pwATRLength = (uint16_t)pDataParams->dwAtrLength;

        /* Set ATR in phToolsAtrParser to parse certain values later */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(&pDataParams->pAtr[0], pDataParams->dwAtrLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));
    }
    else
    {
        *pwATRLength = 0;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_Deactivate(
                        phhalHwContact_RdScr01_DataParams_t * pDataParams
                        )
{
    phStatus_t statusTmp;
    uint32_t dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C communication */
    if (dwCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(pDataParams->pBalDataParams->pf7816Deactivate()));
    pDataParams->dwAtrLength = 0;
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_ClockStop(
                        phhalHwContact_RdScr01_DataParams_t * pDataParams,
                        uint16_t wTgClockCount,
                        uint16_t wThClockCount,
                        uint32_t dwPinState
                        )
{
    phStatus_t statusTmp;
    uint32_t   PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C communication */
    if (dwCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* just to satisfy compiler - command not available yet in scr01dll */
    if(pDataParams || wTgClockCount || wThClockCount || dwPinState);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHwContact_RdScr01_InitContactReader(
                                                    phhalHwContact_RdScr01_DataParams_t * pDataParams
                                                    )
{
    phStatus_t statusTmp;

    /* Set internal default values for communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL, PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));
    /* Voltage class is suppled outside no need to do */
    /* PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_VCC, PHHAL_HW_CONTACT_CLASS_A)); */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ, PHHAL_HW_CONTACT_CLK_DEFAULT));
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR, PHHAL_HW_CONTACT_MAX_CLK_ATR_DEFAULT));*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_I2C_CLK_STRETCH, PH_ON));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_I2C_HS_MODE, PH_ON));

    /* ApplyProtocolSettings to have default values applies */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    /* Deactivate set power and ATR length to zero */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Deactivate(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_RDSCR01 */
