/************************************************************************
                        CFG
                Configuration Interface
 ************************************************************************
 18/01/2003 - Marc LEGOUX - c6r06 : Put line buffer into cfg_FType
 04/12/2000 - Marc LEGOUX - c6r05 : Add ModeSLine
 26/05/2000 - Marc LEGOUX - c6r03 : Add _FileReopen()
 17/09/1997 - Marc LEGOUX - c4r01 : Copy from own module
 ************************************************************************
 A configuration is recorded as a tree of keys. A value is stored at each
 key node. 
 It is possible to identify a key with a path name of the key from
 the configuration root:
        :rootkey:key1:key2: ... :keyN,  
 or from the current key :
        key2: ... :keyN if current key is :rootkey:key1.
 A key path is made of key names separated with ':'.
 If the path does not begin with ':', it starts from the current key, 
 otherwise it starts from the root.
 
 It is possible to read or write the value of a key.
 It is also possible to iter on the subkeys of a key.
 ************************************************************************
 File implementation
 ************************************************************************/
#ifndef cfgf_H
#define cfgf_H

#include <err.h>
#include <config.h>
#include <txf.h>

#define cfg_FilePathMax	256	/* Max length of path names */
#define cfg_TxfLineMaxL	255	// Max line length

#define cfg_NameMaxL	32	// Max name length
#define cfg_ValueMaxL	255	// Max name length

/* Configuration type (text file) */
typedef 
  struct {
    txf_iter_type Txf;
    char node[cfg_NameMaxL+1];		/* Current node key last or "" */
    char rfpath[cfg_FilePathMax+1];	/* read file path */
    char wfpath[cfg_FilePathMax+1];	/* temp write file path */
    char next;			/* Next is done */
    unsigned mode;
    char TxfLine[cfg_TxfLineMaxL+1];	// Line buffer for Txf
  } cfg_FType;

#define cfg_FModeAlloc		1	/* Allocated */
#define cfg_FModeTmpFile	2	/* Read file but be deleted at close*/
/*-------------------------- ERRORS -----------------------------------*/
extern
err_Type cfg__Err;
#define cfg__KeyNotFound (cfg__Err+1)	/* Key not found */
#define cfg__TooLong	 (cfg__Err+2)	/* String or buffer too long */
#define cfg__KeyNode	 (cfg__Err+3)	/* Node key (not a leaf key) */
#define cfg__KeyNoMore	 (cfg__Err+4)	/* No more keys */
#define cfg__KeyType	 (cfg__Err+5)	/* Wrong key value type */
#define cfg__Syntax	 (cfg__Err+6)	/* Config syntax error */
#define cfg__Name	 (cfg__Err+7)	/* Wrong config name */
#define cfg__KeyLevel	 (cfg__Err+8)	/* Too many key levels */
#define cfg__KeyName	 (cfg__Err+9)	/* Bad key */
#define cfg__KeyExists	 (cfg__Err+10)	/* Already defined key */
#define cfg__Write	 (cfg__Err+11)	/* Write is impossible */
#define cfg__KeyMode	 (cfg__Err+12)	/* Wrong mode */

/*-------------------------- PRIVATE-----------------------------------*/
/* Deph of key */
#define cfg_KeyTypeMask		0x000F0
#define cfg_KeyTypeNode		0x00000	/* Node key mode */
#define cfg_KeyTypeLeaf		0x00010	/* Leaf key mode */

/* Types of values */
#define cfg_KeyValMask		0x0F000
#define cfg_KeyValNode		0x00000
#define cfg_KeyValSt		0x01000	/* String ...0 */
#define cfg_KeyValMSt		0x02000	/* Multi-String ...0...0...00 */
#define cfg_KeyValLong		0x03000	/* Long */

/*-------------------------- FUNCTIONS---------------------------------*/
/*---------------------------------------------------------------------*
                        Open a cfg on a file
 *---------------------------------------------------------------------*/
err_Type cfg_FileOpen(
        cfg_FType * pCfg,	/* (OUT): Config  */
        char * FilePath,	/* Cfg File Path */
        unsigned ModeWrite	/* 1 = to write config */
	);
/*---------------------------------------------------------------------*
                        Close a Cfg
 *---------------------------------------------------------------------*/
err_Type cfg_FileClose(
	cfg_FType * Cfg		/* Config  */
	);
/*---------------------------------------------------------------------*
                        Reopen a Cfg
 *---------------------------------------------------------------------*/
err_Type cfg_FileReopen(
        cfg_FType * Cfg		/* Cfg  */
        );
/*---------------------------------------------------------------------*
                        Node iter
 Copy current list of sub-keys into another config
 and open it
 *---------------------------------------------------------------------*/
err_Type cfg_FileIterNode(
        cfg_FType * Cfg,	/* Config */
        unsigned Mode,		/* Iter mode */
        char * KeyName,		/* Node key name */
        cfg_FType ** pCfg,	/* (IN/OUT): Config on sub-keys */
        unsigned *pline		/* null or (OUT): No de ligne ou 0 */
        );
/*---------------------------------------------------------------------*
                        Iter on file key definitions
                        according to mode
 The file must be open.
 Leaf mode: on leaf keys until next node key
 Node mode: on node keys only
 NOTE:	If the Input KeyName is "", the next key is returned,
        otherwise the next key of name KeyName is returned
 *---------------------------------------------------------------------*/
/* Basic modes (ORed) */
#define cfg_KeyModeMask		0x0F

#define cfg_KeyModeGetNx	0x00	/* Get key and goto next */
#define cfg_KeyModeDef		0x01	/* Insert key */
#define cfg_KeyModeUndef	0x02	/* Undef key */
#define cfg_KeyModeGet		0x03	/* Get key */
#define cfg_KeyModeSet		0x04	/* Set key */
#define cfg_KeyModeUndefb	0x05	/* Undef before key */
#define cfg_KeyModeUndefa	0x06	/* Undef key and after key */

#define cfg_KeyMode2Mask	0x0F00
#define cfg_KeyModeSLine	0x0100	/* Search line num if GetNx */

err_Type cfg_FileIter(
        cfg_FType * Cfg,	/* (IN): Open Config */
        unsigned Mode,		/* Iter mode */
        char * KeyName,		/* (IN=search,def or ""/OUT) : Key name (32 max) */
        char * Value,		/* null or (IN/OUT): Key Value */
        unsigned MaxLen,	/* Max value length if Value OUT */
        unsigned * pLen,	/* null or (OUT): returned length */
        unsigned long *pType,	/* null or (OUT): returned type ID */
        unsigned *pline		/* null or (IN/OUT): No de ligne ou 0 */
        );
/*---------------------------------------------------------------------*
                        Find a key from file
 One level  : <keyL>
 Two levels : <keyN:keyL>
 *---------------------------------------------------------------------*/
err_Type cfg_FileKeyFind(
        char * FilePath,	/* Cfg File Path */
        char * KeyName,		/* (IN) : Key name */
        char * Value,		/* (OUT): Key Value */
        unsigned MaxLen,	/* Max value length */
        unsigned * pLen,	/* null or (OUT): returned length */
        unsigned long *pType,	/* null or (OUT): returned type ID */
        unsigned *pline		/* null or (OUT): No de ligne ou 0 */
	);
/*---------------------------------------------------------------------*
                        Set a key into file
 One level  : <keyL>
 Two levels : <keyN:keyL>
 *---------------------------------------------------------------------*/
#define cfg_FileKeyWModeDef	cfg_KeyModeDef
#define cfg_FileKeyWModeUndef	cfg_KeyModeUndef
#define cfg_FileKeyWModeSet	cfg_KeyModeSet

err_Type cfg_FileKeyWrite(
        char * FilePath,	/* Cfg File Path */
        unsigned KeyWMode,	/* KeyWModexxx */
        char * KeyName,		/* (IN) : Key name */
        char * Value,		/* Key Value */
        unsigned Len,		/* Value length */
        unsigned long Type,	/* type ID */
        unsigned *pline		/* null or (OUT): No de ligne ou 0 */
	);
/*-----------------------------------------------------------------------*
                        Start
 Usefull only to control verbose and to get version
 *-----------------------------------------------------------------------*/
err_Type cfg_FileStart(
        void * Cfg
        );
/*-----------------------------------------------------------------------*
                        End
 *-----------------------------------------------------------------------*/
err_Type cfg_FileEnd(
        void * Cfg
        );
/*---------------------------------------------------------------------*/
#endif
