/***********************************************************************
                                  CHAR
                           Gestion des caracteres
 ***********************************************************************
 On definit un type de "gros" caractere char_type (8bits ou plus) pour :
 les buffers de texte,
 les caracteres utilises de facon speciale (EOL, NULL, etc),
 les extentions du code ASCII,
 les flags de search (SE_xxx),
 les attributs d'affichage,
 quelques fonctions sur les caracteres et chaines standard.
 
 On definit un type de "petit" caractere char_0type (8bits) pour :
 les chaines standards et leurs fonctions
 les parseurs standards (tcur, parse)
 ***********************************************************************/
#ifndef char_h
#define char_h

#include <string.h>

#include <err.h>
#include <config.h>
/*--------------------------------DATAS--------------------------------*/
/* Erreurs */
#ifndef char__err
extern
err_type char__err;
#endif
#define char__too_long (char__err + 1)
#define char__cut_line (char__err + 2)
#define char__eos	(char__err + 3)
/*---------------------------------------------------------------------*/
/* Gros caractere */
typedef 
  unsigned long char_type;
#define char_ValMax	0x0FFFF

typedef
  char_type * str_type;

/* Chaine standard */
typedef 
  unsigned char char_0type;
typedef 
  char_0type * str_0type;

/* Max default length for string conversions */
#define char_StMax	512
#define char_st_max char_StMax

/* pointeur NULL */
#define char_pnull ((char *)0)
/*----------------------------------------------------------------------*
                        Conversions
 *----------------------------------------------------------------------*/
extern
err_Type char_StFromChar(
        char_type * st, 	// OUT
        char * st0
        ); 
#define char_CharToSt char_StFromChar
/*---------------------------------------------------------------------*/
extern
err_Type char_StFromCharAttr(
        char_type * st, 	// OUT
        char * st0,
        unsigned Attr		// Upper bytes to add to all st0 chars
        ); 
/*---------------------------------------------------------------------*/
extern
err_Type char_CarFromCharAttr(
        char_type * pCar, 	// OUT
        char c,
        unsigned Attr
        ); 
/*---------------------------------------------------------------------*/
extern
err_Type char_StToChar(
        char_type * st, 
        char * st0		// OUT
        ); 
/*----------------------------------------------------------------------*
			Attributs
 *----------------------------------------------------------------------*/
#define char_AttrMask		0x0F00	// Mask in chars
#define char_AttrKeyWord	1
#define char_AttrComment	2

#define char_AttrGet(c)   (((c) &  0x0F00) >> 8)
#define char_AttrSet(c,a) (((c) & ~0x0F00) | ((char_type)(a)<<8))
/*---------------------------------------------------------------------*
			Caracteres speciaux
 *---------------------------------------------------------------------*/
#define char_eol        0x0A        /* End of line */
#define char_eos        0           /* End of string */
#define char_eot        0x04        /* End of text */

#define char_null       0
#define char_tab	0x09
#define char_lf         0x0A
#define char_cr		0x0D
#define char_space      0x020

/* Flags de commande de search */
#define SE_AFTER     0x01	/* Arret curseur apres (avant) */
#define SE_BOL       0x02	/* Cadrage en debut de ligne (pas cadre) */
#define SE_EOL       0x04	/* Cadrage en fin   de ligne (pas cadre) */
#define SE_UPWARD    0x08	/* Recherche vers l'amont (aval) */
#define SE_NEXACT    0x10	/* Assimilation majus. et minus. (Discrimination) */
#define SE_LINE      0x20	/* Recherche uniquement dans la ligne.*/
#define SE_CUR       0x40	/* Recherche uniquement au curseur.*/
#define SE_BIGCHAR   0x80	/* Recherche avec char_type (char).*/
/*------------------------------FUNCTIONS------------------------------*/
/*---------------------------------------------------------------------*
		Chaines de caracteres char_type (ST)
 *---------------------------------------------------------------------*
 Ce sont des tableaux (char_type[]) termines par char_null.
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
                        Longueur chaine
 *---------------------------------------------------------------------*/
extern
unsigned char_stlen(char_type * st);
#define char_strlen char_stlen
/*---------------------------------------------------------------------*
                        Comparaison chaine
 *---------------------------------------------------------------------*/
extern
int char_stcmp(char_type * st1, char_type * st2);
#define char_strcmp char_stcmp
/*----------------------------------------------------------------------*
                        Cherche caractere
 *----------------------------------------------------------------------*/
extern
char_type * char_stchr(char_type * st,char_type c);
#define char_strchr char_stchr
/*---------------------------------------------------------------------*
                               Copie de chaine
 *---------------------------------------------------------------------*
 On copie jusqu`au char_null inclu.
 Si size est depassee, la copie est tronquee et on retourne un status.
 char__too_long.
 *---------------------------------------------------------------------*/
extern 
err_type char_stncpy(
        char_type * strto,
        char_type * strfrom,
        unsigned size	// Taille max copiee
        );
#define char_strncpy char_stncpy
extern 
err_type char_stcpy(
        char_type * strto,
        char_type * strfrom
        );
#define char_strcpy char_stcpy
/*--------------------------------------------------------------------*
                        Traduction d'une string en LOWERCASE 
 *--------------------------------------------------------------------*/
extern 
err_type char_stlower(char_type * st_r);
#define char_strlower char_stlower
/*---------------------------------------------------------------------*
                           Chaines normales (STR)
 *---------------------------------------------------------------------*
 Ce sont des tableaux (char[]) termines par '\0'
 Les fonctions habituelles sont dans la librairie C.
 *---------------------------------------------------------------------*/
/* Fonctions standard */
#if config_SUN
#define strchr(s,c) index(s,c)
extern char * index();
#endif
/*
extern char * strchr(const char *, int);
extern char * strrchr(char *, int);
extern char * strcpy(char *, char *);
extern char * strncpy(char *, char *, int);
*/

#define char_0strchr(s,c)	(char_0type *)(strchr((char *)(s), (char)(c)))
#define char_0strlower(s)	strlwr((char *)(s))
#define char_0strlen(s)		strlen((char *)(s))
#define char_0strcmp(d,s)	strcmp((char *)(d),(char *)(s))
#define char_0strcmpi(d,s)	strcmpi((char *)(d),(char *)(s))
#define char_0strcpy(d,s)	strcpy((char *)(d),(char *)(s))
#define char_0strcat(d,s)	strcat((char *)(d),(char *)(s))
#define char_0strncpy(a,b,c)	(strncpy((a),(b),(c)),0)
/*---------------------------------------------------------------------*
                        Transformation d'entier en chaine
 *---------------------------------------------------------------------*
 On ecrit le signe - si < 0 et on ajoute un nul en fin de chaine.
 *---------------------------------------------------------------------*/
extern 
err_type stritoa(
        int nb,		/* Nombre */
        char * strto,	/* chaine resultat */
        int base	/* base */
        );
/*----------------------------------------------------------------------*
                        Compare Strings (ignore case)
 *----------------------------------------------------------------------*/
#if config_WINDOWS
/* not ANSI */
extern
int strcmpi(const char *st1, const char *st2);
#endif
/*----------------------------------------------------------------------*
                        Pattern matching
 Pattern wild cards:
   * = 0,1 or several chars
   ? = any char
   # = 0..9
 Returns 0 if ok
 *----------------------------------------------------------------------*/
extern
err_Type char_0strcmpp(
        char * st,
        char * pat
        );
/*----------------------------------------------------------------------*
                        Compare strings with decimal numbers
 Internal number are correctly ordered
 ex: IC10 > IC1
 *----------------------------------------------------------------------*/
extern
int char_0strcmpd(
        char * st1,
        char * st2
        );
/*----------------------------------------------------------------------*
                        Read a line from a string
 CR are ignored
 *----------------------------------------------------------------------*/
extern
err_Type char_0slread(
        char ** st,	// IN: String, OUT: Rest
        char * line,	// OUT: Line
        int * psize	// IN: Max line chars, OUT: read chars (0 not counted)
        );
/*----------------------------------------------------------------------*
                        Substitution
 *----------------------------------------------------------------------*/
extern
err_Type char_0strsubst(
        char * st, 	// IN
        char **pst,	// OUT (st if no change or allocated)
        char * st1,	// A substituter
        char * st2	// Avec 
        );
/*----------------------------------------------------------------------*
                        Check a char is in a string
 Returns found position in st or 0
 *----------------------------------------------------------------------*/
extern
char_0type * char_0strnchr(
        char_0type * st,
        char_0type * cset	// Char set
        );
/*----------------------------------------------------------------------*
                        Pack spaces in a string 
 A group of spaces is replaced with a single space (==cset[0])
 The spaces are the beginning and at the end of the string are removed. 
 Returns the string
 *----------------------------------------------------------------------*/
extern
char_0type * char_0strpack(
        char_0type * st,
        char_0type * cset	// Spaces char set
        );
/*----------------------------------------------------------------------*/
#endif
