/***********************************************************************
			ERRORs
 ***********************************************************************
 Copyright (c) SMARTWARE, 1999-2005 All Rights Reserved.
 ***********************************************************************
 Each module has a name <module name>. 
 In the following examples <module name> is mod. 
 
 MODULE PUBLICATION: 
 Each module can publish informations about itself with err_ModPublish().
 Those informations group the module name, error messages as well as 
 verbose levels inside a descriptor. 
 This descriptor must be removed with err_ModRemove()
 when the module terminates. 
 
 ERROR USAGE:
 Every public functions of every modules MUST return an error code.
 Code 0 means OK.
 
 So every functions are declared like:
 extern err_Type <module name>_Function(...);
 
 ERROR CODES : 
 All Error symbols must de named:
 "<module name>__<error name>"
 ex: mod__NotFound
 
 Each module defines its own error codes with an offset 1,2,... added to
 a variable <module name>__Err which is the error base of the module. Ex:
 #define mod__Error1	(mod__Err+1)
 
 Error base identifiers <module name>__Err are declared in the corresponding 
 module header file <module name>.H :
 extern err_Type <module name>__Err;
 
 They are defined inside ERR.C or <module name>.C by a unique code inside 
 the application. 
 These unique values usally grow with a step of 100 in order to give 
 to each module a set of 100 error codes. 
 The choice of error base codes depends on the application and on the
 number of modules in the application.
 
 ERROR MESSAGES : 
 The global translation between error codes to error messages is made by:
 
 err_Type err_Msg(err_Type Erno,	* (IN): Error code *
                  char ** pMsg		* (OUT): Message *
                  );
 This function uses a function given by each module and 
 defined in the error definition file of the module:
 
 err_Type module_Msg(err_Type Erno,	* (IN): Error code *
                     char ** pMsg	* (OUT): Message *
                     );
 Each module publishes its own function by a call to err_ModPublish().
 
 The error message function can be simply defined with macros from 
 the ERR module: see some examples.
 
 The err_Msg2Set and Get functions enables to define the last error number and
 message for a task. The message is not the normal error message that can be 
 obtained with err_Msg() but a specific message that details the last error. 
 
 VERBOSE LEVELS : 
 Each module can define a verbose level. When this level is activated, 
 The functions err_Printx(level,...) display their message. When the level 
 is not activated, those functions are mute. 
 
 MODULE VERSION : 
 Each module can define its source version. 
 
 09/08/2005 - Marc LEGOUX : C6r19b: Add err_MsgX()
 12/07/2005 - Marc LEGOUX : C6r19a: Change err_WinMask to a decimal value
 18/03/2003 - Marc LEGOUX : C6r09 : Add Title field in err_MsgType
 25/10/1999 - Marc LEGOUX : C6r02 : Add _Msg2xx
 06/10/1999 - Marc LEGOUX : C6r02 : Add module version and fend()
 02/03/1999 - Marc LEGOUX : C5r04 : Add uint
 13/01/1999 - Marc LEGOUX : C5r04 : Modif #if config_... for err_Type
 13/06/1998 - Marc LEGOUX : C5r01 : Modif err_MsgType
 04/08/1997 - Marc LEGOUX : C4r02 : Add errapp.h
 08/01/1997 - Marc LEGOUX : V1r00 : Update from own module
 ***********************************************************************/
#ifndef err_h
#define err_h

#ifdef __cplusplus
extern "C" {
#endif

#include "uid/config.h"

/*--------------------------  TYPES -----------------------------------*/
typedef
  unsigned char uchar;
typedef
  unsigned short ushort;
typedef
  unsigned long ulong;
typedef
  unsigned int uint;
#define	null ((void *)0)	/* Null pointer */

/* ERROR TYPE returned by ALL functions */
#if config_WIN
typedef
  unsigned long err_Type;

#define err_WinUserBase	20000000	// Appli error base for NT
#define err_WinBase	10000000	// NT error base for appli
#elif config_IAR
typedef
  unsigned short err_Type;
#else
typedef
  unsigned int err_Type;
#endif

#define ErrType err_Type
#define err_type err_Type

#define err__Ok	0
/***********************************************************************
                        VERBOSE LEVELs
 ***********************************************************************/
typedef
  unsigned err_VerbType;

/*---------------------------------------------------------------------*
                      AFFECTATION of VERBOSE LEVELS
 *----------------------------------------------------------------------*/
				// DISPLAY : 
#define err_VerbMute	0	// No messages
#define err_VerbNormal	1	// Standard and error messages
#define err_VerbOp	2	// Messages on operation calls

/*---------------------------------------------------------------------*
                      Print a Debug message
 verb : verbose level or 1 for print or 0 for no print
        verb can be built with the verbose level of module descriptor
 msg  : printf format
 *---------------------------------------------------------------------*/
extern
void err_Print(char * msg);
extern
void err_Printf(err_VerbType verb, char * fmt, ...);

#define ErrPrintf err_Printf
#define err_print err_Print
#define err_printf err_Printf
#define err_prints err_Printf
#define err_printu err_Printf
#define err_printl err_Printf
#define err_printp err_Printf

/*---------------------------------------------------------------------*
                      Set the current verbose level of a module
 verb : verbose level = 
        0 for no print, x | 1 to enable printing of level x messages
 *---------------------------------------------------------------------*/
extern 
err_Type err_MsgVerbSet(
        char * Name,		/* Module Name or "All" for every modules */
        err_VerbType Verb	/* Verbose level */
        );
#define err_ModVerbSet err_MsgVerbSet
/***********************************************************************
                        ERROR MESSAGEs
 ***********************************************************************/
/*---------------------------------------------------------------------*
                      Get local error message
 Same definition for the message function of a module: ModMsg(...)
 *---------------------------------------------------------------------*/
extern 
err_Type err_Msg(
        err_Type Erno,		/* (IN): Error code */
        char ** pMsg		/* (OUT): Message */
        );
#define err_msg err_Msg
/*---------------------------------------------------------------------*
                      Get error message on a station
 *---------------------------------------------------------------------*/
extern
err_Type err_MsgX(
        err_Type erno, 
        unsigned Mode, 
        ulong Node, 		// Station node or 0 for local station
        unsigned LenMax, 	// Maximum length to return (Including 0)
        char * Msg		// OUT : Message string
        );
/*---------------------------------------------------------------------*
	To define error messages in the error messages function
 *---------------------------------------------------------------------*/
#define err_Msgif(e, m) if (erno == (e)) *pmsg=(m)
#define err_msgif(e, m) if (erno == (e)) *pmsg=(m)

/*---------------------------------------------------------------------*
                        Set Complementary Error Message
                        For current task
 *---------------------------------------------------------------------*/
extern
err_Type err_Msg2Set(
        err_Type Erno, 	/* Error nb */
        char * Msg	/* Message, Max is 255 +1 */
	);
/*---------------------------------------------------------------------*
                        Get Complementary Error Message
                        of current task
 *---------------------------------------------------------------------*/
extern
err_Type err_Msg2Get(
        err_Type * pErno, 	/* (OUT): Last error number */
        char * pMsg		/* (OUT): Last message, Max is 255 +1 */
	);
/************************************************************************
                	MODULE DESCRIPTOR
 ************************************************************************/
typedef
  struct err_MsgType {
    err_Type erno;		// base error of module or 0
    err_Type (* f)();		// error message function
                                // (err_Type erno, char ** pMsg)
                                // Returns 0 if known error, else != 0
    char Name[33];		// Module Name
    err_VerbType Verb;		// Module current verbose level or 0
    
    struct err_MsgType * nx;	// Link: 0 => end of list
    char Version[33];		// Module version
    err_Type (* fend)();	// ModuleEnd  function or 0
    unsigned Mode;		// Current module mode (See Mode flags)
    void * Info;		// Specific module infos or 0
  
    char * Title;		// Module title
  } err_MsgType;		// item of the list of modules

// Mode flags
#define err_MsgModeNotReady	1	/* Module is not initialized yet and 
					   not ready to be called */
#define err_MsgModeTitle	0x10000	// Title field is valid or 0

#define err_msg_type		err_MsgType
#define err_ModType		err_MsgType
#define err_ModNotReady		err_MsgModeNotReady
#define err_ModModeTitle	err_MsgModeTitle
/*---------------------------------------------------------------------*
                      Publication of Module Descriptor
 *---------------------------------------------------------------------*/
extern 
err_Type err_MsgPublic(
        err_MsgType * Desc,	/* (IN): Module descriptor */
        err_Type erno		/* Module error code base to set into Desc 
        			   or 0 if already defined */
        );
#define err_msg_public err_MsgPublic
#define err_ModPublic  err_MsgPublic
#define err_ModPublish err_MsgPublic
/*---------------------------------------------------------------------*
                      Remove a module descriptor
 *---------------------------------------------------------------------*/
extern
err_Type err_MsgRemove(
        char * Name,		/* Module name ("All" for all modules)*/
        err_MsgType ** pDesc	/* null or (OUT): Removed descriptor or 0 */
        );
#define err_ModRemove err_MsgRemove
/*---------------------------------------------------------------------*
                      Iter on module descriptors
 Returns error 4 if end of module list
 *---------------------------------------------------------------------*/
extern
err_Type err_MsgIter(
        err_MsgType ** pDesc	/* (IN): Current descriptor or 0 */
                                /* (OUT): Next decriptor or 0 (First if IN==0) */
        );
#define err_ModIter err_MsgIter

/*---------------------------------------------------------------------*
                      Get module version
 Error 3 if module not found
 *---------------------------------------------------------------------*/
extern 
err_Type err_MsgVersion(
        char * Name,		/* (IN): Module Name */
        char ** pMsg		/* null or (OUT): Version or "" */
        );
#define err_ModVersion err_MsgVersion

/*---------------------------------------------------------------------*
                        Start err module
 *---------------------------------------------------------------------*/
extern
err_Type err_Start(void * Cfg);
/*---------------------------------------------------------------------*
                        End err module
 *---------------------------------------------------------------------*/
extern
err_Type err_End(void * Cfg);
/*---------------------------------------------------------------------*/
#ifdef __cplusplus
}
#endif
#endif
