/***********************************************************************
                                  PARSE
                              Analyse de texte
 ***********************************************************************
 Les fonctions de ce module permettent d'analyser du texte et d'y 
 reconnaitre des elements syntaxiques. 
 Elles ont une forme commune:
 
 	- Elles recoivent en parametre un pointeur de curseur 
          sur un caractere d'une ligne du texte a analyser. 
          
 	- Elles avancent le curseur sur le caractere suivant la forme
          syntaxique reconnue ou le laisse inchange si aucune forme 
          n'est reconnue.
          
 	- Elles retournent 0 (parse__ok) si la forme est reconnue et 
          un code d'erreur !=0 sinon. 
          
        - Le code parse__nomatch indique qu'aucune forme n'est reconnue. 
        - Le code parse__eot signifie que le curseur est a la fin du texte.
          Les fonctions doivent le retourner quand c'est le cas.
        - Le code parse__syntax indique une forme reconnue mais mal formee. 
        - Un autre code ne garanti pas que le curseur soit laisse inchange 
          (forme partiellement reconnue).
 ***********************************************************************
 21/01/2007 - Marc LEGOUX V2r23d: Add parse_float
 28/12/2005 - Marc LEGOUX C6r22 : Add parse_Item
 05/12/2005 - Marc LEGOUX C6r21 : Add parse_Word
 09/02/2004 - Marc LEGOUX C6r14 : Add parse_ByteList
 28/08/2001 - Marc LEGOUX C6r07 : Use char_0type
 ***********************************************************************/
#ifndef parse_h
#define parse_h

#include <err.h>
#include <char.h>
#include <tcur.h>
 
/*--------------------------------DATAS--------------------------------*/
#ifndef parse__err
extern
err_Type parse__err;
#endif
#define parse__ok	0
#define parse__nomatch	parse__err+1	/* Expression not recognized */
#define parse__toolong	parse__err+2
#define parse__nowhite	parse__err+3
#define parse__eol	parse__err+4	/* End of line */
#define parse__eot	parse__err+5	/* End of text */
#define parse__badcomst parse__err+6
#define parse__nokey    parse__err+7
#define parse__noexpr   parse__err+8
#define parse__syntax   parse__err+9	/* Expression recognized but syntax error */
#define parse__badrequest parse__err+10

/*------------------------------FUNCTIONS------------------------------*/
#define parse_ModeLineIn	1	// Stay inside a line
/*---------------------------------------------------------------------*
                Macros de manipulation du curseur (par pointeur)
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
                               Get pt
 *---------------------------------------------------------------------*/
#define gpt(cur) tcur_gpt(cur)
/*---------------------------------------------------------------------*
                               Set pt
 *---------------------------------------------------------------------*/
#define spt(cur, ptn) tcur_spt(cur, ptn)
 
/*---------------------------------------------------------------------*
                              String terminale ?
 *---------------------------------------------------------------------*/
#define lastpt(cur) tcur_lastl(cur)
 
/*---------------------------------------------------------------------*
                              Fin de texte ?
 *---------------------------------------------------------------------*/
#define eotpt(cur) tcur_eot(cur)
 
/*---------------------------------------------------------------------*
                             Analyseurs
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
                           Term
 *---------------------------------------------------------------------*
 Term = mot du langage (analyse lexicale). 
 Ex: toto:= -> toto := 
 Ex: toto:+ -> toto: +
 Verifie un terme ou retourne le terme courant. 
 Le curseur avance si ok
 *---------------------------------------------------------------------*/
extern
err_Type parse_term(
  tcur_ctype * cur,	// Text cursor  
  char_0type * ident,	// (IN)  Si != "", Chaine a tester 
                        // OUT: Chaine de retour du term  
  int size		// Taille max de *ident >= 3	
  );
/*---------------------------------------------------------------------*
                           C identifier
 *---------------------------------------------------------------------*
 Dans une ligne: suite de caracteres 'a'..'z', 'A'..'Z', '_'.
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__toolong : Nom trop long.
      parse__eol     : Fin de ligne.
      parse__eot     : Fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_cident(
        tcur_ctype * cur,       // Text cursor 
        char_0type * ident,      // OUT: Chaine de retour de l'identificateur
        int size                // Taille max de l'identificateur
	);
/*---------------------------------------------------------------------*
                           General Identifier
 *---------------------------------------------------------------------*
 Dans une ligne: suite de caracteres 'a'..'z', 'A'..'Z', '0',...,'9','_',
                 + ensemble supplementaire (cset)
 Le premier caractere n'est pas un chiffre.
 Si erreur, on retourne dans ident les premiers caracteres reconnus.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_ident( 
        tcur_ctype * cur,	// Text cursor  
        char_0type * ident,	// OUT: Chaine de retour de l'identificateur  
        int size,		// Taille max de *ident.	
        char * cset		// Ensemble de caracteres supplementaires 
        );
/*---------------------------------------------------------------------*
                 Char string defined by a char set
 *---------------------------------------------------------------------*
 Les fins de lignes sont representees par char_eol.
 La fin de texte est representee par char_eot.
 ERRORS: 
      parse__toolong : Nom trop long.
      parse__eot     : Fin de texte.
 *---------------------------------------------------------------------*/
#define parse_WordModeIn	1	// Chars only In the set, Else not In

extern
err_Type parse_Word(
        tcur_ctype * cur,	// Text cursor
        unsigned Mode, 		// Mode
        char_0type * CSet,	// Char set
        char_0type * Str,	// null or OUT: result string
        int MaxL		// Max result length
        );
/*---------------------------------------------------------------------*
                 Chaine de caracteres avec terminateurs
 *---------------------------------------------------------------------*
 L'un des terminateurs termine la chaine.
 Les fins de lignes sont representees par char_eol.
 La fin de texte est representee par char_eot.
 Le separateur n'est pas saute.
 Calls parse_Word()
 *---------------------------------------------------------------------*/
extern 
err_Type parse_word(
        tcur_ctype * cur,       // Text cursor 
        char_0type * tst,	// Chaine des terminateurs possibles 
        char_0type * ident,     // Null or OUT: Result string
        int size                // Taille max de la chaine 
	);
/*---------------------------------------------------------------------*
                 Char string defined by separators
 *---------------------------------------------------------------------*
 An item is ended by a single Sep char, except at the end of line. 
 Spaces are parts of an items but can be packed. 
 Property format : <Prop> = <Value>
 At property is 32 char max
 *---------------------------------------------------------------------*/
#define parse_ItemModeSpacePack		1	// Pack spaces 
#define parse_ItemModeNoSepAtEOL	2	// No sep at EOL
#define parse_ItemModeProperty		4	// Parse property format
#define parse_ItemModeSepN		8	// Skip group of seps (else only one at a time)

extern
err_Type parse_Item(
        tcur_ctype * cur,	// Text cursor
        unsigned Mode, 		// Mode
        char_0type * Sep,	// Char set of Item separators
        char_0type * Space,	// Char set of Spaces
        char_0type * Prop,	// null or OUT : result property (LenMax=32)
        char_0type * Str,	// null or OUT : result string
        unsigned MaxL		// Max result length
        );
/*---------------------------------------------------------------------*
                   Chaine de caracteres delimitee
 *---------------------------------------------------------------------*
 Le delimiteur encadre la chaine.
 Le delimiteur peut etre double dans la chaine.
 La chaine doit tenir sur une ligne.
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__toolong : Nom trop long.
      parse__eol     : Fin de ligne.
      parse__eot     : Fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_dstr(
        tcur_ctype * cur,       // Text cursor 
        char_0type delim,        // Delimiteur de la chaine 
        char_0type * ident,      // OUT: Chaine de retour de l'identificateur
        int size                // Taille max de l'identificateur 
	);
/* 
*/
/*---------------------------------------------------------------------*
                   Chaine de caracteres delimitee de type c
 *---------------------------------------------------------------------*
 Le delimiteur encadre la chaine.
 Le delimiteur ne peut pas etre double dans la chaine.
 Le caractere d'echappement introduit des caracteres speciaux
 La chaine doit tenir sur une ligne.
 esc b -> BS	esc n -> LF	esc v -> VT	esc f -> FF
 esc t -> TAB	esc r -> CR
 esc xDD -> ascii(NN)
 esc uDDDD -> unicode(NN)
 esc <char> -> <char>
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__syntax  : Mauvaise syntax interne.
      parse__toolong : Nom trop long.
      parse__eol     : Fin de ligne.
      parse__eot     : Fin de texte.
 *---------------------------------------------------------------------*/
extern
err_Type parse_dstrc(
	tcur_ctype * cur,	// Text cursor
	char_0type delim,	// Delimiteur de la chaine
	char_0type * ident,	// OUT: Zone de retour de la chaine
	int size,		// Taille max de la zone
	char_0type esc		// Echappement
	);
/*---------------------------------------------------------------------*
                           Saut de blancs
 *---------------------------------------------------------------------*
 Blancs equivaux a SPACE, TAB et EOL pour fin de ligne
 ERRORS: 
      parse__nowhite : Aucun blancs.
      parse__eol     : Fin de ligne (si mode LineIn)
      parse__eot : fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_white(
	tcur_ctype * cur,       // Text cursor 
        unsigned mode
	);
/*---------------------------------------------------------------------*
                           Saut de commentaires
 *---------------------------------------------------------------------*
 Les commentaires sont les blancs, les zones comprises entre les
 deux chaines st1 et st2, et les fins de ligne qui de'butent par steol.
 Les zones st1--st2 sont emboitables.
 ERRORS: 
      parse__nowhite : Aucun blancs ni commentaires
      parse__badcomst: st1 ou st2 ou steol vide.
 *---------------------------------------------------------------------*/
#define parse_comment(cur,st1,st2,ste) parse_commentx(cur,0,st1,st2,ste,0)
extern 
err_Type parse_commentx(
        tcur_ctype * cur,	      // Text cursor 
        unsigned    mode,	      // Mode 
        char_0type * st1,              // Debut de zone 
        char_0type * st2,              // Fin de zone 
        char_0type * steol,            // Debut de fin de ligne 
        char_0type * stpfx             // Null ou prefix de non commentaire 
	);
/*---------------------------------------------------------------------*
                           Reconnait une chaine
 *---------------------------------------------------------------------*
 La chaine ne peut pas contenir des delimiteurs de lignes.
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__eol : Fin de ligne.
      parse__eot : Fin de texte.
 *---------------------------------------------------------------------*/
extern err_Type parse_strm(
        tcur_ctype * cur,       // Text cursor 
        char_0type * str         // Chaine a reconnaitre 
	);
/*---------------------------------------------------------------------*
                           Test une chaine (sans avancer)
 *---------------------------------------------------------------------*
 Une chaine vide est toujours reconnue sauf en fin de ligne ou de texte.
 La chaine ne peut pas contenir des delimiteurs de lignes.
 Le caractere suivant la string ne doit pas appartenir a set (si non null). 
 Dans set, dc1 = 0x11 signifie caractere non alphabetique ni _ ni numerique
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__eol : Fin de ligne.
      parse__eot : Fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_str_check(
        tcur_ctype * cur,       // Text cursor 
        char_0type * str,        // Chaine a reconnaitre 
        char_0type * set         // null ou Caracteres suivant la string 
	); 
/*---------------------------------------------------------------------*
                           Reconnait un caractere
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__eot : Fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_charm(
        tcur_ctype * cur,       // Text cursor 
        char_0type c             // Caractere 
	);
/*---------------------------------------------------------------------*
                           Test un caractere (sans avancer)
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__eot : Fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_char_check(
        tcur_ctype * cur,       // Text cursor 
        char_0type c             // Caractere 
	);
/*---------------------------------------------------------------------*
                           Reconnait un caractere dans un ensemble
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__eot : Fin de texte.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_char_set(
        tcur_ctype * cur,       // Text cursor 
        char_0type * str         // Caractere 
	);
/*---------------------------------------------------------------------*
                           Entier
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
 *---------------------------------------------------------------------*/
extern err_Type parse_int(
        tcur_ctype * cur,       // Text cursor 
        int * nb                // OUT: Parsed integer
	);
/*---------------------------------------------------------------------*
                          Nombre Entier long
 Sign		: [+][-]
 Decimal	: nnn
 Auto Hexa	: 0xnnn or /nnn
 Auto Binary	: 0bnnn
 *---------------------------------------------------------------------*/
#define parse_NumModeNoSign	1	// No sign 
#define parse_NumModeOld	2	// Hexa /nnn 

extern
err_Type parse_long(
  tcur_ctype * cur,	// Text cursor 
  long * pnb,		// OUT: Parsed integer
  unsigned mode,	// Mode 
  unsigned base,	// Base, 0 for auto 
  unsigned nmax		// Nombre max de chiffre, 0 for auto 
  );
/*---------------------------------------------------------------------*
                          Nombre Flottant
 Sign		: [+][-]<mantisse>{.|,}<decima>[E[+][-]<expo>]
 Decimal	: nnn
 Auto Hexa	: 0xnnn or /nnn
 Auto Binary	: 0bnnn
 *---------------------------------------------------------------------*/
#define parse_FloatModeNoSign	1	// No sign 

#define parse_FloatModeComma	0x10	// With comma (else dot only)

#define parse_FloatType	long double
extern
err_Type parse_float(
  tcur_ctype * cur,	// Text cursor 
  parse_FloatType * pnb,// OUT: Parsed value
  unsigned mode,	// Mode 
  unsigned nmax		// Nombre max de chiffre, 0 for auto 
  );
/*---------------------------------------------------------------------*
                          Recherche chaine
 *---------------------------------------------------------------------*
 Modes de recherches (composables).
   SE_AFTER     Arret curseur apres (avant)
   SE_BOL       Cadrage en debut de ligne (pas cadre)
   SE_EOL       Cadrage en fin   de ligne (pas cadre)
   SE_NEXACT    Assimilation majus. et minus. (discrimination)
   SE_LINE      Search uniquement dans la ligne.     
   SE_CUR       Search uniquement au curseur.
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_search(
        tcur_ctype * cur,	// Text cursor 
        char_0type * str,	// Chaine recherchee 
        char flags		// Mode de recherche 
        );
/*---------------------------------------------------------------------*
                          Recherche N chaines
 La premiere trouvee stoppe la recherche
 *---------------------------------------------------------------------*
 Modes de recherches (composables).
   SE_AFTER     Arret curseur apres (avant)
   SE_BOL       Cadrage en debut de ligne (pas cadre)
   SE_EOL       NON IMPLEMENTE
   SE_NEXACT    Assimilation majus. et minus. (discrimination)
   SE_LINE      Search uniquement dans la ligne.     
   SE_CUR       Search uniquement au curseur.
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
 *---------------------------------------------------------------------*/
extern 
err_Type parse_searchn(
        tcur_ctype * cur,	// Text cursor 
        char_0type ** strlst,	// Tableau des Chaines a rechercher (..NULL)
        char flags,		// Mode de recherche 
        unsigned * pidx		// OUT: Index de chaine trouvee 0.. 
        );
/*---------------------------------------------------------------------*
                          Parse liste de bytes
 Syntax : 
 ByteList :: <term> | <term> , <ByteList>
 <term> :: <int> | <String>
 ERRORS: 
      parse__nomatch : Mauvaise forme syntaxique.
      parse__syntax : Erreur apres reconnaissance partielle
 *---------------------------------------------------------------------*/
#define parse_ByteModeAppend0	0x100	// Append 0 after the last string
extern
err_Type parse_ByteList(
	tcur_ctype * cur,
	unsigned Mode, 
	uchar * Buf,		// OUT: Byte list
	unsigned *pLenMax	// IN, OUT: list length, Max length
	);

/************************************************************************
			MODULE
 ************************************************************************/
/*-----------------------------------------------------------------------*
                        Start Module
 *-----------------------------------------------------------------------*/
extern
err_Type parse_Start(
        void * Cfg
        );
/*-----------------------------------------------------------------------*
                        End Module
 *-----------------------------------------------------------------------*/
extern
err_Type parse_End(
        void * Cfg
        );
/*-----------------------------------------------------------------------*/
#endif
