/*************************************************************************
                SYStem interface for basic ObJects
 *************************************************************************
 Copyright (c) SMARTWARE, 1999-2007 All Rights Reserved.
 *************************************************************************
 21/01/2009 - Marc LEGOUX : V2r23p: Add Call Mode AbortR
 23/05/2007 - Marc LEGOUX : V2r23h: Add sysobj_ObjMove
 28/12/2006 - Marc LEGOUX : V2r23c: Add _IdNASet()
 27/11/2005 - Marc LEGOUX : C6r21: Add _NodeGet() and _NodeSet()
 03/08/2005 - Marc LEGOUX : C6r19: Extract SYSOBJ8.H
 01/01/2005 - Marc LEGOUX : C6r18: Improve macros on object idenfifiers
 10/08/2004 - Marc LEGOUX : C6r16: Add sysobj_CallX()
 03/04/2001 - Marc LEGOUX : C6r07: Add Object Id on 32 bits (_IdObjType)
 31/05/2000 - Marc LEGOUX : C6r05: Add Node macros
 20/12/1999 - Marc LEGOUX : C6r03: Add _ObjIter
 12/08/1999 - Marc LEGOUX : C6r02: Add _L32Type, _NO32Type
 01/01/1999 - Marc LEGOUX : C6r01: sysobj 32
 31/01/1998 - Marc LEGOUX : C5r03: Add sysobj_ObjEnd
 08/12/1997 - Marc LEGOUX : C5r01: bug dans sysobj_Type
 26/10/1997 - Marc LEGOUX : C4R03
 08/01/1997 - Marc LEGOUX : C4r01 : Update from own module
 *************************************************************************/
#ifndef sysobj_h
#define sysobj_h

#include "err.h"
#include "sys.h"

#if config_MLOS8
#include <sysobj8.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif
/*-----------------------------------------------------------------------*
                        Common error codes
 *-----------------------------------------------------------------------*/
#define sys__err 0
#define sys__ok			0		/* Every thing's OK */
#define sys__noobj		1+sys__err	/* Object not found */
#define sys__noop		2+sys__err	/* Operation not found */
#define sys__arg		3+sys__err	/* Wrong parameter */
#define sys__timeout		4+sys__err	/* Timeout */
#define sys__resp		5+sys__err	/* Too long answer length */
#define sys__escape		6+sys__err	/* User escape */
#define sys__mem		7+sys__err	/* Wrong memory model */

#define sys__nodrv sys__noobj
/*-----------------------------------------------------------------------*
                        APPLICATION INTERFACE
 *-----------------------------------------------------------------------*/
#define sysct_wait	sysobj_Wait
#define sysct_exec	sysobj_Call
/*-----------------------------------------------------------------------*
                        Common Object ids
 *-----------------------------------------------------------------------*/
#if !config_MLOS8

typedef
 ulong sysobj_TType;			/* Type of Timeout (ms) */
typedef
 ulong sysobj_NOType;			/* Type of Node+Object+Operation Id */
typedef
 ulong sysobj_LType;			/* Type of Lengths (argl and respl) */
typedef
 ulong sysobj_NodeType;			/* Type of Node Id (old) */
typedef
 ulong sysobj_ObjType;			/* Type of Object Id (old) */
#endif

/* common : IAR, NT and MSC mode */
typedef
 ulong sysobj_NO32Type;			/* Type of Node+Object+Operation Id */
typedef
 ulong sysobj_L32Type;			/* Type of Lengths (argl and respl) */

/* SYSTEM OBJECT CODES */
#define	sysobj_KrnId		0x00	/* Kernel Object */
/*----------------------------------------------------------------------*
			OBJECT ID
 *----------------------------------------------------------------------*/
#define	sysobj_OpMask		0x07

// Returns Node from Id
#define sysobj_IdNode(id)	((sysobj_NodeType)(0x007FFF00 & (id)) >> 8)

// Returns Obj from Id
#define sysobj_IdObj(id)	((sysobj_NodeType)(0x7F0000F8 & (id)))

// Returns ObjOp from Id
#define sysobj_IdObjOp(id)	((sysobj_NodeType)(0x7F0000FF & (id)))

// Returns Op from Id
#define sysobj_IdOp(id)		((sysobj_ObjType)(sysobj_OpMask & (id)))

// Check Identifier with No Answer
#define sysobj_IdIsNA(id)	(((id) & 0x00800000) || (0x007FFF00 & (id))==0xFF00)
// Set NA flag into Identifier
#define sysobj_IdSetNA(id)	((id) |  0x00800000)
#define sysobj_IdClrNA(id)	((id) & ~0x00800000)

// Set Node into Id : Returns 0 if error
extern
sysobj_NOType sysobj_IdNodeSet(sysobj_NOType Id, sysobj_NodeType n);

#define sysobj_NodeBCast	0	// Broadcast
#define sysobj_NodeBCastNA	0xFF	// Broadcast (No Answer)
#define sysobj_NodeMax		0x7FFF	// 15 bits at max

/* Local node is defined (else has a default value) */
#define sysobj_NodeIsDef(n)	((n)!=0 && (n)!=1 && (n)!= 0x4F)

/*----------------------------------------------------------------------*
                        Get Local node value
 *----------------------------------------------------------------------*/
extern
err_Type sysobj_NodeGet(
        sysobj_NodeType * pNode,		// OUT
        unsigned Mode				// 0
        );
/*----------------------------------------------------------------------*
                        Set Local node value
 *----------------------------------------------------------------------*/
#define sysobj_NodeModeNotDef	1	// Set only if not already defined
extern
err_Type sysobj_NodeSet(
        sysobj_NodeType Node,		// New node
        unsigned Mode			// 0
        );
/*-----------------------------------------------------------------------*
                        Local Object execution
                        (same link unit)
 objop should be the name of the function
 *-----------------------------------------------------------------------*/
#define sysobj_Call0(pb,a,pr,objop,t) objop(pb,a,pr)

#if !config_MLOS8
/*-----------------------------------------------------------------------*
                        Object Operation Call
 WARNING: Timeout is ignored for local calls
 *-----------------------------------------------------------------------*/
extern
err_Type sysobj_Call(
        uchar * pBuf,		// Arguments/response buffer
        sysobj_LType Argl,	// Argument length
        sysobj_LType * pRespl,	// (OUT) response length
        sysobj_NOType NObjOp,	// (Node,Object,Operation) code
        sysobj_TType Timeout	// Timeout (ms)
        );
#define sysobj_Calld sysobj_Call
#define sysobj_Callx sysobj_Call
/*-----------------------------------------------------------------------*
                        Object Operation Extended Call
 *-----------------------------------------------------------------------*/
// Type of call back functions
typedef
  err_Type (* sysobj_CallBackFuncRef)(
  	void * ctx, 		// Object context
  	void * Params,		// Parameters
        uchar * Buf, 		// Buffer with result
        sysobj_LType * pRespl, 	// Result length
  	sysobj_NOType NObjOp, 	// Object + Op called Id 
        err_Type * pErno, 	// Error code to return
        void * RFU		// RFU
  	);

typedef
  struct {
    unsigned			Mode;
    sysobj_CallBackFuncRef	CallBackFunc;	// Call back function
    void *			CallBackArgs;	// Call back arguments
    ulong			RFU;
  } sysobj_CallXType;

// Modes : ORed flags
#define sysobj_CallModeAsy	1	// Call is asynchronous
#define sysobj_CallModeAbort	2	// Abort execution if timeout
#define sysobj_CallModeLostR	4	// Lost result after execution 
#define sysobj_CallModeAbortR	8	// Abort (only remote) execution if timeout 

// Operation modes
#define sysobj_OpModeNotDef	0x80	// Op is not defined

extern
err_Type sysobj_CallX(
        uchar * pBuf,		// Arguments/response buffer
        sysobj_LType Argl,	// Argument length
        sysobj_LType * pRespl,	// (OUT) response length
        sysobj_NOType NObjOp,	// (Node,Object,Operation) code
        sysobj_TType Timeout,	// Timeout (ms)
        sysobj_CallXType * Cfg	// 0 or call configuration
        );
/*-----------------------------------------------------------------------*
                        Make task waiting MSC
 if timeout==0, the task skips one cycle and 
 the current timeout of the task (if any) is down counted. 
 Returns an error if the current timeout makes a 1 -> 0 transition. 
 *-----------------------------------------------------------------------*/
extern
err_Type sysobj_Wait(sysobj_TType timeout);
/*-----------------------------------------------------------------------*
                        Critical section
 Use a semaphore
 *-----------------------------------------------------------------------*/
/* Begin critical section */
extern
err_Type sysobj_Cs(uchar * psem		/* semaphor */); 
/* End critical section */
extern
err_Type sysobj_Cse(uchar * psem	/* semaphor */); 
#endif
/*-----------------------------------------------------------------------*
                        Object create
 This function is RESERVED to the system
 *-----------------------------------------------------------------------*/
#if !config_MLOS8
#define sysobj_StackU	50			/* Stack length unit */
#endif

typedef
  err_Type (* sysobj_FuncRef)();		/* Pointer on function */

#define sysobj_FuncNull ((sysobj_FuncRef)0)	/* NULL function */
/*-----------------------------------------------------------------------*/
extern
err_Type sysobj_ObjNew(
                sysobj_NOType id, 	/* Object id */
                uchar dtype,		/* =0 normal task, =1 filter */
                uchar stack_l,		/* stack length (in _StackU units) */
                sysobj_FuncRef ops,
                sysobj_FuncRef ope,
                sysobj_FuncRef op2,
                sysobj_FuncRef op3,
                sysobj_FuncRef op4,
                sysobj_FuncRef op5,
                sysobj_FuncRef op6,
                sysobj_FuncRef op7,
                void * ctx		/* Context of object */
                ); 
/*-----------------------------------------------------------------------*
                        Object descriptor
 *-----------------------------------------------------------------------*/
#define sysobj_ModeFilter	1
#define sysobj_Null ((void *)0)

#if !config_MLOS8
typedef 
  struct {
    void * ctx;			/* Object specific context (first field )*/
    char name[32];		/* Object name */
    
  } sysobj_Type, * sysobj_Ref;
/*-----------------------------------------------------------------------*
                        Object End
                        First instance is removed from object list
                        End operation is executed
 *-----------------------------------------------------------------------*/
extern
err_Type sysobj_ObjEnd(
                sysobj_NOType id, 	/* [Node+] Object id */
                sysobj_TType Timeout,	/* Timeout (ms) for End operation */
                void ** pctx		/* (OUT) : Context or NULL */
                ); 
/*-----------------------------------------------------------------------*
                        Object End
                        Selection by ctx
                        End operation is executed
 *-----------------------------------------------------------------------*/
extern
err_Type sysobj_ObjEndCtx(
                void * ctx, 		/* Object own context */
                sysobj_TType Timeout	/* Timeout (ms) for End operation */
                ); 
/*-----------------------------------------------------------------------*
                        Move First object to last position
 *-----------------------------------------------------------------------*/
extern
err_Type sysobj_ObjMoveFTL(); 
/*-----------------------------------------------------------------------*
                        Move object to another position
 *-----------------------------------------------------------------------*/
#define sysobj_ObjMoveModeFirst	1	// Move object to first position
#define sysobj_ObjMoveModeLast	2	// Move object to last position

extern
err_Type sysobj_ObjMove(
        sysobj_NOType id, 	// Object id
        ulong Mode
	); 
/*-----------------------------------------------------------------------*
                        Change Object priority
 *-----------------------------------------------------------------------*/
#define sysobj_ObjPriorHigh	0xFF
#define sysobj_ObjPriorLow	0xFE
#define sysobj_ObjPriorDef	0x00

extern
err_Type sysobj_ObjPriorSet(
        sysobj_Ref SysObj,
        int Prior		/* relative priority */
        );
/*----------------------------------------------------------------------*
                        Task Context
 *----------------------------------------------------------------------*/
typedef
  struct {
    sysobj_NOType	Cur;	// Current object
    sysobj_NOType	Caller;	// Caller (undef node==0)(undef objop==0)
    unsigned		CallMode;
    unsigned		Rfu2;
  
  } sysobj_TaskCtxType;
/*----------------------------------------------------------------------*
                        New/Take Task context
 MANDATORY: _TaskCtxDel must be called when context is no more used. 
 _TaskCtxNew ... _TaskCtxDel sequences can be nested. 
 The context must not be handled by another task.
 *----------------------------------------------------------------------*/
extern
err_Type sysobj_TaskCtxNew(
        sysobj_TaskCtxType ** pCtx,	// null or OUT
        unsigned Mode			// 0 (reserved)
        );
/*----------------------------------------------------------------------*
                        Delete/Release Task context
 *----------------------------------------------------------------------*/
extern
err_Type sysobj_TaskCtxDel();
/*----------------------------------------------------------------------*
                        Change Data Buffer
 *----------------------------------------------------------------------*
 Called inside an object operation. 
 If allowed, a new buffer is defined to support the response data, 
 otherwise the error sys__arg is returned. 
 In case of success, the input data buffer of the operation can be kept 
 after the operation has returned and must be released when no more used with 
 syst_MemFree(). 
 *----------------------------------------------------------------------*/
#define sysobj_BufModeRespAlloc	1	// New response buffer is allocated and must be freed

extern
err_Type sysobj_BufDef(
        uchar * pBuf,		// New response buffer
        unsigned Mode		// Mode or 0
        );
/*-----------------------------------------------------------------------*
                        Iter on defined objects
 NOTE: Check the object identity with the ->ctx field.
 *-----------------------------------------------------------------------*/
typedef
  struct {
    unsigned Rank;	/* (IN/OUT): Object rank (0 to get the first) */
    sysobj_NOType NObj;	/* (OUT): Object id */
    ulong OpMask;	/* (OUT): Defined operation mask */
    char * Name;	/* (OUT): Object Name */
    uchar Mode;		/* (OUT): Mode */
    sysobj_Type *ObjRef;/* (OUT): Object reference (wizard only) */
    void * Private1;
  } sysobj_ObjIterType;

extern
err_Type sysobj_ObjIter(
        sysobj_ObjIterType * Iter	/* (IN/OUT): Iterator on objects */
        );
/*-----------------------------------------------------------------------*/
#endif
/*-----------------------------------------------------------------------*
                        Start all objects
 This function should be defined by the user (should call sysobj_ObjNew())
 *-----------------------------------------------------------------------*/
extern
err_Type syscfg_tasks_create();
/*-----------------------------------------------------------------------*/
#ifdef __cplusplus
}
#endif
#endif
